/*
 * JCusparse - Java bindings for CUSPARSE, the NVIDIA CUDA sparse
 * matrix library, to be used with JCuda
 *
 * Copyright (c) 2010-2015 Marco Hutter - http://www.jcuda.org
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

#include "JCusparse.hpp"
#include "JCusparse_common.hpp"
#include "JNIUtils.hpp"
#include <iostream>
#include <string>
#include <map>

jfieldID cuComplex_x; // float
jfieldID cuComplex_y; // float

jfieldID cuDoubleComplex_x; // double
jfieldID cuDoubleComplex_y; // double


/**
 * Called when the library is loaded. Will initialize all
 * required field and method IDs
 */
JNIEXPORT jint JNICALL JNI_OnLoad(JavaVM *jvm, void *reserved)
{
    JNIEnv *env = NULL;
    if (jvm->GetEnv((void **)&env, JNI_VERSION_1_4))
    {
        return JNI_ERR;
    }

    Logger::log(LOG_TRACE, "Initializing JCusparse\n");

    jclass cls = NULL;

    // Initialize the JNIUtils and PointerUtils
    if (initJNIUtils(env) == JNI_ERR) return JNI_ERR;
    if (initPointerUtils(env) == JNI_ERR) return JNI_ERR;


    // Obtain the fieldIDs for cuComplex#x and cuComplex#y
    if (!init(env, cls, "jcuda/cuComplex")) return JNI_ERR;
    if (!init(env, cls, cuComplex_x, "x", "F")) return JNI_ERR;
    if (!init(env, cls, cuComplex_y, "y", "F")) return JNI_ERR;


    // Obtain the fieldIDs for cuDoubleComplex#x and cuDoubleComplex#y
    if (!init(env, cls, "jcuda/cuDoubleComplex")) return JNI_ERR;
    if (!init(env, cls, cuDoubleComplex_x, "x", "D")) return JNI_ERR;
    if (!init(env, cls, cuDoubleComplex_y, "y", "D")) return JNI_ERR;

    return JNI_VERSION_1_4;
}


/*
 * Set the log level
 *
 * Class:     jcuda_jcusparse_JCusparse
 * Method:    setLogLevelNative
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_jcuda_jcusparse_JCusparse_setLogLevelNative
  (JNIEnv *env, jclass cla, jint logLevel)
{
    Logger::setLogLevel((LogLevel)logLevel);
}





/**
 * Converts the given jobject
 * into a cuComplex
 * and returns whether this conversion succeeded.
 */
bool initNative(JNIEnv *env, jobject &input, cuComplex &output)
{
    output.x = env->GetFloatField(input, cuComplex_x);
    output.y = env->GetFloatField(input, cuComplex_y);
    if (env->ExceptionCheck())
    {
        return false;
    }
    return true;
}

/**
 * Converts the given jobject
 * into a cuDoubleComplex
 * and returns whether this conversion succeeded.
 */
bool initNative(JNIEnv *env, jobject &input, cuDoubleComplex &output)
{
    output.x = env->GetDoubleField(input, cuDoubleComplex_x);
    output.y = env->GetDoubleField(input, cuDoubleComplex_y);
    if (env->ExceptionCheck())
    {
        return false;
    }
    return true;
}


//##############################################################################
//# INITIALIZATION AND MANAGEMENT ROUTINES
//##############################################################################
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateNative(JNIEnv *env, jclass cls, jobject handle)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCreate");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreate(handle=%p)\n",
        handle);

    // Native variable declarations
    cusparseHandle_t handle_native;

    // Obtain native variable values
    // handle is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreate(&handle_native);

    // Write back native variable values
    setNativePointerValue(env, handle, (jlong)handle_native);

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyNative(JNIEnv *env, jclass cls, jobject handle)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDestroy");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroy(handle=%p)\n",
        handle);

    // Native variable declarations
    cusparseHandle_t handle_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroy(handle_native);

    // Write back native variable values
    // handle is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseGetVersionNative(JNIEnv *env, jclass cls, jobject handle, jintArray version)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseGetVersion");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (version == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'version' is null for cusparseGetVersion");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseGetVersion(handle=%p, version=%p)\n",
        handle, version);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int version_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    // version is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseGetVersion(handle_native, &version_native);

    // Write back native variable values
    // handle is read-only
    if (!set(env, version, 0, (jint)version_native)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseGetPropertyNative(JNIEnv *env, jclass cls, jint type, jintArray value)
{
    // Null-checks for non-primitive arguments
    // type is primitive
    if (value == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'value' is null for cusparseGetProperty");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseGetProperty(type=%d, value=%p)\n",
        type, value);

    // Native variable declarations
    libraryPropertyType type_native;
    int value_native;

    // Obtain native variable values
    type_native = (libraryPropertyType)type;
    // value is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseGetProperty(type_native, &value_native);

    // Write back native variable values
    // type is primitive
    if (!set(env, value, 0, (jint)value_native)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jstring JNICALL Java_jcuda_jcusparse_JCusparse_cusparseGetErrorNameNative(JNIEnv *env, jclass cls, jint status)
{
    // Null-checks for non-primitive arguments
    // status is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseGetErrorName(status=%d)\n",
        status);

    // Native variable declarations
    cusparseStatus_t status_native = CUSPARSE_STATUS_SUCCESS;

    // Obtain native variable values
    status_native = (cusparseStatus_t)status;

    // Native function call
    char const * jniResult_native = cusparseGetErrorName(status_native);

    // Write back native variable values
    // status is primitive

    // Return the result
    jstring jniResult = (jstring)jniResult_native;
    return jniResult;
}

JNIEXPORT jstring JNICALL Java_jcuda_jcusparse_JCusparse_cusparseGetErrorStringNative(JNIEnv *env, jclass cls, jint status)
{
    // Null-checks for non-primitive arguments
    // status is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseGetErrorString(status=%d)\n",
        status);

    // Native variable declarations
    cusparseStatus_t status_native = CUSPARSE_STATUS_SUCCESS;

    // Obtain native variable values
    status_native = (cusparseStatus_t)status;

    // Native function call
    char const * jniResult_native = cusparseGetErrorString(status_native);

    // Write back native variable values
    // status is primitive

    // Return the result
    jstring jniResult = (jstring)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSetStreamNative(JNIEnv *env, jclass cls, jobject handle, jobject streamId)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSetStream");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // streamId may be NULL

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSetStream(handle=%p, streamId=%p)\n",
        handle, streamId);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cudaStream_t streamId_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    streamId_native = (cudaStream_t)getNativePointerValue(env, streamId);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSetStream(handle_native, streamId_native);

    // Write back native variable values
    // handle is read-only
    // streamId is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseGetStreamNative(JNIEnv *env, jclass cls, jobject handle, jobject streamId)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseGetStream");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (streamId == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'streamId' is null for cusparseGetStream");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseGetStream(handle=%p, streamId=%p)\n",
        handle, streamId);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cudaStream_t streamId_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    // streamId is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseGetStream(handle_native, &streamId_native);

    // Write back native variable values
    // handle is read-only
    setNativePointerValue(env, streamId, (jlong)streamId_native);

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseGetPointerModeNative(JNIEnv *env, jclass cls, jobject handle, jintArray mode)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseGetPointerMode");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (mode == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'mode' is null for cusparseGetPointerMode");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseGetPointerMode(handle=%p, mode=%p)\n",
        handle, mode);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparsePointerMode_t mode_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    // mode is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseGetPointerMode(handle_native, &mode_native);

    // Write back native variable values
    // handle is read-only
    if (!set(env, mode, 0, (jint)mode_native)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSetPointerModeNative(JNIEnv *env, jclass cls, jobject handle, jint mode)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSetPointerMode");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // mode is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSetPointerMode(handle=%p, mode=%d)\n",
        handle, mode);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparsePointerMode_t mode_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    mode_native = (cusparsePointerMode_t)mode;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSetPointerMode(handle_native, mode_native);

    // Write back native variable values
    // handle is read-only
    // mode is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

//##############################################################################
//# HELPER ROUTINES
//##############################################################################
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateMatDescrNative(JNIEnv *env, jclass cls, jobject descrA)
{
    // Null-checks for non-primitive arguments
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCreateMatDescr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateMatDescr(descrA=%p)\n",
        descrA);

    // Native variable declarations
    cusparseMatDescr_t descrA_native;

    // Obtain native variable values
    // descrA is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateMatDescr(&descrA_native);

    // Write back native variable values
    setNativePointerValue(env, descrA, (jlong)descrA_native);

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyMatDescrNative(JNIEnv *env, jclass cls, jobject descrA)
{
    // Null-checks for non-primitive arguments
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDestroyMatDescr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyMatDescr(descrA=%p)\n",
        descrA);

    // Native variable declarations
    cusparseMatDescr_t descrA_native;

    // Obtain native variable values
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyMatDescr(descrA_native);

    // Write back native variable values
    // descrA is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

//cusparseStatus_t CUSPARSEAPI
//cusparseCopyMatDescr(cusparseMatDescr_t       dest,
//                     const cusparseMatDescr_t src);
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSetMatTypeNative(JNIEnv *env, jclass cls, jobject descrA, jint type)
{
    // Null-checks for non-primitive arguments
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSetMatType");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // type is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSetMatType(descrA=%p, type=%d)\n",
        descrA, type);

    // Native variable declarations
    cusparseMatDescr_t descrA_native;
    cusparseMatrixType_t type_native;

    // Obtain native variable values
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    type_native = (cusparseMatrixType_t)type;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSetMatType(descrA_native, type_native);

    // Write back native variable values
    // descrA is read-only
    // type is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseGetMatTypeNative(JNIEnv *env, jclass cls, jobject descrA)
{
    // Null-checks for non-primitive arguments
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseGetMatType");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseGetMatType(descrA=%p)\n",
        descrA);

    // Native variable declarations
    cusparseMatDescr_t descrA_native;

    // Obtain native variable values
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);

    // Native function call
    cusparseMatrixType_t jniResult_native = cusparseGetMatType(descrA_native);

    // Write back native variable values
    // descrA is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSetMatFillModeNative(JNIEnv *env, jclass cls, jobject descrA, jint fillMode)
{
    // Null-checks for non-primitive arguments
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSetMatFillMode");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // fillMode is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSetMatFillMode(descrA=%p, fillMode=%d)\n",
        descrA, fillMode);

    // Native variable declarations
    cusparseMatDescr_t descrA_native;
    cusparseFillMode_t fillMode_native;

    // Obtain native variable values
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    fillMode_native = (cusparseFillMode_t)fillMode;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSetMatFillMode(descrA_native, fillMode_native);

    // Write back native variable values
    // descrA is read-only
    // fillMode is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseGetMatFillModeNative(JNIEnv *env, jclass cls, jobject descrA)
{
    // Null-checks for non-primitive arguments
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseGetMatFillMode");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseGetMatFillMode(descrA=%p)\n",
        descrA);

    // Native variable declarations
    cusparseMatDescr_t descrA_native;

    // Obtain native variable values
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);

    // Native function call
    cusparseFillMode_t jniResult_native = cusparseGetMatFillMode(descrA_native);

    // Write back native variable values
    // descrA is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSetMatDiagTypeNative(JNIEnv *env, jclass cls, jobject descrA, jint diagType)
{
    // Null-checks for non-primitive arguments
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSetMatDiagType");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // diagType is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSetMatDiagType(descrA=%p, diagType=%d)\n",
        descrA, diagType);

    // Native variable declarations
    cusparseMatDescr_t descrA_native;
    cusparseDiagType_t diagType_native;

    // Obtain native variable values
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    diagType_native = (cusparseDiagType_t)diagType;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSetMatDiagType(descrA_native, diagType_native);

    // Write back native variable values
    // descrA is read-only
    // diagType is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseGetMatDiagTypeNative(JNIEnv *env, jclass cls, jobject descrA)
{
    // Null-checks for non-primitive arguments
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseGetMatDiagType");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseGetMatDiagType(descrA=%p)\n",
        descrA);

    // Native variable declarations
    cusparseMatDescr_t descrA_native;

    // Obtain native variable values
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);

    // Native function call
    cusparseDiagType_t jniResult_native = cusparseGetMatDiagType(descrA_native);

    // Write back native variable values
    // descrA is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSetMatIndexBaseNative(JNIEnv *env, jclass cls, jobject descrA, jint base)
{
    // Null-checks for non-primitive arguments
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSetMatIndexBase");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // base is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSetMatIndexBase(descrA=%p, base=%d)\n",
        descrA, base);

    // Native variable declarations
    cusparseMatDescr_t descrA_native;
    cusparseIndexBase_t base_native;

    // Obtain native variable values
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    base_native = (cusparseIndexBase_t)base;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSetMatIndexBase(descrA_native, base_native);

    // Write back native variable values
    // descrA is read-only
    // base is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseGetMatIndexBaseNative(JNIEnv *env, jclass cls, jobject descrA)
{
    // Null-checks for non-primitive arguments
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseGetMatIndexBase");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseGetMatIndexBase(descrA=%p)\n",
        descrA);

    // Native variable declarations
    cusparseMatDescr_t descrA_native;

    // Obtain native variable values
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);

    // Native function call
    cusparseIndexBase_t jniResult_native = cusparseGetMatIndexBase(descrA_native);

    // Write back native variable values
    // descrA is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateCsric02InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCreateCsric02Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateCsric02Info(info=%p)\n",
        info);

    // Native variable declarations
    csric02Info_t info_native;

    // Obtain native variable values
    // info is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateCsric02Info(&info_native);

    // Write back native variable values
    setNativePointerValue(env, info, (jlong)info_native);

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyCsric02InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDestroyCsric02Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyCsric02Info(info=%p)\n",
        info);

    // Native variable declarations
    csric02Info_t info_native;

    // Obtain native variable values
    info_native = (csric02Info_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyCsric02Info(info_native);

    // Write back native variable values
    // info is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateBsric02InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCreateBsric02Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateBsric02Info(info=%p)\n",
        info);

    // Native variable declarations
    bsric02Info_t info_native;

    // Obtain native variable values
    // info is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateBsric02Info(&info_native);

    // Write back native variable values
    setNativePointerValue(env, info, (jlong)info_native);

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyBsric02InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDestroyBsric02Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyBsric02Info(info=%p)\n",
        info);

    // Native variable declarations
    bsric02Info_t info_native;

    // Obtain native variable values
    info_native = (bsric02Info_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyBsric02Info(info_native);

    // Write back native variable values
    // info is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateCsrilu02InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCreateCsrilu02Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateCsrilu02Info(info=%p)\n",
        info);

    // Native variable declarations
    csrilu02Info_t info_native;

    // Obtain native variable values
    // info is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateCsrilu02Info(&info_native);

    // Write back native variable values
    setNativePointerValue(env, info, (jlong)info_native);

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyCsrilu02InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDestroyCsrilu02Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyCsrilu02Info(info=%p)\n",
        info);

    // Native variable declarations
    csrilu02Info_t info_native;

    // Obtain native variable values
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyCsrilu02Info(info_native);

    // Write back native variable values
    // info is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateBsrilu02InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCreateBsrilu02Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateBsrilu02Info(info=%p)\n",
        info);

    // Native variable declarations
    bsrilu02Info_t info_native;

    // Obtain native variable values
    // info is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateBsrilu02Info(&info_native);

    // Write back native variable values
    setNativePointerValue(env, info, (jlong)info_native);

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyBsrilu02InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDestroyBsrilu02Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyBsrilu02Info(info=%p)\n",
        info);

    // Native variable declarations
    bsrilu02Info_t info_native;

    // Obtain native variable values
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyBsrilu02Info(info_native);

    // Write back native variable values
    // info is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateBsrsv2InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCreateBsrsv2Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateBsrsv2Info(info=%p)\n",
        info);

    // Native variable declarations
    bsrsv2Info_t info_native;

    // Obtain native variable values
    // info is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateBsrsv2Info(&info_native);

    // Write back native variable values
    setNativePointerValue(env, info, (jlong)info_native);

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyBsrsv2InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDestroyBsrsv2Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyBsrsv2Info(info=%p)\n",
        info);

    // Native variable declarations
    bsrsv2Info_t info_native;

    // Obtain native variable values
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyBsrsv2Info(info_native);

    // Write back native variable values
    // info is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateBsrsm2InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCreateBsrsm2Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateBsrsm2Info(info=%p)\n",
        info);

    // Native variable declarations
    bsrsm2Info_t info_native;

    // Obtain native variable values
    // info is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateBsrsm2Info(&info_native);

    // Write back native variable values
    setNativePointerValue(env, info, (jlong)info_native);

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyBsrsm2InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDestroyBsrsm2Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyBsrsm2Info(info=%p)\n",
        info);

    // Native variable declarations
    bsrsm2Info_t info_native;

    // Obtain native variable values
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyBsrsm2Info(info_native);

    // Write back native variable values
    // info is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateCsru2csrInfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCreateCsru2csrInfo");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateCsru2csrInfo(info=%p)\n",
        info);

    // Native variable declarations
    csru2csrInfo_t info_native;

    // Obtain native variable values
    // info is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateCsru2csrInfo(&info_native);

    // Write back native variable values
    setNativePointerValue(env, info, (jlong)info_native);

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyCsru2csrInfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDestroyCsru2csrInfo");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyCsru2csrInfo(info=%p)\n",
        info);

    // Native variable declarations
    csru2csrInfo_t info_native;

    // Obtain native variable values
    info_native = (csru2csrInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyCsru2csrInfo(info_native);

    // Write back native variable values
    // info is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateColorInfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCreateColorInfo");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateColorInfo(info=%p)\n",
        info);

    // Native variable declarations
    cusparseColorInfo_t info_native;

    // Obtain native variable values
    // info is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateColorInfo(&info_native);

    // Write back native variable values
    setNativePointerValue(env, info, (jlong)info_native);

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyColorInfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDestroyColorInfo");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyColorInfo(info=%p)\n",
        info);

    // Native variable declarations
    cusparseColorInfo_t info_native;

    // Obtain native variable values
    info_native = (cusparseColorInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyColorInfo(info_native);

    // Write back native variable values
    // info is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreatePruneInfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCreatePruneInfo");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreatePruneInfo(info=%p)\n",
        info);

    // Native variable declarations
    pruneInfo_t info_native;

    // Obtain native variable values
    // info is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreatePruneInfo(&info_native);

    // Write back native variable values
    setNativePointerValue(env, info, (jlong)info_native);

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyPruneInfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDestroyPruneInfo");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyPruneInfo(info=%p)\n",
        info);

    // Native variable declarations
    pruneInfo_t info_native;

    // Obtain native variable values
    info_native = (pruneInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyPruneInfo(info_native);

    // Write back native variable values
    // info is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

//##############################################################################
//# SPARSE LEVEL 2 ROUTINES
//##############################################################################
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgemviNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint n, jobject alpha, jobject A, jint lda, jint nnz, jobject xVal, jobject xInd, jobject beta, jobject y, jint idxBase, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // n is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseSgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusparseSgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    // nnz is primitive
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseSgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseSgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseSgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseSgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgemvi(handle=%p, transA=%d, m=%d, n=%d, alpha=%p, A=%p, lda=%d, nnz=%d, xVal=%p, xInd=%p, beta=%p, y=%p, idxBase=%d, pBuffer=%p)\n",
        handle, transA, m, n, alpha, A, lda, nnz, xVal, xInd, beta, y, idxBase, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int n_native = 0;
    float * alpha_native = NULL;
    float * A_native = NULL;
    int lda_native = 0;
    int nnz_native = 0;
    float * xVal_native = NULL;
    int * xInd_native = NULL;
    float * beta_native = NULL;
    float * y_native = NULL;
    cusparseIndexBase_t idxBase_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    n_native = (int)n;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float *)alpha_pointerData->getPointer(env);
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    nnz_native = (int)nnz;
    xVal_native = (float *)getPointer(env, xVal);
    xInd_native = (int *)getPointer(env, xInd);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (float *)beta_pointerData->getPointer(env);
    y_native = (float *)getPointer(env, y);
    idxBase_native = (cusparseIndexBase_t)idxBase;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgemvi(handle_native, transA_native, m_native, n_native, alpha_native, A_native, lda_native, nnz_native, xVal_native, xInd_native, beta_native, y_native, idxBase_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // transA is primitive
    // m is primitive
    // n is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // A is a native pointer
    // lda is primitive
    // nnz is primitive
    // xVal is a native pointer
    // xInd is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer
    // idxBase is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgemvi_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint n, jint nnz, jintArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgemvi_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseSgemvi_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgemvi_bufferSize(handle=%p, transA=%d, m=%d, n=%d, nnz=%d, pBufferSize=%p)\n",
        handle, transA, m, n, nnz, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int n_native = 0;
    int nnz_native = 0;
    int * pBufferSize_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    n_native = (int)n;
    nnz_native = (int)nnz;
    if (!initNative(env, pBufferSize, pBufferSize_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgemvi_bufferSize(handle_native, transA_native, m_native, n_native, nnz_native, pBufferSize_native);

    // Write back native variable values
    // handle is read-only
    // transA is primitive
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (!releaseNative(env, pBufferSize_native, pBufferSize, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgemviNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint n, jobject alpha, jobject A, jint lda, jint nnz, jobject xVal, jobject xInd, jobject beta, jobject y, jint idxBase, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // n is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusparseDgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    // nnz is primitive
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseDgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseDgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseDgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseDgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgemvi(handle=%p, transA=%d, m=%d, n=%d, alpha=%p, A=%p, lda=%d, nnz=%d, xVal=%p, xInd=%p, beta=%p, y=%p, idxBase=%d, pBuffer=%p)\n",
        handle, transA, m, n, alpha, A, lda, nnz, xVal, xInd, beta, y, idxBase, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int n_native = 0;
    double * alpha_native = NULL;
    double * A_native = NULL;
    int lda_native = 0;
    int nnz_native = 0;
    double * xVal_native = NULL;
    int * xInd_native = NULL;
    double * beta_native = NULL;
    double * y_native = NULL;
    cusparseIndexBase_t idxBase_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    n_native = (int)n;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double *)alpha_pointerData->getPointer(env);
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    nnz_native = (int)nnz;
    xVal_native = (double *)getPointer(env, xVal);
    xInd_native = (int *)getPointer(env, xInd);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (double *)beta_pointerData->getPointer(env);
    y_native = (double *)getPointer(env, y);
    idxBase_native = (cusparseIndexBase_t)idxBase;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgemvi(handle_native, transA_native, m_native, n_native, alpha_native, A_native, lda_native, nnz_native, xVal_native, xInd_native, beta_native, y_native, idxBase_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // transA is primitive
    // m is primitive
    // n is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // A is a native pointer
    // lda is primitive
    // nnz is primitive
    // xVal is a native pointer
    // xInd is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer
    // idxBase is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgemvi_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint n, jint nnz, jintArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgemvi_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseDgemvi_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgemvi_bufferSize(handle=%p, transA=%d, m=%d, n=%d, nnz=%d, pBufferSize=%p)\n",
        handle, transA, m, n, nnz, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int n_native = 0;
    int nnz_native = 0;
    int * pBufferSize_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    n_native = (int)n;
    nnz_native = (int)nnz;
    if (!initNative(env, pBufferSize, pBufferSize_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgemvi_bufferSize(handle_native, transA_native, m_native, n_native, nnz_native, pBufferSize_native);

    // Write back native variable values
    // handle is read-only
    // transA is primitive
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (!releaseNative(env, pBufferSize_native, pBufferSize, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgemviNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint n, jobject alpha, jobject A, jint lda, jint nnz, jobject xVal, jobject xInd, jobject beta, jobject y, jint idxBase, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // n is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseCgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusparseCgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    // nnz is primitive
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseCgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseCgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseCgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseCgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgemvi(handle=%p, transA=%d, m=%d, n=%d, alpha=%p, A=%p, lda=%d, nnz=%d, xVal=%p, xInd=%p, beta=%p, y=%p, idxBase=%d, pBuffer=%p)\n",
        handle, transA, m, n, alpha, A, lda, nnz, xVal, xInd, beta, y, idxBase, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int n_native = 0;
    cuComplex * alpha_native = NULL;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    int nnz_native = 0;
    cuComplex * xVal_native = NULL;
    int * xInd_native = NULL;
    cuComplex * beta_native = NULL;
    cuComplex * y_native = NULL;
    cusparseIndexBase_t idxBase_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    n_native = (int)n;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex *)alpha_pointerData->getPointer(env);
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    nnz_native = (int)nnz;
    xVal_native = (cuComplex *)getPointer(env, xVal);
    xInd_native = (int *)getPointer(env, xInd);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuComplex *)beta_pointerData->getPointer(env);
    y_native = (cuComplex *)getPointer(env, y);
    idxBase_native = (cusparseIndexBase_t)idxBase;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgemvi(handle_native, transA_native, m_native, n_native, alpha_native, A_native, lda_native, nnz_native, xVal_native, xInd_native, beta_native, y_native, idxBase_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // transA is primitive
    // m is primitive
    // n is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // A is a native pointer
    // lda is primitive
    // nnz is primitive
    // xVal is a native pointer
    // xInd is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer
    // idxBase is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgemvi_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint n, jint nnz, jintArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgemvi_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseCgemvi_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgemvi_bufferSize(handle=%p, transA=%d, m=%d, n=%d, nnz=%d, pBufferSize=%p)\n",
        handle, transA, m, n, nnz, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int n_native = 0;
    int nnz_native = 0;
    int * pBufferSize_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    n_native = (int)n;
    nnz_native = (int)nnz;
    if (!initNative(env, pBufferSize, pBufferSize_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgemvi_bufferSize(handle_native, transA_native, m_native, n_native, nnz_native, pBufferSize_native);

    // Write back native variable values
    // handle is read-only
    // transA is primitive
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (!releaseNative(env, pBufferSize_native, pBufferSize, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgemviNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint n, jobject alpha, jobject A, jint lda, jint nnz, jobject xVal, jobject xInd, jobject beta, jobject y, jint idxBase, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // n is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusparseZgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    // nnz is primitive
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseZgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseZgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseZgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseZgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZgemvi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgemvi(handle=%p, transA=%d, m=%d, n=%d, alpha=%p, A=%p, lda=%d, nnz=%d, xVal=%p, xInd=%p, beta=%p, y=%p, idxBase=%d, pBuffer=%p)\n",
        handle, transA, m, n, alpha, A, lda, nnz, xVal, xInd, beta, y, idxBase, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int n_native = 0;
    cuDoubleComplex * alpha_native = NULL;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    int nnz_native = 0;
    cuDoubleComplex * xVal_native = NULL;
    int * xInd_native = NULL;
    cuDoubleComplex * beta_native = NULL;
    cuDoubleComplex * y_native = NULL;
    cusparseIndexBase_t idxBase_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    n_native = (int)n;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex *)alpha_pointerData->getPointer(env);
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    nnz_native = (int)nnz;
    xVal_native = (cuDoubleComplex *)getPointer(env, xVal);
    xInd_native = (int *)getPointer(env, xInd);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuDoubleComplex *)beta_pointerData->getPointer(env);
    y_native = (cuDoubleComplex *)getPointer(env, y);
    idxBase_native = (cusparseIndexBase_t)idxBase;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgemvi(handle_native, transA_native, m_native, n_native, alpha_native, A_native, lda_native, nnz_native, xVal_native, xInd_native, beta_native, y_native, idxBase_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // transA is primitive
    // m is primitive
    // n is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // A is a native pointer
    // lda is primitive
    // nnz is primitive
    // xVal is a native pointer
    // xInd is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer
    // idxBase is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgemvi_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint n, jint nnz, jintArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgemvi_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseZgemvi_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgemvi_bufferSize(handle=%p, transA=%d, m=%d, n=%d, nnz=%d, pBufferSize=%p)\n",
        handle, transA, m, n, nnz, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int n_native = 0;
    int nnz_native = 0;
    int * pBufferSize_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    n_native = (int)n;
    nnz_native = (int)nnz;
    if (!initNative(env, pBufferSize, pBufferSize_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgemvi_bufferSize(handle_native, transA_native, m_native, n_native, nnz_native, pBufferSize_native);

    // Write back native variable values
    // handle is read-only
    // transA is primitive
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (!releaseNative(env, pBufferSize_native, pBufferSize, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrmvNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseSbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseSbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseSbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseSbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseSbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseSbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseSbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrmv(handle=%p, dirA=%d, transA=%d, mb=%d, nb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, x=%p, beta=%p, y=%p)\n",
        handle, dirA, transA, mb, nb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    float * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    float * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    float * x_native = NULL;
    float * beta_native = NULL;
    float * y_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (float *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    x_native = (float *)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (float *)beta_pointerData->getPointer(env);
    y_native = (float *)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrmv(handle_native, dirA_native, transA_native, mb_native, nb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrmvNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseDbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseDbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseDbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseDbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseDbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseDbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrmv(handle=%p, dirA=%d, transA=%d, mb=%d, nb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, x=%p, beta=%p, y=%p)\n",
        handle, dirA, transA, mb, nb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    double * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    double * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    double * x_native = NULL;
    double * beta_native = NULL;
    double * y_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (double *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    x_native = (double *)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (double *)beta_pointerData->getPointer(env);
    y_native = (double *)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrmv(handle_native, dirA_native, transA_native, mb_native, nb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrmvNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseCbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseCbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseCbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseCbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseCbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseCbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseCbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrmv(handle=%p, dirA=%d, transA=%d, mb=%d, nb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, x=%p, beta=%p, y=%p)\n",
        handle, dirA, transA, mb, nb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    cuComplex * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    cuComplex * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    cuComplex * x_native = NULL;
    cuComplex * beta_native = NULL;
    cuComplex * y_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuComplex *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    x_native = (cuComplex *)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuComplex *)beta_pointerData->getPointer(env);
    y_native = (cuComplex *)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrmv(handle_native, dirA_native, transA_native, mb_native, nb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrmvNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseZbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseZbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseZbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseZbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseZbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseZbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrmv(handle=%p, dirA=%d, transA=%d, mb=%d, nb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, x=%p, beta=%p, y=%p)\n",
        handle, dirA, transA, mb, nb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    cuDoubleComplex * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    cuDoubleComplex * x_native = NULL;
    cuDoubleComplex * beta_native = NULL;
    cuDoubleComplex * y_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuDoubleComplex *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    x_native = (cuDoubleComplex *)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuDoubleComplex *)beta_pointerData->getPointer(env);
    y_native = (cuDoubleComplex *)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrmv(handle_native, dirA_native, transA_native, mb_native, nb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrxmvNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint sizeOfMask, jint mb, jint nb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedMaskPtrA, jobject bsrSortedRowPtrA, jobject bsrSortedEndPtrA, jobject bsrSortedColIndA, jint blockDim, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // sizeOfMask is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedMaskPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedMaskPtrA' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedEndPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedEndPtrA' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrxmv(handle=%p, dirA=%d, transA=%d, sizeOfMask=%d, mb=%d, nb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedMaskPtrA=%p, bsrSortedRowPtrA=%p, bsrSortedEndPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, x=%p, beta=%p, y=%p)\n",
        handle, dirA, transA, sizeOfMask, mb, nb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedMaskPtrA, bsrSortedRowPtrA, bsrSortedEndPtrA, bsrSortedColIndA, blockDim, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int sizeOfMask_native = 0;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    float * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    float * bsrSortedValA_native = NULL;
    int * bsrSortedMaskPtrA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedEndPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    float * x_native = NULL;
    float * beta_native = NULL;
    float * y_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    sizeOfMask_native = (int)sizeOfMask;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (float *)getPointer(env, bsrSortedValA);
    bsrSortedMaskPtrA_native = (int *)getPointer(env, bsrSortedMaskPtrA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedEndPtrA_native = (int *)getPointer(env, bsrSortedEndPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    x_native = (float *)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (float *)beta_pointerData->getPointer(env);
    y_native = (float *)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrxmv(handle_native, dirA_native, transA_native, sizeOfMask_native, mb_native, nb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedMaskPtrA_native, bsrSortedRowPtrA_native, bsrSortedEndPtrA_native, bsrSortedColIndA_native, blockDim_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // sizeOfMask is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedMaskPtrA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedEndPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrxmvNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint sizeOfMask, jint mb, jint nb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedMaskPtrA, jobject bsrSortedRowPtrA, jobject bsrSortedEndPtrA, jobject bsrSortedColIndA, jint blockDim, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // sizeOfMask is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedMaskPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedMaskPtrA' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedEndPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedEndPtrA' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrxmv(handle=%p, dirA=%d, transA=%d, sizeOfMask=%d, mb=%d, nb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedMaskPtrA=%p, bsrSortedRowPtrA=%p, bsrSortedEndPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, x=%p, beta=%p, y=%p)\n",
        handle, dirA, transA, sizeOfMask, mb, nb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedMaskPtrA, bsrSortedRowPtrA, bsrSortedEndPtrA, bsrSortedColIndA, blockDim, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int sizeOfMask_native = 0;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    double * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    double * bsrSortedValA_native = NULL;
    int * bsrSortedMaskPtrA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedEndPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    double * x_native = NULL;
    double * beta_native = NULL;
    double * y_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    sizeOfMask_native = (int)sizeOfMask;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (double *)getPointer(env, bsrSortedValA);
    bsrSortedMaskPtrA_native = (int *)getPointer(env, bsrSortedMaskPtrA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedEndPtrA_native = (int *)getPointer(env, bsrSortedEndPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    x_native = (double *)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (double *)beta_pointerData->getPointer(env);
    y_native = (double *)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrxmv(handle_native, dirA_native, transA_native, sizeOfMask_native, mb_native, nb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedMaskPtrA_native, bsrSortedRowPtrA_native, bsrSortedEndPtrA_native, bsrSortedColIndA_native, blockDim_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // sizeOfMask is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedMaskPtrA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedEndPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrxmvNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint sizeOfMask, jint mb, jint nb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedMaskPtrA, jobject bsrSortedRowPtrA, jobject bsrSortedEndPtrA, jobject bsrSortedColIndA, jint blockDim, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // sizeOfMask is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedMaskPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedMaskPtrA' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedEndPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedEndPtrA' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrxmv(handle=%p, dirA=%d, transA=%d, sizeOfMask=%d, mb=%d, nb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedMaskPtrA=%p, bsrSortedRowPtrA=%p, bsrSortedEndPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, x=%p, beta=%p, y=%p)\n",
        handle, dirA, transA, sizeOfMask, mb, nb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedMaskPtrA, bsrSortedRowPtrA, bsrSortedEndPtrA, bsrSortedColIndA, blockDim, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int sizeOfMask_native = 0;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    cuComplex * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    cuComplex * bsrSortedValA_native = NULL;
    int * bsrSortedMaskPtrA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedEndPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    cuComplex * x_native = NULL;
    cuComplex * beta_native = NULL;
    cuComplex * y_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    sizeOfMask_native = (int)sizeOfMask;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuComplex *)getPointer(env, bsrSortedValA);
    bsrSortedMaskPtrA_native = (int *)getPointer(env, bsrSortedMaskPtrA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedEndPtrA_native = (int *)getPointer(env, bsrSortedEndPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    x_native = (cuComplex *)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuComplex *)beta_pointerData->getPointer(env);
    y_native = (cuComplex *)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrxmv(handle_native, dirA_native, transA_native, sizeOfMask_native, mb_native, nb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedMaskPtrA_native, bsrSortedRowPtrA_native, bsrSortedEndPtrA_native, bsrSortedColIndA_native, blockDim_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // sizeOfMask is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedMaskPtrA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedEndPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrxmvNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint sizeOfMask, jint mb, jint nb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedMaskPtrA, jobject bsrSortedRowPtrA, jobject bsrSortedEndPtrA, jobject bsrSortedColIndA, jint blockDim, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // sizeOfMask is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedMaskPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedMaskPtrA' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedEndPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedEndPtrA' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrxmv(handle=%p, dirA=%d, transA=%d, sizeOfMask=%d, mb=%d, nb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedMaskPtrA=%p, bsrSortedRowPtrA=%p, bsrSortedEndPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, x=%p, beta=%p, y=%p)\n",
        handle, dirA, transA, sizeOfMask, mb, nb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedMaskPtrA, bsrSortedRowPtrA, bsrSortedEndPtrA, bsrSortedColIndA, blockDim, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int sizeOfMask_native = 0;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    cuDoubleComplex * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * bsrSortedValA_native = NULL;
    int * bsrSortedMaskPtrA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedEndPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    cuDoubleComplex * x_native = NULL;
    cuDoubleComplex * beta_native = NULL;
    cuDoubleComplex * y_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    sizeOfMask_native = (int)sizeOfMask;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuDoubleComplex *)getPointer(env, bsrSortedValA);
    bsrSortedMaskPtrA_native = (int *)getPointer(env, bsrSortedMaskPtrA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedEndPtrA_native = (int *)getPointer(env, bsrSortedEndPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    x_native = (cuDoubleComplex *)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuDoubleComplex *)beta_pointerData->getPointer(env);
    y_native = (cuDoubleComplex *)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrxmv(handle_native, dirA_native, transA_native, sizeOfMask_native, mb_native, nb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedMaskPtrA_native, bsrSortedRowPtrA_native, bsrSortedEndPtrA_native, bsrSortedColIndA_native, blockDim_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // sizeOfMask is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedMaskPtrA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedEndPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXbsrsv2_1zeroPivotNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jobject position)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXbsrsv2_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseXbsrsv2_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (position == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'position' is null for cusparseXbsrsv2_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXbsrsv2_zeroPivot(handle=%p, info=%p, position=%p)\n",
        handle, info, position);

    // Native variable declarations
    cusparseHandle_t handle_native;
    bsrsv2Info_t info_native;
    int * position_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    position_native = (int *)getPointer(env, position);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXbsrsv2_zeroPivot(handle_native, info_native, position_native);

    // Write back native variable values
    // handle is read-only
    // info is read-only
    // position is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrsv2_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseSbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseSbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseSbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseSbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrsv2_bufferSize(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, transA, mb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (float *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrsv2_bufferSize(handle_native, dirA_native, transA_native, mb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrsv2_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseDbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseDbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseDbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseDbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrsv2_bufferSize(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, transA, mb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (double *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrsv2_bufferSize(handle_native, dirA_native, transA_native, mb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrsv2_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseCbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseCbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseCbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseCbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrsv2_bufferSize(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, transA, mb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuComplex *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrsv2_bufferSize(handle_native, dirA_native, transA_native, mb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrsv2_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseZbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseZbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseZbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseZbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrsv2_bufferSize(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, transA, mb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuDoubleComplex *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrsv2_bufferSize(handle_native, dirA_native, transA_native, mb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrsv2_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseSbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseSbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseSbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrsv2_analysis(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, mb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (float *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrsv2_analysis(handle_native, dirA_native, transA_native, mb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrsv2_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseDbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseDbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseDbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrsv2_analysis(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, mb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (double *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrsv2_analysis(handle_native, dirA_native, transA_native, mb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrsv2_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseCbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseCbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseCbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrsv2_analysis(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, mb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuComplex *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrsv2_analysis(handle_native, dirA_native, transA_native, mb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrsv2_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseZbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseZbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseZbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrsv2_analysis(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, mb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuDoubleComplex *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrsv2_analysis(handle_native, dirA_native, transA_native, mb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrsv2_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jobject f, jobject x, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseSbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseSbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseSbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseSbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (f == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'f' is null for cusparseSbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseSbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrsv2_solve(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, f=%p, x=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, mb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, f, x, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    float * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    float * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    float * f_native = NULL;
    float * x_native = NULL;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (float *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    f_native = (float *)getPointer(env, f);
    x_native = (float *)getPointer(env, x);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrsv2_solve(handle_native, dirA_native, transA_native, mb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, f_native, x_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is read-only
    // f is a native pointer
    // x is a native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrsv2_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jobject f, jobject x, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseDbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseDbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseDbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (f == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'f' is null for cusparseDbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseDbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrsv2_solve(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, f=%p, x=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, mb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, f, x, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    double * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    double * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    double * f_native = NULL;
    double * x_native = NULL;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (double *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    f_native = (double *)getPointer(env, f);
    x_native = (double *)getPointer(env, x);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrsv2_solve(handle_native, dirA_native, transA_native, mb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, f_native, x_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is read-only
    // f is a native pointer
    // x is a native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrsv2_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jobject f, jobject x, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseCbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseCbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseCbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseCbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (f == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'f' is null for cusparseCbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseCbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrsv2_solve(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, f=%p, x=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, mb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, f, x, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cuComplex * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    cuComplex * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    cuComplex * f_native = NULL;
    cuComplex * x_native = NULL;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuComplex *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    f_native = (cuComplex *)getPointer(env, f);
    x_native = (cuComplex *)getPointer(env, x);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrsv2_solve(handle_native, dirA_native, transA_native, mb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, f_native, x_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is read-only
    // f is a native pointer
    // x is a native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrsv2_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jobject f, jobject x, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseZbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseZbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseZbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (f == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'f' is null for cusparseZbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseZbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrsv2_solve(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, f=%p, x=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, mb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, f, x, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cuDoubleComplex * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    cuDoubleComplex * f_native = NULL;
    cuDoubleComplex * x_native = NULL;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuDoubleComplex *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    f_native = (cuDoubleComplex *)getPointer(env, f);
    x_native = (cuDoubleComplex *)getPointer(env, x);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrsv2_solve(handle_native, dirA_native, transA_native, mb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, f_native, x_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is read-only
    // f is a native pointer
    // x is a native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

//##############################################################################
//# SPARSE LEVEL 3 ROUTINES
//##############################################################################
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrmmNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transB, jint mb, jint n, jint kb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockSize, jobject B, jint ldb, jobject beta, jobject C, jint ldc)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // kb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseSbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseSbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseSbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseSbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseSbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseSbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusparseSbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrmm(handle=%p, dirA=%d, transA=%d, transB=%d, mb=%d, n=%d, kb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockSize=%d, B=%p, ldb=%d, beta=%p, C=%p, ldc=%d)\n",
        handle, dirA, transA, transB, mb, n, kb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockSize, B, ldb, beta, C, ldc);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transB_native;
    int mb_native = 0;
    int n_native = 0;
    int kb_native = 0;
    int nnzb_native = 0;
    float * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    float * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockSize_native = 0;
    float * B_native = NULL;
    int ldb_native = 0;
    float * beta_native = NULL;
    float * C_native = NULL;
    int ldc_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transB_native = (cusparseOperation_t)transB;
    mb_native = (int)mb;
    n_native = (int)n;
    kb_native = (int)kb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (float *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockSize_native = (int)blockSize;
    B_native = (float *)getPointer(env, B);
    ldb_native = (int)ldb;
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (float *)beta_pointerData->getPointer(env);
    C_native = (float *)getPointer(env, C);
    ldc_native = (int)ldc;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrmm(handle_native, dirA_native, transA_native, transB_native, mb_native, n_native, kb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockSize_native, B_native, ldb_native, beta_native, C_native, ldc_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // kb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockSize is primitive
    // B is a native pointer
    // ldb is primitive
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // C is a native pointer
    // ldc is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrmmNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transB, jint mb, jint n, jint kb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockSize, jobject B, jint ldb, jobject beta, jobject C, jint ldc)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // kb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseDbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseDbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseDbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseDbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseDbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusparseDbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrmm(handle=%p, dirA=%d, transA=%d, transB=%d, mb=%d, n=%d, kb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockSize=%d, B=%p, ldb=%d, beta=%p, C=%p, ldc=%d)\n",
        handle, dirA, transA, transB, mb, n, kb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockSize, B, ldb, beta, C, ldc);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transB_native;
    int mb_native = 0;
    int n_native = 0;
    int kb_native = 0;
    int nnzb_native = 0;
    double * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    double * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockSize_native = 0;
    double * B_native = NULL;
    int ldb_native = 0;
    double * beta_native = NULL;
    double * C_native = NULL;
    int ldc_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transB_native = (cusparseOperation_t)transB;
    mb_native = (int)mb;
    n_native = (int)n;
    kb_native = (int)kb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (double *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockSize_native = (int)blockSize;
    B_native = (double *)getPointer(env, B);
    ldb_native = (int)ldb;
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (double *)beta_pointerData->getPointer(env);
    C_native = (double *)getPointer(env, C);
    ldc_native = (int)ldc;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrmm(handle_native, dirA_native, transA_native, transB_native, mb_native, n_native, kb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockSize_native, B_native, ldb_native, beta_native, C_native, ldc_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // kb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockSize is primitive
    // B is a native pointer
    // ldb is primitive
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // C is a native pointer
    // ldc is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrmmNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transB, jint mb, jint n, jint kb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockSize, jobject B, jint ldb, jobject beta, jobject C, jint ldc)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // kb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseCbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseCbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseCbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseCbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseCbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseCbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusparseCbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrmm(handle=%p, dirA=%d, transA=%d, transB=%d, mb=%d, n=%d, kb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockSize=%d, B=%p, ldb=%d, beta=%p, C=%p, ldc=%d)\n",
        handle, dirA, transA, transB, mb, n, kb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockSize, B, ldb, beta, C, ldc);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transB_native;
    int mb_native = 0;
    int n_native = 0;
    int kb_native = 0;
    int nnzb_native = 0;
    cuComplex * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    cuComplex * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockSize_native = 0;
    cuComplex * B_native = NULL;
    int ldb_native = 0;
    cuComplex * beta_native = NULL;
    cuComplex * C_native = NULL;
    int ldc_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transB_native = (cusparseOperation_t)transB;
    mb_native = (int)mb;
    n_native = (int)n;
    kb_native = (int)kb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuComplex *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockSize_native = (int)blockSize;
    B_native = (cuComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuComplex *)beta_pointerData->getPointer(env);
    C_native = (cuComplex *)getPointer(env, C);
    ldc_native = (int)ldc;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrmm(handle_native, dirA_native, transA_native, transB_native, mb_native, n_native, kb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockSize_native, B_native, ldb_native, beta_native, C_native, ldc_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // kb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockSize is primitive
    // B is a native pointer
    // ldb is primitive
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // C is a native pointer
    // ldc is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrmmNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transB, jint mb, jint n, jint kb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockSize, jobject B, jint ldb, jobject beta, jobject C, jint ldc)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // kb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseZbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseZbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseZbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseZbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseZbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusparseZbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrmm(handle=%p, dirA=%d, transA=%d, transB=%d, mb=%d, n=%d, kb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockSize=%d, B=%p, ldb=%d, beta=%p, C=%p, ldc=%d)\n",
        handle, dirA, transA, transB, mb, n, kb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockSize, B, ldb, beta, C, ldc);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transB_native;
    int mb_native = 0;
    int n_native = 0;
    int kb_native = 0;
    int nnzb_native = 0;
    cuDoubleComplex * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockSize_native = 0;
    cuDoubleComplex * B_native = NULL;
    int ldb_native = 0;
    cuDoubleComplex * beta_native = NULL;
    cuDoubleComplex * C_native = NULL;
    int ldc_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transB_native = (cusparseOperation_t)transB;
    mb_native = (int)mb;
    n_native = (int)n;
    kb_native = (int)kb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuDoubleComplex *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockSize_native = (int)blockSize;
    B_native = (cuDoubleComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuDoubleComplex *)beta_pointerData->getPointer(env);
    C_native = (cuDoubleComplex *)getPointer(env, C);
    ldc_native = (int)ldc;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrmm(handle_native, dirA_native, transA_native, transB_native, mb_native, n_native, kb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockSize_native, B_native, ldb_native, beta_native, C_native, ldc_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // kb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockSize is primitive
    // B is a native pointer
    // ldb is primitive
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // C is a native pointer
    // ldc is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXbsrsm2_1zeroPivotNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jobject position)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXbsrsm2_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseXbsrsm2_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (position == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'position' is null for cusparseXbsrsm2_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXbsrsm2_zeroPivot(handle=%p, info=%p, position=%p)\n",
        handle, info, position);

    // Native variable declarations
    cusparseHandle_t handle_native;
    bsrsm2Info_t info_native;
    int * position_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    position_native = (int *)getPointer(env, position);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXbsrsm2_zeroPivot(handle_native, info_native, position_native);

    // Write back native variable values
    // handle is read-only
    // info is read-only
    // position is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrsm2_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseSbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrsm2_bufferSize(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (float *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrsm2_bufferSize(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrsm2_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseDbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrsm2_bufferSize(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (double *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrsm2_bufferSize(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrsm2_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseCbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrsm2_bufferSize(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrsm2_bufferSize(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrsm2_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseZbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrsm2_bufferSize(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuDoubleComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrsm2_bufferSize(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrsm2_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrsm2_analysis(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (float *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrsm2_analysis(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrsm2_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrsm2_analysis(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (double *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrsm2_analysis(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrsm2_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrsm2_analysis(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrsm2_analysis(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrsm2_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrsm2_analysis(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuDoubleComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrsm2_analysis(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrsm2_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jobject B, jint ldb, jobject X, jint ldx, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseSbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseSbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (X == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'X' is null for cusparseSbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldx is primitive
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrsm2_solve(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, B=%p, ldb=%d, X=%p, ldx=%d, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, alpha, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, B, ldb, X, ldx, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    float * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    float * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    float * B_native = NULL;
    int ldb_native = 0;
    float * X_native = NULL;
    int ldx_native = 0;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (float *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    B_native = (float *)getPointer(env, B);
    ldb_native = (int)ldb;
    X_native = (float *)getPointer(env, X);
    ldx_native = (int)ldx;
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrsm2_solve(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, alpha_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, B_native, ldb_native, X_native, ldx_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is read-only
    // B is a native pointer
    // ldb is primitive
    // X is a native pointer
    // ldx is primitive
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrsm2_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jobject B, jint ldb, jobject X, jint ldx, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseDbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (X == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'X' is null for cusparseDbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldx is primitive
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrsm2_solve(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, B=%p, ldb=%d, X=%p, ldx=%d, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, alpha, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, B, ldb, X, ldx, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    double * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    double * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    double * B_native = NULL;
    int ldb_native = 0;
    double * X_native = NULL;
    int ldx_native = 0;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (double *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    B_native = (double *)getPointer(env, B);
    ldb_native = (int)ldb;
    X_native = (double *)getPointer(env, X);
    ldx_native = (int)ldx;
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrsm2_solve(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, alpha_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, B_native, ldb_native, X_native, ldx_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is read-only
    // B is a native pointer
    // ldb is primitive
    // X is a native pointer
    // ldx is primitive
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrsm2_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jobject B, jint ldb, jobject X, jint ldx, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseCbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseCbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (X == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'X' is null for cusparseCbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldx is primitive
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrsm2_solve(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, B=%p, ldb=%d, X=%p, ldx=%d, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, alpha, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, B, ldb, X, ldx, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cuComplex * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    cuComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    cuComplex * B_native = NULL;
    int ldb_native = 0;
    cuComplex * X_native = NULL;
    int ldx_native = 0;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    B_native = (cuComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    X_native = (cuComplex *)getPointer(env, X);
    ldx_native = (int)ldx;
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrsm2_solve(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, alpha_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, B_native, ldb_native, X_native, ldx_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is read-only
    // B is a native pointer
    // ldb is primitive
    // X is a native pointer
    // ldx is primitive
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrsm2_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jobject B, jint ldb, jobject X, jint ldx, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseZbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (X == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'X' is null for cusparseZbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldx is primitive
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrsm2_solve(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, B=%p, ldb=%d, X=%p, ldx=%d, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, alpha, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, B, ldb, X, ldx, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cuDoubleComplex * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    cuDoubleComplex * B_native = NULL;
    int ldb_native = 0;
    cuDoubleComplex * X_native = NULL;
    int ldx_native = 0;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuDoubleComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    B_native = (cuDoubleComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    X_native = (cuDoubleComplex *)getPointer(env, X);
    ldx_native = (int)ldx;
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrsm2_solve(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, alpha_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, B_native, ldb_native, X_native, ldx_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is read-only
    // B is a native pointer
    // ldb is primitive
    // X is a native pointer
    // ldx is primitive
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

//##############################################################################
//# PRECONDITIONERS
//##############################################################################
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrilu02_1numericBoostNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jint enable_boost, jobject tol, jobject boost_val)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // enable_boost is primitive
    if (tol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tol' is null for cusparseScsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (boost_val == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'boost_val' is null for cusparseScsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrilu02_numericBoost(handle=%p, info=%p, enable_boost=%d, tol=%p, boost_val=%p)\n",
        handle, info, enable_boost, tol, boost_val);

    // Native variable declarations
    cusparseHandle_t handle_native;
    csrilu02Info_t info_native;
    int enable_boost_native = 0;
    double * tol_native = NULL;
    float * boost_val_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    enable_boost_native = (int)enable_boost;
    PointerData *tol_pointerData = initPointerData(env, tol);
    if (tol_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    tol_native = (double *)tol_pointerData->getPointer(env);
    PointerData *boost_val_pointerData = initPointerData(env, boost_val);
    if (boost_val_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    boost_val_native = (float *)boost_val_pointerData->getPointer(env);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrilu02_numericBoost(handle_native, info_native, enable_boost_native, tol_native, boost_val_native);

    // Write back native variable values
    // handle is read-only
    // info is read-only
    // enable_boost is primitive
    if (!releasePointerData(env, tol_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, boost_val_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrilu02_1numericBoostNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jint enable_boost, jobject tol, jobject boost_val)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // enable_boost is primitive
    if (tol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tol' is null for cusparseDcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (boost_val == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'boost_val' is null for cusparseDcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrilu02_numericBoost(handle=%p, info=%p, enable_boost=%d, tol=%p, boost_val=%p)\n",
        handle, info, enable_boost, tol, boost_val);

    // Native variable declarations
    cusparseHandle_t handle_native;
    csrilu02Info_t info_native;
    int enable_boost_native = 0;
    double * tol_native = NULL;
    double * boost_val_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    enable_boost_native = (int)enable_boost;
    PointerData *tol_pointerData = initPointerData(env, tol);
    if (tol_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    tol_native = (double *)tol_pointerData->getPointer(env);
    PointerData *boost_val_pointerData = initPointerData(env, boost_val);
    if (boost_val_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    boost_val_native = (double *)boost_val_pointerData->getPointer(env);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrilu02_numericBoost(handle_native, info_native, enable_boost_native, tol_native, boost_val_native);

    // Write back native variable values
    // handle is read-only
    // info is read-only
    // enable_boost is primitive
    if (!releasePointerData(env, tol_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, boost_val_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrilu02_1numericBoostNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jint enable_boost, jobject tol, jobject boost_val)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // enable_boost is primitive
    if (tol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tol' is null for cusparseCcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (boost_val == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'boost_val' is null for cusparseCcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrilu02_numericBoost(handle=%p, info=%p, enable_boost=%d, tol=%p, boost_val=%p)\n",
        handle, info, enable_boost, tol, boost_val);

    // Native variable declarations
    cusparseHandle_t handle_native;
    csrilu02Info_t info_native;
    int enable_boost_native = 0;
    double * tol_native = NULL;
    cuComplex * boost_val_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    enable_boost_native = (int)enable_boost;
    PointerData *tol_pointerData = initPointerData(env, tol);
    if (tol_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    tol_native = (double *)tol_pointerData->getPointer(env);
    PointerData *boost_val_pointerData = initPointerData(env, boost_val);
    if (boost_val_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    boost_val_native = (cuComplex *)boost_val_pointerData->getPointer(env);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrilu02_numericBoost(handle_native, info_native, enable_boost_native, tol_native, boost_val_native);

    // Write back native variable values
    // handle is read-only
    // info is read-only
    // enable_boost is primitive
    if (!releasePointerData(env, tol_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, boost_val_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrilu02_1numericBoostNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jint enable_boost, jobject tol, jobject boost_val)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // enable_boost is primitive
    if (tol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tol' is null for cusparseZcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (boost_val == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'boost_val' is null for cusparseZcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrilu02_numericBoost(handle=%p, info=%p, enable_boost=%d, tol=%p, boost_val=%p)\n",
        handle, info, enable_boost, tol, boost_val);

    // Native variable declarations
    cusparseHandle_t handle_native;
    csrilu02Info_t info_native;
    int enable_boost_native = 0;
    double * tol_native = NULL;
    cuDoubleComplex * boost_val_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    enable_boost_native = (int)enable_boost;
    PointerData *tol_pointerData = initPointerData(env, tol);
    if (tol_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    tol_native = (double *)tol_pointerData->getPointer(env);
    PointerData *boost_val_pointerData = initPointerData(env, boost_val);
    if (boost_val_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    boost_val_native = (cuDoubleComplex *)boost_val_pointerData->getPointer(env);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrilu02_numericBoost(handle_native, info_native, enable_boost_native, tol_native, boost_val_native);

    // Write back native variable values
    // handle is read-only
    // info is read-only
    // enable_boost is primitive
    if (!releasePointerData(env, tol_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, boost_val_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXcsrilu02_1zeroPivotNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jobject position)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXcsrilu02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseXcsrilu02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (position == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'position' is null for cusparseXcsrilu02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXcsrilu02_zeroPivot(handle=%p, info=%p, position=%p)\n",
        handle, info, position);

    // Native variable declarations
    cusparseHandle_t handle_native;
    csrilu02Info_t info_native;
    int * position_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    position_native = (int *)getPointer(env, position);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXcsrilu02_zeroPivot(handle_native, info_native, position_native);

    // Write back native variable values
    // handle is read-only
    // info is read-only
    // position is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrilu02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseScsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrilu02_bufferSize(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    float * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csrilu02Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrilu02_bufferSize(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrilu02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseDcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrilu02_bufferSize(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    double * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csrilu02Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrilu02_bufferSize(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrilu02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseCcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrilu02_bufferSize(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csrilu02Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrilu02_bufferSize(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrilu02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseZcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrilu02_bufferSize(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csrilu02Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrilu02_bufferSize(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrilu02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseScsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrilu02_analysis(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    float * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrilu02_analysis(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrilu02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrilu02_analysis(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    double * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrilu02_analysis(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrilu02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrilu02_analysis(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrilu02_analysis(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrilu02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrilu02_analysis(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrilu02_analysis(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrilu02Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA_valM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_valM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_valM' is null for cusparseScsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseScsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrilu02(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA_valM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA_valM, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    float * csrSortedValA_valM_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_valM_native = (float *)getPointer(env, csrSortedValA_valM);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrilu02(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_valM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA_valM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrilu02Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA_valM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_valM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_valM' is null for cusparseDcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrilu02(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA_valM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA_valM, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    double * csrSortedValA_valM_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_valM_native = (double *)getPointer(env, csrSortedValA_valM);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrilu02(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_valM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA_valM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrilu02Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA_valM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_valM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_valM' is null for cusparseCcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrilu02(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA_valM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA_valM, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * csrSortedValA_valM_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_valM_native = (cuComplex *)getPointer(env, csrSortedValA_valM);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrilu02(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_valM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA_valM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrilu02Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA_valM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_valM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_valM' is null for cusparseZcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrilu02(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA_valM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA_valM, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * csrSortedValA_valM_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_valM_native = (cuDoubleComplex *)getPointer(env, csrSortedValA_valM);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrilu02(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_valM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA_valM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrilu02_1numericBoostNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jint enable_boost, jobject tol, jobject boost_val)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // enable_boost is primitive
    if (tol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tol' is null for cusparseSbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (boost_val == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'boost_val' is null for cusparseSbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrilu02_numericBoost(handle=%p, info=%p, enable_boost=%d, tol=%p, boost_val=%p)\n",
        handle, info, enable_boost, tol, boost_val);

    // Native variable declarations
    cusparseHandle_t handle_native;
    bsrilu02Info_t info_native;
    int enable_boost_native = 0;
    double * tol_native = NULL;
    float * boost_val_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    enable_boost_native = (int)enable_boost;
    PointerData *tol_pointerData = initPointerData(env, tol);
    if (tol_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    tol_native = (double *)tol_pointerData->getPointer(env);
    PointerData *boost_val_pointerData = initPointerData(env, boost_val);
    if (boost_val_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    boost_val_native = (float *)boost_val_pointerData->getPointer(env);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrilu02_numericBoost(handle_native, info_native, enable_boost_native, tol_native, boost_val_native);

    // Write back native variable values
    // handle is read-only
    // info is read-only
    // enable_boost is primitive
    if (!releasePointerData(env, tol_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, boost_val_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrilu02_1numericBoostNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jint enable_boost, jobject tol, jobject boost_val)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // enable_boost is primitive
    if (tol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tol' is null for cusparseDbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (boost_val == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'boost_val' is null for cusparseDbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrilu02_numericBoost(handle=%p, info=%p, enable_boost=%d, tol=%p, boost_val=%p)\n",
        handle, info, enable_boost, tol, boost_val);

    // Native variable declarations
    cusparseHandle_t handle_native;
    bsrilu02Info_t info_native;
    int enable_boost_native = 0;
    double * tol_native = NULL;
    double * boost_val_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    enable_boost_native = (int)enable_boost;
    PointerData *tol_pointerData = initPointerData(env, tol);
    if (tol_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    tol_native = (double *)tol_pointerData->getPointer(env);
    PointerData *boost_val_pointerData = initPointerData(env, boost_val);
    if (boost_val_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    boost_val_native = (double *)boost_val_pointerData->getPointer(env);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrilu02_numericBoost(handle_native, info_native, enable_boost_native, tol_native, boost_val_native);

    // Write back native variable values
    // handle is read-only
    // info is read-only
    // enable_boost is primitive
    if (!releasePointerData(env, tol_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, boost_val_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrilu02_1numericBoostNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jint enable_boost, jobject tol, jobject boost_val)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // enable_boost is primitive
    if (tol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tol' is null for cusparseCbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (boost_val == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'boost_val' is null for cusparseCbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrilu02_numericBoost(handle=%p, info=%p, enable_boost=%d, tol=%p, boost_val=%p)\n",
        handle, info, enable_boost, tol, boost_val);

    // Native variable declarations
    cusparseHandle_t handle_native;
    bsrilu02Info_t info_native;
    int enable_boost_native = 0;
    double * tol_native = NULL;
    cuComplex * boost_val_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    enable_boost_native = (int)enable_boost;
    PointerData *tol_pointerData = initPointerData(env, tol);
    if (tol_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    tol_native = (double *)tol_pointerData->getPointer(env);
    PointerData *boost_val_pointerData = initPointerData(env, boost_val);
    if (boost_val_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    boost_val_native = (cuComplex *)boost_val_pointerData->getPointer(env);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrilu02_numericBoost(handle_native, info_native, enable_boost_native, tol_native, boost_val_native);

    // Write back native variable values
    // handle is read-only
    // info is read-only
    // enable_boost is primitive
    if (!releasePointerData(env, tol_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, boost_val_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrilu02_1numericBoostNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jint enable_boost, jobject tol, jobject boost_val)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // enable_boost is primitive
    if (tol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tol' is null for cusparseZbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (boost_val == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'boost_val' is null for cusparseZbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrilu02_numericBoost(handle=%p, info=%p, enable_boost=%d, tol=%p, boost_val=%p)\n",
        handle, info, enable_boost, tol, boost_val);

    // Native variable declarations
    cusparseHandle_t handle_native;
    bsrilu02Info_t info_native;
    int enable_boost_native = 0;
    double * tol_native = NULL;
    cuDoubleComplex * boost_val_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    enable_boost_native = (int)enable_boost;
    PointerData *tol_pointerData = initPointerData(env, tol);
    if (tol_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    tol_native = (double *)tol_pointerData->getPointer(env);
    PointerData *boost_val_pointerData = initPointerData(env, boost_val);
    if (boost_val_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    boost_val_native = (cuDoubleComplex *)boost_val_pointerData->getPointer(env);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrilu02_numericBoost(handle_native, info_native, enable_boost_native, tol_native, boost_val_native);

    // Write back native variable values
    // handle is read-only
    // info is read-only
    // enable_boost is primitive
    if (!releasePointerData(env, tol_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, boost_val_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXbsrilu02_1zeroPivotNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jobject position)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXbsrilu02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseXbsrilu02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (position == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'position' is null for cusparseXbsrilu02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXbsrilu02_zeroPivot(handle=%p, info=%p, position=%p)\n",
        handle, info, position);

    // Native variable declarations
    cusparseHandle_t handle_native;
    bsrilu02Info_t info_native;
    int * position_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    position_native = (int *)getPointer(env, position);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXbsrilu02_zeroPivot(handle_native, info_native, position_native);

    // Write back native variable values
    // handle is read-only
    // info is read-only
    // position is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrilu02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseSbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrilu02_bufferSize(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (float *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrilu02_bufferSize(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrilu02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseDbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrilu02_bufferSize(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (double *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrilu02_bufferSize(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrilu02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseCbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrilu02_bufferSize(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrilu02_bufferSize(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrilu02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseZbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrilu02_bufferSize(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuDoubleComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrilu02_bufferSize(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrilu02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrilu02_analysis(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (float *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrilu02_analysis(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrilu02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrilu02_analysis(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (double *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrilu02_analysis(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrilu02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrilu02_analysis(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrilu02_analysis(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrilu02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrilu02_analysis(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuDoubleComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrilu02_analysis(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrilu02Native(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrilu02(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (float *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrilu02(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrilu02Native(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrilu02(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (double *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrilu02(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrilu02Native(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrilu02(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrilu02(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrilu02Native(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrilu02(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuDoubleComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrilu02(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXcsric02_1zeroPivotNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jobject position)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXcsric02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseXcsric02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (position == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'position' is null for cusparseXcsric02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXcsric02_zeroPivot(handle=%p, info=%p, position=%p)\n",
        handle, info, position);

    // Native variable declarations
    cusparseHandle_t handle_native;
    csric02Info_t info_native;
    int * position_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    position_native = (int *)getPointer(env, position);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXcsric02_zeroPivot(handle_native, info_native, position_native);

    // Write back native variable values
    // handle is read-only
    // info is read-only
    // position is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsric02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseScsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsric02_bufferSize(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    float * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csric02Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsric02_bufferSize(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsric02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseDcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsric02_bufferSize(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    double * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csric02Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsric02_bufferSize(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsric02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseCcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsric02_bufferSize(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csric02Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsric02_bufferSize(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsric02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseZcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsric02_bufferSize(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csric02Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsric02_bufferSize(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsric02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseScsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsric02_analysis(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    float * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsric02_analysis(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsric02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsric02_analysis(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    double * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsric02_analysis(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsric02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsric02_analysis(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsric02_analysis(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsric02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsric02_analysis(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsric02_analysis(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsric02Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA_valM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_valM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_valM' is null for cusparseScsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseScsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsric02(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA_valM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA_valM, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    float * csrSortedValA_valM_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_valM_native = (float *)getPointer(env, csrSortedValA_valM);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsric02(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_valM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA_valM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsric02Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA_valM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_valM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_valM' is null for cusparseDcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsric02(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA_valM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA_valM, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    double * csrSortedValA_valM_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_valM_native = (double *)getPointer(env, csrSortedValA_valM);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsric02(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_valM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA_valM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsric02Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA_valM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_valM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_valM' is null for cusparseCcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsric02(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA_valM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA_valM, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * csrSortedValA_valM_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_valM_native = (cuComplex *)getPointer(env, csrSortedValA_valM);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsric02(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_valM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA_valM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsric02Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA_valM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_valM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_valM' is null for cusparseZcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsric02(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA_valM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA_valM, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * csrSortedValA_valM_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    csric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_valM_native = (cuDoubleComplex *)getPointer(env, csrSortedValA_valM);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsric02(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_valM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA_valM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXbsric02_1zeroPivotNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jobject position)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXbsric02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseXbsric02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (position == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'position' is null for cusparseXbsric02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXbsric02_zeroPivot(handle=%p, info=%p, position=%p)\n",
        handle, info, position);

    // Native variable declarations
    cusparseHandle_t handle_native;
    bsric02Info_t info_native;
    int * position_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    position_native = (int *)getPointer(env, position);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXbsric02_zeroPivot(handle_native, info_native, position_native);

    // Write back native variable values
    // handle is read-only
    // info is read-only
    // position is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsric02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseSbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsric02_bufferSize(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (float *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsric02_bufferSize(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsric02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseDbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsric02_bufferSize(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (double *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsric02_bufferSize(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsric02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseCbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsric02_bufferSize(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsric02_bufferSize(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsric02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseZbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsric02_bufferSize(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuDoubleComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsric02_bufferSize(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsric02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pInputBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pInputBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pInputBuffer' is null for cusparseSbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsric02_analysis(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pInputBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pInputBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pInputBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (float *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pInputBuffer_native = (void *)getPointer(env, pInputBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsric02_analysis(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pInputBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    // policy is primitive
    // pInputBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsric02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pInputBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pInputBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pInputBuffer' is null for cusparseDbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsric02_analysis(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pInputBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pInputBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pInputBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (double *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pInputBuffer_native = (void *)getPointer(env, pInputBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsric02_analysis(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pInputBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    // policy is primitive
    // pInputBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsric02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pInputBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pInputBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pInputBuffer' is null for cusparseCbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsric02_analysis(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pInputBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pInputBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pInputBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pInputBuffer_native = (void *)getPointer(env, pInputBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsric02_analysis(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pInputBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    // policy is primitive
    // pInputBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsric02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pInputBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pInputBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pInputBuffer' is null for cusparseZbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsric02_analysis(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pInputBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pInputBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pInputBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuDoubleComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pInputBuffer_native = (void *)getPointer(env, pInputBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsric02_analysis(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pInputBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    // policy is primitive
    // pInputBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsric02Native(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsric02(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (float *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsric02(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsric02Native(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsric02(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (double *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsric02(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsric02Native(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsric02(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsric02(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsric02Native(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsric02(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuDoubleComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsric02(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is read-only
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgtsv2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb, jlongArray bufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseSgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseSgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseSgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseSgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (bufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bufferSizeInBytes' is null for cusparseSgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgtsv2_bufferSizeExt(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d, bufferSizeInBytes=%p)\n",
        handle, m, n, dl, d, du, B, ldb, bufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    float * dl_native = NULL;
    float * d_native = NULL;
    float * du_native = NULL;
    float * B_native = NULL;
    int ldb_native = 0;
    size_t * bufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (float *)getPointer(env, dl);
    d_native = (float *)getPointer(env, d);
    du_native = (float *)getPointer(env, du);
    B_native = (float *)getPointer(env, B);
    ldb_native = (int)ldb;
    if (!initNative(env, bufferSizeInBytes, bufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgtsv2_bufferSizeExt(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native, bufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive
    if (!releaseNative(env, bufferSizeInBytes_native, bufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgtsv2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb, jlongArray bufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseDgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseDgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseDgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseDgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (bufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bufferSizeInBytes' is null for cusparseDgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgtsv2_bufferSizeExt(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d, bufferSizeInBytes=%p)\n",
        handle, m, n, dl, d, du, B, ldb, bufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    double * dl_native = NULL;
    double * d_native = NULL;
    double * du_native = NULL;
    double * B_native = NULL;
    int ldb_native = 0;
    size_t * bufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (double *)getPointer(env, dl);
    d_native = (double *)getPointer(env, d);
    du_native = (double *)getPointer(env, du);
    B_native = (double *)getPointer(env, B);
    ldb_native = (int)ldb;
    if (!initNative(env, bufferSizeInBytes, bufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgtsv2_bufferSizeExt(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native, bufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive
    if (!releaseNative(env, bufferSizeInBytes_native, bufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgtsv2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb, jlongArray bufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseCgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseCgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseCgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseCgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (bufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bufferSizeInBytes' is null for cusparseCgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgtsv2_bufferSizeExt(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d, bufferSizeInBytes=%p)\n",
        handle, m, n, dl, d, du, B, ldb, bufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuComplex * dl_native = NULL;
    cuComplex * d_native = NULL;
    cuComplex * du_native = NULL;
    cuComplex * B_native = NULL;
    int ldb_native = 0;
    size_t * bufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (cuComplex *)getPointer(env, dl);
    d_native = (cuComplex *)getPointer(env, d);
    du_native = (cuComplex *)getPointer(env, du);
    B_native = (cuComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    if (!initNative(env, bufferSizeInBytes, bufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgtsv2_bufferSizeExt(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native, bufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive
    if (!releaseNative(env, bufferSizeInBytes_native, bufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgtsv2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb, jlongArray bufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseZgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseZgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseZgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseZgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (bufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bufferSizeInBytes' is null for cusparseZgtsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgtsv2_bufferSizeExt(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d, bufferSizeInBytes=%p)\n",
        handle, m, n, dl, d, du, B, ldb, bufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuDoubleComplex * dl_native = NULL;
    cuDoubleComplex * d_native = NULL;
    cuDoubleComplex * du_native = NULL;
    cuDoubleComplex * B_native = NULL;
    int ldb_native = 0;
    size_t * bufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (cuDoubleComplex *)getPointer(env, dl);
    d_native = (cuDoubleComplex *)getPointer(env, d);
    du_native = (cuDoubleComplex *)getPointer(env, du);
    B_native = (cuDoubleComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    if (!initNative(env, bufferSizeInBytes, bufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgtsv2_bufferSizeExt(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native, bufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive
    if (!releaseNative(env, bufferSizeInBytes_native, bufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgtsv2Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseSgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseSgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseSgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseSgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgtsv2(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d, pBuffer=%p)\n",
        handle, m, n, dl, d, du, B, ldb, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    float * dl_native = NULL;
    float * d_native = NULL;
    float * du_native = NULL;
    float * B_native = NULL;
    int ldb_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (float *)getPointer(env, dl);
    d_native = (float *)getPointer(env, d);
    du_native = (float *)getPointer(env, du);
    B_native = (float *)getPointer(env, B);
    ldb_native = (int)ldb;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgtsv2(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgtsv2Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseDgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseDgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseDgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseDgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgtsv2(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d, pBuffer=%p)\n",
        handle, m, n, dl, d, du, B, ldb, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    double * dl_native = NULL;
    double * d_native = NULL;
    double * du_native = NULL;
    double * B_native = NULL;
    int ldb_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (double *)getPointer(env, dl);
    d_native = (double *)getPointer(env, d);
    du_native = (double *)getPointer(env, du);
    B_native = (double *)getPointer(env, B);
    ldb_native = (int)ldb;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgtsv2(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgtsv2Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseCgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseCgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseCgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseCgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgtsv2(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d, pBuffer=%p)\n",
        handle, m, n, dl, d, du, B, ldb, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuComplex * dl_native = NULL;
    cuComplex * d_native = NULL;
    cuComplex * du_native = NULL;
    cuComplex * B_native = NULL;
    int ldb_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (cuComplex *)getPointer(env, dl);
    d_native = (cuComplex *)getPointer(env, d);
    du_native = (cuComplex *)getPointer(env, du);
    B_native = (cuComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgtsv2(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgtsv2Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseZgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseZgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseZgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseZgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZgtsv2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgtsv2(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d, pBuffer=%p)\n",
        handle, m, n, dl, d, du, B, ldb, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuDoubleComplex * dl_native = NULL;
    cuDoubleComplex * d_native = NULL;
    cuDoubleComplex * du_native = NULL;
    cuDoubleComplex * B_native = NULL;
    int ldb_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (cuDoubleComplex *)getPointer(env, dl);
    d_native = (cuDoubleComplex *)getPointer(env, d);
    du_native = (cuDoubleComplex *)getPointer(env, du);
    B_native = (cuDoubleComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgtsv2(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgtsv2_1nopivot_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb, jlongArray bufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseSgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseSgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseSgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseSgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (bufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bufferSizeInBytes' is null for cusparseSgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgtsv2_nopivot_bufferSizeExt(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d, bufferSizeInBytes=%p)\n",
        handle, m, n, dl, d, du, B, ldb, bufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    float * dl_native = NULL;
    float * d_native = NULL;
    float * du_native = NULL;
    float * B_native = NULL;
    int ldb_native = 0;
    size_t * bufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (float *)getPointer(env, dl);
    d_native = (float *)getPointer(env, d);
    du_native = (float *)getPointer(env, du);
    B_native = (float *)getPointer(env, B);
    ldb_native = (int)ldb;
    if (!initNative(env, bufferSizeInBytes, bufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgtsv2_nopivot_bufferSizeExt(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native, bufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive
    if (!releaseNative(env, bufferSizeInBytes_native, bufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgtsv2_1nopivot_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb, jlongArray bufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseDgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseDgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseDgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseDgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (bufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bufferSizeInBytes' is null for cusparseDgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgtsv2_nopivot_bufferSizeExt(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d, bufferSizeInBytes=%p)\n",
        handle, m, n, dl, d, du, B, ldb, bufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    double * dl_native = NULL;
    double * d_native = NULL;
    double * du_native = NULL;
    double * B_native = NULL;
    int ldb_native = 0;
    size_t * bufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (double *)getPointer(env, dl);
    d_native = (double *)getPointer(env, d);
    du_native = (double *)getPointer(env, du);
    B_native = (double *)getPointer(env, B);
    ldb_native = (int)ldb;
    if (!initNative(env, bufferSizeInBytes, bufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgtsv2_nopivot_bufferSizeExt(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native, bufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive
    if (!releaseNative(env, bufferSizeInBytes_native, bufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgtsv2_1nopivot_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb, jlongArray bufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseCgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseCgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseCgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseCgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (bufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bufferSizeInBytes' is null for cusparseCgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgtsv2_nopivot_bufferSizeExt(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d, bufferSizeInBytes=%p)\n",
        handle, m, n, dl, d, du, B, ldb, bufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuComplex * dl_native = NULL;
    cuComplex * d_native = NULL;
    cuComplex * du_native = NULL;
    cuComplex * B_native = NULL;
    int ldb_native = 0;
    size_t * bufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (cuComplex *)getPointer(env, dl);
    d_native = (cuComplex *)getPointer(env, d);
    du_native = (cuComplex *)getPointer(env, du);
    B_native = (cuComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    if (!initNative(env, bufferSizeInBytes, bufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgtsv2_nopivot_bufferSizeExt(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native, bufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive
    if (!releaseNative(env, bufferSizeInBytes_native, bufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgtsv2_1nopivot_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb, jlongArray bufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseZgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseZgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseZgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseZgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (bufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bufferSizeInBytes' is null for cusparseZgtsv2_nopivot_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgtsv2_nopivot_bufferSizeExt(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d, bufferSizeInBytes=%p)\n",
        handle, m, n, dl, d, du, B, ldb, bufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuDoubleComplex * dl_native = NULL;
    cuDoubleComplex * d_native = NULL;
    cuDoubleComplex * du_native = NULL;
    cuDoubleComplex * B_native = NULL;
    int ldb_native = 0;
    size_t * bufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (cuDoubleComplex *)getPointer(env, dl);
    d_native = (cuDoubleComplex *)getPointer(env, d);
    du_native = (cuDoubleComplex *)getPointer(env, du);
    B_native = (cuDoubleComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    if (!initNative(env, bufferSizeInBytes, bufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgtsv2_nopivot_bufferSizeExt(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native, bufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive
    if (!releaseNative(env, bufferSizeInBytes_native, bufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgtsv2_1nopivotNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseSgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseSgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseSgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseSgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgtsv2_nopivot(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d, pBuffer=%p)\n",
        handle, m, n, dl, d, du, B, ldb, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    float * dl_native = NULL;
    float * d_native = NULL;
    float * du_native = NULL;
    float * B_native = NULL;
    int ldb_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (float *)getPointer(env, dl);
    d_native = (float *)getPointer(env, d);
    du_native = (float *)getPointer(env, du);
    B_native = (float *)getPointer(env, B);
    ldb_native = (int)ldb;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgtsv2_nopivot(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgtsv2_1nopivotNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseDgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseDgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseDgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseDgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgtsv2_nopivot(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d, pBuffer=%p)\n",
        handle, m, n, dl, d, du, B, ldb, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    double * dl_native = NULL;
    double * d_native = NULL;
    double * du_native = NULL;
    double * B_native = NULL;
    int ldb_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (double *)getPointer(env, dl);
    d_native = (double *)getPointer(env, d);
    du_native = (double *)getPointer(env, du);
    B_native = (double *)getPointer(env, B);
    ldb_native = (int)ldb;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgtsv2_nopivot(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgtsv2_1nopivotNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseCgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseCgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseCgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseCgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgtsv2_nopivot(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d, pBuffer=%p)\n",
        handle, m, n, dl, d, du, B, ldb, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuComplex * dl_native = NULL;
    cuComplex * d_native = NULL;
    cuComplex * du_native = NULL;
    cuComplex * B_native = NULL;
    int ldb_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (cuComplex *)getPointer(env, dl);
    d_native = (cuComplex *)getPointer(env, d);
    du_native = (cuComplex *)getPointer(env, du);
    B_native = (cuComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgtsv2_nopivot(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgtsv2_1nopivotNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseZgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseZgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseZgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseZgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZgtsv2_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgtsv2_nopivot(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d, pBuffer=%p)\n",
        handle, m, n, dl, d, du, B, ldb, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuDoubleComplex * dl_native = NULL;
    cuDoubleComplex * d_native = NULL;
    cuDoubleComplex * du_native = NULL;
    cuDoubleComplex * B_native = NULL;
    int ldb_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (cuDoubleComplex *)getPointer(env, dl);
    d_native = (cuDoubleComplex *)getPointer(env, d);
    du_native = (cuDoubleComplex *)getPointer(env, du);
    B_native = (cuDoubleComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgtsv2_nopivot(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgtsv2StridedBatch_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jobject dl, jobject d, jobject du, jobject x, jint batchCount, jint batchStride, jlongArray bufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseSgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseSgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseSgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseSgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    // batchStride is primitive
    if (bufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bufferSizeInBytes' is null for cusparseSgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgtsv2StridedBatch_bufferSizeExt(handle=%p, m=%d, dl=%p, d=%p, du=%p, x=%p, batchCount=%d, batchStride=%d, bufferSizeInBytes=%p)\n",
        handle, m, dl, d, du, x, batchCount, batchStride, bufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    float * dl_native = NULL;
    float * d_native = NULL;
    float * du_native = NULL;
    float * x_native = NULL;
    int batchCount_native = 0;
    int batchStride_native = 0;
    size_t * bufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    dl_native = (float *)getPointer(env, dl);
    d_native = (float *)getPointer(env, d);
    du_native = (float *)getPointer(env, du);
    x_native = (float *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    batchStride_native = (int)batchStride;
    if (!initNative(env, bufferSizeInBytes, bufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgtsv2StridedBatch_bufferSizeExt(handle_native, m_native, dl_native, d_native, du_native, x_native, batchCount_native, batchStride_native, bufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // x is a native pointer
    // batchCount is primitive
    // batchStride is primitive
    if (!releaseNative(env, bufferSizeInBytes_native, bufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgtsv2StridedBatch_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jobject dl, jobject d, jobject du, jobject x, jint batchCount, jint batchStride, jlongArray bufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseDgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseDgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseDgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseDgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    // batchStride is primitive
    if (bufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bufferSizeInBytes' is null for cusparseDgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgtsv2StridedBatch_bufferSizeExt(handle=%p, m=%d, dl=%p, d=%p, du=%p, x=%p, batchCount=%d, batchStride=%d, bufferSizeInBytes=%p)\n",
        handle, m, dl, d, du, x, batchCount, batchStride, bufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    double * dl_native = NULL;
    double * d_native = NULL;
    double * du_native = NULL;
    double * x_native = NULL;
    int batchCount_native = 0;
    int batchStride_native = 0;
    size_t * bufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    dl_native = (double *)getPointer(env, dl);
    d_native = (double *)getPointer(env, d);
    du_native = (double *)getPointer(env, du);
    x_native = (double *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    batchStride_native = (int)batchStride;
    if (!initNative(env, bufferSizeInBytes, bufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgtsv2StridedBatch_bufferSizeExt(handle_native, m_native, dl_native, d_native, du_native, x_native, batchCount_native, batchStride_native, bufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // x is a native pointer
    // batchCount is primitive
    // batchStride is primitive
    if (!releaseNative(env, bufferSizeInBytes_native, bufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgtsv2StridedBatch_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jobject dl, jobject d, jobject du, jobject x, jint batchCount, jint batchStride, jlongArray bufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseCgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseCgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseCgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseCgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    // batchStride is primitive
    if (bufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bufferSizeInBytes' is null for cusparseCgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgtsv2StridedBatch_bufferSizeExt(handle=%p, m=%d, dl=%p, d=%p, du=%p, x=%p, batchCount=%d, batchStride=%d, bufferSizeInBytes=%p)\n",
        handle, m, dl, d, du, x, batchCount, batchStride, bufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    cuComplex * dl_native = NULL;
    cuComplex * d_native = NULL;
    cuComplex * du_native = NULL;
    cuComplex * x_native = NULL;
    int batchCount_native = 0;
    int batchStride_native = 0;
    size_t * bufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    dl_native = (cuComplex *)getPointer(env, dl);
    d_native = (cuComplex *)getPointer(env, d);
    du_native = (cuComplex *)getPointer(env, du);
    x_native = (cuComplex *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    batchStride_native = (int)batchStride;
    if (!initNative(env, bufferSizeInBytes, bufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgtsv2StridedBatch_bufferSizeExt(handle_native, m_native, dl_native, d_native, du_native, x_native, batchCount_native, batchStride_native, bufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // x is a native pointer
    // batchCount is primitive
    // batchStride is primitive
    if (!releaseNative(env, bufferSizeInBytes_native, bufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgtsv2StridedBatch_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jobject dl, jobject d, jobject du, jobject x, jint batchCount, jint batchStride, jlongArray bufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseZgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseZgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseZgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseZgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    // batchStride is primitive
    if (bufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bufferSizeInBytes' is null for cusparseZgtsv2StridedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgtsv2StridedBatch_bufferSizeExt(handle=%p, m=%d, dl=%p, d=%p, du=%p, x=%p, batchCount=%d, batchStride=%d, bufferSizeInBytes=%p)\n",
        handle, m, dl, d, du, x, batchCount, batchStride, bufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    cuDoubleComplex * dl_native = NULL;
    cuDoubleComplex * d_native = NULL;
    cuDoubleComplex * du_native = NULL;
    cuDoubleComplex * x_native = NULL;
    int batchCount_native = 0;
    int batchStride_native = 0;
    size_t * bufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    dl_native = (cuDoubleComplex *)getPointer(env, dl);
    d_native = (cuDoubleComplex *)getPointer(env, d);
    du_native = (cuDoubleComplex *)getPointer(env, du);
    x_native = (cuDoubleComplex *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    batchStride_native = (int)batchStride;
    if (!initNative(env, bufferSizeInBytes, bufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgtsv2StridedBatch_bufferSizeExt(handle_native, m_native, dl_native, d_native, du_native, x_native, batchCount_native, batchStride_native, bufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // x is a native pointer
    // batchCount is primitive
    // batchStride is primitive
    if (!releaseNative(env, bufferSizeInBytes_native, bufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgtsv2StridedBatchNative(JNIEnv *env, jclass cls, jobject handle, jint m, jobject dl, jobject d, jobject du, jobject x, jint batchCount, jint batchStride, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseSgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseSgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseSgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseSgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    // batchStride is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgtsv2StridedBatch(handle=%p, m=%d, dl=%p, d=%p, du=%p, x=%p, batchCount=%d, batchStride=%d, pBuffer=%p)\n",
        handle, m, dl, d, du, x, batchCount, batchStride, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    float * dl_native = NULL;
    float * d_native = NULL;
    float * du_native = NULL;
    float * x_native = NULL;
    int batchCount_native = 0;
    int batchStride_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    dl_native = (float *)getPointer(env, dl);
    d_native = (float *)getPointer(env, d);
    du_native = (float *)getPointer(env, du);
    x_native = (float *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    batchStride_native = (int)batchStride;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgtsv2StridedBatch(handle_native, m_native, dl_native, d_native, du_native, x_native, batchCount_native, batchStride_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // x is a native pointer
    // batchCount is primitive
    // batchStride is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgtsv2StridedBatchNative(JNIEnv *env, jclass cls, jobject handle, jint m, jobject dl, jobject d, jobject du, jobject x, jint batchCount, jint batchStride, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseDgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseDgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseDgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseDgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    // batchStride is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgtsv2StridedBatch(handle=%p, m=%d, dl=%p, d=%p, du=%p, x=%p, batchCount=%d, batchStride=%d, pBuffer=%p)\n",
        handle, m, dl, d, du, x, batchCount, batchStride, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    double * dl_native = NULL;
    double * d_native = NULL;
    double * du_native = NULL;
    double * x_native = NULL;
    int batchCount_native = 0;
    int batchStride_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    dl_native = (double *)getPointer(env, dl);
    d_native = (double *)getPointer(env, d);
    du_native = (double *)getPointer(env, du);
    x_native = (double *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    batchStride_native = (int)batchStride;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgtsv2StridedBatch(handle_native, m_native, dl_native, d_native, du_native, x_native, batchCount_native, batchStride_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // x is a native pointer
    // batchCount is primitive
    // batchStride is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgtsv2StridedBatchNative(JNIEnv *env, jclass cls, jobject handle, jint m, jobject dl, jobject d, jobject du, jobject x, jint batchCount, jint batchStride, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseCgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseCgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseCgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseCgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    // batchStride is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgtsv2StridedBatch(handle=%p, m=%d, dl=%p, d=%p, du=%p, x=%p, batchCount=%d, batchStride=%d, pBuffer=%p)\n",
        handle, m, dl, d, du, x, batchCount, batchStride, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    cuComplex * dl_native = NULL;
    cuComplex * d_native = NULL;
    cuComplex * du_native = NULL;
    cuComplex * x_native = NULL;
    int batchCount_native = 0;
    int batchStride_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    dl_native = (cuComplex *)getPointer(env, dl);
    d_native = (cuComplex *)getPointer(env, d);
    du_native = (cuComplex *)getPointer(env, du);
    x_native = (cuComplex *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    batchStride_native = (int)batchStride;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgtsv2StridedBatch(handle_native, m_native, dl_native, d_native, du_native, x_native, batchCount_native, batchStride_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // x is a native pointer
    // batchCount is primitive
    // batchStride is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgtsv2StridedBatchNative(JNIEnv *env, jclass cls, jobject handle, jint m, jobject dl, jobject d, jobject du, jobject x, jint batchCount, jint batchStride, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseZgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseZgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseZgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseZgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    // batchStride is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZgtsv2StridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgtsv2StridedBatch(handle=%p, m=%d, dl=%p, d=%p, du=%p, x=%p, batchCount=%d, batchStride=%d, pBuffer=%p)\n",
        handle, m, dl, d, du, x, batchCount, batchStride, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    cuDoubleComplex * dl_native = NULL;
    cuDoubleComplex * d_native = NULL;
    cuDoubleComplex * du_native = NULL;
    cuDoubleComplex * x_native = NULL;
    int batchCount_native = 0;
    int batchStride_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    dl_native = (cuDoubleComplex *)getPointer(env, dl);
    d_native = (cuDoubleComplex *)getPointer(env, d);
    du_native = (cuDoubleComplex *)getPointer(env, du);
    x_native = (cuDoubleComplex *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    batchStride_native = (int)batchStride;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgtsv2StridedBatch(handle_native, m_native, dl_native, d_native, du_native, x_native, batchCount_native, batchStride_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // x is a native pointer
    // batchCount is primitive
    // batchStride is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgtsvInterleavedBatch_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint algo, jint m, jobject dl, jobject d, jobject du, jobject x, jint batchCount, jlongArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // algo is primitive
    // m is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseSgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseSgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseSgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseSgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseSgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgtsvInterleavedBatch_bufferSizeExt(handle=%p, algo=%d, m=%d, dl=%p, d=%p, du=%p, x=%p, batchCount=%d, pBufferSizeInBytes=%p)\n",
        handle, algo, m, dl, d, du, x, batchCount, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int algo_native = 0;
    int m_native = 0;
    float * dl_native = NULL;
    float * d_native = NULL;
    float * du_native = NULL;
    float * x_native = NULL;
    int batchCount_native = 0;
    size_t * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    algo_native = (int)algo;
    m_native = (int)m;
    dl_native = (float *)getPointer(env, dl);
    d_native = (float *)getPointer(env, d);
    du_native = (float *)getPointer(env, du);
    x_native = (float *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgtsvInterleavedBatch_bufferSizeExt(handle_native, algo_native, m_native, dl_native, d_native, du_native, x_native, batchCount_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // algo is primitive
    // m is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // x is a native pointer
    // batchCount is primitive
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgtsvInterleavedBatch_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint algo, jint m, jobject dl, jobject d, jobject du, jobject x, jint batchCount, jlongArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // algo is primitive
    // m is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseDgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseDgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseDgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseDgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseDgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgtsvInterleavedBatch_bufferSizeExt(handle=%p, algo=%d, m=%d, dl=%p, d=%p, du=%p, x=%p, batchCount=%d, pBufferSizeInBytes=%p)\n",
        handle, algo, m, dl, d, du, x, batchCount, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int algo_native = 0;
    int m_native = 0;
    double * dl_native = NULL;
    double * d_native = NULL;
    double * du_native = NULL;
    double * x_native = NULL;
    int batchCount_native = 0;
    size_t * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    algo_native = (int)algo;
    m_native = (int)m;
    dl_native = (double *)getPointer(env, dl);
    d_native = (double *)getPointer(env, d);
    du_native = (double *)getPointer(env, du);
    x_native = (double *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgtsvInterleavedBatch_bufferSizeExt(handle_native, algo_native, m_native, dl_native, d_native, du_native, x_native, batchCount_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // algo is primitive
    // m is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // x is a native pointer
    // batchCount is primitive
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgtsvInterleavedBatch_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint algo, jint m, jobject dl, jobject d, jobject du, jobject x, jint batchCount, jlongArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // algo is primitive
    // m is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseCgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseCgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseCgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseCgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseCgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgtsvInterleavedBatch_bufferSizeExt(handle=%p, algo=%d, m=%d, dl=%p, d=%p, du=%p, x=%p, batchCount=%d, pBufferSizeInBytes=%p)\n",
        handle, algo, m, dl, d, du, x, batchCount, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int algo_native = 0;
    int m_native = 0;
    cuComplex * dl_native = NULL;
    cuComplex * d_native = NULL;
    cuComplex * du_native = NULL;
    cuComplex * x_native = NULL;
    int batchCount_native = 0;
    size_t * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    algo_native = (int)algo;
    m_native = (int)m;
    dl_native = (cuComplex *)getPointer(env, dl);
    d_native = (cuComplex *)getPointer(env, d);
    du_native = (cuComplex *)getPointer(env, du);
    x_native = (cuComplex *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgtsvInterleavedBatch_bufferSizeExt(handle_native, algo_native, m_native, dl_native, d_native, du_native, x_native, batchCount_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // algo is primitive
    // m is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // x is a native pointer
    // batchCount is primitive
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgtsvInterleavedBatch_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint algo, jint m, jobject dl, jobject d, jobject du, jobject x, jint batchCount, jlongArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // algo is primitive
    // m is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseZgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseZgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseZgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseZgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseZgtsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgtsvInterleavedBatch_bufferSizeExt(handle=%p, algo=%d, m=%d, dl=%p, d=%p, du=%p, x=%p, batchCount=%d, pBufferSizeInBytes=%p)\n",
        handle, algo, m, dl, d, du, x, batchCount, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int algo_native = 0;
    int m_native = 0;
    cuDoubleComplex * dl_native = NULL;
    cuDoubleComplex * d_native = NULL;
    cuDoubleComplex * du_native = NULL;
    cuDoubleComplex * x_native = NULL;
    int batchCount_native = 0;
    size_t * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    algo_native = (int)algo;
    m_native = (int)m;
    dl_native = (cuDoubleComplex *)getPointer(env, dl);
    d_native = (cuDoubleComplex *)getPointer(env, d);
    du_native = (cuDoubleComplex *)getPointer(env, du);
    x_native = (cuDoubleComplex *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgtsvInterleavedBatch_bufferSizeExt(handle_native, algo_native, m_native, dl_native, d_native, du_native, x_native, batchCount_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // algo is primitive
    // m is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // x is a native pointer
    // batchCount is primitive
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgtsvInterleavedBatchNative(JNIEnv *env, jclass cls, jobject handle, jint algo, jint m, jobject dl, jobject d, jobject du, jobject x, jint batchCount, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // algo is primitive
    // m is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseSgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseSgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseSgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseSgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgtsvInterleavedBatch(handle=%p, algo=%d, m=%d, dl=%p, d=%p, du=%p, x=%p, batchCount=%d, pBuffer=%p)\n",
        handle, algo, m, dl, d, du, x, batchCount, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int algo_native = 0;
    int m_native = 0;
    float * dl_native = NULL;
    float * d_native = NULL;
    float * du_native = NULL;
    float * x_native = NULL;
    int batchCount_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    algo_native = (int)algo;
    m_native = (int)m;
    dl_native = (float *)getPointer(env, dl);
    d_native = (float *)getPointer(env, d);
    du_native = (float *)getPointer(env, du);
    x_native = (float *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgtsvInterleavedBatch(handle_native, algo_native, m_native, dl_native, d_native, du_native, x_native, batchCount_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // algo is primitive
    // m is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // x is a native pointer
    // batchCount is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgtsvInterleavedBatchNative(JNIEnv *env, jclass cls, jobject handle, jint algo, jint m, jobject dl, jobject d, jobject du, jobject x, jint batchCount, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // algo is primitive
    // m is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseDgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseDgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseDgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseDgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgtsvInterleavedBatch(handle=%p, algo=%d, m=%d, dl=%p, d=%p, du=%p, x=%p, batchCount=%d, pBuffer=%p)\n",
        handle, algo, m, dl, d, du, x, batchCount, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int algo_native = 0;
    int m_native = 0;
    double * dl_native = NULL;
    double * d_native = NULL;
    double * du_native = NULL;
    double * x_native = NULL;
    int batchCount_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    algo_native = (int)algo;
    m_native = (int)m;
    dl_native = (double *)getPointer(env, dl);
    d_native = (double *)getPointer(env, d);
    du_native = (double *)getPointer(env, du);
    x_native = (double *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgtsvInterleavedBatch(handle_native, algo_native, m_native, dl_native, d_native, du_native, x_native, batchCount_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // algo is primitive
    // m is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // x is a native pointer
    // batchCount is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgtsvInterleavedBatchNative(JNIEnv *env, jclass cls, jobject handle, jint algo, jint m, jobject dl, jobject d, jobject du, jobject x, jint batchCount, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // algo is primitive
    // m is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseCgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseCgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseCgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseCgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgtsvInterleavedBatch(handle=%p, algo=%d, m=%d, dl=%p, d=%p, du=%p, x=%p, batchCount=%d, pBuffer=%p)\n",
        handle, algo, m, dl, d, du, x, batchCount, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int algo_native = 0;
    int m_native = 0;
    cuComplex * dl_native = NULL;
    cuComplex * d_native = NULL;
    cuComplex * du_native = NULL;
    cuComplex * x_native = NULL;
    int batchCount_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    algo_native = (int)algo;
    m_native = (int)m;
    dl_native = (cuComplex *)getPointer(env, dl);
    d_native = (cuComplex *)getPointer(env, d);
    du_native = (cuComplex *)getPointer(env, du);
    x_native = (cuComplex *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgtsvInterleavedBatch(handle_native, algo_native, m_native, dl_native, d_native, du_native, x_native, batchCount_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // algo is primitive
    // m is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // x is a native pointer
    // batchCount is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgtsvInterleavedBatchNative(JNIEnv *env, jclass cls, jobject handle, jint algo, jint m, jobject dl, jobject d, jobject du, jobject x, jint batchCount, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // algo is primitive
    // m is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseZgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseZgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseZgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseZgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZgtsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgtsvInterleavedBatch(handle=%p, algo=%d, m=%d, dl=%p, d=%p, du=%p, x=%p, batchCount=%d, pBuffer=%p)\n",
        handle, algo, m, dl, d, du, x, batchCount, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int algo_native = 0;
    int m_native = 0;
    cuDoubleComplex * dl_native = NULL;
    cuDoubleComplex * d_native = NULL;
    cuDoubleComplex * du_native = NULL;
    cuDoubleComplex * x_native = NULL;
    int batchCount_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    algo_native = (int)algo;
    m_native = (int)m;
    dl_native = (cuDoubleComplex *)getPointer(env, dl);
    d_native = (cuDoubleComplex *)getPointer(env, d);
    du_native = (cuDoubleComplex *)getPointer(env, du);
    x_native = (cuDoubleComplex *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgtsvInterleavedBatch(handle_native, algo_native, m_native, dl_native, d_native, du_native, x_native, batchCount_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // algo is primitive
    // m is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // x is a native pointer
    // batchCount is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgpsvInterleavedBatch_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint algo, jint m, jobject ds, jobject dl, jobject d, jobject du, jobject dw, jobject x, jint batchCount, jlongArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // algo is primitive
    // m is primitive
    if (ds == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ds' is null for cusparseSgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseSgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseSgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseSgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (dw == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dw' is null for cusparseSgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseSgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseSgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgpsvInterleavedBatch_bufferSizeExt(handle=%p, algo=%d, m=%d, ds=%p, dl=%p, d=%p, du=%p, dw=%p, x=%p, batchCount=%d, pBufferSizeInBytes=%p)\n",
        handle, algo, m, ds, dl, d, du, dw, x, batchCount, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int algo_native = 0;
    int m_native = 0;
    float * ds_native = NULL;
    float * dl_native = NULL;
    float * d_native = NULL;
    float * du_native = NULL;
    float * dw_native = NULL;
    float * x_native = NULL;
    int batchCount_native = 0;
    size_t * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    algo_native = (int)algo;
    m_native = (int)m;
    ds_native = (float *)getPointer(env, ds);
    dl_native = (float *)getPointer(env, dl);
    d_native = (float *)getPointer(env, d);
    du_native = (float *)getPointer(env, du);
    dw_native = (float *)getPointer(env, dw);
    x_native = (float *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgpsvInterleavedBatch_bufferSizeExt(handle_native, algo_native, m_native, ds_native, dl_native, d_native, du_native, dw_native, x_native, batchCount_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // algo is primitive
    // m is primitive
    // ds is a native pointer
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // dw is a native pointer
    // x is a native pointer
    // batchCount is primitive
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgpsvInterleavedBatch_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint algo, jint m, jobject ds, jobject dl, jobject d, jobject du, jobject dw, jobject x, jint batchCount, jlongArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // algo is primitive
    // m is primitive
    if (ds == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ds' is null for cusparseDgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseDgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseDgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseDgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (dw == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dw' is null for cusparseDgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseDgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseDgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgpsvInterleavedBatch_bufferSizeExt(handle=%p, algo=%d, m=%d, ds=%p, dl=%p, d=%p, du=%p, dw=%p, x=%p, batchCount=%d, pBufferSizeInBytes=%p)\n",
        handle, algo, m, ds, dl, d, du, dw, x, batchCount, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int algo_native = 0;
    int m_native = 0;
    double * ds_native = NULL;
    double * dl_native = NULL;
    double * d_native = NULL;
    double * du_native = NULL;
    double * dw_native = NULL;
    double * x_native = NULL;
    int batchCount_native = 0;
    size_t * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    algo_native = (int)algo;
    m_native = (int)m;
    ds_native = (double *)getPointer(env, ds);
    dl_native = (double *)getPointer(env, dl);
    d_native = (double *)getPointer(env, d);
    du_native = (double *)getPointer(env, du);
    dw_native = (double *)getPointer(env, dw);
    x_native = (double *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgpsvInterleavedBatch_bufferSizeExt(handle_native, algo_native, m_native, ds_native, dl_native, d_native, du_native, dw_native, x_native, batchCount_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // algo is primitive
    // m is primitive
    // ds is a native pointer
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // dw is a native pointer
    // x is a native pointer
    // batchCount is primitive
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgpsvInterleavedBatch_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint algo, jint m, jobject ds, jobject dl, jobject d, jobject du, jobject dw, jobject x, jint batchCount, jlongArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // algo is primitive
    // m is primitive
    if (ds == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ds' is null for cusparseCgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseCgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseCgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseCgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (dw == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dw' is null for cusparseCgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseCgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseCgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgpsvInterleavedBatch_bufferSizeExt(handle=%p, algo=%d, m=%d, ds=%p, dl=%p, d=%p, du=%p, dw=%p, x=%p, batchCount=%d, pBufferSizeInBytes=%p)\n",
        handle, algo, m, ds, dl, d, du, dw, x, batchCount, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int algo_native = 0;
    int m_native = 0;
    cuComplex * ds_native = NULL;
    cuComplex * dl_native = NULL;
    cuComplex * d_native = NULL;
    cuComplex * du_native = NULL;
    cuComplex * dw_native = NULL;
    cuComplex * x_native = NULL;
    int batchCount_native = 0;
    size_t * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    algo_native = (int)algo;
    m_native = (int)m;
    ds_native = (cuComplex *)getPointer(env, ds);
    dl_native = (cuComplex *)getPointer(env, dl);
    d_native = (cuComplex *)getPointer(env, d);
    du_native = (cuComplex *)getPointer(env, du);
    dw_native = (cuComplex *)getPointer(env, dw);
    x_native = (cuComplex *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgpsvInterleavedBatch_bufferSizeExt(handle_native, algo_native, m_native, ds_native, dl_native, d_native, du_native, dw_native, x_native, batchCount_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // algo is primitive
    // m is primitive
    // ds is a native pointer
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // dw is a native pointer
    // x is a native pointer
    // batchCount is primitive
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgpsvInterleavedBatch_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint algo, jint m, jobject ds, jobject dl, jobject d, jobject du, jobject dw, jobject x, jint batchCount, jlongArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // algo is primitive
    // m is primitive
    if (ds == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ds' is null for cusparseZgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseZgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseZgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseZgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (dw == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dw' is null for cusparseZgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseZgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseZgpsvInterleavedBatch_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgpsvInterleavedBatch_bufferSizeExt(handle=%p, algo=%d, m=%d, ds=%p, dl=%p, d=%p, du=%p, dw=%p, x=%p, batchCount=%d, pBufferSizeInBytes=%p)\n",
        handle, algo, m, ds, dl, d, du, dw, x, batchCount, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int algo_native = 0;
    int m_native = 0;
    cuDoubleComplex * ds_native = NULL;
    cuDoubleComplex * dl_native = NULL;
    cuDoubleComplex * d_native = NULL;
    cuDoubleComplex * du_native = NULL;
    cuDoubleComplex * dw_native = NULL;
    cuDoubleComplex * x_native = NULL;
    int batchCount_native = 0;
    size_t * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    algo_native = (int)algo;
    m_native = (int)m;
    ds_native = (cuDoubleComplex *)getPointer(env, ds);
    dl_native = (cuDoubleComplex *)getPointer(env, dl);
    d_native = (cuDoubleComplex *)getPointer(env, d);
    du_native = (cuDoubleComplex *)getPointer(env, du);
    dw_native = (cuDoubleComplex *)getPointer(env, dw);
    x_native = (cuDoubleComplex *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgpsvInterleavedBatch_bufferSizeExt(handle_native, algo_native, m_native, ds_native, dl_native, d_native, du_native, dw_native, x_native, batchCount_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // algo is primitive
    // m is primitive
    // ds is a native pointer
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // dw is a native pointer
    // x is a native pointer
    // batchCount is primitive
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgpsvInterleavedBatchNative(JNIEnv *env, jclass cls, jobject handle, jint algo, jint m, jobject ds, jobject dl, jobject d, jobject du, jobject dw, jobject x, jint batchCount, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // algo is primitive
    // m is primitive
    if (ds == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ds' is null for cusparseSgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseSgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseSgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseSgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (dw == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dw' is null for cusparseSgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseSgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgpsvInterleavedBatch(handle=%p, algo=%d, m=%d, ds=%p, dl=%p, d=%p, du=%p, dw=%p, x=%p, batchCount=%d, pBuffer=%p)\n",
        handle, algo, m, ds, dl, d, du, dw, x, batchCount, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int algo_native = 0;
    int m_native = 0;
    float * ds_native = NULL;
    float * dl_native = NULL;
    float * d_native = NULL;
    float * du_native = NULL;
    float * dw_native = NULL;
    float * x_native = NULL;
    int batchCount_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    algo_native = (int)algo;
    m_native = (int)m;
    ds_native = (float *)getPointer(env, ds);
    dl_native = (float *)getPointer(env, dl);
    d_native = (float *)getPointer(env, d);
    du_native = (float *)getPointer(env, du);
    dw_native = (float *)getPointer(env, dw);
    x_native = (float *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgpsvInterleavedBatch(handle_native, algo_native, m_native, ds_native, dl_native, d_native, du_native, dw_native, x_native, batchCount_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // algo is primitive
    // m is primitive
    // ds is a native pointer
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // dw is a native pointer
    // x is a native pointer
    // batchCount is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgpsvInterleavedBatchNative(JNIEnv *env, jclass cls, jobject handle, jint algo, jint m, jobject ds, jobject dl, jobject d, jobject du, jobject dw, jobject x, jint batchCount, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // algo is primitive
    // m is primitive
    if (ds == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ds' is null for cusparseDgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseDgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseDgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseDgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (dw == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dw' is null for cusparseDgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseDgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgpsvInterleavedBatch(handle=%p, algo=%d, m=%d, ds=%p, dl=%p, d=%p, du=%p, dw=%p, x=%p, batchCount=%d, pBuffer=%p)\n",
        handle, algo, m, ds, dl, d, du, dw, x, batchCount, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int algo_native = 0;
    int m_native = 0;
    double * ds_native = NULL;
    double * dl_native = NULL;
    double * d_native = NULL;
    double * du_native = NULL;
    double * dw_native = NULL;
    double * x_native = NULL;
    int batchCount_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    algo_native = (int)algo;
    m_native = (int)m;
    ds_native = (double *)getPointer(env, ds);
    dl_native = (double *)getPointer(env, dl);
    d_native = (double *)getPointer(env, d);
    du_native = (double *)getPointer(env, du);
    dw_native = (double *)getPointer(env, dw);
    x_native = (double *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgpsvInterleavedBatch(handle_native, algo_native, m_native, ds_native, dl_native, d_native, du_native, dw_native, x_native, batchCount_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // algo is primitive
    // m is primitive
    // ds is a native pointer
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // dw is a native pointer
    // x is a native pointer
    // batchCount is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgpsvInterleavedBatchNative(JNIEnv *env, jclass cls, jobject handle, jint algo, jint m, jobject ds, jobject dl, jobject d, jobject du, jobject dw, jobject x, jint batchCount, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // algo is primitive
    // m is primitive
    if (ds == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ds' is null for cusparseCgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseCgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseCgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseCgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (dw == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dw' is null for cusparseCgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseCgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgpsvInterleavedBatch(handle=%p, algo=%d, m=%d, ds=%p, dl=%p, d=%p, du=%p, dw=%p, x=%p, batchCount=%d, pBuffer=%p)\n",
        handle, algo, m, ds, dl, d, du, dw, x, batchCount, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int algo_native = 0;
    int m_native = 0;
    cuComplex * ds_native = NULL;
    cuComplex * dl_native = NULL;
    cuComplex * d_native = NULL;
    cuComplex * du_native = NULL;
    cuComplex * dw_native = NULL;
    cuComplex * x_native = NULL;
    int batchCount_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    algo_native = (int)algo;
    m_native = (int)m;
    ds_native = (cuComplex *)getPointer(env, ds);
    dl_native = (cuComplex *)getPointer(env, dl);
    d_native = (cuComplex *)getPointer(env, d);
    du_native = (cuComplex *)getPointer(env, du);
    dw_native = (cuComplex *)getPointer(env, dw);
    x_native = (cuComplex *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgpsvInterleavedBatch(handle_native, algo_native, m_native, ds_native, dl_native, d_native, du_native, dw_native, x_native, batchCount_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // algo is primitive
    // m is primitive
    // ds is a native pointer
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // dw is a native pointer
    // x is a native pointer
    // batchCount is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgpsvInterleavedBatchNative(JNIEnv *env, jclass cls, jobject handle, jint algo, jint m, jobject ds, jobject dl, jobject d, jobject du, jobject dw, jobject x, jint batchCount, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // algo is primitive
    // m is primitive
    if (ds == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ds' is null for cusparseZgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseZgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseZgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseZgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (dw == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dw' is null for cusparseZgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseZgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZgpsvInterleavedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgpsvInterleavedBatch(handle=%p, algo=%d, m=%d, ds=%p, dl=%p, d=%p, du=%p, dw=%p, x=%p, batchCount=%d, pBuffer=%p)\n",
        handle, algo, m, ds, dl, d, du, dw, x, batchCount, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int algo_native = 0;
    int m_native = 0;
    cuDoubleComplex * ds_native = NULL;
    cuDoubleComplex * dl_native = NULL;
    cuDoubleComplex * d_native = NULL;
    cuDoubleComplex * du_native = NULL;
    cuDoubleComplex * dw_native = NULL;
    cuDoubleComplex * x_native = NULL;
    int batchCount_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    algo_native = (int)algo;
    m_native = (int)m;
    ds_native = (cuDoubleComplex *)getPointer(env, ds);
    dl_native = (cuDoubleComplex *)getPointer(env, dl);
    d_native = (cuDoubleComplex *)getPointer(env, d);
    du_native = (cuDoubleComplex *)getPointer(env, du);
    dw_native = (cuDoubleComplex *)getPointer(env, dw);
    x_native = (cuDoubleComplex *)getPointer(env, x);
    batchCount_native = (int)batchCount;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgpsvInterleavedBatch(handle_native, algo_native, m_native, ds_native, dl_native, d_native, du_native, dw_native, x_native, batchCount_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // algo is primitive
    // m is primitive
    // ds is a native pointer
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // dw is a native pointer
    // x is a native pointer
    // batchCount is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

//##############################################################################
//# EXTRA ROUTINES
//##############################################################################
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrgeam2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject alpha, jobject descrA, jint nnzA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject beta, jobject descrB, jint nnzB, jobject csrSortedValB, jobject csrSortedRowPtrB, jobject csrSortedColIndB, jobject descrC, jobject csrSortedValC, jobject csrSortedRowPtrC, jobject csrSortedColIndC, jlongArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseScsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzA is primitive
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseScsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrB' is null for cusparseScsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzB is primitive
    if (csrSortedValB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValB' is null for cusparseScsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrB' is null for cusparseScsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndB' is null for cusparseScsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseScsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseScsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseScsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseScsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseScsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrgeam2_bufferSizeExt(handle=%p, m=%d, n=%d, alpha=%p, descrA=%p, nnzA=%d, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, beta=%p, descrB=%p, nnzB=%d, csrSortedValB=%p, csrSortedRowPtrB=%p, csrSortedColIndB=%p, descrC=%p, csrSortedValC=%p, csrSortedRowPtrC=%p, csrSortedColIndC=%p, pBufferSizeInBytes=%p)\n",
        handle, m, n, alpha, descrA, nnzA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, beta, descrB, nnzB, csrSortedValB, csrSortedRowPtrB, csrSortedColIndB, descrC, csrSortedValC, csrSortedRowPtrC, csrSortedColIndC, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    float * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    int nnzA_native = 0;
    float * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    float * beta_native = NULL;
    cusparseMatDescr_t descrB_native;
    int nnzB_native = 0;
    float * csrSortedValB_native = NULL;
    int * csrSortedRowPtrB_native = NULL;
    int * csrSortedColIndB_native = NULL;
    cusparseMatDescr_t descrC_native;
    float * csrSortedValC_native = NULL;
    int * csrSortedRowPtrC_native = NULL;
    int * csrSortedColIndC_native = NULL;
    size_t * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    nnzA_native = (int)nnzA;
    csrSortedValA_native = (float *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (float *)beta_pointerData->getPointer(env);
    descrB_native = (cusparseMatDescr_t)getNativePointerValue(env, descrB);
    nnzB_native = (int)nnzB;
    csrSortedValB_native = (float *)getPointer(env, csrSortedValB);
    csrSortedRowPtrB_native = (int *)getPointer(env, csrSortedRowPtrB);
    csrSortedColIndB_native = (int *)getPointer(env, csrSortedColIndB);
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedValC_native = (float *)getPointer(env, csrSortedValC);
    csrSortedRowPtrC_native = (int *)getPointer(env, csrSortedRowPtrC);
    csrSortedColIndC_native = (int *)getPointer(env, csrSortedColIndC);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrgeam2_bufferSizeExt(handle_native, m_native, n_native, alpha_native, descrA_native, nnzA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, beta_native, descrB_native, nnzB_native, csrSortedValB_native, csrSortedRowPtrB_native, csrSortedColIndB_native, descrC_native, csrSortedValC_native, csrSortedRowPtrC_native, csrSortedColIndC_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // nnzA is primitive
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrB is read-only
    // nnzB is primitive
    // csrSortedValB is a native pointer
    // csrSortedRowPtrB is a native pointer
    // csrSortedColIndB is a native pointer
    // descrC is read-only
    // csrSortedValC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // csrSortedColIndC is a native pointer
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrgeam2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject alpha, jobject descrA, jint nnzA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject beta, jobject descrB, jint nnzB, jobject csrSortedValB, jobject csrSortedRowPtrB, jobject csrSortedColIndB, jobject descrC, jobject csrSortedValC, jobject csrSortedRowPtrC, jobject csrSortedColIndC, jlongArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzA is primitive
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseDcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrB' is null for cusparseDcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzB is primitive
    if (csrSortedValB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValB' is null for cusparseDcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrB' is null for cusparseDcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndB' is null for cusparseDcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseDcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseDcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseDcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseDcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseDcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrgeam2_bufferSizeExt(handle=%p, m=%d, n=%d, alpha=%p, descrA=%p, nnzA=%d, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, beta=%p, descrB=%p, nnzB=%d, csrSortedValB=%p, csrSortedRowPtrB=%p, csrSortedColIndB=%p, descrC=%p, csrSortedValC=%p, csrSortedRowPtrC=%p, csrSortedColIndC=%p, pBufferSizeInBytes=%p)\n",
        handle, m, n, alpha, descrA, nnzA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, beta, descrB, nnzB, csrSortedValB, csrSortedRowPtrB, csrSortedColIndB, descrC, csrSortedValC, csrSortedRowPtrC, csrSortedColIndC, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    double * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    int nnzA_native = 0;
    double * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    double * beta_native = NULL;
    cusparseMatDescr_t descrB_native;
    int nnzB_native = 0;
    double * csrSortedValB_native = NULL;
    int * csrSortedRowPtrB_native = NULL;
    int * csrSortedColIndB_native = NULL;
    cusparseMatDescr_t descrC_native;
    double * csrSortedValC_native = NULL;
    int * csrSortedRowPtrC_native = NULL;
    int * csrSortedColIndC_native = NULL;
    size_t * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    nnzA_native = (int)nnzA;
    csrSortedValA_native = (double *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (double *)beta_pointerData->getPointer(env);
    descrB_native = (cusparseMatDescr_t)getNativePointerValue(env, descrB);
    nnzB_native = (int)nnzB;
    csrSortedValB_native = (double *)getPointer(env, csrSortedValB);
    csrSortedRowPtrB_native = (int *)getPointer(env, csrSortedRowPtrB);
    csrSortedColIndB_native = (int *)getPointer(env, csrSortedColIndB);
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedValC_native = (double *)getPointer(env, csrSortedValC);
    csrSortedRowPtrC_native = (int *)getPointer(env, csrSortedRowPtrC);
    csrSortedColIndC_native = (int *)getPointer(env, csrSortedColIndC);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrgeam2_bufferSizeExt(handle_native, m_native, n_native, alpha_native, descrA_native, nnzA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, beta_native, descrB_native, nnzB_native, csrSortedValB_native, csrSortedRowPtrB_native, csrSortedColIndB_native, descrC_native, csrSortedValC_native, csrSortedRowPtrC_native, csrSortedColIndC_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // nnzA is primitive
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrB is read-only
    // nnzB is primitive
    // csrSortedValB is a native pointer
    // csrSortedRowPtrB is a native pointer
    // csrSortedColIndB is a native pointer
    // descrC is read-only
    // csrSortedValC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // csrSortedColIndC is a native pointer
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrgeam2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject alpha, jobject descrA, jint nnzA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject beta, jobject descrB, jint nnzB, jobject csrSortedValB, jobject csrSortedRowPtrB, jobject csrSortedColIndB, jobject descrC, jobject csrSortedValC, jobject csrSortedRowPtrC, jobject csrSortedColIndC, jlongArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseCcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzA is primitive
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseCcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrB' is null for cusparseCcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzB is primitive
    if (csrSortedValB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValB' is null for cusparseCcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrB' is null for cusparseCcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndB' is null for cusparseCcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseCcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseCcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseCcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseCcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseCcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrgeam2_bufferSizeExt(handle=%p, m=%d, n=%d, alpha=%p, descrA=%p, nnzA=%d, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, beta=%p, descrB=%p, nnzB=%d, csrSortedValB=%p, csrSortedRowPtrB=%p, csrSortedColIndB=%p, descrC=%p, csrSortedValC=%p, csrSortedRowPtrC=%p, csrSortedColIndC=%p, pBufferSizeInBytes=%p)\n",
        handle, m, n, alpha, descrA, nnzA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, beta, descrB, nnzB, csrSortedValB, csrSortedRowPtrB, csrSortedColIndB, descrC, csrSortedValC, csrSortedRowPtrC, csrSortedColIndC, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuComplex * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    int nnzA_native = 0;
    cuComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    cuComplex * beta_native = NULL;
    cusparseMatDescr_t descrB_native;
    int nnzB_native = 0;
    cuComplex * csrSortedValB_native = NULL;
    int * csrSortedRowPtrB_native = NULL;
    int * csrSortedColIndB_native = NULL;
    cusparseMatDescr_t descrC_native;
    cuComplex * csrSortedValC_native = NULL;
    int * csrSortedRowPtrC_native = NULL;
    int * csrSortedColIndC_native = NULL;
    size_t * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    nnzA_native = (int)nnzA;
    csrSortedValA_native = (cuComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuComplex *)beta_pointerData->getPointer(env);
    descrB_native = (cusparseMatDescr_t)getNativePointerValue(env, descrB);
    nnzB_native = (int)nnzB;
    csrSortedValB_native = (cuComplex *)getPointer(env, csrSortedValB);
    csrSortedRowPtrB_native = (int *)getPointer(env, csrSortedRowPtrB);
    csrSortedColIndB_native = (int *)getPointer(env, csrSortedColIndB);
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedValC_native = (cuComplex *)getPointer(env, csrSortedValC);
    csrSortedRowPtrC_native = (int *)getPointer(env, csrSortedRowPtrC);
    csrSortedColIndC_native = (int *)getPointer(env, csrSortedColIndC);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrgeam2_bufferSizeExt(handle_native, m_native, n_native, alpha_native, descrA_native, nnzA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, beta_native, descrB_native, nnzB_native, csrSortedValB_native, csrSortedRowPtrB_native, csrSortedColIndB_native, descrC_native, csrSortedValC_native, csrSortedRowPtrC_native, csrSortedColIndC_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // nnzA is primitive
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrB is read-only
    // nnzB is primitive
    // csrSortedValB is a native pointer
    // csrSortedRowPtrB is a native pointer
    // csrSortedColIndB is a native pointer
    // descrC is read-only
    // csrSortedValC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // csrSortedColIndC is a native pointer
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrgeam2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject alpha, jobject descrA, jint nnzA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject beta, jobject descrB, jint nnzB, jobject csrSortedValB, jobject csrSortedRowPtrB, jobject csrSortedColIndB, jobject descrC, jobject csrSortedValC, jobject csrSortedRowPtrC, jobject csrSortedColIndC, jlongArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzA is primitive
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseZcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrB' is null for cusparseZcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzB is primitive
    if (csrSortedValB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValB' is null for cusparseZcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrB' is null for cusparseZcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndB' is null for cusparseZcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseZcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseZcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseZcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseZcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseZcsrgeam2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrgeam2_bufferSizeExt(handle=%p, m=%d, n=%d, alpha=%p, descrA=%p, nnzA=%d, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, beta=%p, descrB=%p, nnzB=%d, csrSortedValB=%p, csrSortedRowPtrB=%p, csrSortedColIndB=%p, descrC=%p, csrSortedValC=%p, csrSortedRowPtrC=%p, csrSortedColIndC=%p, pBufferSizeInBytes=%p)\n",
        handle, m, n, alpha, descrA, nnzA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, beta, descrB, nnzB, csrSortedValB, csrSortedRowPtrB, csrSortedColIndB, descrC, csrSortedValC, csrSortedRowPtrC, csrSortedColIndC, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuDoubleComplex * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    int nnzA_native = 0;
    cuDoubleComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    cuDoubleComplex * beta_native = NULL;
    cusparseMatDescr_t descrB_native;
    int nnzB_native = 0;
    cuDoubleComplex * csrSortedValB_native = NULL;
    int * csrSortedRowPtrB_native = NULL;
    int * csrSortedColIndB_native = NULL;
    cusparseMatDescr_t descrC_native;
    cuDoubleComplex * csrSortedValC_native = NULL;
    int * csrSortedRowPtrC_native = NULL;
    int * csrSortedColIndC_native = NULL;
    size_t * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    nnzA_native = (int)nnzA;
    csrSortedValA_native = (cuDoubleComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuDoubleComplex *)beta_pointerData->getPointer(env);
    descrB_native = (cusparseMatDescr_t)getNativePointerValue(env, descrB);
    nnzB_native = (int)nnzB;
    csrSortedValB_native = (cuDoubleComplex *)getPointer(env, csrSortedValB);
    csrSortedRowPtrB_native = (int *)getPointer(env, csrSortedRowPtrB);
    csrSortedColIndB_native = (int *)getPointer(env, csrSortedColIndB);
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedValC_native = (cuDoubleComplex *)getPointer(env, csrSortedValC);
    csrSortedRowPtrC_native = (int *)getPointer(env, csrSortedRowPtrC);
    csrSortedColIndC_native = (int *)getPointer(env, csrSortedColIndC);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrgeam2_bufferSizeExt(handle_native, m_native, n_native, alpha_native, descrA_native, nnzA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, beta_native, descrB_native, nnzB_native, csrSortedValB_native, csrSortedRowPtrB_native, csrSortedColIndB_native, descrC_native, csrSortedValC_native, csrSortedRowPtrC_native, csrSortedColIndC_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // nnzA is primitive
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrB is read-only
    // nnzB is primitive
    // csrSortedValB is a native pointer
    // csrSortedRowPtrB is a native pointer
    // csrSortedColIndB is a native pointer
    // descrC is read-only
    // csrSortedValC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // csrSortedColIndC is a native pointer
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXcsrgeam2NnzNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject descrA, jint nnzA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject descrB, jint nnzB, jobject csrSortedRowPtrB, jobject csrSortedColIndB, jobject descrC, jobject csrSortedRowPtrC, jobject nnzTotalDevHostPtr, jobject workspace)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXcsrgeam2Nnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseXcsrgeam2Nnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzA is primitive
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseXcsrgeam2Nnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseXcsrgeam2Nnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrB' is null for cusparseXcsrgeam2Nnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzB is primitive
    if (csrSortedRowPtrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrB' is null for cusparseXcsrgeam2Nnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndB' is null for cusparseXcsrgeam2Nnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseXcsrgeam2Nnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseXcsrgeam2Nnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (nnzTotalDevHostPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzTotalDevHostPtr' is null for cusparseXcsrgeam2Nnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (workspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'workspace' is null for cusparseXcsrgeam2Nnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXcsrgeam2Nnz(handle=%p, m=%d, n=%d, descrA=%p, nnzA=%d, csrSortedRowPtrA=%p, csrSortedColIndA=%p, descrB=%p, nnzB=%d, csrSortedRowPtrB=%p, csrSortedColIndB=%p, descrC=%p, csrSortedRowPtrC=%p, nnzTotalDevHostPtr=%p, workspace=%p)\n",
        handle, m, n, descrA, nnzA, csrSortedRowPtrA, csrSortedColIndA, descrB, nnzB, csrSortedRowPtrB, csrSortedColIndB, descrC, csrSortedRowPtrC, nnzTotalDevHostPtr, workspace);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    int nnzA_native = 0;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    cusparseMatDescr_t descrB_native;
    int nnzB_native = 0;
    int * csrSortedRowPtrB_native = NULL;
    int * csrSortedColIndB_native = NULL;
    cusparseMatDescr_t descrC_native;
    int * csrSortedRowPtrC_native = NULL;
    int * nnzTotalDevHostPtr_native = NULL;
    void * workspace_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    nnzA_native = (int)nnzA;
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    descrB_native = (cusparseMatDescr_t)getNativePointerValue(env, descrB);
    nnzB_native = (int)nnzB;
    csrSortedRowPtrB_native = (int *)getPointer(env, csrSortedRowPtrB);
    csrSortedColIndB_native = (int *)getPointer(env, csrSortedColIndB);
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedRowPtrC_native = (int *)getPointer(env, csrSortedRowPtrC);
    PointerData *nnzTotalDevHostPtr_pointerData = initPointerData(env, nnzTotalDevHostPtr);
    if (nnzTotalDevHostPtr_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    nnzTotalDevHostPtr_native = (int *)nnzTotalDevHostPtr_pointerData->getPointer(env);
    workspace_native = (void *)getPointer(env, workspace);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXcsrgeam2Nnz(handle_native, m_native, n_native, descrA_native, nnzA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, descrB_native, nnzB_native, csrSortedRowPtrB_native, csrSortedColIndB_native, descrC_native, csrSortedRowPtrC_native, nnzTotalDevHostPtr_native, workspace_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // descrA is read-only
    // nnzA is primitive
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // descrB is read-only
    // nnzB is primitive
    // csrSortedRowPtrB is a native pointer
    // csrSortedColIndB is a native pointer
    // descrC is read-only
    // csrSortedRowPtrC is a native pointer
    // If the PointerData is not backed by native memory, then this call has to block
    if (!isPointerBackedByNativeMemory(env, nnzTotalDevHostPtr))
    {
        cudaDeviceSynchronize();
    }
    if (!releasePointerData(env, nnzTotalDevHostPtr_pointerData, 0)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // workspace is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrgeam2Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject alpha, jobject descrA, jint nnzA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject beta, jobject descrB, jint nnzB, jobject csrSortedValB, jobject csrSortedRowPtrB, jobject csrSortedColIndB, jobject descrC, jobject csrSortedValC, jobject csrSortedRowPtrC, jobject csrSortedColIndC, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseScsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzA is primitive
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseScsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrB' is null for cusparseScsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzB is primitive
    if (csrSortedValB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValB' is null for cusparseScsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrB' is null for cusparseScsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndB' is null for cusparseScsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseScsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseScsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseScsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseScsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseScsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrgeam2(handle=%p, m=%d, n=%d, alpha=%p, descrA=%p, nnzA=%d, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, beta=%p, descrB=%p, nnzB=%d, csrSortedValB=%p, csrSortedRowPtrB=%p, csrSortedColIndB=%p, descrC=%p, csrSortedValC=%p, csrSortedRowPtrC=%p, csrSortedColIndC=%p, pBuffer=%p)\n",
        handle, m, n, alpha, descrA, nnzA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, beta, descrB, nnzB, csrSortedValB, csrSortedRowPtrB, csrSortedColIndB, descrC, csrSortedValC, csrSortedRowPtrC, csrSortedColIndC, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    float * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    int nnzA_native = 0;
    float * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    float * beta_native = NULL;
    cusparseMatDescr_t descrB_native;
    int nnzB_native = 0;
    float * csrSortedValB_native = NULL;
    int * csrSortedRowPtrB_native = NULL;
    int * csrSortedColIndB_native = NULL;
    cusparseMatDescr_t descrC_native;
    float * csrSortedValC_native = NULL;
    int * csrSortedRowPtrC_native = NULL;
    int * csrSortedColIndC_native = NULL;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    nnzA_native = (int)nnzA;
    csrSortedValA_native = (float *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (float *)beta_pointerData->getPointer(env);
    descrB_native = (cusparseMatDescr_t)getNativePointerValue(env, descrB);
    nnzB_native = (int)nnzB;
    csrSortedValB_native = (float *)getPointer(env, csrSortedValB);
    csrSortedRowPtrB_native = (int *)getPointer(env, csrSortedRowPtrB);
    csrSortedColIndB_native = (int *)getPointer(env, csrSortedColIndB);
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedValC_native = (float *)getPointer(env, csrSortedValC);
    csrSortedRowPtrC_native = (int *)getPointer(env, csrSortedRowPtrC);
    csrSortedColIndC_native = (int *)getPointer(env, csrSortedColIndC);
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrgeam2(handle_native, m_native, n_native, alpha_native, descrA_native, nnzA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, beta_native, descrB_native, nnzB_native, csrSortedValB_native, csrSortedRowPtrB_native, csrSortedColIndB_native, descrC_native, csrSortedValC_native, csrSortedRowPtrC_native, csrSortedColIndC_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // nnzA is primitive
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrB is read-only
    // nnzB is primitive
    // csrSortedValB is a native pointer
    // csrSortedRowPtrB is a native pointer
    // csrSortedColIndB is a native pointer
    // descrC is read-only
    // csrSortedValC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // csrSortedColIndC is a native pointer
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrgeam2Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject alpha, jobject descrA, jint nnzA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject beta, jobject descrB, jint nnzB, jobject csrSortedValB, jobject csrSortedRowPtrB, jobject csrSortedColIndB, jobject descrC, jobject csrSortedValC, jobject csrSortedRowPtrC, jobject csrSortedColIndC, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzA is primitive
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseDcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrB' is null for cusparseDcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzB is primitive
    if (csrSortedValB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValB' is null for cusparseDcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrB' is null for cusparseDcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndB' is null for cusparseDcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseDcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseDcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseDcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseDcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrgeam2(handle=%p, m=%d, n=%d, alpha=%p, descrA=%p, nnzA=%d, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, beta=%p, descrB=%p, nnzB=%d, csrSortedValB=%p, csrSortedRowPtrB=%p, csrSortedColIndB=%p, descrC=%p, csrSortedValC=%p, csrSortedRowPtrC=%p, csrSortedColIndC=%p, pBuffer=%p)\n",
        handle, m, n, alpha, descrA, nnzA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, beta, descrB, nnzB, csrSortedValB, csrSortedRowPtrB, csrSortedColIndB, descrC, csrSortedValC, csrSortedRowPtrC, csrSortedColIndC, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    double * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    int nnzA_native = 0;
    double * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    double * beta_native = NULL;
    cusparseMatDescr_t descrB_native;
    int nnzB_native = 0;
    double * csrSortedValB_native = NULL;
    int * csrSortedRowPtrB_native = NULL;
    int * csrSortedColIndB_native = NULL;
    cusparseMatDescr_t descrC_native;
    double * csrSortedValC_native = NULL;
    int * csrSortedRowPtrC_native = NULL;
    int * csrSortedColIndC_native = NULL;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    nnzA_native = (int)nnzA;
    csrSortedValA_native = (double *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (double *)beta_pointerData->getPointer(env);
    descrB_native = (cusparseMatDescr_t)getNativePointerValue(env, descrB);
    nnzB_native = (int)nnzB;
    csrSortedValB_native = (double *)getPointer(env, csrSortedValB);
    csrSortedRowPtrB_native = (int *)getPointer(env, csrSortedRowPtrB);
    csrSortedColIndB_native = (int *)getPointer(env, csrSortedColIndB);
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedValC_native = (double *)getPointer(env, csrSortedValC);
    csrSortedRowPtrC_native = (int *)getPointer(env, csrSortedRowPtrC);
    csrSortedColIndC_native = (int *)getPointer(env, csrSortedColIndC);
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrgeam2(handle_native, m_native, n_native, alpha_native, descrA_native, nnzA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, beta_native, descrB_native, nnzB_native, csrSortedValB_native, csrSortedRowPtrB_native, csrSortedColIndB_native, descrC_native, csrSortedValC_native, csrSortedRowPtrC_native, csrSortedColIndC_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // nnzA is primitive
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrB is read-only
    // nnzB is primitive
    // csrSortedValB is a native pointer
    // csrSortedRowPtrB is a native pointer
    // csrSortedColIndB is a native pointer
    // descrC is read-only
    // csrSortedValC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // csrSortedColIndC is a native pointer
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrgeam2Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject alpha, jobject descrA, jint nnzA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject beta, jobject descrB, jint nnzB, jobject csrSortedValB, jobject csrSortedRowPtrB, jobject csrSortedColIndB, jobject descrC, jobject csrSortedValC, jobject csrSortedRowPtrC, jobject csrSortedColIndC, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseCcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzA is primitive
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseCcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrB' is null for cusparseCcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzB is primitive
    if (csrSortedValB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValB' is null for cusparseCcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrB' is null for cusparseCcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndB' is null for cusparseCcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseCcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseCcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseCcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseCcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrgeam2(handle=%p, m=%d, n=%d, alpha=%p, descrA=%p, nnzA=%d, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, beta=%p, descrB=%p, nnzB=%d, csrSortedValB=%p, csrSortedRowPtrB=%p, csrSortedColIndB=%p, descrC=%p, csrSortedValC=%p, csrSortedRowPtrC=%p, csrSortedColIndC=%p, pBuffer=%p)\n",
        handle, m, n, alpha, descrA, nnzA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, beta, descrB, nnzB, csrSortedValB, csrSortedRowPtrB, csrSortedColIndB, descrC, csrSortedValC, csrSortedRowPtrC, csrSortedColIndC, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuComplex * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    int nnzA_native = 0;
    cuComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    cuComplex * beta_native = NULL;
    cusparseMatDescr_t descrB_native;
    int nnzB_native = 0;
    cuComplex * csrSortedValB_native = NULL;
    int * csrSortedRowPtrB_native = NULL;
    int * csrSortedColIndB_native = NULL;
    cusparseMatDescr_t descrC_native;
    cuComplex * csrSortedValC_native = NULL;
    int * csrSortedRowPtrC_native = NULL;
    int * csrSortedColIndC_native = NULL;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    nnzA_native = (int)nnzA;
    csrSortedValA_native = (cuComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuComplex *)beta_pointerData->getPointer(env);
    descrB_native = (cusparseMatDescr_t)getNativePointerValue(env, descrB);
    nnzB_native = (int)nnzB;
    csrSortedValB_native = (cuComplex *)getPointer(env, csrSortedValB);
    csrSortedRowPtrB_native = (int *)getPointer(env, csrSortedRowPtrB);
    csrSortedColIndB_native = (int *)getPointer(env, csrSortedColIndB);
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedValC_native = (cuComplex *)getPointer(env, csrSortedValC);
    csrSortedRowPtrC_native = (int *)getPointer(env, csrSortedRowPtrC);
    csrSortedColIndC_native = (int *)getPointer(env, csrSortedColIndC);
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrgeam2(handle_native, m_native, n_native, alpha_native, descrA_native, nnzA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, beta_native, descrB_native, nnzB_native, csrSortedValB_native, csrSortedRowPtrB_native, csrSortedColIndB_native, descrC_native, csrSortedValC_native, csrSortedRowPtrC_native, csrSortedColIndC_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // nnzA is primitive
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrB is read-only
    // nnzB is primitive
    // csrSortedValB is a native pointer
    // csrSortedRowPtrB is a native pointer
    // csrSortedColIndB is a native pointer
    // descrC is read-only
    // csrSortedValC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // csrSortedColIndC is a native pointer
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrgeam2Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject alpha, jobject descrA, jint nnzA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject beta, jobject descrB, jint nnzB, jobject csrSortedValB, jobject csrSortedRowPtrB, jobject csrSortedColIndB, jobject descrC, jobject csrSortedValC, jobject csrSortedRowPtrC, jobject csrSortedColIndC, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzA is primitive
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseZcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrB' is null for cusparseZcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzB is primitive
    if (csrSortedValB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValB' is null for cusparseZcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrB' is null for cusparseZcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndB' is null for cusparseZcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseZcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseZcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseZcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseZcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZcsrgeam2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrgeam2(handle=%p, m=%d, n=%d, alpha=%p, descrA=%p, nnzA=%d, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, beta=%p, descrB=%p, nnzB=%d, csrSortedValB=%p, csrSortedRowPtrB=%p, csrSortedColIndB=%p, descrC=%p, csrSortedValC=%p, csrSortedRowPtrC=%p, csrSortedColIndC=%p, pBuffer=%p)\n",
        handle, m, n, alpha, descrA, nnzA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, beta, descrB, nnzB, csrSortedValB, csrSortedRowPtrB, csrSortedColIndB, descrC, csrSortedValC, csrSortedRowPtrC, csrSortedColIndC, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuDoubleComplex * alpha_native = NULL;
    cusparseMatDescr_t descrA_native;
    int nnzA_native = 0;
    cuDoubleComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    cuDoubleComplex * beta_native = NULL;
    cusparseMatDescr_t descrB_native;
    int nnzB_native = 0;
    cuDoubleComplex * csrSortedValB_native = NULL;
    int * csrSortedRowPtrB_native = NULL;
    int * csrSortedColIndB_native = NULL;
    cusparseMatDescr_t descrC_native;
    cuDoubleComplex * csrSortedValC_native = NULL;
    int * csrSortedRowPtrC_native = NULL;
    int * csrSortedColIndC_native = NULL;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex *)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    nnzA_native = (int)nnzA;
    csrSortedValA_native = (cuDoubleComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuDoubleComplex *)beta_pointerData->getPointer(env);
    descrB_native = (cusparseMatDescr_t)getNativePointerValue(env, descrB);
    nnzB_native = (int)nnzB;
    csrSortedValB_native = (cuDoubleComplex *)getPointer(env, csrSortedValB);
    csrSortedRowPtrB_native = (int *)getPointer(env, csrSortedRowPtrB);
    csrSortedColIndB_native = (int *)getPointer(env, csrSortedColIndB);
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedValC_native = (cuDoubleComplex *)getPointer(env, csrSortedValC);
    csrSortedRowPtrC_native = (int *)getPointer(env, csrSortedRowPtrC);
    csrSortedColIndC_native = (int *)getPointer(env, csrSortedColIndC);
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrgeam2(handle_native, m_native, n_native, alpha_native, descrA_native, nnzA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, beta_native, descrB_native, nnzB_native, csrSortedValB_native, csrSortedRowPtrB_native, csrSortedColIndB_native, descrC_native, csrSortedValC_native, csrSortedRowPtrC_native, csrSortedColIndC_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is read-only
    // nnzA is primitive
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrB is read-only
    // nnzB is primitive
    // csrSortedValB is a native pointer
    // csrSortedRowPtrB is a native pointer
    // csrSortedColIndB is a native pointer
    // descrC is read-only
    // csrSortedValC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // csrSortedColIndC is a native pointer
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

//##############################################################################
//# SPARSE MATRIX REORDERING
//##############################################################################
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrcolorNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject fractionToColor, jobject ncolors, jobject coloring, jobject reordering, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (fractionToColor == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'fractionToColor' is null for cusparseScsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (ncolors == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ncolors' is null for cusparseScsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (coloring == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'coloring' is null for cusparseScsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (reordering == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'reordering' is null for cusparseScsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrcolor(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, fractionToColor=%p, ncolors=%p, coloring=%p, reordering=%p, info=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, fractionToColor, ncolors, coloring, reordering, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    float * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    float * fractionToColor_native = NULL;
    int * ncolors_native = NULL;
    int * coloring_native = NULL;
    int * reordering_native = NULL;
    cusparseColorInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    fractionToColor_native = (float *)getPointer(env, fractionToColor);
    ncolors_native = (int *)getPointer(env, ncolors);
    coloring_native = (int *)getPointer(env, coloring);
    reordering_native = (int *)getPointer(env, reordering);
    info_native = (cusparseColorInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrcolor(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, fractionToColor_native, ncolors_native, coloring_native, reordering_native, info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // fractionToColor is a native pointer
    // ncolors is a native pointer
    // coloring is a native pointer
    // reordering is a native pointer
    // info is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrcolorNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject fractionToColor, jobject ncolors, jobject coloring, jobject reordering, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (fractionToColor == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'fractionToColor' is null for cusparseDcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (ncolors == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ncolors' is null for cusparseDcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (coloring == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'coloring' is null for cusparseDcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (reordering == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'reordering' is null for cusparseDcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrcolor(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, fractionToColor=%p, ncolors=%p, coloring=%p, reordering=%p, info=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, fractionToColor, ncolors, coloring, reordering, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    double * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    double * fractionToColor_native = NULL;
    int * ncolors_native = NULL;
    int * coloring_native = NULL;
    int * reordering_native = NULL;
    cusparseColorInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    fractionToColor_native = (double *)getPointer(env, fractionToColor);
    ncolors_native = (int *)getPointer(env, ncolors);
    coloring_native = (int *)getPointer(env, coloring);
    reordering_native = (int *)getPointer(env, reordering);
    info_native = (cusparseColorInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrcolor(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, fractionToColor_native, ncolors_native, coloring_native, reordering_native, info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // fractionToColor is a native pointer
    // ncolors is a native pointer
    // coloring is a native pointer
    // reordering is a native pointer
    // info is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrcolorNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject fractionToColor, jobject ncolors, jobject coloring, jobject reordering, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (fractionToColor == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'fractionToColor' is null for cusparseCcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (ncolors == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ncolors' is null for cusparseCcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (coloring == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'coloring' is null for cusparseCcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (reordering == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'reordering' is null for cusparseCcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrcolor(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, fractionToColor=%p, ncolors=%p, coloring=%p, reordering=%p, info=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, fractionToColor, ncolors, coloring, reordering, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    float * fractionToColor_native = NULL;
    int * ncolors_native = NULL;
    int * coloring_native = NULL;
    int * reordering_native = NULL;
    cusparseColorInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    fractionToColor_native = (float *)getPointer(env, fractionToColor);
    ncolors_native = (int *)getPointer(env, ncolors);
    coloring_native = (int *)getPointer(env, coloring);
    reordering_native = (int *)getPointer(env, reordering);
    info_native = (cusparseColorInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrcolor(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, fractionToColor_native, ncolors_native, coloring_native, reordering_native, info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // fractionToColor is a native pointer
    // ncolors is a native pointer
    // coloring is a native pointer
    // reordering is a native pointer
    // info is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrcolorNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject fractionToColor, jobject ncolors, jobject coloring, jobject reordering, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (fractionToColor == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'fractionToColor' is null for cusparseZcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (ncolors == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ncolors' is null for cusparseZcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (coloring == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'coloring' is null for cusparseZcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (reordering == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'reordering' is null for cusparseZcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsrcolor");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrcolor(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, fractionToColor=%p, ncolors=%p, coloring=%p, reordering=%p, info=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, fractionToColor, ncolors, coloring, reordering, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    double * fractionToColor_native = NULL;
    int * ncolors_native = NULL;
    int * coloring_native = NULL;
    int * reordering_native = NULL;
    cusparseColorInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    fractionToColor_native = (double *)getPointer(env, fractionToColor);
    ncolors_native = (int *)getPointer(env, ncolors);
    coloring_native = (int *)getPointer(env, coloring);
    reordering_native = (int *)getPointer(env, reordering);
    info_native = (cusparseColorInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrcolor(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, fractionToColor_native, ncolors_native, coloring_native, reordering_native, info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // nnz is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // fractionToColor is a native pointer
    // ncolors is a native pointer
    // coloring is a native pointer
    // reordering is a native pointer
    // info is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

//##############################################################################
//# SPARSE FORMAT CONVERSION
//##############################################################################
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSnnzNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint m, jint n, jobject descrA, jobject A, jint lda, jobject nnzPerRowCol, jobject nnzTotalDevHostPtr)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSnnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSnnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusparseSnnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (nnzPerRowCol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzPerRowCol' is null for cusparseSnnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (nnzTotalDevHostPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzTotalDevHostPtr' is null for cusparseSnnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSnnz(handle=%p, dirA=%d, m=%d, n=%d, descrA=%p, A=%p, lda=%d, nnzPerRowCol=%p, nnzTotalDevHostPtr=%p)\n",
        handle, dirA, m, n, descrA, A, lda, nnzPerRowCol, nnzTotalDevHostPtr);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    float * A_native = NULL;
    int lda_native = 0;
    int * nnzPerRowCol_native = NULL;
    int * nnzTotalDevHostPtr_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    nnzPerRowCol_native = (int *)getPointer(env, nnzPerRowCol);
    PointerData *nnzTotalDevHostPtr_pointerData = initPointerData(env, nnzTotalDevHostPtr);
    if (nnzTotalDevHostPtr_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    nnzTotalDevHostPtr_native = (int *)nnzTotalDevHostPtr_pointerData->getPointer(env);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSnnz(handle_native, dirA_native, m_native, n_native, descrA_native, A_native, lda_native, nnzPerRowCol_native, nnzTotalDevHostPtr_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // m is primitive
    // n is primitive
    // descrA is read-only
    // A is a native pointer
    // lda is primitive
    // nnzPerRowCol is a native pointer
    // If the PointerData is not backed by native memory, then this call has to block
    if (!isPointerBackedByNativeMemory(env, nnzTotalDevHostPtr))
    {
        cudaDeviceSynchronize();
    }
    if (!releasePointerData(env, nnzTotalDevHostPtr_pointerData, 0)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDnnzNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint m, jint n, jobject descrA, jobject A, jint lda, jobject nnzPerRowCol, jobject nnzTotalDevHostPtr)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDnnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDnnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusparseDnnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (nnzPerRowCol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzPerRowCol' is null for cusparseDnnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (nnzTotalDevHostPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzTotalDevHostPtr' is null for cusparseDnnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDnnz(handle=%p, dirA=%d, m=%d, n=%d, descrA=%p, A=%p, lda=%d, nnzPerRowCol=%p, nnzTotalDevHostPtr=%p)\n",
        handle, dirA, m, n, descrA, A, lda, nnzPerRowCol, nnzTotalDevHostPtr);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    double * A_native = NULL;
    int lda_native = 0;
    int * nnzPerRowCol_native = NULL;
    int * nnzTotalDevHostPtr_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    nnzPerRowCol_native = (int *)getPointer(env, nnzPerRowCol);
    PointerData *nnzTotalDevHostPtr_pointerData = initPointerData(env, nnzTotalDevHostPtr);
    if (nnzTotalDevHostPtr_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    nnzTotalDevHostPtr_native = (int *)nnzTotalDevHostPtr_pointerData->getPointer(env);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDnnz(handle_native, dirA_native, m_native, n_native, descrA_native, A_native, lda_native, nnzPerRowCol_native, nnzTotalDevHostPtr_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // m is primitive
    // n is primitive
    // descrA is read-only
    // A is a native pointer
    // lda is primitive
    // nnzPerRowCol is a native pointer
    // If the PointerData is not backed by native memory, then this call has to block
    if (!isPointerBackedByNativeMemory(env, nnzTotalDevHostPtr))
    {
        cudaDeviceSynchronize();
    }
    if (!releasePointerData(env, nnzTotalDevHostPtr_pointerData, 0)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCnnzNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint m, jint n, jobject descrA, jobject A, jint lda, jobject nnzPerRowCol, jobject nnzTotalDevHostPtr)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCnnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCnnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusparseCnnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (nnzPerRowCol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzPerRowCol' is null for cusparseCnnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (nnzTotalDevHostPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzTotalDevHostPtr' is null for cusparseCnnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCnnz(handle=%p, dirA=%d, m=%d, n=%d, descrA=%p, A=%p, lda=%d, nnzPerRowCol=%p, nnzTotalDevHostPtr=%p)\n",
        handle, dirA, m, n, descrA, A, lda, nnzPerRowCol, nnzTotalDevHostPtr);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    int * nnzPerRowCol_native = NULL;
    int * nnzTotalDevHostPtr_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    nnzPerRowCol_native = (int *)getPointer(env, nnzPerRowCol);
    PointerData *nnzTotalDevHostPtr_pointerData = initPointerData(env, nnzTotalDevHostPtr);
    if (nnzTotalDevHostPtr_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    nnzTotalDevHostPtr_native = (int *)nnzTotalDevHostPtr_pointerData->getPointer(env);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCnnz(handle_native, dirA_native, m_native, n_native, descrA_native, A_native, lda_native, nnzPerRowCol_native, nnzTotalDevHostPtr_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // m is primitive
    // n is primitive
    // descrA is read-only
    // A is a native pointer
    // lda is primitive
    // nnzPerRowCol is a native pointer
    // If the PointerData is not backed by native memory, then this call has to block
    if (!isPointerBackedByNativeMemory(env, nnzTotalDevHostPtr))
    {
        cudaDeviceSynchronize();
    }
    if (!releasePointerData(env, nnzTotalDevHostPtr_pointerData, 0)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZnnzNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint m, jint n, jobject descrA, jobject A, jint lda, jobject nnzPerRowCol, jobject nnzTotalDevHostPtr)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZnnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZnnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusparseZnnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (nnzPerRowCol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzPerRowCol' is null for cusparseZnnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (nnzTotalDevHostPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzTotalDevHostPtr' is null for cusparseZnnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZnnz(handle=%p, dirA=%d, m=%d, n=%d, descrA=%p, A=%p, lda=%d, nnzPerRowCol=%p, nnzTotalDevHostPtr=%p)\n",
        handle, dirA, m, n, descrA, A, lda, nnzPerRowCol, nnzTotalDevHostPtr);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    int * nnzPerRowCol_native = NULL;
    int * nnzTotalDevHostPtr_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    nnzPerRowCol_native = (int *)getPointer(env, nnzPerRowCol);
    PointerData *nnzTotalDevHostPtr_pointerData = initPointerData(env, nnzTotalDevHostPtr);
    if (nnzTotalDevHostPtr_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    nnzTotalDevHostPtr_native = (int *)nnzTotalDevHostPtr_pointerData->getPointer(env);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZnnz(handle_native, dirA_native, m_native, n_native, descrA_native, A_native, lda_native, nnzPerRowCol_native, nnzTotalDevHostPtr_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // m is primitive
    // n is primitive
    // descrA is read-only
    // A is a native pointer
    // lda is primitive
    // nnzPerRowCol is a native pointer
    // If the PointerData is not backed by native memory, then this call has to block
    if (!isPointerBackedByNativeMemory(env, nnzTotalDevHostPtr))
    {
        cudaDeviceSynchronize();
    }
    if (!releasePointerData(env, nnzTotalDevHostPtr_pointerData, 0)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

//##############################################################################
//# SPARSE FORMAT CONVERSION
//##############################################################################
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSnnz_1compressNative(JNIEnv *env, jclass cls, jobject handle, jint m, jobject descr, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject nnzPerRow, jobject nnzC, jfloat tol)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    if (descr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descr' is null for cusparseSnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseSnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseSnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (nnzPerRow == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzPerRow' is null for cusparseSnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (nnzC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzC' is null for cusparseSnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // tol is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSnnz_compress(handle=%p, m=%d, descr=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, nnzPerRow=%p, nnzC=%p, tol=%f)\n",
        handle, m, descr, csrSortedValA, csrSortedRowPtrA, nnzPerRow, nnzC, tol);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    cusparseMatDescr_t descr_native;
    float * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * nnzPerRow_native = NULL;
    int * nnzC_native = NULL;
    float tol_native = 0.0f;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    descr_native = (cusparseMatDescr_t)getNativePointerValue(env, descr);
    csrSortedValA_native = (float *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    nnzPerRow_native = (int *)getPointer(env, nnzPerRow);
    nnzC_native = (int *)getPointer(env, nnzC);
    tol_native = (float)tol;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSnnz_compress(handle_native, m_native, descr_native, csrSortedValA_native, csrSortedRowPtrA_native, nnzPerRow_native, nnzC_native, tol_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // descr is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // nnzPerRow is a native pointer
    // nnzC is a native pointer
    // tol is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDnnz_1compressNative(JNIEnv *env, jclass cls, jobject handle, jint m, jobject descr, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject nnzPerRow, jobject nnzC, jdouble tol)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    if (descr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descr' is null for cusparseDnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (nnzPerRow == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzPerRow' is null for cusparseDnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (nnzC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzC' is null for cusparseDnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // tol is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDnnz_compress(handle=%p, m=%d, descr=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, nnzPerRow=%p, nnzC=%p, tol=%lf)\n",
        handle, m, descr, csrSortedValA, csrSortedRowPtrA, nnzPerRow, nnzC, tol);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    cusparseMatDescr_t descr_native;
    double * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * nnzPerRow_native = NULL;
    int * nnzC_native = NULL;
    double tol_native = 0.0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    descr_native = (cusparseMatDescr_t)getNativePointerValue(env, descr);
    csrSortedValA_native = (double *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    nnzPerRow_native = (int *)getPointer(env, nnzPerRow);
    nnzC_native = (int *)getPointer(env, nnzC);
    tol_native = (double)tol;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDnnz_compress(handle_native, m_native, descr_native, csrSortedValA_native, csrSortedRowPtrA_native, nnzPerRow_native, nnzC_native, tol_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // descr is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // nnzPerRow is a native pointer
    // nnzC is a native pointer
    // tol is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCnnz_1compressNative(JNIEnv *env, jclass cls, jobject handle, jint m, jobject descr, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject nnzPerRow, jobject nnzC, jobject tol)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    if (descr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descr' is null for cusparseCnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (nnzPerRow == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzPerRow' is null for cusparseCnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (nnzC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzC' is null for cusparseCnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (tol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tol' is null for cusparseCnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCnnz_compress(handle=%p, m=%d, descr=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, nnzPerRow=%p, nnzC=%p, tol=%p)\n",
        handle, m, descr, csrSortedValA, csrSortedRowPtrA, nnzPerRow, nnzC, tol);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    cusparseMatDescr_t descr_native;
    cuComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * nnzPerRow_native = NULL;
    int * nnzC_native = NULL;
    cuComplex tol_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    descr_native = (cusparseMatDescr_t)getNativePointerValue(env, descr);
    csrSortedValA_native = (cuComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    nnzPerRow_native = (int *)getPointer(env, nnzPerRow);
    nnzC_native = (int *)getPointer(env, nnzC);
    if (!initNative(env, tol, tol_native)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCnnz_compress(handle_native, m_native, descr_native, csrSortedValA_native, csrSortedRowPtrA_native, nnzPerRow_native, nnzC_native, tol_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // descr is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // nnzPerRow is a native pointer
    // nnzC is a native pointer
    // tol is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZnnz_1compressNative(JNIEnv *env, jclass cls, jobject handle, jint m, jobject descr, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject nnzPerRow, jobject nnzC, jobject tol)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    if (descr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descr' is null for cusparseZnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (nnzPerRow == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzPerRow' is null for cusparseZnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (nnzC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzC' is null for cusparseZnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (tol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tol' is null for cusparseZnnz_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZnnz_compress(handle=%p, m=%d, descr=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, nnzPerRow=%p, nnzC=%p, tol=%p)\n",
        handle, m, descr, csrSortedValA, csrSortedRowPtrA, nnzPerRow, nnzC, tol);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    cusparseMatDescr_t descr_native;
    cuDoubleComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * nnzPerRow_native = NULL;
    int * nnzC_native = NULL;
    cuDoubleComplex tol_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    descr_native = (cusparseMatDescr_t)getNativePointerValue(env, descr);
    csrSortedValA_native = (cuDoubleComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    nnzPerRow_native = (int *)getPointer(env, nnzPerRow);
    nnzC_native = (int *)getPointer(env, nnzC);
    if (!initNative(env, tol, tol_native)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZnnz_compress(handle_native, m_native, descr_native, csrSortedValA_native, csrSortedRowPtrA_native, nnzPerRow_native, nnzC_native, tol_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // descr is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // nnzPerRow is a native pointer
    // nnzC is a native pointer
    // tol is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsr2csr_1compressNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject descrA, jobject csrSortedValA, jobject csrSortedColIndA, jobject csrSortedRowPtrA, jint nnzA, jobject nnzPerRow, jobject csrSortedValC, jobject csrSortedColIndC, jobject csrSortedRowPtrC, jfloat tol)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzA is primitive
    if (nnzPerRow == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzPerRow' is null for cusparseScsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseScsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseScsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseScsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // tol is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsr2csr_compress(handle=%p, m=%d, n=%d, descrA=%p, csrSortedValA=%p, csrSortedColIndA=%p, csrSortedRowPtrA=%p, nnzA=%d, nnzPerRow=%p, csrSortedValC=%p, csrSortedColIndC=%p, csrSortedRowPtrC=%p, tol=%f)\n",
        handle, m, n, descrA, csrSortedValA, csrSortedColIndA, csrSortedRowPtrA, nnzA, nnzPerRow, csrSortedValC, csrSortedColIndC, csrSortedRowPtrC, tol);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    float * csrSortedValA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int nnzA_native = 0;
    int * nnzPerRow_native = NULL;
    float * csrSortedValC_native = NULL;
    int * csrSortedColIndC_native = NULL;
    int * csrSortedRowPtrC_native = NULL;
    float tol_native = 0.0f;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float *)getPointer(env, csrSortedValA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    nnzA_native = (int)nnzA;
    nnzPerRow_native = (int *)getPointer(env, nnzPerRow);
    csrSortedValC_native = (float *)getPointer(env, csrSortedValC);
    csrSortedColIndC_native = (int *)getPointer(env, csrSortedColIndC);
    csrSortedRowPtrC_native = (int *)getPointer(env, csrSortedRowPtrC);
    tol_native = (float)tol;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsr2csr_compress(handle_native, m_native, n_native, descrA_native, csrSortedValA_native, csrSortedColIndA_native, csrSortedRowPtrA_native, nnzA_native, nnzPerRow_native, csrSortedValC_native, csrSortedColIndC_native, csrSortedRowPtrC_native, tol_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedColIndA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // nnzA is primitive
    // nnzPerRow is a native pointer
    // csrSortedValC is a native pointer
    // csrSortedColIndC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // tol is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsr2csr_1compressNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject descrA, jobject csrSortedValA, jobject csrSortedColIndA, jobject csrSortedRowPtrA, jint nnzA, jobject nnzPerRow, jobject csrSortedValC, jobject csrSortedColIndC, jobject csrSortedRowPtrC, jdouble tol)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzA is primitive
    if (nnzPerRow == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzPerRow' is null for cusparseDcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseDcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseDcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseDcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // tol is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsr2csr_compress(handle=%p, m=%d, n=%d, descrA=%p, csrSortedValA=%p, csrSortedColIndA=%p, csrSortedRowPtrA=%p, nnzA=%d, nnzPerRow=%p, csrSortedValC=%p, csrSortedColIndC=%p, csrSortedRowPtrC=%p, tol=%lf)\n",
        handle, m, n, descrA, csrSortedValA, csrSortedColIndA, csrSortedRowPtrA, nnzA, nnzPerRow, csrSortedValC, csrSortedColIndC, csrSortedRowPtrC, tol);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    double * csrSortedValA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int nnzA_native = 0;
    int * nnzPerRow_native = NULL;
    double * csrSortedValC_native = NULL;
    int * csrSortedColIndC_native = NULL;
    int * csrSortedRowPtrC_native = NULL;
    double tol_native = 0.0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double *)getPointer(env, csrSortedValA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    nnzA_native = (int)nnzA;
    nnzPerRow_native = (int *)getPointer(env, nnzPerRow);
    csrSortedValC_native = (double *)getPointer(env, csrSortedValC);
    csrSortedColIndC_native = (int *)getPointer(env, csrSortedColIndC);
    csrSortedRowPtrC_native = (int *)getPointer(env, csrSortedRowPtrC);
    tol_native = (double)tol;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsr2csr_compress(handle_native, m_native, n_native, descrA_native, csrSortedValA_native, csrSortedColIndA_native, csrSortedRowPtrA_native, nnzA_native, nnzPerRow_native, csrSortedValC_native, csrSortedColIndC_native, csrSortedRowPtrC_native, tol_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedColIndA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // nnzA is primitive
    // nnzPerRow is a native pointer
    // csrSortedValC is a native pointer
    // csrSortedColIndC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // tol is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsr2csr_1compressNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject descrA, jobject csrSortedValA, jobject csrSortedColIndA, jobject csrSortedRowPtrA, jint nnzA, jobject nnzPerRow, jobject csrSortedValC, jobject csrSortedColIndC, jobject csrSortedRowPtrC, jobject tol)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzA is primitive
    if (nnzPerRow == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzPerRow' is null for cusparseCcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseCcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseCcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseCcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (tol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tol' is null for cusparseCcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsr2csr_compress(handle=%p, m=%d, n=%d, descrA=%p, csrSortedValA=%p, csrSortedColIndA=%p, csrSortedRowPtrA=%p, nnzA=%d, nnzPerRow=%p, csrSortedValC=%p, csrSortedColIndC=%p, csrSortedRowPtrC=%p, tol=%p)\n",
        handle, m, n, descrA, csrSortedValA, csrSortedColIndA, csrSortedRowPtrA, nnzA, nnzPerRow, csrSortedValC, csrSortedColIndC, csrSortedRowPtrC, tol);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * csrSortedValA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int nnzA_native = 0;
    int * nnzPerRow_native = NULL;
    cuComplex * csrSortedValC_native = NULL;
    int * csrSortedColIndC_native = NULL;
    int * csrSortedRowPtrC_native = NULL;
    cuComplex tol_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex *)getPointer(env, csrSortedValA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    nnzA_native = (int)nnzA;
    nnzPerRow_native = (int *)getPointer(env, nnzPerRow);
    csrSortedValC_native = (cuComplex *)getPointer(env, csrSortedValC);
    csrSortedColIndC_native = (int *)getPointer(env, csrSortedColIndC);
    csrSortedRowPtrC_native = (int *)getPointer(env, csrSortedRowPtrC);
    if (!initNative(env, tol, tol_native)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsr2csr_compress(handle_native, m_native, n_native, descrA_native, csrSortedValA_native, csrSortedColIndA_native, csrSortedRowPtrA_native, nnzA_native, nnzPerRow_native, csrSortedValC_native, csrSortedColIndC_native, csrSortedRowPtrC_native, tol_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedColIndA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // nnzA is primitive
    // nnzPerRow is a native pointer
    // csrSortedValC is a native pointer
    // csrSortedColIndC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // tol is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsr2csr_1compressNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject descrA, jobject csrSortedValA, jobject csrSortedColIndA, jobject csrSortedRowPtrA, jint nnzA, jobject nnzPerRow, jobject csrSortedValC, jobject csrSortedColIndC, jobject csrSortedRowPtrC, jobject tol)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzA is primitive
    if (nnzPerRow == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzPerRow' is null for cusparseZcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseZcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseZcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseZcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (tol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tol' is null for cusparseZcsr2csr_compress");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsr2csr_compress(handle=%p, m=%d, n=%d, descrA=%p, csrSortedValA=%p, csrSortedColIndA=%p, csrSortedRowPtrA=%p, nnzA=%d, nnzPerRow=%p, csrSortedValC=%p, csrSortedColIndC=%p, csrSortedRowPtrC=%p, tol=%p)\n",
        handle, m, n, descrA, csrSortedValA, csrSortedColIndA, csrSortedRowPtrA, nnzA, nnzPerRow, csrSortedValC, csrSortedColIndC, csrSortedRowPtrC, tol);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * csrSortedValA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int nnzA_native = 0;
    int * nnzPerRow_native = NULL;
    cuDoubleComplex * csrSortedValC_native = NULL;
    int * csrSortedColIndC_native = NULL;
    int * csrSortedRowPtrC_native = NULL;
    cuDoubleComplex tol_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex *)getPointer(env, csrSortedValA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    nnzA_native = (int)nnzA;
    nnzPerRow_native = (int *)getPointer(env, nnzPerRow);
    csrSortedValC_native = (cuDoubleComplex *)getPointer(env, csrSortedValC);
    csrSortedColIndC_native = (int *)getPointer(env, csrSortedColIndC);
    csrSortedRowPtrC_native = (int *)getPointer(env, csrSortedRowPtrC);
    if (!initNative(env, tol, tol_native)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsr2csr_compress(handle_native, m_native, n_native, descrA_native, csrSortedValA_native, csrSortedColIndA_native, csrSortedRowPtrA_native, nnzA_native, nnzPerRow_native, csrSortedValC_native, csrSortedColIndC_native, csrSortedRowPtrC_native, tol_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedColIndA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // nnzA is primitive
    // nnzPerRow is a native pointer
    // csrSortedValC is a native pointer
    // csrSortedColIndC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // tol is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXcoo2csrNative(JNIEnv *env, jclass cls, jobject handle, jobject cooRowInd, jint nnz, jint m, jobject csrSortedRowPtr, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXcoo2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (cooRowInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'cooRowInd' is null for cusparseXcoo2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    // m is primitive
    if (csrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtr' is null for cusparseXcoo2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXcoo2csr(handle=%p, cooRowInd=%p, nnz=%d, m=%d, csrSortedRowPtr=%p, idxBase=%d)\n",
        handle, cooRowInd, nnz, m, csrSortedRowPtr, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int * cooRowInd_native = NULL;
    int nnz_native = 0;
    int m_native = 0;
    int * csrSortedRowPtr_native = NULL;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    cooRowInd_native = (int *)getPointer(env, cooRowInd);
    nnz_native = (int)nnz;
    m_native = (int)m;
    csrSortedRowPtr_native = (int *)getPointer(env, csrSortedRowPtr);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXcoo2csr(handle_native, cooRowInd_native, nnz_native, m_native, csrSortedRowPtr_native, idxBase_native);

    // Write back native variable values
    // handle is read-only
    // cooRowInd is a native pointer
    // nnz is primitive
    // m is primitive
    // csrSortedRowPtr is a native pointer
    // idxBase is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXcsr2cooNative(JNIEnv *env, jclass cls, jobject handle, jobject csrSortedRowPtr, jint nnz, jint m, jobject cooRowInd, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXcsr2coo");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtr' is null for cusparseXcsr2coo");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    // m is primitive
    if (cooRowInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'cooRowInd' is null for cusparseXcsr2coo");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXcsr2coo(handle=%p, csrSortedRowPtr=%p, nnz=%d, m=%d, cooRowInd=%p, idxBase=%d)\n",
        handle, csrSortedRowPtr, nnz, m, cooRowInd, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int * csrSortedRowPtr_native = NULL;
    int nnz_native = 0;
    int m_native = 0;
    int * cooRowInd_native = NULL;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    csrSortedRowPtr_native = (int *)getPointer(env, csrSortedRowPtr);
    nnz_native = (int)nnz;
    m_native = (int)m;
    cooRowInd_native = (int *)getPointer(env, cooRowInd);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXcsr2coo(handle_native, csrSortedRowPtr_native, nnz_native, m_native, cooRowInd_native, idxBase_native);

    // Write back native variable values
    // handle is read-only
    // csrSortedRowPtr is a native pointer
    // nnz is primitive
    // m is primitive
    // cooRowInd is a native pointer
    // idxBase is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXcsr2bsrNnzNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint m, jint n, jobject descrA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jint blockDim, jobject descrC, jobject bsrSortedRowPtrC, jobject nnzTotalDevHostPtr)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXcsr2bsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseXcsr2bsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseXcsr2bsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseXcsr2bsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseXcsr2bsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrC' is null for cusparseXcsr2bsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (nnzTotalDevHostPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzTotalDevHostPtr' is null for cusparseXcsr2bsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXcsr2bsrNnz(handle=%p, dirA=%d, m=%d, n=%d, descrA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, blockDim=%d, descrC=%p, bsrSortedRowPtrC=%p, nnzTotalDevHostPtr=%p)\n",
        handle, dirA, m, n, descrA, csrSortedRowPtrA, csrSortedColIndA, blockDim, descrC, bsrSortedRowPtrC, nnzTotalDevHostPtr);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    cusparseMatDescr_t descrC_native;
    int * bsrSortedRowPtrC_native = NULL;
    int * nnzTotalDevHostPtr_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    blockDim_native = (int)blockDim;
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    bsrSortedRowPtrC_native = (int *)getPointer(env, bsrSortedRowPtrC);
    PointerData *nnzTotalDevHostPtr_pointerData = initPointerData(env, nnzTotalDevHostPtr);
    if (nnzTotalDevHostPtr_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    nnzTotalDevHostPtr_native = (int *)nnzTotalDevHostPtr_pointerData->getPointer(env);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXcsr2bsrNnz(handle_native, dirA_native, m_native, n_native, descrA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, blockDim_native, descrC_native, bsrSortedRowPtrC_native, nnzTotalDevHostPtr_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // m is primitive
    // n is primitive
    // descrA is read-only
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // blockDim is primitive
    // descrC is read-only
    // bsrSortedRowPtrC is a native pointer
    // If the PointerData is not backed by native memory, then this call has to block
    if (!isPointerBackedByNativeMemory(env, nnzTotalDevHostPtr))
    {
        cudaDeviceSynchronize();
    }
    if (!releasePointerData(env, nnzTotalDevHostPtr_pointerData, 0)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsr2bsrNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint m, jint n, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jint blockDim, jobject descrC, jobject bsrSortedValC, jobject bsrSortedRowPtrC, jobject bsrSortedColIndC)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseScsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValC' is null for cusparseScsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrC' is null for cusparseScsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndC' is null for cusparseScsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsr2bsr(handle=%p, dirA=%d, m=%d, n=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, blockDim=%d, descrC=%p, bsrSortedValC=%p, bsrSortedRowPtrC=%p, bsrSortedColIndC=%p)\n",
        handle, dirA, m, n, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, blockDim, descrC, bsrSortedValC, bsrSortedRowPtrC, bsrSortedColIndC);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    float * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    cusparseMatDescr_t descrC_native;
    float * bsrSortedValC_native = NULL;
    int * bsrSortedRowPtrC_native = NULL;
    int * bsrSortedColIndC_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    blockDim_native = (int)blockDim;
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    bsrSortedValC_native = (float *)getPointer(env, bsrSortedValC);
    bsrSortedRowPtrC_native = (int *)getPointer(env, bsrSortedRowPtrC);
    bsrSortedColIndC_native = (int *)getPointer(env, bsrSortedColIndC);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsr2bsr(handle_native, dirA_native, m_native, n_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, blockDim_native, descrC_native, bsrSortedValC_native, bsrSortedRowPtrC_native, bsrSortedColIndC_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // m is primitive
    // n is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // blockDim is primitive
    // descrC is read-only
    // bsrSortedValC is a native pointer
    // bsrSortedRowPtrC is a native pointer
    // bsrSortedColIndC is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsr2bsrNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint m, jint n, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jint blockDim, jobject descrC, jobject bsrSortedValC, jobject bsrSortedRowPtrC, jobject bsrSortedColIndC)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseDcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValC' is null for cusparseDcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrC' is null for cusparseDcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndC' is null for cusparseDcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsr2bsr(handle=%p, dirA=%d, m=%d, n=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, blockDim=%d, descrC=%p, bsrSortedValC=%p, bsrSortedRowPtrC=%p, bsrSortedColIndC=%p)\n",
        handle, dirA, m, n, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, blockDim, descrC, bsrSortedValC, bsrSortedRowPtrC, bsrSortedColIndC);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    double * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    cusparseMatDescr_t descrC_native;
    double * bsrSortedValC_native = NULL;
    int * bsrSortedRowPtrC_native = NULL;
    int * bsrSortedColIndC_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    blockDim_native = (int)blockDim;
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    bsrSortedValC_native = (double *)getPointer(env, bsrSortedValC);
    bsrSortedRowPtrC_native = (int *)getPointer(env, bsrSortedRowPtrC);
    bsrSortedColIndC_native = (int *)getPointer(env, bsrSortedColIndC);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsr2bsr(handle_native, dirA_native, m_native, n_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, blockDim_native, descrC_native, bsrSortedValC_native, bsrSortedRowPtrC_native, bsrSortedColIndC_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // m is primitive
    // n is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // blockDim is primitive
    // descrC is read-only
    // bsrSortedValC is a native pointer
    // bsrSortedRowPtrC is a native pointer
    // bsrSortedColIndC is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsr2bsrNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint m, jint n, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jint blockDim, jobject descrC, jobject bsrSortedValC, jobject bsrSortedRowPtrC, jobject bsrSortedColIndC)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseCcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValC' is null for cusparseCcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrC' is null for cusparseCcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndC' is null for cusparseCcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsr2bsr(handle=%p, dirA=%d, m=%d, n=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, blockDim=%d, descrC=%p, bsrSortedValC=%p, bsrSortedRowPtrC=%p, bsrSortedColIndC=%p)\n",
        handle, dirA, m, n, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, blockDim, descrC, bsrSortedValC, bsrSortedRowPtrC, bsrSortedColIndC);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    cusparseMatDescr_t descrC_native;
    cuComplex * bsrSortedValC_native = NULL;
    int * bsrSortedRowPtrC_native = NULL;
    int * bsrSortedColIndC_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    blockDim_native = (int)blockDim;
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    bsrSortedValC_native = (cuComplex *)getPointer(env, bsrSortedValC);
    bsrSortedRowPtrC_native = (int *)getPointer(env, bsrSortedRowPtrC);
    bsrSortedColIndC_native = (int *)getPointer(env, bsrSortedColIndC);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsr2bsr(handle_native, dirA_native, m_native, n_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, blockDim_native, descrC_native, bsrSortedValC_native, bsrSortedRowPtrC_native, bsrSortedColIndC_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // m is primitive
    // n is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // blockDim is primitive
    // descrC is read-only
    // bsrSortedValC is a native pointer
    // bsrSortedRowPtrC is a native pointer
    // bsrSortedColIndC is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsr2bsrNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint m, jint n, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jint blockDim, jobject descrC, jobject bsrSortedValC, jobject bsrSortedRowPtrC, jobject bsrSortedColIndC)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseZcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValC' is null for cusparseZcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrC' is null for cusparseZcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndC' is null for cusparseZcsr2bsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsr2bsr(handle=%p, dirA=%d, m=%d, n=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, blockDim=%d, descrC=%p, bsrSortedValC=%p, bsrSortedRowPtrC=%p, bsrSortedColIndC=%p)\n",
        handle, dirA, m, n, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, blockDim, descrC, bsrSortedValC, bsrSortedRowPtrC, bsrSortedColIndC);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    cusparseMatDescr_t descrC_native;
    cuDoubleComplex * bsrSortedValC_native = NULL;
    int * bsrSortedRowPtrC_native = NULL;
    int * bsrSortedColIndC_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    blockDim_native = (int)blockDim;
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    bsrSortedValC_native = (cuDoubleComplex *)getPointer(env, bsrSortedValC);
    bsrSortedRowPtrC_native = (int *)getPointer(env, bsrSortedRowPtrC);
    bsrSortedColIndC_native = (int *)getPointer(env, bsrSortedColIndC);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsr2bsr(handle_native, dirA_native, m_native, n_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, blockDim_native, descrC_native, bsrSortedValC_native, bsrSortedRowPtrC_native, bsrSortedColIndC_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // m is primitive
    // n is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // blockDim is primitive
    // descrC is read-only
    // bsrSortedValC is a native pointer
    // bsrSortedRowPtrC is a native pointer
    // bsrSortedColIndC is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsr2csrNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject descrC, jobject csrSortedValC, jobject csrSortedRowPtrC, jobject csrSortedColIndC)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseSbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseSbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseSbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseSbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseSbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseSbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseSbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsr2csr(handle=%p, dirA=%d, mb=%d, nb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, descrC=%p, csrSortedValC=%p, csrSortedRowPtrC=%p, csrSortedColIndC=%p)\n",
        handle, dirA, mb, nb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, descrC, csrSortedValC, csrSortedRowPtrC, csrSortedColIndC);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nb_native = 0;
    cusparseMatDescr_t descrA_native;
    float * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    cusparseMatDescr_t descrC_native;
    float * csrSortedValC_native = NULL;
    int * csrSortedRowPtrC_native = NULL;
    int * csrSortedColIndC_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (float *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedValC_native = (float *)getPointer(env, csrSortedValC);
    csrSortedRowPtrC_native = (int *)getPointer(env, csrSortedRowPtrC);
    csrSortedColIndC_native = (int *)getPointer(env, csrSortedColIndC);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsr2csr(handle_native, dirA_native, mb_native, nb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, descrC_native, csrSortedValC_native, csrSortedRowPtrC_native, csrSortedColIndC_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // descrC is read-only
    // csrSortedValC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // csrSortedColIndC is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsr2csrNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject descrC, jobject csrSortedValC, jobject csrSortedRowPtrC, jobject csrSortedColIndC)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseDbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseDbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseDbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseDbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseDbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseDbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseDbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsr2csr(handle=%p, dirA=%d, mb=%d, nb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, descrC=%p, csrSortedValC=%p, csrSortedRowPtrC=%p, csrSortedColIndC=%p)\n",
        handle, dirA, mb, nb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, descrC, csrSortedValC, csrSortedRowPtrC, csrSortedColIndC);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nb_native = 0;
    cusparseMatDescr_t descrA_native;
    double * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    cusparseMatDescr_t descrC_native;
    double * csrSortedValC_native = NULL;
    int * csrSortedRowPtrC_native = NULL;
    int * csrSortedColIndC_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (double *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedValC_native = (double *)getPointer(env, csrSortedValC);
    csrSortedRowPtrC_native = (int *)getPointer(env, csrSortedRowPtrC);
    csrSortedColIndC_native = (int *)getPointer(env, csrSortedColIndC);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsr2csr(handle_native, dirA_native, mb_native, nb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, descrC_native, csrSortedValC_native, csrSortedRowPtrC_native, csrSortedColIndC_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // descrC is read-only
    // csrSortedValC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // csrSortedColIndC is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsr2csrNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject descrC, jobject csrSortedValC, jobject csrSortedRowPtrC, jobject csrSortedColIndC)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseCbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseCbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseCbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseCbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseCbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseCbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseCbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsr2csr(handle=%p, dirA=%d, mb=%d, nb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, descrC=%p, csrSortedValC=%p, csrSortedRowPtrC=%p, csrSortedColIndC=%p)\n",
        handle, dirA, mb, nb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, descrC, csrSortedValC, csrSortedRowPtrC, csrSortedColIndC);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    cusparseMatDescr_t descrC_native;
    cuComplex * csrSortedValC_native = NULL;
    int * csrSortedRowPtrC_native = NULL;
    int * csrSortedColIndC_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuComplex *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedValC_native = (cuComplex *)getPointer(env, csrSortedValC);
    csrSortedRowPtrC_native = (int *)getPointer(env, csrSortedRowPtrC);
    csrSortedColIndC_native = (int *)getPointer(env, csrSortedColIndC);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsr2csr(handle_native, dirA_native, mb_native, nb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, descrC_native, csrSortedValC_native, csrSortedRowPtrC_native, csrSortedColIndC_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // descrC is read-only
    // csrSortedValC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // csrSortedColIndC is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsr2csrNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject descrC, jobject csrSortedValC, jobject csrSortedRowPtrC, jobject csrSortedColIndC)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseZbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseZbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseZbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseZbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseZbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseZbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseZbsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsr2csr(handle=%p, dirA=%d, mb=%d, nb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, descrC=%p, csrSortedValC=%p, csrSortedRowPtrC=%p, csrSortedColIndC=%p)\n",
        handle, dirA, mb, nb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, descrC, csrSortedValC, csrSortedRowPtrC, csrSortedColIndC);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int blockDim_native = 0;
    cusparseMatDescr_t descrC_native;
    cuDoubleComplex * csrSortedValC_native = NULL;
    int * csrSortedRowPtrC_native = NULL;
    int * csrSortedColIndC_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuDoubleComplex *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedValC_native = (cuDoubleComplex *)getPointer(env, csrSortedValC);
    csrSortedRowPtrC_native = (int *)getPointer(env, csrSortedRowPtrC);
    csrSortedColIndC_native = (int *)getPointer(env, csrSortedColIndC);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsr2csr(handle_native, dirA_native, mb_native, nb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, descrC_native, csrSortedValC_native, csrSortedRowPtrC_native, csrSortedColIndC_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // descrC is read-only
    // csrSortedValC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // csrSortedColIndC is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgebsr2gebsc_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint mb, jint nb, jint nnzb, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint rowBlockDim, jint colBlockDim, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgebsr2gebsc_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSgebsr2gebsc_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSgebsr2gebsc_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSgebsr2gebsc_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseSgebsr2gebsc_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgebsr2gebsc_bufferSize(handle=%p, mb=%d, nb=%d, nnzb=%d, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, rowBlockDim=%d, colBlockDim=%d, pBufferSizeInBytes=%p)\n",
        handle, mb, nb, nnzb, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, rowBlockDim, colBlockDim, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    float * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int rowBlockDim_native = 0;
    int colBlockDim_native = 0;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    bsrSortedVal_native = (float *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    rowBlockDim_native = (int)rowBlockDim;
    colBlockDim_native = (int)colBlockDim;
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgebsr2gebsc_bufferSize(handle_native, mb_native, nb_native, nnzb_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, rowBlockDim_native, colBlockDim_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgebsr2gebsc_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint mb, jint nb, jint nnzb, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint rowBlockDim, jint colBlockDim, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgebsr2gebsc_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDgebsr2gebsc_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDgebsr2gebsc_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDgebsr2gebsc_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseDgebsr2gebsc_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgebsr2gebsc_bufferSize(handle=%p, mb=%d, nb=%d, nnzb=%d, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, rowBlockDim=%d, colBlockDim=%d, pBufferSizeInBytes=%p)\n",
        handle, mb, nb, nnzb, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, rowBlockDim, colBlockDim, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    double * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int rowBlockDim_native = 0;
    int colBlockDim_native = 0;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    bsrSortedVal_native = (double *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    rowBlockDim_native = (int)rowBlockDim;
    colBlockDim_native = (int)colBlockDim;
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgebsr2gebsc_bufferSize(handle_native, mb_native, nb_native, nnzb_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, rowBlockDim_native, colBlockDim_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgebsr2gebsc_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint mb, jint nb, jint nnzb, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint rowBlockDim, jint colBlockDim, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgebsr2gebsc_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCgebsr2gebsc_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCgebsr2gebsc_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCgebsr2gebsc_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseCgebsr2gebsc_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgebsr2gebsc_bufferSize(handle=%p, mb=%d, nb=%d, nnzb=%d, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, rowBlockDim=%d, colBlockDim=%d, pBufferSizeInBytes=%p)\n",
        handle, mb, nb, nnzb, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, rowBlockDim, colBlockDim, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    cuComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int rowBlockDim_native = 0;
    int colBlockDim_native = 0;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    bsrSortedVal_native = (cuComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    rowBlockDim_native = (int)rowBlockDim;
    colBlockDim_native = (int)colBlockDim;
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgebsr2gebsc_bufferSize(handle_native, mb_native, nb_native, nnzb_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, rowBlockDim_native, colBlockDim_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgebsr2gebsc_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint mb, jint nb, jint nnzb, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint rowBlockDim, jint colBlockDim, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgebsr2gebsc_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZgebsr2gebsc_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZgebsr2gebsc_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZgebsr2gebsc_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseZgebsr2gebsc_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgebsr2gebsc_bufferSize(handle=%p, mb=%d, nb=%d, nnzb=%d, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, rowBlockDim=%d, colBlockDim=%d, pBufferSizeInBytes=%p)\n",
        handle, mb, nb, nnzb, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, rowBlockDim, colBlockDim, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    cuDoubleComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int rowBlockDim_native = 0;
    int colBlockDim_native = 0;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    bsrSortedVal_native = (cuDoubleComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    rowBlockDim_native = (int)rowBlockDim;
    colBlockDim_native = (int)colBlockDim;
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgebsr2gebsc_bufferSize(handle_native, mb_native, nb_native, nnzb_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, rowBlockDim_native, colBlockDim_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgebsr2gebscNative(JNIEnv *env, jclass cls, jobject handle, jint mb, jint nb, jint nnzb, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint rowBlockDim, jint colBlockDim, jobject bscVal, jobject bscRowInd, jobject bscColPtr, jint copyValues, jint idxBase, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (bscVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bscVal' is null for cusparseSgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bscRowInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bscRowInd' is null for cusparseSgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bscColPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bscColPtr' is null for cusparseSgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // copyValues is primitive
    // idxBase is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgebsr2gebsc(handle=%p, mb=%d, nb=%d, nnzb=%d, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, rowBlockDim=%d, colBlockDim=%d, bscVal=%p, bscRowInd=%p, bscColPtr=%p, copyValues=%d, idxBase=%d, pBuffer=%p)\n",
        handle, mb, nb, nnzb, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, rowBlockDim, colBlockDim, bscVal, bscRowInd, bscColPtr, copyValues, idxBase, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    float * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int rowBlockDim_native = 0;
    int colBlockDim_native = 0;
    float * bscVal_native = NULL;
    int * bscRowInd_native = NULL;
    int * bscColPtr_native = NULL;
    cusparseAction_t copyValues_native;
    cusparseIndexBase_t idxBase_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    bsrSortedVal_native = (float *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    rowBlockDim_native = (int)rowBlockDim;
    colBlockDim_native = (int)colBlockDim;
    bscVal_native = (float *)getPointer(env, bscVal);
    bscRowInd_native = (int *)getPointer(env, bscRowInd);
    bscColPtr_native = (int *)getPointer(env, bscColPtr);
    copyValues_native = (cusparseAction_t)copyValues;
    idxBase_native = (cusparseIndexBase_t)idxBase;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgebsr2gebsc(handle_native, mb_native, nb_native, nnzb_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, rowBlockDim_native, colBlockDim_native, bscVal_native, bscRowInd_native, bscColPtr_native, copyValues_native, idxBase_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // rowBlockDim is primitive
    // colBlockDim is primitive
    // bscVal is a native pointer
    // bscRowInd is a native pointer
    // bscColPtr is a native pointer
    // copyValues is primitive
    // idxBase is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgebsr2gebscNative(JNIEnv *env, jclass cls, jobject handle, jint mb, jint nb, jint nnzb, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint rowBlockDim, jint colBlockDim, jobject bscVal, jobject bscRowInd, jobject bscColPtr, jint copyValues, jint idxBase, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (bscVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bscVal' is null for cusparseDgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bscRowInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bscRowInd' is null for cusparseDgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bscColPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bscColPtr' is null for cusparseDgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // copyValues is primitive
    // idxBase is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgebsr2gebsc(handle=%p, mb=%d, nb=%d, nnzb=%d, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, rowBlockDim=%d, colBlockDim=%d, bscVal=%p, bscRowInd=%p, bscColPtr=%p, copyValues=%d, idxBase=%d, pBuffer=%p)\n",
        handle, mb, nb, nnzb, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, rowBlockDim, colBlockDim, bscVal, bscRowInd, bscColPtr, copyValues, idxBase, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    double * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int rowBlockDim_native = 0;
    int colBlockDim_native = 0;
    double * bscVal_native = NULL;
    int * bscRowInd_native = NULL;
    int * bscColPtr_native = NULL;
    cusparseAction_t copyValues_native;
    cusparseIndexBase_t idxBase_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    bsrSortedVal_native = (double *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    rowBlockDim_native = (int)rowBlockDim;
    colBlockDim_native = (int)colBlockDim;
    bscVal_native = (double *)getPointer(env, bscVal);
    bscRowInd_native = (int *)getPointer(env, bscRowInd);
    bscColPtr_native = (int *)getPointer(env, bscColPtr);
    copyValues_native = (cusparseAction_t)copyValues;
    idxBase_native = (cusparseIndexBase_t)idxBase;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgebsr2gebsc(handle_native, mb_native, nb_native, nnzb_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, rowBlockDim_native, colBlockDim_native, bscVal_native, bscRowInd_native, bscColPtr_native, copyValues_native, idxBase_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // rowBlockDim is primitive
    // colBlockDim is primitive
    // bscVal is a native pointer
    // bscRowInd is a native pointer
    // bscColPtr is a native pointer
    // copyValues is primitive
    // idxBase is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgebsr2gebscNative(JNIEnv *env, jclass cls, jobject handle, jint mb, jint nb, jint nnzb, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint rowBlockDim, jint colBlockDim, jobject bscVal, jobject bscRowInd, jobject bscColPtr, jint copyValues, jint idxBase, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (bscVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bscVal' is null for cusparseCgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bscRowInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bscRowInd' is null for cusparseCgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bscColPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bscColPtr' is null for cusparseCgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // copyValues is primitive
    // idxBase is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgebsr2gebsc(handle=%p, mb=%d, nb=%d, nnzb=%d, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, rowBlockDim=%d, colBlockDim=%d, bscVal=%p, bscRowInd=%p, bscColPtr=%p, copyValues=%d, idxBase=%d, pBuffer=%p)\n",
        handle, mb, nb, nnzb, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, rowBlockDim, colBlockDim, bscVal, bscRowInd, bscColPtr, copyValues, idxBase, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    cuComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int rowBlockDim_native = 0;
    int colBlockDim_native = 0;
    cuComplex * bscVal_native = NULL;
    int * bscRowInd_native = NULL;
    int * bscColPtr_native = NULL;
    cusparseAction_t copyValues_native;
    cusparseIndexBase_t idxBase_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    bsrSortedVal_native = (cuComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    rowBlockDim_native = (int)rowBlockDim;
    colBlockDim_native = (int)colBlockDim;
    bscVal_native = (cuComplex *)getPointer(env, bscVal);
    bscRowInd_native = (int *)getPointer(env, bscRowInd);
    bscColPtr_native = (int *)getPointer(env, bscColPtr);
    copyValues_native = (cusparseAction_t)copyValues;
    idxBase_native = (cusparseIndexBase_t)idxBase;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgebsr2gebsc(handle_native, mb_native, nb_native, nnzb_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, rowBlockDim_native, colBlockDim_native, bscVal_native, bscRowInd_native, bscColPtr_native, copyValues_native, idxBase_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // rowBlockDim is primitive
    // colBlockDim is primitive
    // bscVal is a native pointer
    // bscRowInd is a native pointer
    // bscColPtr is a native pointer
    // copyValues is primitive
    // idxBase is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgebsr2gebscNative(JNIEnv *env, jclass cls, jobject handle, jint mb, jint nb, jint nnzb, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint rowBlockDim, jint colBlockDim, jobject bscVal, jobject bscRowInd, jobject bscColPtr, jint copyValues, jint idxBase, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (bscVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bscVal' is null for cusparseZgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bscRowInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bscRowInd' is null for cusparseZgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bscColPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bscColPtr' is null for cusparseZgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // copyValues is primitive
    // idxBase is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZgebsr2gebsc");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgebsr2gebsc(handle=%p, mb=%d, nb=%d, nnzb=%d, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, rowBlockDim=%d, colBlockDim=%d, bscVal=%p, bscRowInd=%p, bscColPtr=%p, copyValues=%d, idxBase=%d, pBuffer=%p)\n",
        handle, mb, nb, nnzb, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, rowBlockDim, colBlockDim, bscVal, bscRowInd, bscColPtr, copyValues, idxBase, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    cuDoubleComplex * bsrSortedVal_native = NULL;
    int * bsrSortedRowPtr_native = NULL;
    int * bsrSortedColInd_native = NULL;
    int rowBlockDim_native = 0;
    int colBlockDim_native = 0;
    cuDoubleComplex * bscVal_native = NULL;
    int * bscRowInd_native = NULL;
    int * bscColPtr_native = NULL;
    cusparseAction_t copyValues_native;
    cusparseIndexBase_t idxBase_native;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    bsrSortedVal_native = (cuDoubleComplex *)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int *)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int *)getPointer(env, bsrSortedColInd);
    rowBlockDim_native = (int)rowBlockDim;
    colBlockDim_native = (int)colBlockDim;
    bscVal_native = (cuDoubleComplex *)getPointer(env, bscVal);
    bscRowInd_native = (int *)getPointer(env, bscRowInd);
    bscColPtr_native = (int *)getPointer(env, bscColPtr);
    copyValues_native = (cusparseAction_t)copyValues;
    idxBase_native = (cusparseIndexBase_t)idxBase;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgebsr2gebsc(handle_native, mb_native, nb_native, nnzb_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, rowBlockDim_native, colBlockDim_native, bscVal_native, bscRowInd_native, bscColPtr_native, copyValues_native, idxBase_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // rowBlockDim is primitive
    // colBlockDim is primitive
    // bscVal is a native pointer
    // bscRowInd is a native pointer
    // bscColPtr is a native pointer
    // copyValues is primitive
    // idxBase is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgebsr2csrNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint rowBlockDim, jint colBlockDim, jobject descrC, jobject csrSortedValC, jobject csrSortedRowPtrC, jobject csrSortedColIndC)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseSgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseSgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseSgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseSgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseSgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseSgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseSgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgebsr2csr(handle=%p, dirA=%d, mb=%d, nb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, rowBlockDim=%d, colBlockDim=%d, descrC=%p, csrSortedValC=%p, csrSortedRowPtrC=%p, csrSortedColIndC=%p)\n",
        handle, dirA, mb, nb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, rowBlockDim, colBlockDim, descrC, csrSortedValC, csrSortedRowPtrC, csrSortedColIndC);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nb_native = 0;
    cusparseMatDescr_t descrA_native;
    float * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int rowBlockDim_native = 0;
    int colBlockDim_native = 0;
    cusparseMatDescr_t descrC_native;
    float * csrSortedValC_native = NULL;
    int * csrSortedRowPtrC_native = NULL;
    int * csrSortedColIndC_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (float *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    rowBlockDim_native = (int)rowBlockDim;
    colBlockDim_native = (int)colBlockDim;
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedValC_native = (float *)getPointer(env, csrSortedValC);
    csrSortedRowPtrC_native = (int *)getPointer(env, csrSortedRowPtrC);
    csrSortedColIndC_native = (int *)getPointer(env, csrSortedColIndC);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgebsr2csr(handle_native, dirA_native, mb_native, nb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, rowBlockDim_native, colBlockDim_native, descrC_native, csrSortedValC_native, csrSortedRowPtrC_native, csrSortedColIndC_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // rowBlockDim is primitive
    // colBlockDim is primitive
    // descrC is read-only
    // csrSortedValC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // csrSortedColIndC is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgebsr2csrNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint rowBlockDim, jint colBlockDim, jobject descrC, jobject csrSortedValC, jobject csrSortedRowPtrC, jobject csrSortedColIndC)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseDgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseDgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseDgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseDgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseDgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseDgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseDgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgebsr2csr(handle=%p, dirA=%d, mb=%d, nb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, rowBlockDim=%d, colBlockDim=%d, descrC=%p, csrSortedValC=%p, csrSortedRowPtrC=%p, csrSortedColIndC=%p)\n",
        handle, dirA, mb, nb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, rowBlockDim, colBlockDim, descrC, csrSortedValC, csrSortedRowPtrC, csrSortedColIndC);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nb_native = 0;
    cusparseMatDescr_t descrA_native;
    double * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int rowBlockDim_native = 0;
    int colBlockDim_native = 0;
    cusparseMatDescr_t descrC_native;
    double * csrSortedValC_native = NULL;
    int * csrSortedRowPtrC_native = NULL;
    int * csrSortedColIndC_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (double *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    rowBlockDim_native = (int)rowBlockDim;
    colBlockDim_native = (int)colBlockDim;
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedValC_native = (double *)getPointer(env, csrSortedValC);
    csrSortedRowPtrC_native = (int *)getPointer(env, csrSortedRowPtrC);
    csrSortedColIndC_native = (int *)getPointer(env, csrSortedColIndC);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgebsr2csr(handle_native, dirA_native, mb_native, nb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, rowBlockDim_native, colBlockDim_native, descrC_native, csrSortedValC_native, csrSortedRowPtrC_native, csrSortedColIndC_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // rowBlockDim is primitive
    // colBlockDim is primitive
    // descrC is read-only
    // csrSortedValC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // csrSortedColIndC is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgebsr2csrNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint rowBlockDim, jint colBlockDim, jobject descrC, jobject csrSortedValC, jobject csrSortedRowPtrC, jobject csrSortedColIndC)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseCgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseCgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseCgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseCgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseCgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseCgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseCgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgebsr2csr(handle=%p, dirA=%d, mb=%d, nb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, rowBlockDim=%d, colBlockDim=%d, descrC=%p, csrSortedValC=%p, csrSortedRowPtrC=%p, csrSortedColIndC=%p)\n",
        handle, dirA, mb, nb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, rowBlockDim, colBlockDim, descrC, csrSortedValC, csrSortedRowPtrC, csrSortedColIndC);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int rowBlockDim_native = 0;
    int colBlockDim_native = 0;
    cusparseMatDescr_t descrC_native;
    cuComplex * csrSortedValC_native = NULL;
    int * csrSortedRowPtrC_native = NULL;
    int * csrSortedColIndC_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuComplex *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    rowBlockDim_native = (int)rowBlockDim;
    colBlockDim_native = (int)colBlockDim;
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedValC_native = (cuComplex *)getPointer(env, csrSortedValC);
    csrSortedRowPtrC_native = (int *)getPointer(env, csrSortedRowPtrC);
    csrSortedColIndC_native = (int *)getPointer(env, csrSortedColIndC);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgebsr2csr(handle_native, dirA_native, mb_native, nb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, rowBlockDim_native, colBlockDim_native, descrC_native, csrSortedValC_native, csrSortedRowPtrC_native, csrSortedColIndC_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // rowBlockDim is primitive
    // colBlockDim is primitive
    // descrC is read-only
    // csrSortedValC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // csrSortedColIndC is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgebsr2csrNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint rowBlockDim, jint colBlockDim, jobject descrC, jobject csrSortedValC, jobject csrSortedRowPtrC, jobject csrSortedColIndC)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseZgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseZgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseZgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseZgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseZgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseZgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseZgebsr2csr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgebsr2csr(handle=%p, dirA=%d, mb=%d, nb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, rowBlockDim=%d, colBlockDim=%d, descrC=%p, csrSortedValC=%p, csrSortedRowPtrC=%p, csrSortedColIndC=%p)\n",
        handle, dirA, mb, nb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, rowBlockDim, colBlockDim, descrC, csrSortedValC, csrSortedRowPtrC, csrSortedColIndC);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int rowBlockDim_native = 0;
    int colBlockDim_native = 0;
    cusparseMatDescr_t descrC_native;
    cuDoubleComplex * csrSortedValC_native = NULL;
    int * csrSortedRowPtrC_native = NULL;
    int * csrSortedColIndC_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuDoubleComplex *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    rowBlockDim_native = (int)rowBlockDim;
    colBlockDim_native = (int)colBlockDim;
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedValC_native = (cuDoubleComplex *)getPointer(env, csrSortedValC);
    csrSortedRowPtrC_native = (int *)getPointer(env, csrSortedRowPtrC);
    csrSortedColIndC_native = (int *)getPointer(env, csrSortedColIndC);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgebsr2csr(handle_native, dirA_native, mb_native, nb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, rowBlockDim_native, colBlockDim_native, descrC_native, csrSortedValC_native, csrSortedRowPtrC_native, csrSortedColIndC_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // rowBlockDim is primitive
    // colBlockDim is primitive
    // descrC is read-only
    // csrSortedValC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // csrSortedColIndC is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsr2gebsr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint m, jint n, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jint rowBlockDim, jint colBlockDim, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseScsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsr2gebsr_bufferSize(handle=%p, dirA=%d, m=%d, n=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, rowBlockDim=%d, colBlockDim=%d, pBufferSizeInBytes=%p)\n",
        handle, dirA, m, n, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, rowBlockDim, colBlockDim, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    float * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    int rowBlockDim_native = 0;
    int colBlockDim_native = 0;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    rowBlockDim_native = (int)rowBlockDim;
    colBlockDim_native = (int)colBlockDim;
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsr2gebsr_bufferSize(handle_native, dirA_native, m_native, n_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, rowBlockDim_native, colBlockDim_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // m is primitive
    // n is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsr2gebsr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint m, jint n, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jint rowBlockDim, jint colBlockDim, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseDcsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsr2gebsr_bufferSize(handle=%p, dirA=%d, m=%d, n=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, rowBlockDim=%d, colBlockDim=%d, pBufferSizeInBytes=%p)\n",
        handle, dirA, m, n, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, rowBlockDim, colBlockDim, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    double * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    int rowBlockDim_native = 0;
    int colBlockDim_native = 0;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    rowBlockDim_native = (int)rowBlockDim;
    colBlockDim_native = (int)colBlockDim;
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsr2gebsr_bufferSize(handle_native, dirA_native, m_native, n_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, rowBlockDim_native, colBlockDim_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // m is primitive
    // n is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsr2gebsr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint m, jint n, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jint rowBlockDim, jint colBlockDim, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseCcsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsr2gebsr_bufferSize(handle=%p, dirA=%d, m=%d, n=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, rowBlockDim=%d, colBlockDim=%d, pBufferSizeInBytes=%p)\n",
        handle, dirA, m, n, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, rowBlockDim, colBlockDim, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    int rowBlockDim_native = 0;
    int colBlockDim_native = 0;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    rowBlockDim_native = (int)rowBlockDim;
    colBlockDim_native = (int)colBlockDim;
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsr2gebsr_bufferSize(handle_native, dirA_native, m_native, n_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, rowBlockDim_native, colBlockDim_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // m is primitive
    // n is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsr2gebsr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint m, jint n, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jint rowBlockDim, jint colBlockDim, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseZcsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsr2gebsr_bufferSize(handle=%p, dirA=%d, m=%d, n=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, rowBlockDim=%d, colBlockDim=%d, pBufferSizeInBytes=%p)\n",
        handle, dirA, m, n, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, rowBlockDim, colBlockDim, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    int rowBlockDim_native = 0;
    int colBlockDim_native = 0;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    rowBlockDim_native = (int)rowBlockDim;
    colBlockDim_native = (int)colBlockDim;
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsr2gebsr_bufferSize(handle_native, dirA_native, m_native, n_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, rowBlockDim_native, colBlockDim_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // m is primitive
    // n is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXcsr2gebsrNnzNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint m, jint n, jobject descrA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject descrC, jobject bsrSortedRowPtrC, jint rowBlockDim, jint colBlockDim, jobject nnzTotalDevHostPtr, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXcsr2gebsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseXcsr2gebsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseXcsr2gebsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseXcsr2gebsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseXcsr2gebsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrC' is null for cusparseXcsr2gebsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (nnzTotalDevHostPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzTotalDevHostPtr' is null for cusparseXcsr2gebsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseXcsr2gebsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXcsr2gebsrNnz(handle=%p, dirA=%d, m=%d, n=%d, descrA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, descrC=%p, bsrSortedRowPtrC=%p, rowBlockDim=%d, colBlockDim=%d, nnzTotalDevHostPtr=%p, pBuffer=%p)\n",
        handle, dirA, m, n, descrA, csrSortedRowPtrA, csrSortedColIndA, descrC, bsrSortedRowPtrC, rowBlockDim, colBlockDim, nnzTotalDevHostPtr, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    cusparseMatDescr_t descrC_native;
    int * bsrSortedRowPtrC_native = NULL;
    int rowBlockDim_native = 0;
    int colBlockDim_native = 0;
    int * nnzTotalDevHostPtr_native = NULL;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    bsrSortedRowPtrC_native = (int *)getPointer(env, bsrSortedRowPtrC);
    rowBlockDim_native = (int)rowBlockDim;
    colBlockDim_native = (int)colBlockDim;
    PointerData *nnzTotalDevHostPtr_pointerData = initPointerData(env, nnzTotalDevHostPtr);
    if (nnzTotalDevHostPtr_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    nnzTotalDevHostPtr_native = (int *)nnzTotalDevHostPtr_pointerData->getPointer(env);
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXcsr2gebsrNnz(handle_native, dirA_native, m_native, n_native, descrA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, descrC_native, bsrSortedRowPtrC_native, rowBlockDim_native, colBlockDim_native, nnzTotalDevHostPtr_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // m is primitive
    // n is primitive
    // descrA is read-only
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // descrC is read-only
    // bsrSortedRowPtrC is a native pointer
    // rowBlockDim is primitive
    // colBlockDim is primitive
    // If the PointerData is not backed by native memory, then this call has to block
    if (!isPointerBackedByNativeMemory(env, nnzTotalDevHostPtr))
    {
        cudaDeviceSynchronize();
    }
    if (!releasePointerData(env, nnzTotalDevHostPtr_pointerData, 0)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsr2gebsrNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint m, jint n, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject descrC, jobject bsrSortedValC, jobject bsrSortedRowPtrC, jobject bsrSortedColIndC, jint rowBlockDim, jint colBlockDim, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseScsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValC' is null for cusparseScsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrC' is null for cusparseScsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndC' is null for cusparseScsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseScsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsr2gebsr(handle=%p, dirA=%d, m=%d, n=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, descrC=%p, bsrSortedValC=%p, bsrSortedRowPtrC=%p, bsrSortedColIndC=%p, rowBlockDim=%d, colBlockDim=%d, pBuffer=%p)\n",
        handle, dirA, m, n, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, descrC, bsrSortedValC, bsrSortedRowPtrC, bsrSortedColIndC, rowBlockDim, colBlockDim, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    float * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    cusparseMatDescr_t descrC_native;
    float * bsrSortedValC_native = NULL;
    int * bsrSortedRowPtrC_native = NULL;
    int * bsrSortedColIndC_native = NULL;
    int rowBlockDim_native = 0;
    int colBlockDim_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    bsrSortedValC_native = (float *)getPointer(env, bsrSortedValC);
    bsrSortedRowPtrC_native = (int *)getPointer(env, bsrSortedRowPtrC);
    bsrSortedColIndC_native = (int *)getPointer(env, bsrSortedColIndC);
    rowBlockDim_native = (int)rowBlockDim;
    colBlockDim_native = (int)colBlockDim;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsr2gebsr(handle_native, dirA_native, m_native, n_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, descrC_native, bsrSortedValC_native, bsrSortedRowPtrC_native, bsrSortedColIndC_native, rowBlockDim_native, colBlockDim_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // m is primitive
    // n is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // descrC is read-only
    // bsrSortedValC is a native pointer
    // bsrSortedRowPtrC is a native pointer
    // bsrSortedColIndC is a native pointer
    // rowBlockDim is primitive
    // colBlockDim is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsr2gebsrNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint m, jint n, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject descrC, jobject bsrSortedValC, jobject bsrSortedRowPtrC, jobject bsrSortedColIndC, jint rowBlockDim, jint colBlockDim, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseDcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValC' is null for cusparseDcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrC' is null for cusparseDcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndC' is null for cusparseDcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsr2gebsr(handle=%p, dirA=%d, m=%d, n=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, descrC=%p, bsrSortedValC=%p, bsrSortedRowPtrC=%p, bsrSortedColIndC=%p, rowBlockDim=%d, colBlockDim=%d, pBuffer=%p)\n",
        handle, dirA, m, n, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, descrC, bsrSortedValC, bsrSortedRowPtrC, bsrSortedColIndC, rowBlockDim, colBlockDim, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    double * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    cusparseMatDescr_t descrC_native;
    double * bsrSortedValC_native = NULL;
    int * bsrSortedRowPtrC_native = NULL;
    int * bsrSortedColIndC_native = NULL;
    int rowBlockDim_native = 0;
    int colBlockDim_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    bsrSortedValC_native = (double *)getPointer(env, bsrSortedValC);
    bsrSortedRowPtrC_native = (int *)getPointer(env, bsrSortedRowPtrC);
    bsrSortedColIndC_native = (int *)getPointer(env, bsrSortedColIndC);
    rowBlockDim_native = (int)rowBlockDim;
    colBlockDim_native = (int)colBlockDim;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsr2gebsr(handle_native, dirA_native, m_native, n_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, descrC_native, bsrSortedValC_native, bsrSortedRowPtrC_native, bsrSortedColIndC_native, rowBlockDim_native, colBlockDim_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // m is primitive
    // n is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // descrC is read-only
    // bsrSortedValC is a native pointer
    // bsrSortedRowPtrC is a native pointer
    // bsrSortedColIndC is a native pointer
    // rowBlockDim is primitive
    // colBlockDim is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsr2gebsrNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint m, jint n, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject descrC, jobject bsrSortedValC, jobject bsrSortedRowPtrC, jobject bsrSortedColIndC, jint rowBlockDim, jint colBlockDim, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseCcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValC' is null for cusparseCcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrC' is null for cusparseCcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndC' is null for cusparseCcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsr2gebsr(handle=%p, dirA=%d, m=%d, n=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, descrC=%p, bsrSortedValC=%p, bsrSortedRowPtrC=%p, bsrSortedColIndC=%p, rowBlockDim=%d, colBlockDim=%d, pBuffer=%p)\n",
        handle, dirA, m, n, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, descrC, bsrSortedValC, bsrSortedRowPtrC, bsrSortedColIndC, rowBlockDim, colBlockDim, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    cusparseMatDescr_t descrC_native;
    cuComplex * bsrSortedValC_native = NULL;
    int * bsrSortedRowPtrC_native = NULL;
    int * bsrSortedColIndC_native = NULL;
    int rowBlockDim_native = 0;
    int colBlockDim_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    bsrSortedValC_native = (cuComplex *)getPointer(env, bsrSortedValC);
    bsrSortedRowPtrC_native = (int *)getPointer(env, bsrSortedRowPtrC);
    bsrSortedColIndC_native = (int *)getPointer(env, bsrSortedColIndC);
    rowBlockDim_native = (int)rowBlockDim;
    colBlockDim_native = (int)colBlockDim;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsr2gebsr(handle_native, dirA_native, m_native, n_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, descrC_native, bsrSortedValC_native, bsrSortedRowPtrC_native, bsrSortedColIndC_native, rowBlockDim_native, colBlockDim_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // m is primitive
    // n is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // descrC is read-only
    // bsrSortedValC is a native pointer
    // bsrSortedRowPtrC is a native pointer
    // bsrSortedColIndC is a native pointer
    // rowBlockDim is primitive
    // colBlockDim is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsr2gebsrNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint m, jint n, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject descrC, jobject bsrSortedValC, jobject bsrSortedRowPtrC, jobject bsrSortedColIndC, jint rowBlockDim, jint colBlockDim, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // m is primitive
    // n is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseZcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValC' is null for cusparseZcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrC' is null for cusparseZcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndC' is null for cusparseZcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDim is primitive
    // colBlockDim is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZcsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsr2gebsr(handle=%p, dirA=%d, m=%d, n=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, descrC=%p, bsrSortedValC=%p, bsrSortedRowPtrC=%p, bsrSortedColIndC=%p, rowBlockDim=%d, colBlockDim=%d, pBuffer=%p)\n",
        handle, dirA, m, n, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, descrC, bsrSortedValC, bsrSortedRowPtrC, bsrSortedColIndC, rowBlockDim, colBlockDim, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int m_native = 0;
    int n_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * csrSortedValA_native = NULL;
    int * csrSortedRowPtrA_native = NULL;
    int * csrSortedColIndA_native = NULL;
    cusparseMatDescr_t descrC_native;
    cuDoubleComplex * bsrSortedValC_native = NULL;
    int * bsrSortedRowPtrC_native = NULL;
    int * bsrSortedColIndC_native = NULL;
    int rowBlockDim_native = 0;
    int colBlockDim_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    m_native = (int)m;
    n_native = (int)n;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex *)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int *)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int *)getPointer(env, csrSortedColIndA);
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    bsrSortedValC_native = (cuDoubleComplex *)getPointer(env, bsrSortedValC);
    bsrSortedRowPtrC_native = (int *)getPointer(env, bsrSortedRowPtrC);
    bsrSortedColIndC_native = (int *)getPointer(env, bsrSortedColIndC);
    rowBlockDim_native = (int)rowBlockDim;
    colBlockDim_native = (int)colBlockDim;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsr2gebsr(handle_native, dirA_native, m_native, n_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, descrC_native, bsrSortedValC_native, bsrSortedRowPtrC_native, bsrSortedColIndC_native, rowBlockDim_native, colBlockDim_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // m is primitive
    // n is primitive
    // descrA is read-only
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // descrC is read-only
    // bsrSortedValC is a native pointer
    // bsrSortedRowPtrC is a native pointer
    // bsrSortedColIndC is a native pointer
    // rowBlockDim is primitive
    // colBlockDim is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgebsr2gebsr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint rowBlockDimA, jint colBlockDimA, jint rowBlockDimC, jint colBlockDimC, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseSgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseSgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseSgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDimA is primitive
    // colBlockDimA is primitive
    // rowBlockDimC is primitive
    // colBlockDimC is primitive
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseSgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgebsr2gebsr_bufferSize(handle=%p, dirA=%d, mb=%d, nb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, rowBlockDimA=%d, colBlockDimA=%d, rowBlockDimC=%d, colBlockDimC=%d, pBufferSizeInBytes=%p)\n",
        handle, dirA, mb, nb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, rowBlockDimA, colBlockDimA, rowBlockDimC, colBlockDimC, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int rowBlockDimA_native = 0;
    int colBlockDimA_native = 0;
    int rowBlockDimC_native = 0;
    int colBlockDimC_native = 0;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (float *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    rowBlockDimA_native = (int)rowBlockDimA;
    colBlockDimA_native = (int)colBlockDimA;
    rowBlockDimC_native = (int)rowBlockDimC;
    colBlockDimC_native = (int)colBlockDimC;
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgebsr2gebsr_bufferSize(handle_native, dirA_native, mb_native, nb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, rowBlockDimA_native, colBlockDimA_native, rowBlockDimC_native, colBlockDimC_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // rowBlockDimA is primitive
    // colBlockDimA is primitive
    // rowBlockDimC is primitive
    // colBlockDimC is primitive
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgebsr2gebsr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint rowBlockDimA, jint colBlockDimA, jint rowBlockDimC, jint colBlockDimC, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseDgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseDgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseDgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDimA is primitive
    // colBlockDimA is primitive
    // rowBlockDimC is primitive
    // colBlockDimC is primitive
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseDgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgebsr2gebsr_bufferSize(handle=%p, dirA=%d, mb=%d, nb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, rowBlockDimA=%d, colBlockDimA=%d, rowBlockDimC=%d, colBlockDimC=%d, pBufferSizeInBytes=%p)\n",
        handle, dirA, mb, nb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, rowBlockDimA, colBlockDimA, rowBlockDimC, colBlockDimC, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int rowBlockDimA_native = 0;
    int colBlockDimA_native = 0;
    int rowBlockDimC_native = 0;
    int colBlockDimC_native = 0;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (double *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    rowBlockDimA_native = (int)rowBlockDimA;
    colBlockDimA_native = (int)colBlockDimA;
    rowBlockDimC_native = (int)rowBlockDimC;
    colBlockDimC_native = (int)colBlockDimC;
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgebsr2gebsr_bufferSize(handle_native, dirA_native, mb_native, nb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, rowBlockDimA_native, colBlockDimA_native, rowBlockDimC_native, colBlockDimC_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // rowBlockDimA is primitive
    // colBlockDimA is primitive
    // rowBlockDimC is primitive
    // colBlockDimC is primitive
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgebsr2gebsr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint rowBlockDimA, jint colBlockDimA, jint rowBlockDimC, jint colBlockDimC, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseCgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseCgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseCgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDimA is primitive
    // colBlockDimA is primitive
    // rowBlockDimC is primitive
    // colBlockDimC is primitive
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseCgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgebsr2gebsr_bufferSize(handle=%p, dirA=%d, mb=%d, nb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, rowBlockDimA=%d, colBlockDimA=%d, rowBlockDimC=%d, colBlockDimC=%d, pBufferSizeInBytes=%p)\n",
        handle, dirA, mb, nb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, rowBlockDimA, colBlockDimA, rowBlockDimC, colBlockDimC, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int rowBlockDimA_native = 0;
    int colBlockDimA_native = 0;
    int rowBlockDimC_native = 0;
    int colBlockDimC_native = 0;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuComplex *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    rowBlockDimA_native = (int)rowBlockDimA;
    colBlockDimA_native = (int)colBlockDimA;
    rowBlockDimC_native = (int)rowBlockDimC;
    colBlockDimC_native = (int)colBlockDimC;
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgebsr2gebsr_bufferSize(handle_native, dirA_native, mb_native, nb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, rowBlockDimA_native, colBlockDimA_native, rowBlockDimC_native, colBlockDimC_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // rowBlockDimA is primitive
    // colBlockDimA is primitive
    // rowBlockDimC is primitive
    // colBlockDimC is primitive
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgebsr2gebsr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint rowBlockDimA, jint colBlockDimA, jint rowBlockDimC, jint colBlockDimC, jintArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseZgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseZgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseZgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDimA is primitive
    // colBlockDimA is primitive
    // rowBlockDimC is primitive
    // colBlockDimC is primitive
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseZgebsr2gebsr_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgebsr2gebsr_bufferSize(handle=%p, dirA=%d, mb=%d, nb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, rowBlockDimA=%d, colBlockDimA=%d, rowBlockDimC=%d, colBlockDimC=%d, pBufferSizeInBytes=%p)\n",
        handle, dirA, mb, nb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, rowBlockDimA, colBlockDimA, rowBlockDimC, colBlockDimC, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int rowBlockDimA_native = 0;
    int colBlockDimA_native = 0;
    int rowBlockDimC_native = 0;
    int colBlockDimC_native = 0;
    int * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuDoubleComplex *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    rowBlockDimA_native = (int)rowBlockDimA;
    colBlockDimA_native = (int)colBlockDimA;
    rowBlockDimC_native = (int)rowBlockDimC;
    colBlockDimC_native = (int)colBlockDimC;
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgebsr2gebsr_bufferSize(handle_native, dirA_native, mb_native, nb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, rowBlockDimA_native, colBlockDimA_native, rowBlockDimC_native, colBlockDimC_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // rowBlockDimA is primitive
    // colBlockDimA is primitive
    // rowBlockDimC is primitive
    // colBlockDimC is primitive
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXgebsr2gebsrNnzNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nb, jint nnzb, jobject descrA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint rowBlockDimA, jint colBlockDimA, jobject descrC, jobject bsrSortedRowPtrC, jint rowBlockDimC, jint colBlockDimC, jobject nnzTotalDevHostPtr, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXgebsr2gebsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseXgebsr2gebsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseXgebsr2gebsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseXgebsr2gebsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDimA is primitive
    // colBlockDimA is primitive
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseXgebsr2gebsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrC' is null for cusparseXgebsr2gebsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDimC is primitive
    // colBlockDimC is primitive
    if (nnzTotalDevHostPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzTotalDevHostPtr' is null for cusparseXgebsr2gebsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseXgebsr2gebsrNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXgebsr2gebsrNnz(handle=%p, dirA=%d, mb=%d, nb=%d, nnzb=%d, descrA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, rowBlockDimA=%d, colBlockDimA=%d, descrC=%p, bsrSortedRowPtrC=%p, rowBlockDimC=%d, colBlockDimC=%d, nnzTotalDevHostPtr=%p, pBuffer=%p)\n",
        handle, dirA, mb, nb, nnzb, descrA, bsrSortedRowPtrA, bsrSortedColIndA, rowBlockDimA, colBlockDimA, descrC, bsrSortedRowPtrC, rowBlockDimC, colBlockDimC, nnzTotalDevHostPtr, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int rowBlockDimA_native = 0;
    int colBlockDimA_native = 0;
    cusparseMatDescr_t descrC_native;
    int * bsrSortedRowPtrC_native = NULL;
    int rowBlockDimC_native = 0;
    int colBlockDimC_native = 0;
    int * nnzTotalDevHostPtr_native = NULL;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    rowBlockDimA_native = (int)rowBlockDimA;
    colBlockDimA_native = (int)colBlockDimA;
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    bsrSortedRowPtrC_native = (int *)getPointer(env, bsrSortedRowPtrC);
    rowBlockDimC_native = (int)rowBlockDimC;
    colBlockDimC_native = (int)colBlockDimC;
    PointerData *nnzTotalDevHostPtr_pointerData = initPointerData(env, nnzTotalDevHostPtr);
    if (nnzTotalDevHostPtr_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    nnzTotalDevHostPtr_native = (int *)nnzTotalDevHostPtr_pointerData->getPointer(env);
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXgebsr2gebsrNnz(handle_native, dirA_native, mb_native, nb_native, nnzb_native, descrA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, rowBlockDimA_native, colBlockDimA_native, descrC_native, bsrSortedRowPtrC_native, rowBlockDimC_native, colBlockDimC_native, nnzTotalDevHostPtr_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // rowBlockDimA is primitive
    // colBlockDimA is primitive
    // descrC is read-only
    // bsrSortedRowPtrC is a native pointer
    // rowBlockDimC is primitive
    // colBlockDimC is primitive
    // If the PointerData is not backed by native memory, then this call has to block
    if (!isPointerBackedByNativeMemory(env, nnzTotalDevHostPtr))
    {
        cudaDeviceSynchronize();
    }
    if (!releasePointerData(env, nnzTotalDevHostPtr_pointerData, 0)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgebsr2gebsrNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint rowBlockDimA, jint colBlockDimA, jobject descrC, jobject bsrSortedValC, jobject bsrSortedRowPtrC, jobject bsrSortedColIndC, jint rowBlockDimC, jint colBlockDimC, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseSgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseSgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseSgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDimA is primitive
    // colBlockDimA is primitive
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseSgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValC' is null for cusparseSgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrC' is null for cusparseSgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndC' is null for cusparseSgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDimC is primitive
    // colBlockDimC is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgebsr2gebsr(handle=%p, dirA=%d, mb=%d, nb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, rowBlockDimA=%d, colBlockDimA=%d, descrC=%p, bsrSortedValC=%p, bsrSortedRowPtrC=%p, bsrSortedColIndC=%p, rowBlockDimC=%d, colBlockDimC=%d, pBuffer=%p)\n",
        handle, dirA, mb, nb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, rowBlockDimA, colBlockDimA, descrC, bsrSortedValC, bsrSortedRowPtrC, bsrSortedColIndC, rowBlockDimC, colBlockDimC, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int rowBlockDimA_native = 0;
    int colBlockDimA_native = 0;
    cusparseMatDescr_t descrC_native;
    float * bsrSortedValC_native = NULL;
    int * bsrSortedRowPtrC_native = NULL;
    int * bsrSortedColIndC_native = NULL;
    int rowBlockDimC_native = 0;
    int colBlockDimC_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (float *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    rowBlockDimA_native = (int)rowBlockDimA;
    colBlockDimA_native = (int)colBlockDimA;
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    bsrSortedValC_native = (float *)getPointer(env, bsrSortedValC);
    bsrSortedRowPtrC_native = (int *)getPointer(env, bsrSortedRowPtrC);
    bsrSortedColIndC_native = (int *)getPointer(env, bsrSortedColIndC);
    rowBlockDimC_native = (int)rowBlockDimC;
    colBlockDimC_native = (int)colBlockDimC;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgebsr2gebsr(handle_native, dirA_native, mb_native, nb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, rowBlockDimA_native, colBlockDimA_native, descrC_native, bsrSortedValC_native, bsrSortedRowPtrC_native, bsrSortedColIndC_native, rowBlockDimC_native, colBlockDimC_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // rowBlockDimA is primitive
    // colBlockDimA is primitive
    // descrC is read-only
    // bsrSortedValC is a native pointer
    // bsrSortedRowPtrC is a native pointer
    // bsrSortedColIndC is a native pointer
    // rowBlockDimC is primitive
    // colBlockDimC is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgebsr2gebsrNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint rowBlockDimA, jint colBlockDimA, jobject descrC, jobject bsrSortedValC, jobject bsrSortedRowPtrC, jobject bsrSortedColIndC, jint rowBlockDimC, jint colBlockDimC, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseDgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseDgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseDgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDimA is primitive
    // colBlockDimA is primitive
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseDgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValC' is null for cusparseDgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrC' is null for cusparseDgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndC' is null for cusparseDgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDimC is primitive
    // colBlockDimC is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgebsr2gebsr(handle=%p, dirA=%d, mb=%d, nb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, rowBlockDimA=%d, colBlockDimA=%d, descrC=%p, bsrSortedValC=%p, bsrSortedRowPtrC=%p, bsrSortedColIndC=%p, rowBlockDimC=%d, colBlockDimC=%d, pBuffer=%p)\n",
        handle, dirA, mb, nb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, rowBlockDimA, colBlockDimA, descrC, bsrSortedValC, bsrSortedRowPtrC, bsrSortedColIndC, rowBlockDimC, colBlockDimC, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int rowBlockDimA_native = 0;
    int colBlockDimA_native = 0;
    cusparseMatDescr_t descrC_native;
    double * bsrSortedValC_native = NULL;
    int * bsrSortedRowPtrC_native = NULL;
    int * bsrSortedColIndC_native = NULL;
    int rowBlockDimC_native = 0;
    int colBlockDimC_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (double *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    rowBlockDimA_native = (int)rowBlockDimA;
    colBlockDimA_native = (int)colBlockDimA;
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    bsrSortedValC_native = (double *)getPointer(env, bsrSortedValC);
    bsrSortedRowPtrC_native = (int *)getPointer(env, bsrSortedRowPtrC);
    bsrSortedColIndC_native = (int *)getPointer(env, bsrSortedColIndC);
    rowBlockDimC_native = (int)rowBlockDimC;
    colBlockDimC_native = (int)colBlockDimC;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgebsr2gebsr(handle_native, dirA_native, mb_native, nb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, rowBlockDimA_native, colBlockDimA_native, descrC_native, bsrSortedValC_native, bsrSortedRowPtrC_native, bsrSortedColIndC_native, rowBlockDimC_native, colBlockDimC_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // rowBlockDimA is primitive
    // colBlockDimA is primitive
    // descrC is read-only
    // bsrSortedValC is a native pointer
    // bsrSortedRowPtrC is a native pointer
    // bsrSortedColIndC is a native pointer
    // rowBlockDimC is primitive
    // colBlockDimC is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgebsr2gebsrNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint rowBlockDimA, jint colBlockDimA, jobject descrC, jobject bsrSortedValC, jobject bsrSortedRowPtrC, jobject bsrSortedColIndC, jint rowBlockDimC, jint colBlockDimC, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseCgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseCgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseCgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDimA is primitive
    // colBlockDimA is primitive
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseCgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValC' is null for cusparseCgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrC' is null for cusparseCgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndC' is null for cusparseCgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDimC is primitive
    // colBlockDimC is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgebsr2gebsr(handle=%p, dirA=%d, mb=%d, nb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, rowBlockDimA=%d, colBlockDimA=%d, descrC=%p, bsrSortedValC=%p, bsrSortedRowPtrC=%p, bsrSortedColIndC=%p, rowBlockDimC=%d, colBlockDimC=%d, pBuffer=%p)\n",
        handle, dirA, mb, nb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, rowBlockDimA, colBlockDimA, descrC, bsrSortedValC, bsrSortedRowPtrC, bsrSortedColIndC, rowBlockDimC, colBlockDimC, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int rowBlockDimA_native = 0;
    int colBlockDimA_native = 0;
    cusparseMatDescr_t descrC_native;
    cuComplex * bsrSortedValC_native = NULL;
    int * bsrSortedRowPtrC_native = NULL;
    int * bsrSortedColIndC_native = NULL;
    int rowBlockDimC_native = 0;
    int colBlockDimC_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuComplex *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    rowBlockDimA_native = (int)rowBlockDimA;
    colBlockDimA_native = (int)colBlockDimA;
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    bsrSortedValC_native = (cuComplex *)getPointer(env, bsrSortedValC);
    bsrSortedRowPtrC_native = (int *)getPointer(env, bsrSortedRowPtrC);
    bsrSortedColIndC_native = (int *)getPointer(env, bsrSortedColIndC);
    rowBlockDimC_native = (int)rowBlockDimC;
    colBlockDimC_native = (int)colBlockDimC;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgebsr2gebsr(handle_native, dirA_native, mb_native, nb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, rowBlockDimA_native, colBlockDimA_native, descrC_native, bsrSortedValC_native, bsrSortedRowPtrC_native, bsrSortedColIndC_native, rowBlockDimC_native, colBlockDimC_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // rowBlockDimA is primitive
    // colBlockDimA is primitive
    // descrC is read-only
    // bsrSortedValC is a native pointer
    // bsrSortedRowPtrC is a native pointer
    // bsrSortedColIndC is a native pointer
    // rowBlockDimC is primitive
    // colBlockDimC is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgebsr2gebsrNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint rowBlockDimA, jint colBlockDimA, jobject descrC, jobject bsrSortedValC, jobject bsrSortedRowPtrC, jobject bsrSortedColIndC, jint rowBlockDimC, jint colBlockDimC, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseZgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseZgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseZgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDimA is primitive
    // colBlockDimA is primitive
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseZgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValC' is null for cusparseZgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrC' is null for cusparseZgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndC' is null for cusparseZgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // rowBlockDimC is primitive
    // colBlockDimC is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZgebsr2gebsr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgebsr2gebsr(handle=%p, dirA=%d, mb=%d, nb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, rowBlockDimA=%d, colBlockDimA=%d, descrC=%p, bsrSortedValC=%p, bsrSortedRowPtrC=%p, bsrSortedColIndC=%p, rowBlockDimC=%d, colBlockDimC=%d, pBuffer=%p)\n",
        handle, dirA, mb, nb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, rowBlockDimA, colBlockDimA, descrC, bsrSortedValC, bsrSortedRowPtrC, bsrSortedColIndC, rowBlockDimC, colBlockDimC, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex * bsrSortedValA_native = NULL;
    int * bsrSortedRowPtrA_native = NULL;
    int * bsrSortedColIndA_native = NULL;
    int rowBlockDimA_native = 0;
    int colBlockDimA_native = 0;
    cusparseMatDescr_t descrC_native;
    cuDoubleComplex * bsrSortedValC_native = NULL;
    int * bsrSortedRowPtrC_native = NULL;
    int * bsrSortedColIndC_native = NULL;
    int rowBlockDimC_native = 0;
    int colBlockDimC_native = 0;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuDoubleComplex *)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int *)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int *)getPointer(env, bsrSortedColIndA);
    rowBlockDimA_native = (int)rowBlockDimA;
    colBlockDimA_native = (int)colBlockDimA;
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    bsrSortedValC_native = (cuDoubleComplex *)getPointer(env, bsrSortedValC);
    bsrSortedRowPtrC_native = (int *)getPointer(env, bsrSortedRowPtrC);
    bsrSortedColIndC_native = (int *)getPointer(env, bsrSortedColIndC);
    rowBlockDimC_native = (int)rowBlockDimC;
    colBlockDimC_native = (int)colBlockDimC;
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgebsr2gebsr(handle_native, dirA_native, mb_native, nb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, rowBlockDimA_native, colBlockDimA_native, descrC_native, bsrSortedValC_native, bsrSortedRowPtrC_native, bsrSortedColIndC_native, rowBlockDimC_native, colBlockDimC_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // dirA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    // descrA is read-only
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // rowBlockDimA is primitive
    // colBlockDimA is primitive
    // descrC is read-only
    // bsrSortedValC is a native pointer
    // bsrSortedRowPtrC is a native pointer
    // bsrSortedColIndC is a native pointer
    // rowBlockDimC is primitive
    // colBlockDimC is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

//##############################################################################
//# SPARSE MATRIX SORTING
//##############################################################################
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateIdentityPermutationNative(JNIEnv *env, jclass cls, jobject handle, jint n, jobject p)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCreateIdentityPermutation");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    if (p == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'p' is null for cusparseCreateIdentityPermutation");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateIdentityPermutation(handle=%p, n=%d, p=%p)\n",
        handle, n, p);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int n_native = 0;
    int * p_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    n_native = (int)n;
    p_native = (int *)getPointer(env, p);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateIdentityPermutation(handle_native, n_native, p_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // p is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXcoosort_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nnz, jobject cooRowsA, jobject cooColsA, jlongArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXcoosort_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (cooRowsA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'cooRowsA' is null for cusparseXcoosort_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (cooColsA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'cooColsA' is null for cusparseXcoosort_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseXcoosort_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXcoosort_bufferSizeExt(handle=%p, m=%d, n=%d, nnz=%d, cooRowsA=%p, cooColsA=%p, pBufferSizeInBytes=%p)\n",
        handle, m, n, nnz, cooRowsA, cooColsA, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int nnz_native = 0;
    int * cooRowsA_native = NULL;
    int * cooColsA_native = NULL;
    size_t * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    nnz_native = (int)nnz;
    cooRowsA_native = (int *)getPointer(env, cooRowsA);
    cooColsA_native = (int *)getPointer(env, cooColsA);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXcoosort_bufferSizeExt(handle_native, m_native, n_native, nnz_native, cooRowsA_native, cooColsA_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nnz is primitive
    // cooRowsA is a native pointer
    // cooColsA is a native pointer
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXcoosortByRowNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nnz, jobject cooRowsA, jobject cooColsA, jobject P, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXcoosortByRow");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (cooRowsA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'cooRowsA' is null for cusparseXcoosortByRow");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (cooColsA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'cooColsA' is null for cusparseXcoosortByRow");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (P == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'P' is null for cusparseXcoosortByRow");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseXcoosortByRow");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXcoosortByRow(handle=%p, m=%d, n=%d, nnz=%d, cooRowsA=%p, cooColsA=%p, P=%p, pBuffer=%p)\n",
        handle, m, n, nnz, cooRowsA, cooColsA, P, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int nnz_native = 0;
    int * cooRowsA_native = NULL;
    int * cooColsA_native = NULL;
    int * P_native = NULL;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    nnz_native = (int)nnz;
    cooRowsA_native = (int *)getPointer(env, cooRowsA);
    cooColsA_native = (int *)getPointer(env, cooColsA);
    P_native = (int *)getPointer(env, P);
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXcoosortByRow(handle_native, m_native, n_native, nnz_native, cooRowsA_native, cooColsA_native, P_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nnz is primitive
    // cooRowsA is a native pointer
    // cooColsA is a native pointer
    // P is a native pointer
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXcoosortByColumnNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nnz, jobject cooRowsA, jobject cooColsA, jobject P, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXcoosortByColumn");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (cooRowsA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'cooRowsA' is null for cusparseXcoosortByColumn");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (cooColsA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'cooColsA' is null for cusparseXcoosortByColumn");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (P == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'P' is null for cusparseXcoosortByColumn");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseXcoosortByColumn");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXcoosortByColumn(handle=%p, m=%d, n=%d, nnz=%d, cooRowsA=%p, cooColsA=%p, P=%p, pBuffer=%p)\n",
        handle, m, n, nnz, cooRowsA, cooColsA, P, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int nnz_native = 0;
    int * cooRowsA_native = NULL;
    int * cooColsA_native = NULL;
    int * P_native = NULL;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    nnz_native = (int)nnz;
    cooRowsA_native = (int *)getPointer(env, cooRowsA);
    cooColsA_native = (int *)getPointer(env, cooColsA);
    P_native = (int *)getPointer(env, P);
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXcoosortByColumn(handle_native, m_native, n_native, nnz_native, cooRowsA_native, cooColsA_native, P_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nnz is primitive
    // cooRowsA is a native pointer
    // cooColsA is a native pointer
    // P is a native pointer
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXcsrsort_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nnz, jobject csrRowPtrA, jobject csrColIndA, jlongArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXcsrsort_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (csrRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrRowPtrA' is null for cusparseXcsrsort_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrColIndA' is null for cusparseXcsrsort_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseXcsrsort_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXcsrsort_bufferSizeExt(handle=%p, m=%d, n=%d, nnz=%d, csrRowPtrA=%p, csrColIndA=%p, pBufferSizeInBytes=%p)\n",
        handle, m, n, nnz, csrRowPtrA, csrColIndA, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int nnz_native = 0;
    int * csrRowPtrA_native = NULL;
    int * csrColIndA_native = NULL;
    size_t * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    nnz_native = (int)nnz;
    csrRowPtrA_native = (int *)getPointer(env, csrRowPtrA);
    csrColIndA_native = (int *)getPointer(env, csrColIndA);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXcsrsort_bufferSizeExt(handle_native, m_native, n_native, nnz_native, csrRowPtrA_native, csrColIndA_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nnz is primitive
    // csrRowPtrA is a native pointer
    // csrColIndA is a native pointer
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXcsrsortNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nnz, jobject descrA, jobject csrRowPtrA, jobject csrColIndA, jobject P, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXcsrsort");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseXcsrsort");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrRowPtrA' is null for cusparseXcsrsort");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrColIndA' is null for cusparseXcsrsort");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (P == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'P' is null for cusparseXcsrsort");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseXcsrsort");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXcsrsort(handle=%p, m=%d, n=%d, nnz=%d, descrA=%p, csrRowPtrA=%p, csrColIndA=%p, P=%p, pBuffer=%p)\n",
        handle, m, n, nnz, descrA, csrRowPtrA, csrColIndA, P, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    int * csrRowPtrA_native = NULL;
    int * csrColIndA_native = NULL;
    int * P_native = NULL;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrRowPtrA_native = (int *)getPointer(env, csrRowPtrA);
    csrColIndA_native = (int *)getPointer(env, csrColIndA);
    P_native = (int *)getPointer(env, P);
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXcsrsort(handle_native, m_native, n_native, nnz_native, descrA_native, csrRowPtrA_native, csrColIndA_native, P_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nnz is primitive
    // descrA is read-only
    // csrRowPtrA is a native pointer
    // csrColIndA is a native pointer
    // P is a native pointer
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXcscsort_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nnz, jobject cscColPtrA, jobject cscRowIndA, jlongArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXcscsort_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (cscColPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'cscColPtrA' is null for cusparseXcscsort_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (cscRowIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'cscRowIndA' is null for cusparseXcscsort_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseXcscsort_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXcscsort_bufferSizeExt(handle=%p, m=%d, n=%d, nnz=%d, cscColPtrA=%p, cscRowIndA=%p, pBufferSizeInBytes=%p)\n",
        handle, m, n, nnz, cscColPtrA, cscRowIndA, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int nnz_native = 0;
    int * cscColPtrA_native = NULL;
    int * cscRowIndA_native = NULL;
    size_t * pBufferSizeInBytes_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    nnz_native = (int)nnz;
    cscColPtrA_native = (int *)getPointer(env, cscColPtrA);
    cscRowIndA_native = (int *)getPointer(env, cscRowIndA);
    if (!initNative(env, pBufferSizeInBytes, pBufferSizeInBytes_native, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXcscsort_bufferSizeExt(handle_native, m_native, n_native, nnz_native, cscColPtrA_native, cscRowIndA_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nnz is primitive
    // cscColPtrA is a native pointer
    // cscRowIndA is a native pointer
    if (!releaseNative(env, pBufferSizeInBytes_native, pBufferSizeInBytes, true)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXcscsortNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nnz, jobject descrA, jobject cscColPtrA, jobject cscRowIndA, jobject P, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXcscsort");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseXcscsort");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (cscColPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'cscColPtrA' is null for cusparseXcscsort");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (cscRowIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'cscRowIndA' is null for cusparseXcscsort");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (P == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'P' is null for cusparseXcscsort");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseXcscsort");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXcscsort(handle=%p, m=%d, n=%d, nnz=%d, descrA=%p, cscColPtrA=%p, cscRowIndA=%p, P=%p, pBuffer=%p)\n",
        handle, m, n, nnz, descrA, cscColPtrA, cscRowIndA, P, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    int * cscColPtrA_native = NULL;
    int * cscRowIndA_native = NULL;
    int * P_native = NULL;
    void * pBuffer_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    cscColPtrA_native = (int *)getPointer(env, cscColPtrA);
    cscRowIndA_native = (int *)getPointer(env, cscRowIndA);
    P_native = (int *)getPointer(env, P);
    pBuffer_native = (void *)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXcscsort(handle_native, m_native, n_native, nnz_native, descrA_native, cscColPtrA_native, cscRowIndA_native, P_native, pBuffer_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nnz is primitive
    // descrA is read-only
    // cscColPtrA is a native pointer
    // cscRowIndA is a native pointer
    // P is a native pointer
    // pBuffer is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsru2csr_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nnz, jobject csrVal, jobject csrRowPtr, jobject csrColInd, jobject info, jlongArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsru2csr_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (csrVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrVal' is null for cusparseScsru2csr_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrRowPtr' is null for cusparseScsru2csr_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrColInd' is null for cusparseScsru2csr_bufferSizeExt");
        return JC