/*
 * JDrupes Builder
 * Copyright (C) 2025 Michael N. Lipp
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.jdrupes.builder.eclipse;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.Properties;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import org.jdrupes.builder.api.BuildException;
import org.jdrupes.builder.api.FileTree;
import org.jdrupes.builder.api.Intend;
import org.jdrupes.builder.api.Project;
import org.jdrupes.builder.api.Resource;
import org.jdrupes.builder.api.ResourceRequest;
import org.jdrupes.builder.api.ResourceType;
import org.jdrupes.builder.core.AbstractGenerator;
import org.jdrupes.builder.java.ClasspathElement;
import org.jdrupes.builder.java.JarFile;
import org.jdrupes.builder.java.JavaCompiler;
import org.jdrupes.builder.java.JavaProject;
import org.jdrupes.builder.java.JavaResourceCollector;
import static org.jdrupes.builder.java.JavaTypes.CompilationResourcesType;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

/// The [EclipseConfigurator] provides the resource [EclipseConfiguration].
/// "The configuration" consists of the Eclipse configuration files
/// for a given project. The configurator generates the following
/// files as W3C DOM documents (for XML files) or as [Properties]
/// for a given project:
///
///   * `.project`,
///   * `.classpath`,
///   * `.settings/org.eclipse.core.resources.prefs`,
///   * `.settings/org.eclipse.core.runtime.prefs` and
///   * `.settings/org.eclipse.jdt.core.prefs`.
///
/// Each generated data structure can be post processed by a corresponding
/// `adapt` method before being written to disk.
///
/// Additional resources can be generated by the method
/// [#adaptConfiguration].  
///
public class EclipseConfigurator extends AbstractGenerator {

    /// The Constant GENERATED_BY.
    public static final String GENERATED_BY = "Generated by JDrupes Builder";
    private static DocumentBuilderFactory dbf
        = DocumentBuilderFactory.newInstance();
    private BiConsumer<Document, Node> classpathAdaptor = (_, _) -> {
    };
    private Runnable configurationAdaptor = () -> {
    };
    private Consumer<Properties> jdtCorePrefsAdaptor = _ -> {
    };
    private Consumer<Properties> resourcesPrefsAdaptor = _ -> {
    };
    private Consumer<Properties> runtimePrefsAdaptor = _ -> {
    };
    private ProjectConfigurationAdaptor prjConfigAdaptor = (_, _, _) -> {
    };

    /// Instantiates a new eclipse configurator.
    ///
    /// @param project the project
    ///
    public EclipseConfigurator(Project project) {
        super(project);
    }

    /// Provides an [EclipseConfiguration].
    ///
    /// @param <T> the generic type
    /// @param requested the requested
    /// @return the stream
    ///
    @Override
    protected <T extends Resource> Stream<T>
            doProvide(ResourceRequest<T> requested) {
        if (!requested.includes(new ResourceType<EclipseConfiguration>() {})) {
            return Stream.empty();
        }

        // Make sure that the directories exist.
        project().directory().resolve(".settings").toFile().mkdirs();

        // generate .project
        generateXmlFile(this::generateProjectConfiguration, ".project");

        // generate .classpath
        if (project() instanceof JavaProject) {
            generateXmlFile(this::generateClasspathConfiguration, ".classpath");
        }

        // Generate preferences
        generateResourcesPrefs();
        generateRuntimePrefs();
        if (project() instanceof JavaProject) {
            generateJdtCorePrefs();
        }

        // General overrides
        configurationAdaptor.run();

        // Create result
        @SuppressWarnings({ "unchecked", "PMD.UseDiamondOperator" })
        var result = (Stream<T>) Stream.of(project().newResource(
            new ResourceType<EclipseConfiguration>() {},
            project().directory()));
        return result;
    }

    private void generateXmlFile(Consumer<Document> generator, String name) {
        try {
            var doc = dbf.newDocumentBuilder().newDocument();
            generator.accept(doc);
            var transformer = TransformerFactory.newInstance().newTransformer();
            transformer.setOutputProperty(OutputKeys.INDENT, "yes");
            transformer.setOutputProperty(
                "{http://xml.apache.org/xslt}indent-amount", "4");
            try (var out = Files
                .newBufferedWriter(project().directory().resolve(name))) {
                transformer.transform(new DOMSource(doc),
                    new StreamResult(out));
            }
        } catch (ParserConfigurationException | TransformerException
                | TransformerFactoryConfigurationError | IOException e) {
            throw new BuildException(e);
        }
    }

    /// Generates the content of the `.project` file into the given document.
    ///
    /// @param doc the document
    ///
    @SuppressWarnings("PMD.AvoidDuplicateLiterals")
    protected void generateProjectConfiguration(Document doc) {
        var prjDescr
            = doc.appendChild(doc.createElement("projectDescription"));
        prjDescr.appendChild(doc.createElement("name"))
            .appendChild(doc.createTextNode(project().name()));
        prjDescr.appendChild(doc.createElement("comment")).appendChild(
            doc.createTextNode(GENERATED_BY));
        prjDescr.appendChild(doc.createElement("projects"));
        var buildSpec
            = prjDescr.appendChild(doc.createElement("buildSpec"));
        var natures = prjDescr.appendChild(doc.createElement("natures"));
        if (project() instanceof JavaProject) {
            var cmd
                = buildSpec.appendChild(doc.createElement("buildCommand"));
            cmd.appendChild(doc.createElement("name"))
                .appendChild(doc.createTextNode(
                    "org.eclipse.jdt.core.javabuilder"));
            cmd.appendChild(doc.createElement("arguments"));
            natures.appendChild(doc.createElement("nature")).appendChild(
                doc.createTextNode("org.eclipse.jdt.core.javanature"));
        }

        // Allow derived class to adapt the project configuration
        prjConfigAdaptor.accept(doc, buildSpec, natures);
    }

    /// Allow derived classes to post process the project configuration.
    ///
    @FunctionalInterface
    public interface ProjectConfigurationAdaptor {
        /// Execute the adaptor.
        ///
        /// @param doc the document
        /// @param buildSpec shortcut to the `buildSpec` element
        /// @param natures shortcut to the `natures` element
        ///
        void accept(Document doc, Node buildSpec,
                Node natures);
    }

    /// Adapt project configuration.
    ///
    /// @param adaptor the adaptor
    /// @return the eclipse configurator
    ///
    public EclipseConfigurator adaptProjectConfiguration(
            ProjectConfigurationAdaptor adaptor) {
        prjConfigAdaptor = adaptor;
        return this;
    }

    /// Generates the content of the `.classpath` file into the given
    /// document.
    ///
    /// @param doc the doc
    ///
    @SuppressWarnings({ "PMD.AvoidDuplicateLiterals",
        "PMD.UseDiamondOperator" })
    protected void generateClasspathConfiguration(Document doc) {
        var classpath = doc.appendChild(doc.createElement("classpath"));
        project().providers(Intend.Supply)
            .filter(p -> p instanceof JavaCompiler).map(p -> (JavaCompiler) p)
            .findFirst().ifPresent(jc -> {
                jc.sources().stream().map(FileTree::root)
                    .map(p -> project().relativize(p)).forEach(p -> {
                        var entry = (Element) classpath
                            .appendChild(doc.createElement("classpathentry"));
                        entry.setAttribute("kind", "src");
                        entry.setAttribute("path", p.toString());
                    });
                var entry = (Element) classpath
                    .appendChild(doc.createElement("classpathentry"));
                entry.setAttribute("kind", "output");
                entry.setAttribute("path",
                    project().relativize(jc.destination()).toString());
                jc.optionArgument("-target", "--target", "--release")
                    .ifPresentOrElse(v -> addSpecificJre(doc, classpath, v),
                        () -> addInheritedJre(doc, classpath));
            });

        // Add resources
        project().providers(Intend.Supply)
            .filter(p -> p instanceof JavaResourceCollector)
            .map(p -> (JavaResourceCollector) p)
            .findFirst().ifPresent(rc -> {
                rc.resources().stream().map(FileTree::root)
                    .filter(p -> p.toFile().canRead())
                    .map(p -> project().relativize(p)).forEach(p -> {
                        var entry = (Element) classpath
                            .appendChild(doc.createElement("classpathentry"));
                        entry.setAttribute("kind", "src");
                        entry.setAttribute("path", p.toString());
                    });
            });

        // Add projects
        collectContributing(project()).collect(Collectors.toSet()).stream()
            .forEach(p -> {
                var entry = (Element) classpath
                    .appendChild(doc.createElement("classpathentry"));
                entry.setAttribute("kind", "src");
                entry.setAttribute("path", "/" + p.name());
                var attributes
                    = entry.appendChild(doc.createElement("attributes"));
                var attribute = (Element) attributes
                    .appendChild(doc.createElement("attribute"));
                attribute.setAttribute("without_test_code", "true");
            });

        // Add jars
        project().provided(new ResourceRequest<ClasspathElement>(
            CompilationResourcesType)).filter(p -> p instanceof JarFile)
            .map(jf -> (JarFile) jf).forEach(jf -> {
                var entry = (Element) classpath
                    .appendChild(doc.createElement("classpathentry"));
                entry.setAttribute("kind", "lib");
                var jarPathName = jf.path().toString();
                entry.setAttribute("path", jarPathName);

                // Educated guesses
                var sourcesJar = new File(
                    jarPathName.replaceFirst("\\.jar$", "-sources.jar"));
                if (sourcesJar.canRead()) {
                    entry.setAttribute("sourcepath",
                        sourcesJar.getAbsolutePath());
                }
                var javadocJar = new File(
                    jarPathName.replaceFirst("\\.jar$", "-javadoc.jar"));
                if (javadocJar.canRead()) {
                    var attr = (Element) entry
                        .appendChild(doc.createElement("attributes"))
                        .appendChild(doc.createElement("attribute"));
                    attr.setAttribute("name", "javadoc_location");
                    attr.setAttribute("value",
                        "jar:file:" + javadocJar.getAbsolutePath() + "!/");
                }
            });

        // Allow derived class to override
        classpathAdaptor.accept(doc, classpath);
    }

    private Stream<Project> collectContributing(Project project) {
        return project.providers(Intend.Consume, Intend.Forward, Intend.Expose)
            .filter(p -> p instanceof Project).map(p -> (Project) p)
            .map(p -> Stream.concat(Stream.of(p), collectContributing(p)))
            .flatMap(s -> s);
    }

    private void addSpecificJre(Document doc, Node classpath,
            String version) {
        var entry = (Element) classpath
            .appendChild(doc.createElement("classpathentry"));
        entry.setAttribute("kind", "con");
        entry.setAttribute("path",
            "org.eclipse.jdt.launching.JRE_CONTAINER"
                + "/org.eclipse.jdt.internal.debug.ui.launcher.StandardVMType"
                + "/JavaSE-" + version);
        var attributes = entry.appendChild(doc.createElement("attributes"));
        var attribute
            = (Element) attributes.appendChild(doc.createElement("attribute"));
        attribute.setAttribute("name", "module");
        attribute.setAttribute("value", "true");
    }

    private void addInheritedJre(Document doc, Node classpath) {
        var entry = (Element) classpath
            .appendChild(doc.createElement("classpathentry"));
        entry.setAttribute("kind", "con");
        entry.setAttribute("path",
            "org.eclipse.jdt.launching.JRE_CONTAINER");
        var attributes = entry.appendChild(doc.createElement("attributes"));
        var attribute
            = (Element) attributes.appendChild(doc.createElement("attribute"));
        attribute.setAttribute("name", "module");
        attribute.setAttribute("value", "true");
    }

    /// Allow the user to post process the classpath configuration.
    /// The node passed to the consumer is the `classpath` element.
    ///
    /// @param adaptor the adaptor
    /// @return the eclipse configurator
    ///
    public EclipseConfigurator
            adaptClasspathConfiguration(BiConsumer<Document, Node> adaptor) {
        classpathAdaptor = adaptor;
        return this;
    }

    /// Generate the properties for the
    /// `.settings/org.eclipse.core.resources.prefs` file.
    ///
    @SuppressWarnings("PMD.PreserveStackTrace")
    protected void generateResourcesPrefs() {
        var props = new Properties();
        props.setProperty("eclipse.preferences.version", "1");
        props.setProperty("encoding/<project>", "UTF-8");
        resourcesPrefsAdaptor.accept(props);
        try (var out = new FixCommentsFilter(Files.newBufferedWriter(
            project().directory().resolve(
                ".settings/org.eclipse.core.resources.prefs")),
            GENERATED_BY)) {
            props.store(out, "");
        } catch (IOException e) {
            throw new BuildException(
                "Cannot write eclipse settings: " + e.getMessage());
        }
    }

    /// Allow the user to adapt the properties for the
    /// `.settings/org.eclipse.core.resources.prefs` file.
    ///
    /// @param adaptor the adaptor
    /// @return the eclipse configurator
    ///
    public EclipseConfigurator
            adaptResourcePrefs(Consumer<Properties> adaptor) {
        resourcesPrefsAdaptor = adaptor;
        return this;
    }

    /// Generate the properties for the
    /// `.settings/org.eclipse.core.runtime.prefs` file.
    ///
    @SuppressWarnings("PMD.PreserveStackTrace")
    protected void generateRuntimePrefs() {
        var props = new Properties();
        props.setProperty("eclipse.preferences.version", "1");
        props.setProperty("line.separator", "\n");
        runtimePrefsAdaptor.accept(props);
        try (var out = new FixCommentsFilter(Files.newBufferedWriter(
            project().directory().resolve(
                ".settings/org.eclipse.core.runtime.prefs")),
            GENERATED_BY)) {
            props.store(out, "");
        } catch (IOException e) {
            throw new BuildException(
                "Cannot write eclipse settings: " + e.getMessage());
        }
    }

    /// Allow the user to adapt the properties for the
    /// `.settings/org.eclipse.core.runtime.prefs` file.
    ///
    /// @param adaptor the adaptor
    /// @return the eclipse configurator
    ///
    public EclipseConfigurator adaptRuntimePrefs(Consumer<Properties> adaptor) {
        runtimePrefsAdaptor = adaptor;
        return this;
    }

    /// Generate the properties for the
    /// `.settings/org.eclipse.jdt.core.prefs` file.
    ///
    @SuppressWarnings("PMD.PreserveStackTrace")
    protected void generateJdtCorePrefs() {
        var props = new Properties();
        props.setProperty("eclipse.preferences.version", "1");
        project().providers(Intend.Supply)
            .filter(p -> p instanceof JavaCompiler).map(p -> (JavaCompiler) p)
            .findFirst().ifPresent(jc -> {
                jc.optionArgument("-target", "--target", "--release")
                    .ifPresent(v -> {
                        props.setProperty("org.eclipse.jdt.core.compiler"
                            + ".codegen.targetPlatform", v);
                    });
                jc.optionArgument("-source", "--source", "--release")
                    .ifPresent(v -> {
                        props.setProperty("org.eclipse.jdt.core.compiler"
                            + ".source", v);
                        props.setProperty("org.eclipse.jdt.core.compiler"
                            + ".compliance", v);
                    });
            });
        jdtCorePrefsAdaptor.accept(props);
        try (var out = new FixCommentsFilter(Files.newBufferedWriter(
            project().directory()
                .resolve(".settings/org.eclipse.jdt.core.prefs")),
            GENERATED_BY)) {
            props.store(out, "");
        } catch (IOException e) {
            throw new BuildException(
                "Cannot write eclipse settings: " + e.getMessage());
        }
    }

    /// Allow the user to adapt the properties for the
    /// `.settings/org.eclipse.jdt.core.prefs` file.
    ///
    /// @param adaptor the adaptor
    /// @return the eclipse configurator
    ///
    public EclipseConfigurator adaptJdtCorePrefs(Consumer<Properties> adaptor) {
        jdtCorePrefsAdaptor = adaptor;
        return this;
    }

    /// Allow the user to add additional resources.
    ///
    /// @param adaptor the adaptor
    /// @return the eclipse configurator
    ///
    public EclipseConfigurator adaptConfiguration(Runnable adaptor) {
        configurationAdaptor = adaptor;
        return this;
    }

}
