/*
 * JDrupes Builder
 * Copyright (C) 2025 Michael N. Lipp
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.jdrupes.builder.core;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.stream.Stream;
import org.jdrupes.builder.api.BuildException;
import org.jdrupes.builder.api.Cleanliness;
import org.jdrupes.builder.api.Generator;
import org.jdrupes.builder.api.Intend;
import static org.jdrupes.builder.api.Intend.*;
import org.jdrupes.builder.api.NamedParameter;
import org.jdrupes.builder.api.Project;
import org.jdrupes.builder.api.PropertyKey;
import org.jdrupes.builder.api.Resource;
import org.jdrupes.builder.api.ResourceProvider;
import org.jdrupes.builder.api.ResourceRequest;
import org.jdrupes.builder.api.ResourceType;
import org.jdrupes.builder.api.RootProject;

/// A default implementation of a [Project].
///
@SuppressWarnings({ "PMD.CouplingBetweenObjects", "PMD.GodClass" })
public abstract class AbstractProject extends AbstractProvider
        implements Project {

    private Map<Class<? extends Project>, Future<Project>> projects;
    private static ThreadLocal<AbstractProject> fallbackParent
        = new ThreadLocal<>();
    private static Path jdbldDirectory = Path.of("marker:jdbldDirectory");
    private final AbstractProject parent;
    private final String projectName;
    private final Path projectDirectory;
    private final Map<ResourceProvider, Intend> providers
        = new ConcurrentHashMap<>();
    @SuppressWarnings("PMD.UseConcurrentHashMap")
    private final Map<PropertyKey, Object> properties = new HashMap<>();
    // Only non null in the root project
    private DefaultBuildContext context;
    private Map<String, ResourceRequest<?>[]> commands;

    /// Named parameter for specifying the parent project.
    ///
    /// @param parentProject the parent project
    /// @return the named parameter
    ///
    protected static NamedParameter<Class<? extends Project>>
            parent(Class<? extends Project> parentProject) {
        return new NamedParameter<>("parent", parentProject);
    }

    /// Named parameter for specifying the name.
    ///
    /// @param name the name
    /// @return the named parameter
    ///
    protected static NamedParameter<String> name(String name) {
        return new NamedParameter<>("name", name);
    }

    /// Named parameter for specifying the directory.
    ///
    /// @param directory the directory
    /// @return the named parameter
    ///
    protected static NamedParameter<Path> directory(Path directory) {
        return new NamedParameter<>("directory", directory);
    }

    /// Hack to pass `context().jdbldDirectory()` as named parameter
    /// for the directory to the constructor. This is required because
    /// you cannot "refer to an instance method while explicitly invoking
    /// a constructor". 
    ///
    /// @return the named parameter
    ///
    protected static NamedParameter<Path> jdbldDirectory() {
        return new NamedParameter<>("directory", jdbldDirectory);
    }

    /// Base class constructor for all projects. The behavior depends 
    /// on whether the project is a root project (implements [RootProject])
    /// or a subproject and on whether the project specifies a parent project.
    ///
    /// [RootProject]s must invoke this constructor with a null parent project
    /// class.
    ///
    /// A sub project that wants to specify a parent project must invoke this
    /// constructor with the parent project's class. If a sub project does not
    /// specify a parent project, the root project is used as parent. In both
    /// cases, the constructor adds a [Intend#Forward] dependency between the
    /// parent project and the new project. This can then be overridden in the
    /// sub project's constructor.
    ///
    /// @param params the named parameters
    ///   * parent - the class of the parent project
    ///   * name - the name of the project. If not provided the name is
    ///     set to the (simple) class name
    ///   * directory - the directory of the project. If not provided,
    ///     the directory is set to the name with uppercase letters
    ///     converted to lowercase for subprojects. For root projects
    ///     the directory is always set to the current working
    ///
    @SuppressWarnings({ "PMD.ConstructorCallsOverridableMethod",
        "PMD.UseLocaleWithCaseConversions", "PMD.AvoidCatchingGenericException",
        "PMD.CognitiveComplexity" })
    protected AbstractProject(NamedParameter<?>... params) {
        // Evaluate parent project
        var parentProject = NamedParameter.<
                Class<? extends Project>> get(params, "parent", null);
        if (parentProject == null) {
            parent = fallbackParent.get();
            if (this instanceof RootProject) {
                if (parent != null) {
                    throw new BuildException("Root project of type "
                        + getClass().getSimpleName()
                        + " cannot be a sub project.");
                }
                // ConcurrentHashMap does not support null values.
                projects = Collections.synchronizedMap(new HashMap<>());
                context = new DefaultBuildContext();
                commands = new HashMap<>(Map.of(
                    "clean", new ResourceRequest<?>[] {
                        new ResourceRequest<Cleanliness>(
                            new ResourceType<>() {}) }));
            }
        } else {
            parent = (AbstractProject) project(parentProject);
        }

        // Set name and directory, add fallback dependency
        var name = NamedParameter.<String> get(params, "name",
            () -> getClass().getSimpleName());
        projectName = name;
        var directory = NamedParameter.<Path> get(params, "directory", null);
        if (directory == jdbldDirectory) { // NOPMD
            directory = context().jdbldDirectory();
        }
        if (parent == null) {
            if (directory != null) {
                throw new BuildException("Root project of type "
                    + getClass().getSimpleName()
                    + " cannot specify a directory.");
            }
            projectDirectory = Path.of("").toAbsolutePath();
        } else {
            if (directory == null) {
                directory = Path.of(projectName.toLowerCase());
            }
            projectDirectory = parent.directory().resolve(directory);
            // Fallback, will be replaced when the parent explicitly adds a
            // dependency.
            parent.dependency(Forward, this);
        }
        try {
            rootProject().prepareProject(this);
        } catch (Exception e) {
            throw new BuildException(e);
        }
    }

    @Override
    public final RootProject rootProject() {
        if (this instanceof RootProject root) {
            return root;
        }
        // The method may be called (indirectly) from the constructor
        // of a subproject, that specifies its parent project class, to
        // get the parent project instance. In this case, the new
        // project's parent attribute has not been set yet and we have
        // to use the fallback.
        return Optional.ofNullable(parent).orElse(fallbackParent.get())
            .rootProject();
    }

    @Override
    public ResourceProvider project(Class<? extends Project> prjCls) {
        if (this.getClass().equals(prjCls)) {
            return this;
        }
        if (projects == null) {
            return rootProject().project(prjCls);
        }

        // "this" is the root project.
        try {
            return projects.computeIfAbsent(prjCls, k -> {
                return context().executor().submit(() -> {
                    try {
                        fallbackParent.set(this);
                        return (Project) k.getConstructor().newInstance();
                    } catch (SecurityException | InstantiationException
                            | IllegalAccessException
                            | InvocationTargetException
                            | NoSuchMethodException e) {
                        throw new IllegalArgumentException(e);
                    } finally {
                        fallbackParent.set(null);
                    }
                });
            }).get();
        } catch (InterruptedException | ExecutionException e) {
            throw new BuildException(e);
        }
    }

    @Override
    @SuppressWarnings("checkstyle:OverloadMethodsDeclarationOrder")
    public String name() {
        return projectName;
    }

    @Override
    @SuppressWarnings("checkstyle:OverloadMethodsDeclarationOrder")
    public Path directory() {
        return projectDirectory;
    }

    /// Generator.
    ///
    /// @param provider the provider
    /// @return the project
    ///
    @Override
    public Project generator(Generator provider) {
        providers.put(provider, Supply);
        return this;
    }

    @Override
    public Project dependency(Intend intend, ResourceProvider provider) {
        providers.put(provider, intend);
        return this;
    }

    @Override
    public Stream<ResourceProvider> providers(Set<Intend> intends) {
        return providers.entrySet().stream()
            .filter(e -> intends.contains(e.getValue())).map(Entry::getKey);
    }

    @Override
    public <T extends Resource> Stream<T> getFrom(
            Stream<ResourceProvider> providers, ResourceRequest<T> request) {
        return providers.map(p -> context().<T> get(p, request))
            // Terminate stream to start all tasks for evaluating the futures
            .toList().stream().flatMap(s -> s);
    }

    @Override
    public DefaultBuildContext context() {
        return ((AbstractProject) rootProject()).context;
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T> T get(PropertyKey property) {
        return (T) Optional.ofNullable(properties.get(property))
            .orElseGet(() -> {
                if (parent != null) {
                    return parent.get(property);
                }
                return property.defaultValue();
            });
    }

    @Override
    public AbstractProject set(PropertyKey property, Object value) {
        if (!property.type().isAssignableFrom(value.getClass())) {
            throw new IllegalArgumentException("Value for " + property
                + " must be of type " + property.type());
        }
        properties.put(property, value);
        return this;
    }

    /// A project itself does not provide any resources. Rather, requests
    /// for resources are forwarded to the project's providers with intend
    /// [Intend#Forward], [Intend#Expose] or [Intend#Supply].
    ///
    /// @param <R> the generic type
    /// @param requested the requested
    /// @return the provided resources
    ///
    @Override
    protected <R extends Resource> Stream<R>
            doProvide(ResourceRequest<R> requested) {
        return getFrom(providers(EnumSet.of(Forward, Expose, Supply)),
            requested);
    }

    /// Define command, see [RootProject#commandAlias].
    ///
    /// @param name the name
    /// @param requests the requests
    /// @return the root project
    ///
    public RootProject commandAlias(String name,
            ResourceRequest<?>... requests) {
        if (commands == null) {
            throw new BuildException("Commands can only be defined for"
                + " the root project.");
        }
        commands.put(name, requests);
        return (RootProject) this;
    }

    /* default */ ResourceRequest<?>[] lookupCommand(String name) {
        return commands.getOrDefault(name, new ResourceRequest[0]);
    }

    /// Convenience method for reading the content of a file into a
    /// String.
    ///
    /// @param path the path
    /// @return the string
    ///
    @SuppressWarnings("PMD.PreserveStackTrace")
    public String readString(Path path) {
        try {
            return Files.readString(path);
        } catch (IOException e) {
            throw new BuildException("Cannot read file: " + e.getMessage());
        }
    }

    @Override
    public int hashCode() {
        return Objects.hash(projectDirectory, projectName);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        AbstractProject other = (AbstractProject) obj;
        return Objects.equals(projectDirectory, other.projectDirectory)
            && Objects.equals(projectName, other.projectName);
    }

    /// To string.
    ///
    /// @return the string
    ///
    @Override
    public String toString() {
        var relDir = rootProject().directory().relativize(directory());
        return "Project " + name() + (relDir.toString().isBlank() ? ""
            : (" (in " + relDir + ")"));
    }

}
