/*
 * JDrupes Builder
 * Copyright (C) 2025 Michael N. Lipp
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.jdrupes.builder.mvnrepo;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.stream.Stream;
import org.apache.maven.model.Model;
import org.apache.maven.model.io.DefaultModelWriter;
import org.jdrupes.builder.api.BuildException;
import org.jdrupes.builder.api.Generator;
import org.jdrupes.builder.api.Project;
import static org.jdrupes.builder.api.Project.Properties.*;
import org.jdrupes.builder.api.Resource;
import org.jdrupes.builder.api.ResourceRequest;
import org.jdrupes.builder.core.AbstractGenerator;
import static org.jdrupes.builder.mvnrepo.MvnProperties.*;
import static org.jdrupes.builder.mvnrepo.MvnRepoTypes.*;

/// A [Generator] for POM files. The generator generates a maven
/// [Model] with basic information. The group is set to the value
/// of the property [MvnProperties#GroupId] if it is defined. The
/// artifact id is set to the property [MvnProperties#ArtifactId]
/// or the name of the project, if [MvnProperties#ArtifactId] is not
/// defined. The version is set to the value of the property
/// [Project.Properties#Version].
///
/// This basic model is passed to [#adaptPom] where it can be adapted.
/// as the project requires. Finally, the model is written to the
/// POM file
///
public class PomFileGenerator extends AbstractGenerator {

    /// The Constant GENERATED_BY.
    public static final String GENERATED_BY = "Generated by JDrupes Builder";
    private Supplier<Path> destination
        = () -> project().buildDirectory().resolve("publications/maven");
    private Consumer<Model> pomAdapter = _ -> {
    }; // Do nothing>

    /// Instantiates a new library generator.
    ///
    /// @param project the project
    ///
    public PomFileGenerator(Project project) {
        super(project);
    }

    /// Returns the destination directory. Defaults to sub directory
    /// `publications/maven` in the project's build directory
    /// (see [Project#buildDirectory]).
    ///
    /// @return the destination
    ///
    public Path destination() {
        return destination.get();
    }

    /// Sets the destination directory. The [Path] is resolved against
    /// the project's build directory (see [Project#buildDirectory]).
    ///
    /// @param destination the new destination
    /// @return the jar generator
    ///
    public PomFileGenerator destination(Path destination) {
        this.destination
            = () -> project().buildDirectory().resolve(destination);
        return this;
    }

    /// Sets the destination directory.
    ///
    /// @param destination the new destination
    /// @return the jar generator
    ///
    public PomFileGenerator destination(Supplier<Path> destination) {
        this.destination = destination;
        return this;
    }

    @Override
    protected <T extends Resource> Stream<T>
            doProvide(ResourceRequest<T> requested) {
        var pomPath = destination().resolve(Optional.ofNullable(
            project().get(ArtifactId)).orElse(project().name()) + "-pom.xml");
        if (cleanup(requested, pomPath)) {
            return Stream.empty();
        }

        if (!requested.includes(PomFileType)) {
            return Stream.empty();
        }

        pomPath.getParent().toFile().mkdirs();
        var deps = project().newResource(MvnRepoDependenciesType).addAll(
            project().supplied(new ResourceRequest<>(
                MvnRepoDependenciesType)));
        Model model = generatePom();

        // create, compare and maybe write model
        pomPath.getParent().toFile().mkdirs();
        try {
            // Create new in memory
            ByteArrayOutputStream newPom = new ByteArrayOutputStream();
            new DefaultModelWriter().write(newPom, null, model);
            newPom.close();

            // Read existing
            var existingContent = pomPath.toFile().exists()
                ? Files.readAllBytes(pomPath)
                : new byte[0];

            // Compare and write
            if (Arrays.equals(newPom.toByteArray(), existingContent)) {
                log.fine(() -> "Existing " + project().rootProject()
                    .directory().relativize(pomPath) + " is up to date.");
            } else {
                log.fine(() -> "Updating " + project().rootProject()
                    .directory().relativize(pomPath) + ".");
                Files.write(pomPath, newPom.toByteArray());
            }
        } catch (IOException e) {
            throw new BuildException(e);
        }

        @SuppressWarnings("unchecked")
        var result = (Stream<T>) Stream
            .of(project().newResource(PomFileType, pomPath));
        return result;
    }

    private Model generatePom() {
        Model model = new Model();
        model.setModelVersion("4.0.0");
        var groupId = project().<String> get(GroupId);
        if (groupId != null) {
            model.setGroupId(groupId);
        }
        model.setArtifactId(Optional.ofNullable(project()
            .<String> get(ArtifactId)).orElse(project().name()));
        model.setVersion(project().get(Version));
        model.setName(project().name());

        // Adapt
        pomAdapter.accept(model);
        return model;
    }

    /// Allow derived classes to post process the generated POM.
    ///
    /// @param adaptor the adaptor
    /// @return the pom file generator
    ///
    public PomFileGenerator adaptPom(Consumer<Model> adaptor) {
        this.pomAdapter = adaptor;
        return this;
    }

}
