/*
 * Decompiled with CFR 0.152.
 */
package org.jgroups.protocols.raft;

import java.io.File;
import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.function.ObjIntConsumer;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.commons.io.FileUtils;
import org.jgroups.Address;
import org.jgroups.logging.LogFactory;
import org.jgroups.protocols.raft.Log;
import org.jgroups.protocols.raft.LogEntry;
import org.jgroups.raft.util.IntegerHelper;
import org.jgroups.util.Streamable;
import org.jgroups.util.Util;
import org.rocksdb.Options;
import org.rocksdb.RocksDB;
import org.rocksdb.RocksDBException;
import org.rocksdb.WriteBatch;
import org.rocksdb.WriteOptions;

public class RocksDBLog
implements Log {
    private static final int ITERATION_BATCH_SIZE = 128;
    private final org.jgroups.logging.Log log = LogFactory.getLog(MethodHandles.lookup().lookupClass());
    private static final byte[] FIRSTAPPENDED = "FA".getBytes();
    private static final byte[] LASTAPPENDED = "LA".getBytes();
    private static final byte[] CURRENTTERM = "CT".getBytes();
    private static final byte[] COMMITINDEX = "CX".getBytes();
    private static final byte[] VOTEDFOR = "VF".getBytes();
    private RocksDB db;
    private File dbFileName;
    private int currentTerm = 0;
    private Address votedFor = null;
    private int commitIndex = 0;
    private int lastAppended = 0;
    private int firstAppended = 0;
    private volatile WriteOptions writeOptions;

    @Override
    public void init(String log_name, Map<String, String> args) throws Exception {
        boolean trace = this.log.isTraceEnabled();
        Options options = new Options();
        options.setCreateIfMissing(true);
        this.dbFileName = new File(log_name);
        this.db = RocksDB.open((Options)options, (String)this.dbFileName.getAbsolutePath());
        if (trace) {
            this.log.trace("opened %s", new Object[]{this.db});
        }
        this.writeOptions = new WriteOptions().setDisableWAL(false);
        if (this.isANewRAFTLog()) {
            if (trace) {
                this.log.trace("log %s is new, must be initialized", new Object[]{this.dbFileName});
            }
            this.initLogWithMetadata();
        } else {
            if (trace) {
                this.log.trace("log %s exists, does not have to be initialized", new Object[]{this.dbFileName});
            }
            this.readMetadataFromLog();
        }
        this.checkForConsistency();
    }

    @Override
    public void close() {
        if (this.log.isTraceEnabled()) {
            this.log.trace("closing DB: %s", new Object[]{this.db});
        }
        if (this.db != null) {
            this.db.close();
        }
        this.currentTerm = 0;
        this.votedFor = null;
        this.commitIndex = 0;
        this.lastAppended = 0;
        this.firstAppended = 0;
    }

    @Override
    public void delete() {
        this.close();
        if (this.log.isTraceEnabled()) {
            this.log.trace("deleting DB directory: %s", new Object[]{this.dbFileName});
        }
        try {
            FileUtils.deleteDirectory((File)this.dbFileName);
        }
        catch (IOException e) {
            this.log.error("Failed to delete directory " + this.dbFileName, (Throwable)e);
        }
    }

    @Override
    public int currentTerm() {
        return this.currentTerm;
    }

    @Override
    public Log currentTerm(int new_term) {
        this.setInt("term", CURRENTTERM, new_term);
        this.currentTerm = new_term;
        return this;
    }

    @Override
    public Address votedFor() {
        return this.votedFor;
    }

    @Override
    public Log votedFor(Address member) {
        if (this.log.isTraceEnabled()) {
            this.log.trace("Set voted-for to %s", new Object[]{member});
        }
        try {
            this.db.put(VOTEDFOR, Util.objectToByteBuffer((Object)member));
        }
        catch (Exception e) {
            this.log.error("Failed to set voted-for", (Throwable)e);
        }
        this.votedFor = member;
        return this;
    }

    @Override
    public int firstAppended() {
        return this.firstAppended;
    }

    @Override
    public int commitIndex() {
        return this.commitIndex;
    }

    @Override
    public Log commitIndex(int new_index) {
        this.setInt("commit-index", COMMITINDEX, new_index);
        this.commitIndex = new_index;
        return this;
    }

    @Override
    public int lastAppended() {
        return this.lastAppended;
    }

    @Override
    public void append(int index, boolean overwrite, LogEntry ... entries) {
        boolean trace = this.log.isTraceEnabled();
        if (trace) {
            this.log.trace("Appending %d entries", new Object[]{entries.length});
        }
        int newTerm = this.currentTerm;
        try (WriteBatch batch = new WriteBatch();){
            for (LogEntry entry : entries) {
                if (overwrite) {
                    this.appendEntry(index, entry, batch);
                } else {
                    this.appendEntryIfAbsent(index, entry, batch);
                }
                if (entry.term != newTerm) {
                    newTerm = entry.term;
                }
                ++index;
            }
            if (trace) {
                this.log.trace("Flushing batch to DB: %s", new Object[]{batch});
            }
            if (newTerm != this.currentTerm) {
                this.updateCurrentTerm(newTerm, batch);
            }
            this.updateLastAppended(index - 1, batch);
            this.db.write(this.writeOptions, batch);
        }
        catch (IOException | RocksDBException e) {
            this.log.error("Failed to append entries", e);
        }
    }

    @Override
    public LogEntry get(int index) {
        return this.getLogEntry(index);
    }

    @Override
    public void forEach(ObjIntConsumer<LogEntry> function, int start_index, int end_index) {
        start_index = Math.max(start_index, this.firstAppended);
        end_index = Math.min(end_index, this.lastAppended);
        if (start_index <= 0) {
            start_index = 1;
        }
        if (end_index < start_index) {
            return;
        }
        try {
            while (start_index <= end_index) {
                int iterationEndIndex = Math.min(start_index + 128 + 1, end_index + 1);
                List keys = IntStream.range(start_index, iterationEndIndex).mapToObj(IntegerHelper::fromIntToByteArray).collect(Collectors.toList());
                int index = start_index;
                for (byte[] entry : this.db.multiGetAsList(keys)) {
                    if (entry == null) {
                        return;
                    }
                    LogEntry logEntry = (LogEntry)Util.streamableFromByteBuffer(LogEntry::new, (byte[])entry);
                    function.accept(logEntry, index);
                    ++index;
                }
                start_index = iterationEndIndex;
            }
        }
        catch (Exception e) {
            this.log.error("Error while iterating over entry from [" + start_index + "," + end_index + "]", (Throwable)e);
        }
    }

    @Override
    public void forEach(ObjIntConsumer<LogEntry> function) {
        this.forEach(function, Math.max(1, this.firstAppended), this.lastAppended);
    }

    @Override
    public void truncate(int upto_index) {
        if (upto_index < this.firstAppended || upto_index > this.lastAppended) {
            return;
        }
        try (WriteBatch batch = new WriteBatch();){
            for (int index = this.firstAppended; index < upto_index; ++index) {
                batch.delete(IntegerHelper.fromIntToByteArray(index));
            }
            batch.put(FIRSTAPPENDED, IntegerHelper.fromIntToByteArray(upto_index));
            this.db.write(this.writeOptions, batch);
            this.firstAppended = upto_index;
        }
        catch (RocksDBException e) {
            this.log.error("Failed to truncate log to index " + upto_index, (Throwable)e);
        }
    }

    @Override
    public void deleteAllEntriesStartingFrom(int start_index) {
        if (start_index < this.firstAppended || start_index > this.lastAppended) {
            return;
        }
        try (WriteBatch batch = new WriteBatch();){
            for (int index = start_index; index <= this.lastAppended; ++index) {
                batch.delete(IntegerHelper.fromIntToByteArray(index));
            }
            LogEntry last = this.getLogEntry(start_index - 1);
            if (last == null) {
                this.updateCurrentTerm(0, batch);
            } else {
                this.updateCurrentTerm(last.term, batch);
            }
            this.updateLastAppended(start_index - 1, batch);
            if (this.commitIndex > this.lastAppended) {
                batch.put(COMMITINDEX, IntegerHelper.fromIntToByteArray(this.lastAppended));
                this.commitIndex = this.lastAppended;
            }
            this.db.write(this.writeOptions, batch);
        }
        catch (RocksDBException e) {
            this.log.error("Failed to delete log starting from " + start_index, (Throwable)e);
        }
    }

    public byte[] print(byte[] bytes) throws RocksDBException {
        return this.db.get(bytes);
    }

    public void printMetadata() throws Exception {
        this.log.info("-----------------");
        this.log.info("RAFT Log Metadata");
        this.log.info("-----------------");
        List data = this.db.multiGetAsList(Arrays.asList(FIRSTAPPENDED, LASTAPPENDED, CURRENTTERM, COMMITINDEX, VOTEDFOR));
        this.log.info("First Appended: %d", new Object[]{IntegerHelper.fromByteArrayToInt((byte[])data.get(0))});
        this.log.info("Last Appended: %d", new Object[]{IntegerHelper.fromByteArrayToInt((byte[])data.get(1))});
        this.log.info("Current Term: %d", new Object[]{IntegerHelper.fromByteArrayToInt((byte[])data.get(2))});
        this.log.info("Commit Index: %d", new Object[]{IntegerHelper.fromByteArrayToInt((byte[])data.get(3))});
        this.log.info("Voted for: %s", (Object[])Util.objectFromByteBuffer((byte[])((byte[])data.get(4))));
    }

    public String toString() {
        return "RocksDBLog{currentTerm=" + this.currentTerm + ", votedFor=" + this.votedFor + ", commitIndex=" + this.commitIndex + ", lastAppended=" + this.lastAppended + ", firstAppended=" + this.firstAppended + "}";
    }

    private LogEntry getLogEntry(int index) {
        try {
            byte[] entryBytes = this.db.get(IntegerHelper.fromIntToByteArray(index));
            return entryBytes == null ? null : (LogEntry)Util.streamableFromByteBuffer(LogEntry::new, (byte[])entryBytes);
        }
        catch (Exception e) {
            this.log.error("Failed to read log entry from index " + index, (Throwable)e);
            return null;
        }
    }

    private void appendEntryIfAbsent(int index, LogEntry entry, WriteBatch batch) throws RocksDBException, IOException {
        if (this.db.get(IntegerHelper.fromIntToByteArray(index)) != null) {
            throw new IllegalStateException("Entry at index " + index + " already exists");
        }
        this.appendEntry(index, entry, batch);
    }

    private void appendEntry(int index, LogEntry entry, WriteBatch batch) throws IOException, RocksDBException {
        if (this.log.isTraceEnabled()) {
            this.log.trace("Appending entry %d: %s", new Object[]{index, entry});
        }
        batch.put(IntegerHelper.fromIntToByteArray(index), Util.streamableToByteBuffer((Streamable)entry));
    }

    private void updateCurrentTerm(int updatedCurrentTerm, WriteBatch batch) throws RocksDBException {
        this.currentTerm = updatedCurrentTerm;
        if (this.log.isTraceEnabled()) {
            this.log.trace("Updating currentTerm: %d", new Object[]{updatedCurrentTerm});
        }
        batch.put(CURRENTTERM, IntegerHelper.fromIntToByteArray(this.currentTerm));
    }

    private void updateLastAppended(int updatedLastAppended, WriteBatch batch) throws RocksDBException {
        this.lastAppended = updatedLastAppended;
        if (this.log.isTraceEnabled()) {
            this.log.trace("Updating lastAppended: %d", new Object[]{updatedLastAppended});
        }
        batch.put(LASTAPPENDED, IntegerHelper.fromIntToByteArray(this.lastAppended));
    }

    private boolean isANewRAFTLog() throws RocksDBException {
        return this.db.get(FIRSTAPPENDED) == null;
    }

    private void initLogWithMetadata() throws RocksDBException {
        try (WriteBatch batch = new WriteBatch();){
            batch.put(FIRSTAPPENDED, IntegerHelper.fromIntToByteArray(0));
            batch.put(LASTAPPENDED, IntegerHelper.fromIntToByteArray(0));
            batch.put(CURRENTTERM, IntegerHelper.fromIntToByteArray(0));
            batch.put(COMMITINDEX, IntegerHelper.fromIntToByteArray(0));
            this.db.write(this.writeOptions, batch);
        }
    }

    private void readMetadataFromLog() throws Exception {
        List data = this.db.multiGetAsList(Arrays.asList(FIRSTAPPENDED, LASTAPPENDED, CURRENTTERM, COMMITINDEX, VOTEDFOR));
        this.firstAppended = IntegerHelper.fromByteArrayToInt((byte[])data.get(0));
        this.lastAppended = IntegerHelper.fromByteArrayToInt((byte[])data.get(1));
        this.currentTerm = IntegerHelper.fromByteArrayToInt((byte[])data.get(2));
        this.commitIndex = IntegerHelper.fromByteArrayToInt((byte[])data.get(3));
        this.votedFor = (Address)Util.objectFromByteBuffer((byte[])((byte[])data.get(4)));
        if (this.log.isDebugEnabled()) {
            this.log.debug("read metadata from log: firstAppended=%d, lastAppended=%d, currentTerm=%d, commitIndex=%d, votedFor=%s", new Object[]{this.firstAppended, this.lastAppended, this.currentTerm, this.commitIndex, this.votedFor});
        }
    }

    private void checkForConsistency() throws Exception {
        List data = this.db.multiGetAsList(Arrays.asList(FIRSTAPPENDED, LASTAPPENDED, CURRENTTERM, COMMITINDEX, VOTEDFOR));
        int loggedFirstAppended = IntegerHelper.fromByteArrayToInt((byte[])data.get(0));
        this.log.trace("FirstAppended in DB is: %d", new Object[]{loggedFirstAppended});
        int loggedLastAppended = IntegerHelper.fromByteArrayToInt((byte[])data.get(1));
        this.log.trace("LastAppended in DB is: %d", new Object[]{loggedLastAppended});
        int loggedCurrentTerm = IntegerHelper.fromByteArrayToInt((byte[])data.get(2));
        this.log.trace("CurrentTerm in DB is: %d", new Object[]{loggedCurrentTerm});
        int loggedCommitIndex = IntegerHelper.fromByteArrayToInt((byte[])data.get(3));
        this.log.trace("CommitIndex in DB is: %d", new Object[]{loggedCommitIndex});
        Address loggedVotedForAddress = (Address)Util.objectFromByteBuffer((byte[])((byte[])data.get(4)));
        this.log.trace("VotedFor in DB is: %s", new Object[]{loggedVotedForAddress});
        assert (this.firstAppended == loggedFirstAppended);
        assert (this.lastAppended == loggedLastAppended);
        assert (this.currentTerm == loggedCurrentTerm);
        assert (this.commitIndex == loggedCommitIndex);
        assert (this.votedFor == null || this.votedFor.equals(loggedVotedForAddress));
        LogEntry lastAppendedEntry = this.getLogEntry(this.lastAppended);
        assert (lastAppendedEntry == null || lastAppendedEntry.term <= this.currentTerm);
    }

    private void setInt(String field, byte[] key, int value) {
        if (this.log.isTraceEnabled()) {
            this.log.trace("Set %s to %s", new Object[]{field, value});
        }
        try {
            this.db.put(key, IntegerHelper.fromIntToByteArray(value));
        }
        catch (RocksDBException e) {
            this.log.error("Failed to set " + field, (Throwable)e);
        }
    }
}

