/*
 * Decompiled with CFR 0.152.
 */
package org.jgroups.raft.filelog;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.util.Objects;
import org.jgroups.logging.Log;
import org.jgroups.logging.LogFactory;
import org.jgroups.raft.util.pmem.FileProvider;

public final class FileStorage
implements Closeable {
    private static final Log LOG = LogFactory.getLog(FileStorage.class);
    private final File storageFile;
    private FileChannel channel;
    private RandomAccessFile raf;
    private long fileSize;
    private final int writeAheadBytes;
    private Flush requiredFlush;
    private ByteBuffer ioBuffer;
    private boolean ioBufferReady;

    public FileStorage(File storageFile) {
        this(storageFile, 0);
        this.ioBufferReady = false;
    }

    public FileStorage(File storageFile, int writeAheadBytes) {
        if (writeAheadBytes < 0) {
            throw new IllegalArgumentException("writeAheadBytes must be greater than or equals to 0");
        }
        this.storageFile = Objects.requireNonNull(storageFile);
        this.writeAheadBytes = writeAheadBytes;
        this.fileSize = -1L;
        this.requiredFlush = Flush.None;
    }

    public ByteBuffer ioBufferWith(int requiredCapacity) {
        ByteBuffer newBuffer;
        ByteBuffer availableWriteBuffer = this.ioBuffer;
        if (availableWriteBuffer != null && availableWriteBuffer.capacity() >= requiredCapacity) {
            availableWriteBuffer.position(0).limit(requiredCapacity);
            this.ioBufferReady = true;
            return availableWriteBuffer;
        }
        this.ioBuffer = newBuffer = ByteBuffer.allocateDirect(requiredCapacity);
        this.ioBufferReady = true;
        return newBuffer;
    }

    public long getCachedFileSize() {
        if (this.channel == null) {
            throw new IllegalStateException("fileSize cannot be retrieved if closed");
        }
        return this.fileSize;
    }

    public File getStorageFile() {
        return this.storageFile;
    }

    public void open() throws IOException {
        if (this.channel == null) {
            FileChannel pmemChannel = FileProvider.openPMEMChannel(this.storageFile, 1024, true, true);
            if (this.channel == null) {
                RandomAccessFile raf = new RandomAccessFile(this.storageFile, "rw");
                this.channel = raf.getChannel();
                this.raf = raf;
            } else {
                this.channel = pmemChannel;
                this.raf = null;
            }
            this.fileSize = this.channel.size();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int write(long position) throws IOException {
        boolean growFile;
        if (!this.ioBufferReady) {
            throw new IllegalStateException("must prepare the IO buffer first!");
        }
        FileChannel channel = this.checkOpen();
        int dataLength = this.ioBuffer.remaining();
        long nextPosition = position + (long)dataLength;
        boolean bl = growFile = nextPosition > this.fileSize;
        if (growFile) {
            if (this.writeAheadBytes > 0 && dataLength < this.writeAheadBytes) {
                long writeAheadSize = nextPosition + (long)this.writeAheadBytes;
                this.raf.setLength(writeAheadSize);
                this.fileSize = writeAheadSize;
            } else {
                this.fileSize = nextPosition;
            }
            this.requiredFlush = Flush.Metadata;
        }
        try {
            int written = channel.write(this.ioBuffer, position);
            if (written < dataLength && !growFile) {
                this.fileSize = position + (long)written;
            }
            if (written > 0 && this.requiredFlush == Flush.None) {
                this.requiredFlush = Flush.Data;
            }
            int n = written;
            return n;
        }
        finally {
            this.ioBufferReady = false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ByteBuffer read(long position, int expectedLength) throws IOException {
        FileChannel channel = this.checkOpen();
        ByteBuffer readBuffer = this.ioBufferWith(expectedLength);
        assert (this.ioBufferReady);
        try {
            channel.read(readBuffer, position);
            ByteBuffer byteBuffer = readBuffer.flip();
            return byteBuffer;
        }
        finally {
            this.ioBufferReady = false;
        }
    }

    public boolean isOpened() {
        return this.channel != null && this.channel.isOpen();
    }

    private FileChannel checkOpen() {
        FileChannel channel = this.channel;
        if (channel == null || !channel.isOpen()) {
            throw new IllegalStateException("File " + this.storageFile.getAbsolutePath() + " not open!");
        }
        return channel;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public void flush() throws IOException {
        this.checkOpen();
        if (this.requiredFlush == Flush.None) {
            return;
        }
        try {
            switch (this.requiredFlush) {
                case Metadata: {
                    this.channel.force(true);
                    return;
                }
                case Data: {
                    this.channel.force(false);
                    return;
                }
            }
            return;
        }
        finally {
            this.requiredFlush = Flush.None;
        }
    }

    public void truncateTo(long size) throws IOException {
        FileChannel existing = this.checkOpen();
        existing.truncate(size);
        this.fileSize = size;
        this.requiredFlush = Flush.Metadata;
    }

    public void truncateFrom(long position) throws IOException {
        FileChannel existing = this.checkOpen();
        if (position > this.fileSize) {
            throw new IllegalArgumentException("position must be greater then fileSize");
        }
        File tmpFile = new File(this.storageFile.getParentFile(), this.storageFile.getName() + ".tmp");
        try (FileChannel newChannel = FileStorage.tryCreatePMEMFileChannel(tmpFile, this.fileSize);){
            existing.transferTo(position, this.fileSize, newChannel);
        }
        try {
            Files.move(tmpFile.toPath(), this.storageFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException failedMove) {
            try {
                Files.deleteIfExists(tmpFile.toPath());
            }
            catch (IOException deleteTmpEx) {
                failedMove.addSuppressed(deleteTmpEx);
                throw failedMove;
            }
            throw failedMove;
        }
        this.channel = null;
        this.raf = null;
        this.fileSize = -1L;
        existing.close();
        this.open();
        this.requiredFlush = Flush.Metadata;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public void close() throws IOException {
        if (this.channel == null) {
            return;
        }
        IOException suppressed = null;
        try {
            if (this.raf == null) return;
            this.raf.close();
            return;
        }
        catch (IOException rafEx) {
            suppressed = rafEx;
            return;
        }
        finally {
            try {
                this.channel.close();
            }
            catch (IOException chEx) {
                if (suppressed != null) {
                    chEx.addSuppressed(suppressed);
                    throw chEx;
                }
            }
            finally {
                this.requiredFlush = Flush.None;
                this.raf = null;
                this.channel = null;
                this.fileSize = -1L;
            }
        }
    }

    public void delete() throws IOException {
        if (this.storageFile.exists()) {
            if (!this.storageFile.delete()) {
                LOG.warn("Failed to delete file " + this.storageFile.getAbsolutePath());
            }
            this.close();
        }
    }

    private static FileChannel tryCreatePMEMFileChannel(File tmpFile, long fileSize) throws IOException {
        FileChannel pmemChannel;
        if (FileProvider.isPMEMAvailable() && (pmemChannel = FileProvider.openPMEMChannel(tmpFile, (int)fileSize, true, true)) != null) {
            return pmemChannel;
        }
        return FileChannel.open(tmpFile.toPath(), StandardOpenOption.WRITE, StandardOpenOption.CREATE);
    }

    private static enum Flush {
        Metadata,
        Data,
        None;

    }
}

