/*
 * Decompiled with CFR 0.152.
 */
package org.jgroups.raft.filelog;

import java.io.File;
import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.nio.ByteBuffer;
import java.util.Objects;
import java.util.function.ObjIntConsumer;
import org.jgroups.logging.Log;
import org.jgroups.logging.LogFactory;
import org.jgroups.protocols.raft.LogEntries;
import org.jgroups.protocols.raft.LogEntry;
import org.jgroups.raft.filelog.FilePositionCache;
import org.jgroups.raft.filelog.FileStorage;

public class LogEntryStorage {
    private static final Log log = LogFactory.getLog(MethodHandles.lookup().lookupClass());
    private static final byte MAGIC_NUMBER = 1;
    private static final String FILE_NAME = "entries.raft";
    private static final int HEADER_SIZE = 17;
    private static final int DEFAULT_WRITE_AHEAD_BYTES = 4096;
    private final FileStorage fileStorage;
    private FilePositionCache positionCache;
    private Header lastAppendedHeader;
    private int lastAppended;
    private boolean fsync;

    public LogEntryStorage(File parentDir, boolean fsync) {
        this.fsync = fsync;
        this.positionCache = new FilePositionCache(0);
        this.fileStorage = new FileStorage(new File(parentDir, FILE_NAME), 4096);
    }

    public void open() throws IOException {
        this.fileStorage.open();
    }

    public void close() throws IOException {
        this.fileStorage.close();
    }

    public void delete() throws IOException {
        this.fileStorage.delete();
    }

    public void reload(int commitIndex) throws IOException {
        Header header = this.readHeader(0L);
        if (header == null) {
            this.positionCache = new FilePositionCache(0);
            this.lastAppended = 0;
            return;
        }
        this.positionCache = new FilePositionCache(header.index == 1 ? 0 : header.index);
        this.setFilePosition(header.index, header.position);
        this.lastAppended = header.index;
        long position = header.nextPosition();
        while ((header = this.readHeader(position)) != null) {
            this.setFilePosition(header.index, header.position);
            position = header.nextPosition();
            this.lastAppended = header.index;
        }
        return;
    }

    public int getFirstAppended() {
        return this.positionCache.getFirstAppended();
    }

    public int getLastAppended() {
        return this.lastAppended;
    }

    public LogEntry getLogEntry(int index) throws IOException {
        long position = this.positionCache.getPosition(index);
        if (position < 0L) {
            return null;
        }
        Header header = this.readHeader(position);
        if (header == null) {
            return null;
        }
        return header.readLogEntry(this);
    }

    public int write(int startIndex, LogEntries entries) throws IOException {
        if (startIndex == 1) {
            return this.appendWithoutOverwriteCheck(entries, 1, 0L);
        }
        long previousPosition = this.positionCache.getPosition(startIndex - 1);
        if (previousPosition < 0L) {
            throw new IllegalStateException();
        }
        if (this.lastAppendedHeader == null || this.lastAppendedHeader.position != previousPosition) {
            this.lastAppendedHeader = this.readHeader(previousPosition);
            assert (this.lastAppendedHeader == null || this.lastAppendedHeader.position == previousPosition);
        }
        if (this.lastAppendedHeader == null) {
            throw new IllegalStateException();
        }
        return this.appendWithoutOverwriteCheck(entries, startIndex, this.lastAppendedHeader.nextPosition());
    }

    private void setFilePosition(int index, long position) {
        if (!this.positionCache.set(index, position)) {
            log.warn("Unable to set file position for index " + index + ". LogEntry is too old");
        }
    }

    private int appendOverwriteCheck(LogEntry[] entries, int index, long position) throws IOException {
        int term = 0;
        for (LogEntry entry : entries) {
            Header header = new Header(position, index, entry);
            if (!this.entryExists(header)) {
                ByteBuffer buffer = this.fileStorage.ioBufferWith(header.totalLength);
                header.writeTo(buffer);
                buffer.put(entry.command(), entry.offset(), entry.length());
                buffer.flip();
                this.fileStorage.write(header.position);
                buffer.clear();
                this.setFilePosition(header.index, header.position);
                term = Math.max(entry.term(), term);
            }
            position = header.nextPosition();
            ++index;
        }
        this.lastAppended = index - 1;
        if (this.positionCache.invalidateFrom(index)) {
            this.fileStorage.truncateTo(position);
        }
        if (this.fsync) {
            this.fileStorage.flush();
        }
        return term;
    }

    private int appendWithoutOverwriteCheck(LogEntries entries, int index, long position) throws IOException {
        int term = 0;
        int batchBytes = 0;
        for (LogEntry entry : entries) {
            batchBytes += Header.getTotalLength(entry.length());
        }
        long startPosition = position;
        ByteBuffer batchBuffer = this.fileStorage.ioBufferWith(batchBytes);
        int size = entries.size();
        int i = 0;
        for (LogEntry entry : entries) {
            Header header = new Header(position, index, entry);
            header.writeTo(batchBuffer);
            if (entry.length() > 0) {
                batchBuffer.put(entry.command(), entry.offset(), entry.length());
            }
            this.setFilePosition(header.index, header.position);
            term = Math.max(entry.term(), term);
            position = header.nextPosition();
            if (i == size - 1) {
                this.lastAppendedHeader = header;
            }
            ++i;
            ++index;
        }
        batchBuffer.flip();
        this.fileStorage.write(startPosition);
        this.lastAppended = index - 1;
        if (this.positionCache.invalidateFrom(index)) {
            this.fileStorage.truncateTo(position);
        }
        if (this.fsync) {
            this.fileStorage.flush();
        }
        return term;
    }

    private static Header readHeader(LogEntryStorage logStorage, long position) throws IOException {
        ByteBuffer data = logStorage.fileStorage.read(position, 17);
        if (data.remaining() != 17) {
            return null;
        }
        return new Header(position, data).consistencyCheck();
    }

    private Header readHeader(long position) throws IOException {
        return LogEntryStorage.readHeader(this, position);
    }

    public void removeOld(int index) throws IOException {
        int indexToRemove = Math.min(this.lastAppended, index);
        long pos = this.positionCache.getPosition(indexToRemove);
        if (pos > 0L) {
            this.fileStorage.truncateFrom(pos);
        }
        this.positionCache = this.positionCache.createDeleteCopyFrom(index);
        if (this.lastAppended < index) {
            this.lastAppended = index;
        }
    }

    public void reinitializeTo(int index, LogEntry firstEntry) throws IOException {
        this.fileStorage.truncateTo(0L);
        this.positionCache = new FilePositionCache(index);
        int batchBytes = Header.getTotalLength(firstEntry.length());
        ByteBuffer batchBuffer = this.fileStorage.ioBufferWith(batchBytes);
        Header header = new Header(0L, index, firstEntry);
        header.writeTo(batchBuffer);
        if (firstEntry.length() > 0) {
            batchBuffer.put(firstEntry.command(), firstEntry.offset(), firstEntry.length());
        }
        this.fileStorage.write(0L);
        this.setFilePosition(index, 0L);
        if (this.fsync) {
            this.fileStorage.flush();
        }
        this.lastAppended = index;
        this.lastAppendedHeader = header;
    }

    public int removeNew(int index) throws IOException {
        if (index == 1) {
            this.fileStorage.truncateTo(0L);
            this.lastAppended = 0;
            return 0;
        }
        long position = this.positionCache.getPosition(index - 1);
        Header previousHeader = this.readHeader(position);
        if (previousHeader == null) {
            throw new IllegalStateException();
        }
        this.fileStorage.truncateTo(previousHeader.nextPosition());
        this.positionCache.invalidateFrom(index);
        this.lastAppended = index - 1;
        return previousHeader.term;
    }

    public void forEach(ObjIntConsumer<LogEntry> consumer, int startIndex, int endIndex) throws IOException {
        long position = this.positionCache.getPosition(startIndex = Math.max(Math.max(startIndex, this.getFirstAppended()), 1));
        if (position < 0L) {
            return;
        }
        while (startIndex <= endIndex) {
            Header header = this.readHeader(position);
            if (header == null) {
                return;
            }
            consumer.accept(header.readLogEntry(this), startIndex);
            position = header.nextPosition();
            ++startIndex;
        }
    }

    private boolean entryExists(Header header) throws IOException {
        Header existing = LogEntryStorage.readHeader(this, header.position);
        if (existing == null) {
            return false;
        }
        if (existing.equals(header)) {
            return true;
        }
        throw new IllegalStateException();
    }

    public void useFsync(boolean value) {
        this.fsync = value;
    }

    private static class Header {
        final long position;
        final byte magic;
        final int totalLength;
        final int term;
        final int index;
        final int dataLength;

        Header(long position, int index, LogEntry entry) {
            Objects.requireNonNull(entry);
            this.position = position;
            this.magic = 1;
            this.index = index;
            this.term = entry.term();
            this.dataLength = entry.length();
            this.totalLength = Header.getTotalLength(this.dataLength);
        }

        public static int getTotalLength(int dataLength) {
            return 17 + dataLength;
        }

        Header(long position, ByteBuffer buffer) {
            this.position = position;
            this.magic = buffer.get();
            this.totalLength = buffer.getInt();
            this.term = buffer.getInt();
            this.index = buffer.getInt();
            this.dataLength = buffer.getInt();
        }

        public void writeTo(ByteBuffer buffer) {
            buffer.put(this.magic);
            buffer.putInt(this.totalLength);
            buffer.putInt(this.term);
            buffer.putInt(this.index);
            buffer.putInt(this.dataLength);
        }

        long nextPosition() {
            return 17L + this.position + (long)this.dataLength;
        }

        Header consistencyCheck() {
            return this.magic != 1 || this.term <= 0 || this.index <= 0 || this.dataLength < 0 || this.totalLength < 17 || this.dataLength + 17 != this.totalLength ? null : this;
        }

        LogEntry readLogEntry(LogEntryStorage storage) throws IOException {
            ByteBuffer data = storage.fileStorage.read(this.position + 17L, this.dataLength);
            if (data.remaining() != this.dataLength) {
                return null;
            }
            assert (!data.hasArray());
            byte[] bytes = new byte[this.dataLength];
            data.get(bytes);
            return new LogEntry(this.term, bytes);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Header header = (Header)o;
            return this.position == header.position && this.magic == header.magic && this.totalLength == header.totalLength && this.term == header.term && this.index == header.index && this.dataLength == header.dataLength;
        }

        public int hashCode() {
            return Objects.hash(this.position, this.magic, this.totalLength, this.term, this.index, this.dataLength);
        }
    }
}

