/*
 * Copyright (c) 2010-2016 the original author or authors
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package org.jmxtrans.agent.util.io;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;

import javax.annotation.Nonnull;

/**
 * Inspired by {@code org.springframework.core.io.AbstractResource}
 * @author <a href="mailto:cleclerc@cloudbees.com">Cyrille Le Clerc</a>
 */
public abstract class AbstractResource  implements Resource {

    @Override
    public long lastModified() throws IoRuntimeException {
        long lastModified = getFile().lastModified();
        if (lastModified == 0L) {
            throw new FileNotFoundRuntimeException(getDescription() +
                    " cannot be resolved in the file system for resolving its last-modified timestamp");
        }
        return lastModified;
    }

    @Nonnull
    @Override
    public URL getURL() throws IoRuntimeException {
        throw new FileNotFoundRuntimeException(getDescription() + " cannot be resolved to URL");
    }

    @Nonnull
    @Override
    public File getFile() throws IoRuntimeException {
        throw new FileNotFoundRuntimeException(getDescription() + " cannot be resolved to absolute file path");
    }

    @Nonnull
    @Override
    public URI getURI() throws IoRuntimeException {
        URL url = getURL();
        try {
            return url.toURI();
        } catch (URISyntaxException e) {
            throw new IoRuntimeException("Exception parsing the URI of '" + url + "'", e);
        }
    }

    /**
     * This implementation checks whether a File can be opened,
     * falling back to whether an InputStream can be opened.
     * This will cover both directories and content resources.
     */
    @Override
    public boolean exists() {
        // Try file existence: can we find the file in the file system?
        try {
            return getFile().exists();
        }
        catch (IoRuntimeException ex) {
            // Fall back to stream existence: can we open the stream?
            try {
                try (InputStream is = getInputStream()) {
                	return true;
                }
            }
            catch (Throwable isEx) {
                return false;
            }
        }
    }

    @Override
    public String toString() {
        return getDescription();
    }
}
