/*
 * Decompiled with CFR 0.152.
 */
package com.googlecode.jmxtrans;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.inject.Injector;
import com.google.inject.name.Named;
import com.googlecode.jmxtrans.ConfigurationParser;
import com.googlecode.jmxtrans.ManagedJmxTransformerProcess;
import com.googlecode.jmxtrans.classloader.ClassLoaderEnricher;
import com.googlecode.jmxtrans.cli.JCommanderArgumentParser;
import com.googlecode.jmxtrans.cli.JmxTransConfiguration;
import com.googlecode.jmxtrans.exceptions.LifecycleException;
import com.googlecode.jmxtrans.guice.JmxTransModule;
import com.googlecode.jmxtrans.jobs.ServerJob;
import com.googlecode.jmxtrans.model.JmxProcess;
import com.googlecode.jmxtrans.model.OutputWriter;
import com.googlecode.jmxtrans.model.Query;
import com.googlecode.jmxtrans.model.Server;
import com.googlecode.jmxtrans.model.ValidationException;
import com.googlecode.jmxtrans.monitoring.ManagedThreadPoolExecutor;
import com.googlecode.jmxtrans.util.WatchDir;
import com.googlecode.jmxtrans.util.WatchedCallback;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.File;
import java.lang.management.ManagementFactory;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nonnull;
import javax.inject.Inject;
import javax.management.MBeanServer;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang.RandomStringUtils;
import org.quartz.CronExpression;
import org.quartz.CronTrigger;
import org.quartz.JobDataMap;
import org.quartz.JobDetail;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.Trigger;
import org.quartz.TriggerUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JmxTransformer
implements WatchedCallback {
    private static final Logger log = LoggerFactory.getLogger(JmxTransformer.class);
    private final Scheduler serverScheduler;
    private final JmxTransConfiguration configuration;
    private final ConfigurationParser configurationParser;
    private final Injector injector;
    private WatchDir watcher;
    private ImmutableList<Server> masterServersList = ImmutableList.of();
    private Thread shutdownHook = new ShutdownHook();
    private volatile boolean isRunning = false;
    @Nonnull
    private final ThreadPoolExecutor queryProcessorExecutor;
    @Nonnull
    private final ThreadPoolExecutor resultProcessorExecutor;
    @Nonnull
    private final ThreadLocalRandom random = ThreadLocalRandom.current();

    @Inject
    public JmxTransformer(Scheduler serverScheduler, JmxTransConfiguration configuration, ConfigurationParser configurationParser, Injector injector, @Nonnull @Named(value="queryProcessorExecutor") ThreadPoolExecutor queryProcessorExecutor, @Nonnull @Named(value="resultProcessorExecutor") ThreadPoolExecutor resultProcessorExecutor) {
        this.serverScheduler = serverScheduler;
        this.configuration = configuration;
        this.configurationParser = configurationParser;
        this.injector = injector;
        this.queryProcessorExecutor = queryProcessorExecutor;
        this.resultProcessorExecutor = resultProcessorExecutor;
    }

    public static void main(String[] args) throws Exception {
        JmxTransConfiguration configuration = new JCommanderArgumentParser().parseOptions(args);
        if (configuration.isHelp()) {
            return;
        }
        ClassLoaderEnricher enricher = new ClassLoaderEnricher();
        for (File jar : configuration.getAdditionalJars()) {
            enricher.add(jar);
        }
        Injector injector = JmxTransModule.createInjector(configuration);
        JmxTransformer transformer = (JmxTransformer)injector.getInstance(JmxTransformer.class);
        transformer.doMain();
    }

    private void doMain() throws Exception {
        MBeanServer platformMBeanServer = ManagementFactory.getPlatformMBeanServer();
        ManagedJmxTransformerProcess mbean = new ManagedJmxTransformerProcess(this, this.configuration);
        platformMBeanServer.registerMBean(mbean, mbean.getObjectName());
        ManagedThreadPoolExecutor queryExecutorMBean = new ManagedThreadPoolExecutor(this.queryProcessorExecutor, "queryProcessorExecutor");
        platformMBeanServer.registerMBean(queryExecutorMBean, queryExecutorMBean.getObjectName());
        ManagedThreadPoolExecutor resultExecutorMBean = new ManagedThreadPoolExecutor(this.resultProcessorExecutor, "resultProcessorExecutor");
        platformMBeanServer.registerMBean(resultExecutorMBean, resultExecutorMBean.getObjectName());
        this.start();
        try {
            while (true) {
                Thread.sleep(5L);
            }
        }
        catch (Exception e) {
            log.info("shutting down", (Throwable)e);
            platformMBeanServer.unregisterMBean(mbean.getObjectName());
            platformMBeanServer.unregisterMBean(queryExecutorMBean.getObjectName());
            platformMBeanServer.unregisterMBean(resultExecutorMBean.getObjectName());
            return;
        }
    }

    public synchronized void start() throws LifecycleException {
        if (this.isRunning) {
            throw new LifecycleException("Process already started");
        }
        log.info("Starting Jmxtrans on : {}", (Object)this.configuration.getJsonDirOrFile());
        try {
            this.serverScheduler.start();
            this.startupWatchdir();
            this.startupSystem();
        }
        catch (Exception e) {
            log.error(e.getMessage(), (Throwable)e);
            throw new LifecycleException(e);
        }
        Runtime.getRuntime().addShutdownHook(this.shutdownHook);
        this.isRunning = true;
    }

    public synchronized void stop() throws LifecycleException {
        if (!this.isRunning) {
            throw new LifecycleException("Process already stopped");
        }
        try {
            log.info("Stopping Jmxtrans");
            if (this.shutdownHook != null) {
                Runtime.getRuntime().removeShutdownHook(this.shutdownHook);
            }
            this.stopServices();
            this.isRunning = false;
        }
        catch (LifecycleException e) {
            log.error(e.getMessage(), (Throwable)e);
            throw new LifecycleException(e);
        }
    }

    @SuppressFBWarnings(value={"SWL_SLEEP_WITH_LOCK_HELD"}, justification="Workaround for Quartz issue")
    private synchronized void stopServices() throws LifecycleException {
        try {
            if (this.serverScheduler.isStarted()) {
                this.serverScheduler.shutdown(true);
                log.debug("Shutdown server scheduler");
                try {
                    Thread.sleep(1500L);
                }
                catch (InterruptedException e) {
                    log.error(e.getMessage(), (Throwable)e);
                    Thread.currentThread().interrupt();
                }
            }
            MoreExecutors.shutdownAndAwaitTermination((ExecutorService)this.queryProcessorExecutor, (long)10L, (TimeUnit)TimeUnit.SECONDS);
            MoreExecutors.shutdownAndAwaitTermination((ExecutorService)this.resultProcessorExecutor, (long)10L, (TimeUnit)TimeUnit.SECONDS);
            if (this.watcher != null) {
                this.watcher.stopService();
                this.watcher = null;
                log.debug("Shutdown watch service");
            }
            this.stopWriterAndClearMasterServerList();
        }
        catch (Exception e) {
            throw new LifecycleException(e);
        }
    }

    private void stopWriterAndClearMasterServerList() {
        for (Server server : this.masterServersList) {
            for (OutputWriter writer : server.getOutputWriters()) {
                try {
                    writer.close();
                }
                catch (LifecycleException ex) {
                    log.error("Eror stopping writer: {}", (Object)writer);
                }
            }
            for (Query query : server.getQueries()) {
                for (OutputWriter writer : query.getOutputWriterInstances()) {
                    try {
                        writer.close();
                        log.debug("Stopped writer: {} for query: {}", (Object)writer, (Object)query);
                    }
                    catch (LifecycleException ex) {
                        log.error("Error stopping writer: {} for query: {}", new Object[]{writer, query, ex});
                    }
                }
            }
        }
        this.masterServersList = ImmutableList.of();
    }

    private void startupWatchdir() throws Exception {
        File dirToWatch = this.configuration.getJsonDirOrFile().isFile() ? new File(FilenameUtils.getFullPath((String)this.configuration.getJsonDirOrFile().getAbsolutePath())) : this.configuration.getJsonDirOrFile();
        this.watcher = new WatchDir(dirToWatch, (WatchedCallback)this);
        this.watcher.start();
    }

    public void executeStandalone(JmxProcess process) throws Exception {
        this.masterServersList = process.getServers();
        this.serverScheduler.start();
        this.processServersIntoJobs();
        Thread.sleep(TimeUnit.MILLISECONDS.convert(10L, TimeUnit.SECONDS));
    }

    private void startupSystem() throws LifecycleException {
        this.processFilesIntoServers();
        this.processServersIntoJobs();
    }

    private void validateSetup(Server server, ImmutableSet<Query> queries) throws ValidationException {
        for (Query q : queries) {
            this.validateSetup(server, q);
        }
    }

    private void validateSetup(Server server, Query query) throws ValidationException {
        for (OutputWriter w : query.getOutputWriterInstances()) {
            this.injector.injectMembers((Object)w);
            w.validateSetup(server, query);
        }
    }

    private void processFilesIntoServers() throws LifecycleException {
        try {
            this.stopWriterAndClearMasterServerList();
        }
        catch (Exception e) {
            log.error("Error while clearing master server list: " + e.getMessage(), (Throwable)e);
            throw new LifecycleException(e);
        }
        this.masterServersList = this.configurationParser.parseServers(this.getJsonFiles(), this.configuration.isContinueOnJsonError());
    }

    private void processServersIntoJobs() throws LifecycleException {
        for (Server server : this.masterServersList) {
            try {
                for (Query query : server.getQueries()) {
                    for (OutputWriter writer : query.getOutputWriterInstances()) {
                        writer.start();
                    }
                }
                this.validateSetup(server, server.getQueries());
                this.scheduleJob(server);
            }
            catch (ParseException ex) {
                throw new LifecycleException("Error parsing cron expression: " + server.getCronExpression(), ex);
            }
            catch (SchedulerException ex) {
                throw new LifecycleException("Error scheduling job for server: " + server, ex);
            }
            catch (ValidationException ex) {
                throw new LifecycleException("Error validating json setup for query", ex);
            }
        }
    }

    private void scheduleJob(Server server) throws ParseException, SchedulerException {
        CronTrigger trigger;
        String name = server.getHost() + ":" + server.getPort() + "-" + System.currentTimeMillis() + "-" + RandomStringUtils.randomNumeric((int)10);
        JobDetail jd = new JobDetail(name, "ServerJob", ServerJob.class);
        JobDataMap map = new JobDataMap();
        map.put((Object)Server.class.getName(), (Object)server);
        jd.setJobDataMap(map);
        if (server.getCronExpression() != null && CronExpression.isValidExpression((String)server.getCronExpression())) {
            trigger = new CronTrigger();
            trigger.setCronExpression(server.getCronExpression());
            trigger.setName(server.getHost() + ":" + server.getPort() + "-" + Long.toString(System.currentTimeMillis()));
            trigger.setStartTime(this.computeSpreadStartDate(this.configuration.getRunPeriod()));
        } else {
            int runPeriod = this.configuration.getRunPeriod();
            if (server.getRunPeriodSeconds() != null) {
                runPeriod = server.getRunPeriodSeconds();
            }
            Trigger minuteTrigger = TriggerUtils.makeSecondlyTrigger((int)runPeriod);
            minuteTrigger.setName(server.getHost() + ":" + server.getPort() + "-" + Long.toString(System.currentTimeMillis()));
            minuteTrigger.setStartTime(this.computeSpreadStartDate(runPeriod));
            trigger = minuteTrigger;
        }
        this.serverScheduler.scheduleJob(jd, (Trigger)trigger);
        if (log.isDebugEnabled()) {
            log.debug("Scheduled job: " + jd.getName() + " for server: " + server);
        }
    }

    @VisibleForTesting
    Date computeSpreadStartDate(int runPeriod) {
        long spread = this.random.nextLong(TimeUnit.MILLISECONDS.convert(runPeriod, TimeUnit.SECONDS));
        return new Date(new Date().getTime() + spread);
    }

    private void deleteAllJobs() throws Exception {
        ArrayList<JobDetail> allJobs = new ArrayList<JobDetail>();
        String[] jobGroups = this.serverScheduler.getJobGroupNames();
        for (String jobGroup : jobGroups) {
            String[] jobNames;
            for (String jobName : jobNames = this.serverScheduler.getJobNames(jobGroup)) {
                allJobs.add(this.serverScheduler.getJobDetail(jobName, jobGroup));
            }
        }
        for (JobDetail jd : allJobs) {
            this.serverScheduler.deleteJob(jd.getName(), jd.getGroup());
            if (!log.isDebugEnabled()) continue;
            log.debug("Deleted scheduled job: " + jd.getName() + " group: " + jd.getGroup());
        }
    }

    private List<File> getJsonFiles() {
        File jsonDirOrFile = this.configuration.getJsonDirOrFile();
        if (jsonDirOrFile == null) {
            throw new IllegalStateException("Configuration should specify configuration directory or file, with -j of -f option");
        }
        File[] files = jsonDirOrFile.isFile() ? new File[]{jsonDirOrFile} : (File[])MoreObjects.firstNonNull((Object)jsonDirOrFile.listFiles(), (Object)new File[0]);
        ArrayList<File> result = new ArrayList<File>();
        for (File file : files) {
            if (!this.isJsonFile(file)) continue;
            result.add(file);
        }
        return result;
    }

    private boolean isJsonFile(File file) {
        if (this.configuration.getJsonDirOrFile().isFile()) {
            return file.equals(this.configuration.getJsonDirOrFile());
        }
        return file.isFile() && file.getName().endsWith(".json");
    }

    public void fileModified(File file) throws Exception {
        if (this.isJsonFile(file)) {
            Thread.sleep(1000L);
            log.info("Configuration file modified: " + file);
            this.deleteAllJobs();
            this.startupSystem();
        }
    }

    public void fileDeleted(File file) throws Exception {
        log.info("Configuration file deleted: " + file);
        Thread.sleep(1000L);
        this.deleteAllJobs();
        this.startupSystem();
    }

    public void fileAdded(File file) throws Exception {
        if (this.isJsonFile(file)) {
            Thread.sleep(1000L);
            log.info("Configuration file added: " + file);
            this.deleteAllJobs();
            this.startupSystem();
        }
    }

    protected class ShutdownHook
    extends Thread {
        protected ShutdownHook() {
        }

        @Override
        public void run() {
            try {
                JmxTransformer.this.stopServices();
            }
            catch (LifecycleException e) {
                log.error("Error shutdown hook", (Throwable)e);
            }
        }
    }
}

