/*
 *  Copyright 2001-2014 Stephen Colebourne
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.joda.beans.gen;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.BeanDefinition;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaProperty;
import org.joda.beans.Property;
import org.joda.beans.PropertyDefinition;
import org.joda.beans.impl.direct.DirectBean;
import org.joda.beans.impl.direct.DirectBeanBuilder;
import org.joda.beans.impl.direct.DirectFieldsBeanBuilder;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;

/**
 * Code generator for a bean.
 * 
 * @author Stephen Colebourne
 */
class BeanGen {

    /** Constructor style for none. */
    static final int CONSTRUCTOR_NONE = 0;
    /** Constructor style for builder-based. */
    static final int CONSTRUCTOR_BY_BUILDER = 1;
    /** Constructor style for argument-based. */
    static final int CONSTRUCTOR_BY_ARGS = 2;
    /** Start marker. */
    private static final String AUTOGENERATED_START_TEXT = "AUTOGENERATED START";
    /** Start marker. */
    private static final String AUTOGENERATED_START = "\t//------------------------- AUTOGENERATED START -------------------------";
    /** End marker. */
    private static final String AUTOGENERATED_END = "\t//-------------------------- AUTOGENERATED END --------------------------";
    /** Line separator. */
    private static final String LINE_SEPARATOR = "\t//-----------------------------------------------------------------------";
    /** Line separator. */
    private static final String LINE_SEPARATOR_INDENTED = "\t\t//-----------------------------------------------------------------------";
    /** Pattern to find bean type. */
    private static final Pattern BEAN_TYPE = Pattern.compile(".*class +(" +
            "([A-Z][A-Za-z0-9_]+)" +
                "(?:<" +
                    "([A-Z])( +extends +[A-Za-z0-9_<>]+)?" +
                    "(?:[,] +" +
                        "([A-Z])( +extends +[A-Za-z0-9_<>]+)?" +
                        "(?:[,] +" +
                            "([A-Z])( +extends +[A-Za-z0-9_<>]+)?" +
                        ")?" +
                    ")?" +
                ">)?" +
            ").*");
    /** Pattern to find super type. */
    private static final Pattern SUPER_TYPE = Pattern.compile(".*extends +(" +
            "([A-Z][A-Za-z0-9_]+)" +
                "(?:<" +
                    "([A-Z][A-Za-z0-9_<> ]*)" +
                    "(?:[,] +" +
                        "([A-Z][A-Za-z0-9_<> ]*)?" +
                        "(?:[,] +" +
                            "([A-Z][A-Za-z0-9_<> ]*)?" +
                        ")?" +
                    ")?" +
                ">)?" +
            ").*");
    /** Pattern to find root type. */
    private static final Pattern SUPER_IMPL_TYPE = Pattern.compile(".*implements.*[ ,]((Immutable)?Bean)[ ,].*");
    /** The style pattern. */
    private static final Pattern STYLE_PATTERN = Pattern.compile(".*[ ,(]style[ ]*[=][ ]*[\"]([a-zA-Z]*)[\"].*");
    /** The style pattern. */
    private static final Pattern BUILDER_SCOPE_PATTERN = Pattern.compile(".*[ ,(]builderScope[ ]*[=][ ]*[\"]([a-zA-Z]*)[\"].*");
    /** The root pattern. */
    private static final Pattern HIERARCHY_PATTERN = Pattern.compile(".*[ ,(]hierarchy[ ]*[=][ ]*[\"]([a-zA-Z]*)[\"].*");
    /** The validator pattern. */
    private static final Pattern VALIDATOR_PATTERN = Pattern.compile(".*private[ ]+void[ ]+([a-zA-Z][a-zA-Z0-9]*)[(][ ]*[)].*");
    /** Pattern to find super type. */
    private static final Set<String> PRIMITIVE_EQUALS = new HashSet<String>();
    static {
        PRIMITIVE_EQUALS.add("boolean");
        PRIMITIVE_EQUALS.add("char");
        PRIMITIVE_EQUALS.add("byte");
        PRIMITIVE_EQUALS.add("short");
        PRIMITIVE_EQUALS.add("int");
        PRIMITIVE_EQUALS.add("long");
        // not float or double, as Double.equals is not the same as double ==
    }

    /** The content to process. */
    private final List<String> content;
    /** The config. */
    private final BeanGenConfig config;
    /** The start position of auto-generation. */
    private final int autoStartIndex;
    /** The end position of auto-generation. */
    private final int autoEndIndex;
    /** The region to insert into. */
    private final List<String> insertRegion;
    /** The list of property generators. */
    private final List<PropertyGen> properties;
    /** The data model of the bean. */
    private final GeneratableBean data;

    /**
     * Constructor.
     * @param content  the content to process, not null
     * @param config  the config to use, not null
     */
    BeanGen(List<String> content, BeanGenConfig config) {
        this.content = content;
        this.config = config;
        int beanDefIndex = parseBeanDefinition();
        if (beanDefIndex >= 0) {
            this.data = new GeneratableBean();
            this.data.getCurrentImports().addAll(parseImports(beanDefIndex));
            this.data.setImportInsertLocation(parseImportLocation(beanDefIndex));
            this.data.setBeanStyle(parseBeanStyle(beanDefIndex));
            if (data.isBeanStyleValid() == false) {
                throw new RuntimeException("Invalid bean style: " + data.getBeanStyle());
            }
            this.data.setBeanBuilderScope(parseBeanBuilderScope(beanDefIndex));
            if (data.isBeanBuilderScopeValid() == false) {
                throw new RuntimeException("Invalid bean builder scope: " + data.getBeanStyle());
            }
            this.data.setImmutableConstructor(parseImmutableConstructor(beanDefIndex));
            this.data.setConstructable(parseConstructable(beanDefIndex));
            this.data.setTypeParts(parseBeanType(beanDefIndex));
            this.data.setSuperTypeParts(parseBeanSuperType(beanDefIndex));
            if (parseBeanHierarchy(beanDefIndex).equals("immutable")) {
                this.data.setImmutable(true);
                this.data.setConstructorStyle(CONSTRUCTOR_BY_BUILDER);
            } else if (this.data.getImmutableConstructor() == CONSTRUCTOR_NONE) {
                if (data.isImmutable()) {
                    if (data.isTypeFinal()) {
                        this.data.setConstructorStyle(CONSTRUCTOR_BY_ARGS);
                    } else {
                        this.data.setConstructorStyle(CONSTRUCTOR_BY_BUILDER);
                    }
                } else {
                    this.data.setConstructorStyle(CONSTRUCTOR_BY_BUILDER);
                }
            } else {
                this.data.setConstructorStyle(this.data.getImmutableConstructor());
            }
            if (data.isImmutable()) {
                this.data.setImmutableValidator(parseImmutableValidator(beanDefIndex));
            }
            this.properties = parseProperties(data);
            this.autoStartIndex = parseStartAutogen();
            this.autoEndIndex = parseEndAutogen();
            this.insertRegion = content.subList(autoStartIndex + 1, autoEndIndex);
            this.data.setManualClone(parseManualClone(beanDefIndex));
            this.data.setManualEqualsHashCode(parseManualEqualsHashCode(beanDefIndex));
            this.data.setManualToStringCode(parseManualToStringCode(beanDefIndex));
            if (data.isImmutable()) {
                for (PropertyGen prop : properties) {
                    if (prop.getData().isDerived() == false && prop.getData().isFinal() == false) {
                        throw new RuntimeException("ImmutableBean must have final properties: " + data.getTypeRaw() + "." + prop.getData().getFieldName());
                    }
                }
            } else if (data.getImmutableConstructor() > CONSTRUCTOR_NONE) {
                throw new RuntimeException("Mutable beans must not specify @ImmutableConstructor: " + data.getTypeRaw());
            }
        } else {
            this.autoStartIndex = -1;
            this.autoEndIndex = -1;
            this.insertRegion = null;
            this.data = null;
            this.properties = null;
        }
    }

    //-----------------------------------------------------------------------
    void process() {
        if (insertRegion != null) {
            data.ensureImport(BeanDefinition.class);
            if (properties.size() > 0) {
                data.ensureImport(PropertyDefinition.class);
            }
            removeOld();
            if (data.isRootClass() && data.isExtendsDirectBean()) {
                data.ensureImport(DirectBean.class);
            }
            insertRegion.add("\t///CLOVER:OFF");
            generateMeta();
            generateImmutableBuilderMethod();
            generateArgBasedConstructor();
            generateBuilderBasedConstructor();
            generateMetaBean();
            generatePropertyByName();
            generatePropertyNames();
            generateGettersSetters();
            generateSeparator();
            generateImmutableToBuilder();
            generateClone();
            generateEquals();
            generateHashCode();
            generateToString();
            generateMetaClass();
            generateBuilderClass();
            insertRegion.add("\t///CLOVER:ON");
            resolveImports();
            resolveIndents();
        }
    }

    private void resolveImports() {
        if (data.getNewImports().size() > 0) {
            int pos = data.getImportInsertLocation() + 1;
            for (String imp : data.getNewImports()) {
                content.add(pos++, "import " + imp + ";");
            }
        }
    }

    private void resolveIndents() {
        for (ListIterator<String> it = content.listIterator(); it.hasNext(); ) {
            it.set(it.next().replace("\t", config.getIndent()));
        }
    }

    //-----------------------------------------------------------------------
    private int parseBeanDefinition() {
        for (int index = 0; index < content.size(); index++) {
            String line = content.get(index).trim();
            if (line.startsWith("@BeanDefinition")) {
                return index;
            }
        }
        return -1;
    }

    private Set<String> parseImports(int defLine) {
        Set<String> imports = new HashSet<String>();
        for (int index = 0; index < defLine; index++) {
            if (content.get(index).startsWith("import ")) {
                String imp = content.get(index).substring(7).trim();
                imp = imp.substring(0, imp.indexOf(';'));
                if (imp.endsWith(".*") == false) {
                    imports.add(imp);
                }
            }
        }
        return imports;
    }

    private int parseImportLocation(int defLine) {
        int location = 0;
        for (int index = 0; index < defLine; index++) {
            if (content.get(index).startsWith("import ") || content.get(index).startsWith("package ")) {
                location = index;
            }
        }
        return location;
    }

    private String parseBeanStyle(int defLine) {
        String line = content.get(defLine).trim();
        Matcher matcher = STYLE_PATTERN.matcher(line);
        if (matcher.matches()) {
            return matcher.group(1);
        }
        return "smart";
    }

    private String parseBeanBuilderScope(int defLine) {
        String line = content.get(defLine).trim();
        Matcher matcher = BUILDER_SCOPE_PATTERN.matcher(line);
        if (matcher.matches()) {
            return matcher.group(1);
        }
        return "smart";
    }

    private String parseBeanHierarchy(int defLine) {
        String line = content.get(defLine).trim();
        Matcher matcher = HIERARCHY_PATTERN.matcher(line);
        if (matcher.matches()) {
            return matcher.group(1);
        }
        return "";
    }

    private boolean parseConstructable(int defLine) {
        for (int index = defLine; index < content.size(); index++) {
            if (content.get(index).contains(" abstract class ")) {
                return false;
            }
        }
        return true;
    }

    private String[] parseBeanType(int defLine) {
        Matcher matcher = BEAN_TYPE.matcher("");
        for (int index = defLine; index < content.size(); index++) {
            String line = content.get(index);
            matcher.reset(line);
            if (matcher.matches()) {
                int startName = matcher.start(1);
                String fnl = line.substring(0, startName).contains(" final ") ? " final " : null;
                return new String[] {fnl, matcher.group(1), matcher.group(2), matcher.group(3), matcher.group(4),
                        matcher.group(5), matcher.group(6), matcher.group(7), matcher.group(8)};
            }
            if (line.contains(AUTOGENERATED_START_TEXT)) {
                break;
            }
        }
        throw new RuntimeException("Unable to locate bean class name");
    }

    private String[] parseBeanSuperType(int defLine) {
        // need to start searching beyond the full type to avoid 'extends' having two meanings
        String fullType = data.getType();
        // search for implements
        Matcher matcherImplements = SUPER_IMPL_TYPE.matcher("");
        boolean matchedType = false;
        for (int index = defLine; index < content.size(); index++) {
            String line = content.get(index);
            if (matchedType == false) {
                if (line.contains(fullType) == false) {
                    continue;
                }
                matchedType = true;
                line = line.substring(line.indexOf(fullType) + fullType.length());
            }
            matcherImplements.reset(line);
            if (matcherImplements.matches()) {
                return new String[] {matcherImplements.group(1)};
            }
            if (line.contains(AUTOGENERATED_START_TEXT)) {
                break;
            }
        }
        // search for extends
        Matcher matcherExtends = SUPER_TYPE.matcher("");
        matchedType = false;
        for (int index = defLine; index < content.size(); index++) {
            String line = content.get(index);
            if (matchedType == false) {
                if (line.contains(fullType) == false) {
                    continue;
                }
                matchedType = true;
                line = line.substring(line.indexOf(fullType) + fullType.length());
            }
            matcherExtends.reset(line);
            if (matcherExtends.matches()) {
                return new String[] {matcherExtends.group(1), matcherExtends.group(2), matcherExtends.group(3), matcherExtends.group(4), matcherExtends.group(5)};
            }
            if (line.contains(AUTOGENERATED_START_TEXT)) {
                break;
            }
        }
        throw new RuntimeException("Unable to locate bean superclass");
    }

    private int parseImmutableConstructor(int defLine) {
        int found = CONSTRUCTOR_NONE;
        for (int index = defLine; index < content.size(); index++) {
            if (content.get(index).trim().equals("@ImmutableConstructor")) {
                if (found > 0) {
                    throw new RuntimeException("Only one @ImmutableConstructor may be specified");
                }
                found = CONSTRUCTOR_BY_ARGS;
                if (index + 1 < content.size()) {
                    String nextLine = content.get(index + 1);
                    if (nextLine.contains("Builder ") || nextLine.contains("Builder<")) {
                        found = CONSTRUCTOR_BY_BUILDER;
                    }
                }
            }
        }
        return found;
    }

    private String parseImmutableValidator(int defLine) {
        boolean found = false;
        for (int index = defLine; index < content.size(); index++) {
            if (content.get(index).trim().equals("@ImmutableValidator")) {
                if (found) {
                    throw new RuntimeException("Only one @ImmutableValidator may be specified");
                }
                found = true;
                if (index + 1 < content.size()) {
                    String nextLine = content.get(index + 1);
                    Matcher matcher = VALIDATOR_PATTERN.matcher(nextLine);
                    if (matcher.matches()) {
                        return matcher.group(1);
                    }
                    throw new RuntimeException("@ImmutableValidator method must be private void and no-args");
                }
            }
        }
        return null;
    }

    private List<PropertyGen> parseProperties(GeneratableBean data) {
        List<PropertyGen> props = new ArrayList<PropertyGen>();
        for (int index = 0; index < content.size(); index++) {
            String line = content.get(index).trim();
            if (line.startsWith("@PropertyDefinition")) {
                PropertyGen prop = new PropertyGen(this, content, index, false);
                props.add(prop);
                data.getProperties().add(prop.getData());
            } else if (line.startsWith("@DerivedProperty")) {
                PropertyGen prop = new PropertyGen(this, content, index, true);
                props.add(prop);
                data.getProperties().add(prop.getData());
            }
        }
        return props;
    }

    private int parseStartAutogen() {
        for (int index = 0; index < content.size(); index++) {
            String line = content.get(index).trim();
            if (line.contains(" AUTOGENERATED START ")) {
                content.set(index, AUTOGENERATED_START);
                return index;
            }
        }
        for (int index = content.size() - 1; index >= 0; index--) {
            String line = content.get(index).trim();
            if (line.equals("}")) {
                content.add(index, AUTOGENERATED_START);
                return index;
            }
            if (line.length() > 0) {
                break;
            }
        }
        throw new RuntimeException("Unable to locate start autogeneration point");
    }

    private int parseEndAutogen() {
        for (int index = autoStartIndex; index < content.size(); index++) {
            String line = content.get(index).trim();
            if (line.contains(" AUTOGENERATED END ")) {
                content.set(index, AUTOGENERATED_END);
                return index;
            }
        }
        content.add(autoStartIndex + 1, AUTOGENERATED_END);
        return autoStartIndex + 1;
    }

    private void removeOld() {
        insertRegion.clear();
    }

    private boolean parseManualClone(int defLine) {
        for (int index = defLine; index < autoStartIndex; index++) {
            String line = content.get(index).trim();
            if (line.startsWith("public ") && line.endsWith(" clone() {")) {
                return true;
            }
        }
        for (int index = autoEndIndex; index < content.size(); index++) {
            String line = content.get(index).trim();
            if (line.startsWith("public ") && line.endsWith(" clone() {")) {
                return true;
            }
        }
        return false;
    }

    private boolean parseManualEqualsHashCode(int defLine) {
        for (int index = defLine; index < autoStartIndex; index++) {
            String line = content.get(index).trim();
            if (line.equals("public int hashCode() {") || (line.startsWith("public boolean equals(") && line.endsWith(") {"))) {
                return true;
            }
        }
        for (int index = autoEndIndex; index < content.size(); index++) {
            String line = content.get(index).trim();
            if (line.equals("public int hashCode() {") || (line.startsWith("public boolean equals(") && line.endsWith(") {"))) {
                return true;
            }
        }
        return false;
    }

    private boolean parseManualToStringCode(int defLine) {
        for (int index = defLine; index < autoStartIndex; index++) {
            String line = content.get(index).trim();
            if (line.equals("public String toString() {")) {
                return true;
            }
        }
        for (int index = autoEndIndex; index < content.size(); index++) {
            String line = content.get(index).trim();
            if (line.equals("public String toString() {")) {
                return true;
            }
        }
        return false;
    }

    //-----------------------------------------------------------------------
    private void generateSeparator() {
        if (insertRegion.size() > 0 && insertRegion.get(insertRegion.size() - 1).equals(LINE_SEPARATOR)) {
            return;
        }
        insertRegion.add(LINE_SEPARATOR);
    }

    private void generateIndentedSeparator() {
        if (insertRegion.size() > 0 && insertRegion.get(insertRegion.size() - 1).equals(LINE_SEPARATOR_INDENTED)) {
            return;
        }
        insertRegion.add(LINE_SEPARATOR_INDENTED);
    }

    private void generateImmutableBuilderMethod() {
        if (data.isConstructable() &&
                ((data.isImmutable() && data.isEffectiveBuilderScopePublic()) || (data.isMutable() && data.isBuilderScopePublic()))) {
            insertRegion.add("\t/**");
            insertRegion.add("\t * Returns a builder used to create an instance of the bean.");
            if (data.isTypeGeneric()) {
                for (int j = 0; j < data.getTypeGenericCount(); j++) {
                    insertRegion.add("\t * @param " + data.getTypeGenericName(j, true) + "  the type");
                }
            }
            insertRegion.add("\t * @return the builder, not null");
            insertRegion.add("\t */");
            if (data.isTypeGeneric()) {
                insertRegion.add("\tpublic static " + data.getTypeGeneric(true) + " " + data.getTypeRaw() + ".Builder" + data.getTypeGenericName(true) + " builder() {");
            } else {
                insertRegion.add("\tpublic static " + data.getTypeRaw() + ".Builder builder() {");
            }
            insertRegion.add("\t\treturn new " + data.getTypeRaw() + ".Builder" + data.getTypeGenericName(true) + "();");
            insertRegion.add("\t}");
            insertRegion.add("");
        }
    }

    private void generateBuilderBasedConstructor() {
        if (data.getConstructorStyle() == CONSTRUCTOR_BY_BUILDER && data.getImmutableConstructor() == CONSTRUCTOR_NONE && 
                ((data.isMutable() && data.isBuilderScopePublic()) || data.isImmutable())) {
            List<PropertyGen> nonDerived = nonDerivedProperties();
            String scope = (data.isTypeFinal() ? "private" : "protected");
            // signature
            insertRegion.add("\t/**");
            insertRegion.add("\t * Restricted constructor.");
            insertRegion.add("\t * @param builder  the builder to copy from, not null");
            insertRegion.add("\t */");
            insertRegion.add("\t" + scope + " " + data.getTypeRaw() + "(" + data.getTypeRaw() + ".Builder" + data.getTypeGenericName(true) + " builder) {");
            // super
            if (data.isSubClass()) {
                insertRegion.add("\t\tsuper(builder);");
            }
            // validate
            for (PropertyGen prop : properties) {
                if (prop.getData().isValidated()) {
                    insertRegion.add("\t\t" + prop.getData().getValidationMethodName() +
                            "(builder." + prop.generateBuilderFieldName() +
                            ", \"" + prop.getData().getPropertyName() + "\");");
                }
            }
            // assign
            if (data.isImmutable()) {
                // assign
                for (int i = 0; i < nonDerived.size(); i++) {
                    insertRegion.addAll(nonDerived.get(i).generateConstructorAssign("builder."));
                }
            } else {
                for (int i = 0; i < nonDerived.size(); i++) {
                    PropertyGen propGen = nonDerived.get(i);
                    GeneratableProperty prop = propGen.getData();
                    if (prop.isCollectionType()) {
                        if (prop.isNotNull()) {
                            insertRegion.add("\t\tthis." + prop.getPropertyName() + ".addAll(builder." + propGen.generateBuilderFieldName() + ");");
                        } else {
                            insertRegion.add("\t\tif (" + prop.getPropertyName() + " != null) {");
                            insertRegion.add("\t\t\tthis." + prop.getPropertyName() + ".addAll(builder." + propGen.generateBuilderFieldName() + ");");
                            insertRegion.add("\t\t}");
                        }
                    } else if (prop.isMapType()) {
                        if (prop.isNotNull()) {
                            insertRegion.add("\t\tthis." + prop.getPropertyName() + ".putAll(builder." + propGen.generateBuilderFieldName() + ");");
                        } else {
                            insertRegion.add("\t\tif (" + prop.getPropertyName() + " != null) {");
                            insertRegion.add("\t\t\tthis." + prop.getPropertyName() + ".putAll(builder." + propGen.generateBuilderFieldName() + ");");
                            insertRegion.add("\t\t}");
                        }
                    } else {
                        insertRegion.add("\t\tthis." + prop.getPropertyName() + " = builder." + propGen.generateBuilderFieldName() + ";");
                    }
                }
            }
            if (data.getImmutableValidator() != null) {
                insertRegion.add("\t\t" + data.getImmutableValidator() + "();");
            }
            insertRegion.add("\t}");
            insertRegion.add("");
        }
    }

    private void generateArgBasedConstructor() {
        if (data.getConstructorStyle() == CONSTRUCTOR_BY_ARGS && data.getImmutableConstructor() == CONSTRUCTOR_NONE && 
                ((data.isMutable() && data.isBuilderScopePublic()) || data.isImmutable())) {
            List<PropertyGen> nonDerived = nonDerivedProperties();
            if (nonDerived.size() == 0) {
                insertRegion.add("\tprivate " + data.getTypeRaw() + "() {");
            } else {
                // signature
                insertRegion.add("\tprivate " + data.getTypeRaw() + "(");
                for (int i = 0; i < nonDerived.size(); i++) {
                    PropertyGen prop = nonDerived.get(i);
                    insertRegion.add("\t\t\t" + prop.getBuilderType() + " " + prop.getData().getPropertyName() + (i < nonDerived.size() - 1 ? "," : ") {"));
                }
                // validate
                for (PropertyGen prop : properties) {
                    if (prop.getData().isValidated()) {
                        insertRegion.add("\t\t" + prop.getData().getValidationMethodName() +
                                "(" + prop.getData().getPropertyName() +
                                ", \"" + prop.getData().getPropertyName() + "\");");
                    }
                }
                // assign
                for (int i = 0; i < nonDerived.size(); i++) {
                    insertRegion.addAll(nonDerived.get(i).generateConstructorAssign(""));
                }
            }
            if (data.getImmutableValidator() != null) {
                insertRegion.add("\t\t" + data.getImmutableValidator() + "();");
            }
            insertRegion.add("\t}");
            insertRegion.add("");
        }
    }

    //-----------------------------------------------------------------------
    private void generateMeta() {
        data.ensureImport(JodaBeanUtils.class);
        // this cannot be generified without either Eclipse or javac complaining
        // raw types forever
        insertRegion.add("\t/**");
        insertRegion.add("\t * The meta-bean for {@code " + data.getTypeRaw() + "}.");
        if (data.isTypeGeneric()) {
            insertRegion.add("\t * @return the meta-bean, not null");
            insertRegion.add("\t */");
            insertRegion.add("\t@SuppressWarnings(\"rawtypes\")");
            insertRegion.add("\tpublic static " + data.getTypeRaw() + ".Meta meta() {");
        } else {
            insertRegion.add("\t * @return the meta-bean, not null");
            insertRegion.add("\t */");
            insertRegion.add("\tpublic static " + data.getTypeRaw() + ".Meta meta() {");
        }
        insertRegion.add("\t\treturn " + data.getTypeRaw() + ".Meta.INSTANCE;");
        insertRegion.add("\t}");
        
        if (data.isTypeGeneric()) {
            // this works around an Eclipse bug https://bugs.eclipse.org/bugs/show_bug.cgi?id=397462
            // long name needed for uniqueness as static overriding is borked
            insertRegion.add("");
            insertRegion.add("\t/**");
            insertRegion.add("\t * The meta-bean for {@code " + data.getTypeRaw() + "}.");
            if (data.getTypeGenericCount() == 1) {
                insertRegion.add("\t * @param <R>  the bean's generic type");
                insertRegion.add("\t * @param cls  the bean's generic type");
            } else if (data.getTypeGenericCount() == 2) {
                insertRegion.add("\t * @param <R>  the first generic type");
                insertRegion.add("\t * @param <S>  the second generic type");
                insertRegion.add("\t * @param cls1  the first generic type");
                insertRegion.add("\t * @param cls2  the second generic type");
            } else if (data.getTypeGenericCount() == 3) {
                insertRegion.add("\t * @param <R>  the first generic type");
                insertRegion.add("\t * @param <S>  the second generic type");
                insertRegion.add("\t * @param <T>  the second generic type");
                insertRegion.add("\t * @param cls1  the first generic type");
                insertRegion.add("\t * @param cls2  the second generic type");
                insertRegion.add("\t * @param cls3  the third generic type");
            }
            insertRegion.add("\t * @return the meta-bean, not null");
            insertRegion.add("\t */");
            insertRegion.add("\t@SuppressWarnings(\"unchecked\")");
            if (data.getTypeGenericCount() == 1) {
                insertRegion.add("\tpublic static <R" + data.getTypeGenericExtends(0, "R") + "> " + data.getTypeRaw() +
                        ".Meta<R> meta" + data.getTypeRaw() + "(Class<R> cls) {");
            } else if (data.getTypeGenericCount() == 2) {
                insertRegion.add("\tpublic static <R" + data.getTypeGenericExtends(0, "R") +
                        ", S" + data.getTypeGenericExtends(1, "S") + "> " + data.getTypeRaw() +
                        ".Meta<R, S> meta" + data.getTypeRaw() + "(Class<R> cls1, Class<S> cls2) {");
            } else if (data.getTypeGenericCount() == 3) {
                insertRegion.add("\tpublic static <R" + data.getTypeGenericExtends(0, "R") +
                        ", S" + data.getTypeGenericExtends(1, "S") +
                        ", T" + data.getTypeGenericExtends(2, "T") +
                        "> " + data.getTypeRaw() +
                        ".Meta<R, S, T> meta" + data.getTypeRaw() + "(Class<R> cls1, Class<S> cls2, Class<T> cls3) {");
            }
            insertRegion.add("\t\treturn " + data.getTypeRaw() + ".Meta.INSTANCE;");
            insertRegion.add("\t}");
        }
        
        insertRegion.add("");
        insertRegion.add("\tstatic {");
        insertRegion.add("\t\tJodaBeanUtils.registerMetaBean(" + data.getTypeRaw() + ".Meta.INSTANCE);");
        insertRegion.add("\t}");
        insertRegion.add("");
    }

    private void generateMetaBean() {
        if (data.isTypeGeneric()) {
            insertRegion.add("\t@SuppressWarnings(\"unchecked\")");
        }
        insertRegion.add("\t@Override");
        insertRegion.add("\tpublic " + data.getTypeRaw() + ".Meta" + data.getTypeGenericName(true) + " metaBean() {");
        insertRegion.add("\t\treturn " + data.getTypeRaw() + ".Meta.INSTANCE;");
        insertRegion.add("\t}");
        insertRegion.add("");
    }

    private void generateGettersSetters() {
        for (PropertyGen prop : properties) {
            generateSeparator();
            insertRegion.addAll(prop.generateGetter());
            if (data.isMutable()) {
                insertRegion.addAll(prop.generateSetter());
            }
            if (data.isBeanStyleGenerateProperties()) {
                insertRegion.addAll(prop.generateProperty());
            }
        }
    }

    //-----------------------------------------------------------------------
    private void generatePropertyByName() {
        if (data.isRootClass() && data.isExtendsDirectBean() == false) {
            data.ensureImport(Property.class);
            insertRegion.add("\t@Override");
            insertRegion.add("\tpublic <R> Property<R> property(String propertyName) {");
            insertRegion.add("\t\treturn metaBean().<R>metaProperty(propertyName).createProperty(this);");
            insertRegion.add("\t}");
            insertRegion.add("");
        }
    }

    private void generatePropertyNames() {
        if (data.isRootClass() && data.isExtendsDirectBean() == false) {
            data.ensureImport(Set.class);
            insertRegion.add("\t@Override");
            insertRegion.add("\tpublic Set<String> propertyNames() {");
            insertRegion.add("\t\treturn metaBean().metaPropertyMap().keySet();");
            insertRegion.add("\t}");
            insertRegion.add("");
        }
    }

    //-----------------------------------------------------------------------
    private void generateImmutableToBuilder() {
        if (data.isImmutable() && data.isEffectiveBuilderScopePublic()) {
            if (data.isConstructable()) {
                List<PropertyGen> nonDerived = nonDerivedProperties();
                if (nonDerived.size() > 0) {
                    insertRegion.add("\t/**");
                    insertRegion.add("\t * Returns a builder that allows this bean to be mutated.");
                    insertRegion.add("\t * @return the mutable builder, not null");
                    insertRegion.add("\t */");
                    insertRegion.add("\tpublic Builder" + data.getTypeGenericName(true) + " toBuilder() {");
                    insertRegion.add("\t\treturn new Builder" + data.getTypeGenericName(true) + "(this);");
                    insertRegion.add("\t}");
                    insertRegion.add("");
                }
            } else {
                insertRegion.add("\t/**");
                insertRegion.add("\t * Returns a builder that allows this bean to be mutated.");
                insertRegion.add("\t * @return the mutable builder, not null");
                insertRegion.add("\t */");
                insertRegion.add("\tpublic abstract Builder" + data.getTypeGenericName(true) + " toBuilder();");
                insertRegion.add("");
            }
        }
    }

    private void generateClone() {
        if (data.isManualClone() || data.isImmutable() || (data.isRootClass() == false && data.isConstructable() == false)) {
            return;
        }
        insertRegion.add("\t@Override");
        if (data.isImmutable()) {
            insertRegion.add("\tpublic " + data.getTypeNoExtends() + " clone() {");
            insertRegion.add("\t\treturn this;");
        } else {
            data.ensureImport(JodaBeanUtils.class);
            insertRegion.add("\tpublic " + data.getTypeNoExtends() + " clone() {");
            insertRegion.add("\t\treturn JodaBeanUtils.cloneAlways(this);");
        }
        insertRegion.add("\t}");
        insertRegion.add("");
    }

    private void generateEquals() {
        if (data.isManualEqualsHashCode()) {
            return;
        }
        data.ensureImport(JodaBeanUtils.class);
        insertRegion.add("\t@Override");
        insertRegion.add("\tpublic boolean equals(Object obj) {");
        insertRegion.add("\t\tif (obj == this) {");
        insertRegion.add("\t\t\treturn true;");
        insertRegion.add("\t\t}");
        insertRegion.add("\t\tif (obj != null && obj.getClass() == this.getClass()) {");
        if (properties.size() == 0) {
            if (data.isSubClass()) {
                insertRegion.add("\t\t\treturn super.equals(obj);");
            } else {
                insertRegion.add("\t\t\treturn true;");
            }
        } else {
            insertRegion.add("\t\t\t" + data.getTypeWildcard() + " other = (" + data.getTypeWildcard() + ") obj;");
            for (int i = 0; i < properties.size(); i++) {
                PropertyGen prop = properties.get(i);
                String getter = prop.getData().getGetterGen().generateGetInvoke(prop.getData());
                String equals = "JodaBeanUtils.equal(" + getter + ", other." + getter + ")";
                if (PRIMITIVE_EQUALS.contains(prop.getData().getType())) {
                    equals = "(" + getter + " == other." + getter + ")";
                }
                insertRegion.add(
                        (i == 0 ? "\t\t\treturn " : "\t\t\t\t\t") + equals +
                        (data.isSubClass() || i < properties.size() - 1 ? " &&" : ";"));
            }
            if (data.isSubClass()) {
                insertRegion.add("\t\t\t\t\tsuper.equals(obj);");
            }
        }
        insertRegion.add("\t\t}");
        insertRegion.add("\t\treturn false;");
        insertRegion.add("\t}");
        insertRegion.add("");
    }

    private void generateHashCode() {
        if (data.isManualEqualsHashCode()) {
            return;
        }
        data.ensureImport(JodaBeanUtils.class);
        insertRegion.add("\t@Override");
        insertRegion.add("\tpublic int hashCode() {");
        if (data.isSubClass()) {
            insertRegion.add("\t\tint hash = 7;");
        } else {
            insertRegion.add("\t\tint hash = getClass().hashCode();");
        }
        for (int i = 0; i < properties.size(); i++) {
            PropertyGen prop = properties.get(i);
            String getter = prop.getData().getGetterGen().generateGetInvoke(prop.getData());
            insertRegion.add("\t\thash += hash * 31 + JodaBeanUtils.hashCode(" + getter + ");");
        }
        if (data.isSubClass()) {
            insertRegion.add("\t\treturn hash ^ super.hashCode();");
        } else {
            insertRegion.add("\t\treturn hash;");
        }
        insertRegion.add("\t}");
        insertRegion.add("");
    }

    private void generateToString() {
        if (data.isManualToStringCode()) {
            return;
        }
        if (data.isRootClass() && data.isTypeFinal()) {
            insertRegion.add("\t@Override");
            insertRegion.add("\tpublic String toString() {");
            insertRegion.add("\t\tStringBuilder buf = new StringBuilder(" + (properties.size() * 32 + 32) + ");");
            insertRegion.add("\t\tbuf.append(\"" + data.getTypeRaw() + "{\");");
            for (int i = 0; i < properties.size(); i++) {
                PropertyGen prop = properties.get(i);
                String getter = prop.getData().getGetterGen().generateGetInvoke(prop.getData());
                if (i < properties.size() - 1) {
                    insertRegion.add("\t\tbuf.append(\"" + prop.getData().getPropertyName() +
                            "\").append('=').append(" + getter + ").append(',').append(' ');");
                } else {
                    insertRegion.add("\t\tbuf.append(\"" + prop.getData().getPropertyName() +
                            "\").append('=').append(JodaBeanUtils.toString(" + getter + "));");
                }
            }
            insertRegion.add("\t\tbuf.append('}');");
            insertRegion.add("\t\treturn buf.toString();");
            insertRegion.add("\t}");
            insertRegion.add("");
            return;
        }
        
        insertRegion.add("\t@Override");
        insertRegion.add("\tpublic String toString() {");
        insertRegion.add("\t\tStringBuilder buf = new StringBuilder(" + (properties.size() * 32 + 32) + ");");
        insertRegion.add("\t\tbuf.append(\"" + data.getTypeRaw() + "{\");");
        insertRegion.add("\t\tint len = buf.length();");
        insertRegion.add("\t\ttoString(buf);");
        insertRegion.add("\t\tif (buf.length() > len) {");
        insertRegion.add("\t\t\tbuf.setLength(buf.length() - 2);");
        insertRegion.add("\t\t}");
        insertRegion.add("\t\tbuf.append('}');");
        insertRegion.add("\t\treturn buf.toString();");
        insertRegion.add("\t}");
        insertRegion.add("");
        
        if (data.isSubClass()) {
            insertRegion.add("\t@Override");
        }
        insertRegion.add("\tprotected void toString(StringBuilder buf) {");
        if (data.isSubClass()) {
            insertRegion.add("\t\tsuper.toString(buf);");
        }
        for (int i = 0; i < properties.size(); i++) {
            PropertyGen prop = properties.get(i);
            String getter = prop.getData().getGetterGen().generateGetInvoke(prop.getData());
            insertRegion.add("\t\tbuf.append(\"" + prop.getData().getPropertyName() +
                    "\").append('=').append(JodaBeanUtils.toString(" + getter + ")).append(',').append(' ');");
        }
        insertRegion.add("\t}");
        insertRegion.add("");
    }

    //-----------------------------------------------------------------------
    private void generateMetaClass() {
        generateSeparator();
        insertRegion.add("\t/**");
        insertRegion.add("\t * The meta-bean for {@code " + data.getTypeRaw() + "}.");
        if (data.isTypeGeneric()) {
            for (int j = 0; j < data.getTypeGenericCount(); j++) {
                insertRegion.add("\t * @param " + data.getTypeGenericName(j, true) + "  the type");
            }
        }
        insertRegion.add("\t */");
        String superMeta;
        if (data.isSubClass()) {
            superMeta = data.getSuperTypeRaw() + ".Meta" + data.getSuperTypeGeneric(true);
        } else {
            data.ensureImport(DirectMetaBean.class);
            superMeta = "DirectMetaBean";
        }
        String finalType = data.isTypeFinal() ? "final " : "";
        if (data.isTypeGeneric()) {
            insertRegion.add("\tpublic static " + finalType + "class Meta" + data.getTypeGeneric(true) + " extends " + superMeta + " {");
        } else {
            insertRegion.add("\tpublic static " + finalType + "class Meta extends " + superMeta + " {");
        }
        insertRegion.add("\t\t/**");
        insertRegion.add("\t\t * The singleton instance of the meta-bean.");
        insertRegion.add("\t\t */");
        if (data.isTypeGeneric()) {
            insertRegion.add("\t\t@SuppressWarnings(\"rawtypes\")");
        }
        insertRegion.add("\t\tstatic final Meta INSTANCE = new Meta();");
        insertRegion.add("");
        generateMetaPropertyConstants();
        generateMetaPropertyMapSetup();
        insertRegion.add("\t\t/**");
        insertRegion.add("\t\t * Restricted constructor.");
        insertRegion.add("\t\t */");
        insertRegion.add("\t\t" + data.getNestedClassConstructorScope() + " Meta() {");
        insertRegion.add("\t\t}");
        insertRegion.add("");
        generateMetaPropertyGet();
        generateMetaBuilder();
        generateMetaBeanType();
        generateMetaPropertyMap();
        generateIndentedSeparator();
        generateMetaPropertyMethods();
        generateIndentedSeparator();
        generateMetaGetPropertyValue();
        generateMetaSetPropertyValue();
        generateMetaValidate();
        insertRegion.add("\t}");
        insertRegion.add("");
    }

    private void generateMetaPropertyConstants() {
        for (PropertyGen prop : properties) {
            insertRegion.addAll(prop.generateMetaPropertyConstant());
        }
    }

    private void generateMetaPropertyMapSetup() {
        data.ensureImport(MetaProperty.class);
        data.ensureImport(DirectMetaPropertyMap.class);
        insertRegion.add("\t\t/**");
        insertRegion.add("\t\t * The meta-properties.");
        insertRegion.add("\t\t */");
        insertRegion.add("\t\tprivate final Map<String, MetaProperty<?>> " + config.getPrefix() + "metaPropertyMap$ = new DirectMetaPropertyMap(");
        if (data.isSubClass()) {
            insertRegion.add("\t\t\t\tthis, (DirectMetaPropertyMap) super.metaPropertyMap()" + (properties.size() == 0 ? ");" : ","));
        } else {
            insertRegion.add("\t\t\t\tthis, null" + (properties.size() == 0 ? ");" : ","));
        }
        for (int i = 0; i < properties.size(); i++) {
            String line = "\t\t\t\t\"" + properties.get(i).getData().getPropertyName() + "\"";
            line += (i + 1 == properties.size() ? ");" : ",");
            insertRegion.add(line);
        }
        insertRegion.add("");
    }

    private void generateMetaBuilder() {
        insertRegion.add("\t\t@Override");
        if (data.isImmutable() && data.isEffectiveBuilderScopePublic() == false) {
            insertRegion.add("\t\tpublic BeanBuilder<? extends " + data.getTypeNoExtends() + "> builder() {");
            if (data.isConstructable()) {
                insertRegion.add("\t\t\treturn new " + data.getTypeRaw() + ".Builder" + data.getTypeGenericName(true) + "();");
            } else {
                insertRegion.add("\t\t\tthrow new UnsupportedOperationException(\"" + data.getTypeRaw() + " is an abstract class\");");
            }
        } else if (data.isImmutable() || (data.isMutable() && data.isBuilderScopePublic())) {
            insertRegion.add("\t\tpublic " + data.getTypeRaw() + ".Builder" + data.getTypeGenericName(true) + " builder() {");
            if (data.isConstructable()) {
                insertRegion.add("\t\t\treturn new " + data.getTypeRaw() + ".Builder" + data.getTypeGenericName(true) + "();");
            } else {
                insertRegion.add("\t\t\tthrow new UnsupportedOperationException(\"" + data.getTypeRaw() + " is an abstract class\");");
            }
        } else {
            data.ensureImport(BeanBuilder.class);
            insertRegion.add("\t\tpublic BeanBuilder<? extends " + data.getTypeNoExtends() + "> builder() {");
            if (data.isConstructable()) {
                data.ensureImport(DirectBeanBuilder.class);
                insertRegion.add("\t\t\treturn new DirectBeanBuilder<" + data.getTypeNoExtends() + ">(new " + data.getTypeNoExtends() + "());");
            } else {
                insertRegion.add("\t\t\tthrow new UnsupportedOperationException(\"" + data.getTypeRaw() + " is an abstract class\");");
            }
        }
        insertRegion.add("\t\t}");
        insertRegion.add("");
    }

    private void generateMetaBeanType() {
        if (data.isTypeGeneric()) {
            insertRegion.add("\t\t@SuppressWarnings({\"unchecked\", \"rawtypes\" })");
        }
        insertRegion.add("\t\t@Override");
        insertRegion.add("\t\tpublic Class<? extends " + data.getTypeNoExtends() + "> beanType() {");
        if (data.isTypeGeneric()) {
            insertRegion.add("\t\t\treturn (Class) " + data.getTypeRaw() + ".class;");
        } else {
            insertRegion.add("\t\t\treturn " + data.getTypeNoExtends() + ".class;");
        }
        insertRegion.add("\t\t}");
        insertRegion.add("");
    }

    private void generateMetaPropertyGet() {
        if (properties.size() > 0) {
            data.ensureImport(MetaProperty.class);
            insertRegion.add("\t\t@Override");
            insertRegion.add("\t\tprotected MetaProperty<?> metaPropertyGet(String propertyName) {");
            insertRegion.add("\t\t\tswitch (propertyName.hashCode()) {");
            for (PropertyGen prop : properties) {
                insertRegion.addAll(prop.generateMetaPropertyGetCase());
            }
            insertRegion.add("\t\t\t}");
            insertRegion.add("\t\t\treturn super.metaPropertyGet(propertyName);");
            insertRegion.add("\t\t}");
            insertRegion.add("");
        }
    }

    private void generateMetaPropertyMap() {
        data.ensureImport(Map.class);
        insertRegion.add("\t\t@Override");
        insertRegion.add("\t\tpublic Map<String, MetaProperty<?>> metaPropertyMap() {");
        insertRegion.add("\t\t\treturn " + config.getPrefix() + "metaPropertyMap$;");
        insertRegion.add("\t\t}");
        insertRegion.add("");
    }

    private void generateMetaPropertyMethods() {
        if (data.isBeanStyleGenerateMetaProperties()) {
            for (PropertyGen prop : properties) {
                insertRegion.addAll(prop.generateMetaProperty());
            }
        }
    }

    //-----------------------------------------------------------------------
    private void generateMetaGetPropertyValue() {
        if (properties.size() == 0) {
            return;
        }
        data.ensureImport(Bean.class);
        insertRegion.add("\t\t@Override");
        insertRegion.add("\t\tprotected Object propertyGet(Bean bean, String propertyName, boolean quiet) {");
        insertRegion.add("\t\t\tswitch (propertyName.hashCode()) {");
        for (PropertyGen prop : properties) {
            insertRegion.addAll(prop.generatePropertyGetCase());
        }
        insertRegion.add("\t\t\t}");
        insertRegion.add("\t\t\treturn super.propertyGet(bean, propertyName, quiet);");
        insertRegion.add("\t\t}");
        insertRegion.add("");
    }

    private void generateMetaSetPropertyValue() {
        if (properties.size() == 0) {
            return;
        }
        data.ensureImport(Bean.class);
        if (data.isImmutable()) {
            insertRegion.add("\t\t@Override");
            insertRegion.add("\t\tprotected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {");
            insertRegion.add("\t\t\tmetaProperty(propertyName);");
            insertRegion.add("\t\t\tif (quiet) {");
            insertRegion.add("\t\t\t\treturn;");
            insertRegion.add("\t\t\t}");
            insertRegion.add("\t\t\tthrow new UnsupportedOperationException(\"Property cannot be written: \" + propertyName);");
            insertRegion.add("\t\t}");
            insertRegion.add("");
            return;
        }
        
        boolean generics = false;
        for (GeneratableProperty prop : data.getProperties()) {
            generics |= (prop.getStyle().isWritable() &&
                    ((prop.isGeneric() && prop.isGenericWildcardParamType() == false) || data.isTypeGeneric()));
        }
        if (generics) {
            insertRegion.add("\t\t@SuppressWarnings(\"unchecked\")");
        }
        insertRegion.add("\t\t@Override");
        insertRegion.add("\t\tprotected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {");
        insertRegion.add("\t\t\tswitch (propertyName.hashCode()) {");
        for (PropertyGen prop : properties) {
            insertRegion.addAll(prop.generatePropertySetCase());
        }
        insertRegion.add("\t\t\t}");
        insertRegion.add("\t\t\tsuper.propertySet(bean, propertyName, newValue, quiet);");
        insertRegion.add("\t\t}");
        insertRegion.add("");
    }

    private void generateMetaValidate() {
        if (data.isValidated() == false || data.isImmutable()) {
            return;
        }
        data.ensureImport(Bean.class);
        insertRegion.add("\t\t@Override");
        insertRegion.add("\t\tprotected void validate(Bean bean) {");
        if (data.isValidated()) {
            for (PropertyGen prop : properties) {
                if (prop.getData().isValidated()) {
                    insertRegion.add("\t\t\t" + prop.getData().getValidationMethodName() +
                            "(((" + data.getTypeWildcard() + ") bean)." + prop.getData().getFieldName() +
                            ", \"" + prop.getData().getPropertyName() + "\");");
                }
            }
        }
        if (data.isSubClass()) {
            insertRegion.add("\t\t\tsuper.validate(bean);");
        }
        insertRegion.add("\t\t}");
        insertRegion.add("");
    }

    //-----------------------------------------------------------------------
    private void generateBuilderClass() {
        if (data.isMutable() && data.isBuilderScopePublic() == false) {
            return;
        }
        List<PropertyGen> nonDerived = nonDerivedProperties();
        generateSeparator();
        String finalType = data.isTypeFinal() ? "final " : "";
        insertRegion.add("\t/**");
        insertRegion.add("\t * The bean-builder for {@code " + data.getTypeRaw() + "}.");
        if (data.isTypeGeneric()) {
            for (int j = 0; j < data.getTypeGenericCount(); j++) {
                insertRegion.add("\t * @param " + data.getTypeGenericName(j, true) + "  the type");
            }
        }
        insertRegion.add("\t */");
        String superBuilder;
        if (data.isSubClass()) {
            superBuilder = data.getSuperTypeRaw() + ".Builder" + data.getSuperTypeGeneric(true);
        } else {
            data.ensureImport(DirectFieldsBeanBuilder.class);
            superBuilder = "DirectFieldsBeanBuilder<" + data.getTypeNoExtends() + ">";
        }
        if (data.isConstructable()) {
            insertRegion.add("\t" + data.getEffectiveBuilderScope() + " static " + finalType +
                    "class Builder" + data.getTypeGeneric(true) + " extends " + superBuilder + " {");
        } else {
            insertRegion.add("\t" + data.getEffectiveBuilderScope() + " abstract static " + finalType +
                    "class Builder" + data.getTypeGeneric(true) + " extends " + superBuilder + " {");
        }
        if (nonDerived.size() > 0) {
            insertRegion.add("");
            generateBuilderProperties();
        }
        insertRegion.add("");
        generateBuilderConstructorOneMetaArg();
        generateBuilderConstructorCopy();
        generateIndentedSeparator();
        generateBuilderGet();
        generateBuilderSet();
        generateBuilderOtherSets();
        if (data.isConstructable()) {
            generateBuilderBuilder();
        }
        generateIndentedSeparator();
        generateBuilderPropertySetMethods();
        generateIndentedSeparator();
        generateBuilderToString();
        insertRegion.add("\t}");
        insertRegion.add("");
    }

    private void generateBuilderConstructorOneMetaArg() {
        insertRegion.add("\t\t/**");
        insertRegion.add("\t\t * Restricted constructor.");
        insertRegion.add("\t\t */");
        insertRegion.add("\t\t" + data.getNestedClassConstructorScope() + " Builder() {");
        insertRegion.add("\t\t}");
        insertRegion.add("");
    }

    private void generateBuilderConstructorCopy() {
        if (data.isEffectiveBuilderScopePublic()) {
            List<PropertyGen> nonDerived = nonDerivedProperties();
            if (nonDerived.size() > 0) {
                insertRegion.add("\t\t/**");
                insertRegion.add("\t\t * Restricted copy constructor.");
                insertRegion.add("\t\t * @param beanToCopy  the bean to copy from, not null");
                insertRegion.add("\t\t */");
                insertRegion.add("\t\t" + data.getNestedClassConstructorScope() + " Builder(" + data.getTypeNoExtends() + " beanToCopy) {");
                for (int i = 0; i < nonDerived.size(); i++) {
                    insertRegion.addAll(nonDerived.get(i).generateBuilderConstructorAssign("beanToCopy"));
                }
                insertRegion.add("\t\t}");
                insertRegion.add("");
            }
        }
    }

    private void generateBuilderProperties() {
        for (PropertyGen prop : nonDerivedProperties()) {
            insertRegion.addAll(prop.generateBuilderField());
        }
    }

    private void generateBuilderGet() {
        List<PropertyGen> nonDerived = nonDerivedProperties();
        insertRegion.add("\t\t@Override");
        insertRegion.add("\t\tpublic Object get(String propertyName) {");
        if (nonDerived.size() > 0) {
            insertRegion.add("\t\t\tswitch (propertyName.hashCode()) {");
            for (PropertyGen prop : nonDerived) {
                insertRegion.addAll(prop.generateBuilderFieldGet());
            }
            insertRegion.add("\t\t\t\tdefault:");
            if (data.isRootClass()) {
                data.ensureImport(NoSuchElementException.class);
                insertRegion.add("\t\t\t\t\tthrow new NoSuchElementException(\"Unknown property: \" + propertyName);");
            } else {
                insertRegion.add("\t\t\t\t\treturn super.get(propertyName);");
            }
            insertRegion.add("\t\t\t}");
        } else {
            data.ensureImport(NoSuchElementException.class);
            insertRegion.add("\t\t\tthrow new NoSuchElementException(\"Unknown property: \" + propertyName);");
        }
        insertRegion.add("\t\t}");
        insertRegion.add("");
    }

    private void generateBuilderSet() {
        List<PropertyGen> nonDerived = nonDerivedProperties();
        boolean generics = false;
        for (GeneratableProperty prop : data.getProperties()) {
            generics |= (prop.isGeneric() && prop.isGenericWildcardParamType() == false);
        }
        if (generics) {
            insertRegion.add("\t\t@SuppressWarnings(\"unchecked\")");
        }
        insertRegion.add("\t\t@Override");
        insertRegion.add("\t\tpublic Builder" + data.getTypeGenericName(true) + " set(String propertyName, Object newValue) {");
        if (nonDerived.size() > 0) {
            insertRegion.add("\t\t\tswitch (propertyName.hashCode()) {");
            for (PropertyGen prop : nonDerived) {
                insertRegion.addAll(prop.generateBuilderFieldSet());
            }
            insertRegion.add("\t\t\t\tdefault:");
            if (data.isRootClass()) {
                data.ensureImport(NoSuchElementException.class);
                insertRegion.add("\t\t\t\t\tthrow new NoSuchElementException(\"Unknown property: \" + propertyName);");
            } else {
                insertRegion.add("\t\t\t\t\tsuper.set(propertyName, newValue);");
                insertRegion.add("\t\t\t\t\tbreak;");
            }
            insertRegion.add("\t\t\t}");
            insertRegion.add("\t\t\treturn this;");
        } else {
            data.ensureImport(NoSuchElementException.class);
            insertRegion.add("\t\t\tthrow new NoSuchElementException(\"Unknown property: \" + propertyName);");
        }
        insertRegion.add("\t\t}");
        insertRegion.add("");
    }

    private void generateBuilderOtherSets() {
        insertRegion.add("\t\t@Override");
        insertRegion.add("\t\tpublic Builder" + data.getTypeGenericName(true) + " set(MetaProperty<?> property, Object value) {");
        insertRegion.add("\t\t\tsuper.set(property, value);");
        insertRegion.add("\t\t\treturn this;");
        insertRegion.add("\t\t}");
        insertRegion.add("");
        insertRegion.add("\t\t@Override");
        insertRegion.add("\t\tpublic Builder" + data.getTypeGenericName(true) + " setString(String propertyName, String value) {");
        insertRegion.add("\t\t\tsetString(meta().metaProperty(propertyName), value);");
        insertRegion.add("\t\t\treturn this;");
        insertRegion.add("\t\t}");
        insertRegion.add("");
        insertRegion.add("\t\t@Override");
        insertRegion.add("\t\tpublic Builder" + data.getTypeGenericName(true) + " setString(MetaProperty<?> property, String value) {");
        insertRegion.add("\t\t\tsuper.setString(property, value);");
        insertRegion.add("\t\t\treturn this;");
        insertRegion.add("\t\t}");
        insertRegion.add("");
        insertRegion.add("\t\t@Override");
        insertRegion.add("\t\tpublic Builder" + data.getTypeGenericName(true) + " setAll(Map<String, ? extends Object> propertyValueMap) {");
        insertRegion.add("\t\t\tsuper.setAll(propertyValueMap);");
        insertRegion.add("\t\t\treturn this;");
        insertRegion.add("\t\t}");
        insertRegion.add("");
    }

    private void generateBuilderBuilder() {
        List<PropertyGen> nonDerived = nonDerivedProperties();
        insertRegion.add("\t\t@Override");
        insertRegion.add("\t\tpublic " + data.getTypeRaw() + data.getTypeGenericName(true) + " build() {");
        if (data.getConstructorStyle() == CONSTRUCTOR_BY_ARGS) {
            if (nonDerived.size() == 0) {
                insertRegion.add("\t\t\treturn new " + data.getTypeRaw() + data.getTypeGenericName(true) + "();");
            } else {
                insertRegion.add("\t\t\treturn new " + data.getTypeRaw() + data.getTypeGenericName(true) + "(");
                for (int i = 0; i < nonDerived.size(); i++) {
                    insertRegion.add("\t\t\t\t\t" + nonDerived.get(i).generateBuilderFieldName() + (i < nonDerived.size() - 1 ? "," : ");"));
                }
            }
        } else if (data.getConstructorStyle() == CONSTRUCTOR_BY_BUILDER) {
            insertRegion.add("\t\t\treturn new " + data.getTypeRaw() + data.getTypeGenericName(true) + "(this);");
        }
        insertRegion.add("\t\t}");
        insertRegion.add("");
    }

    private void generateBuilderPropertySetMethods() {
        if (data.isEffectiveBuilderScopePublic()) {
            for (PropertyGen prop : nonDerivedProperties()) {
                insertRegion.addAll(prop.generateBuilderSetMethod());
            }
        }
    }

    private void generateBuilderToString() {
        List<PropertyGen> nonDerived = nonDerivedProperties();
        if (data.isImmutable() && data.isTypeFinal()) {
            insertRegion.add("\t\t@Override");
            insertRegion.add("\t\tpublic String toString() {");
            if (nonDerived.size() == 0) {
                insertRegion.add("\t\t\treturn \"" + data.getTypeRaw() + ".Builder{}\";");
            } else {
                insertRegion.add("\t\t\tStringBuilder buf = new StringBuilder(" + (nonDerived.size() * 32 + 32) + ");");
                insertRegion.add("\t\t\tbuf.append(\"" + data.getTypeRaw() + ".Builder{\");");
                for (int i = 0; i < nonDerived.size(); i++) {
                    PropertyGen prop = nonDerived.get(i);
                    String getter = nonDerived.get(i).generateBuilderFieldName();
                    String base = "\t\t\tbuf.append(\"" + prop.getData().getPropertyName() +
                            "\").append('=').append(JodaBeanUtils.toString(" + getter + "))";
                    if (i < nonDerived.size() - 1) {
                        insertRegion.add(base + ".append(',').append(' ');");
                    } else {
                        insertRegion.add(base + ";");
                    }
                }
                insertRegion.add("\t\t\tbuf.append('}');");
                insertRegion.add("\t\t\treturn buf.toString();");
            }
            insertRegion.add("\t\t}");
            insertRegion.add("");
            return;
        }
        
        insertRegion.add("\t\t@Override");
        insertRegion.add("\t\tpublic String toString() {");
        insertRegion.add("\t\t\tStringBuilder buf = new StringBuilder(" + (nonDerived.size() * 32 + 32) + ");");
        insertRegion.add("\t\t\tbuf.append(\"" + data.getTypeRaw() + ".Builder{\");");
        insertRegion.add("\t\t\tint len = buf.length();");
        insertRegion.add("\t\t\ttoString(buf);");
        insertRegion.add("\t\t\tif (buf.length() > len) {");
        insertRegion.add("\t\t\t\tbuf.setLength(buf.length() - 2);");
        insertRegion.add("\t\t\t}");
        insertRegion.add("\t\t\tbuf.append('}');");
        insertRegion.add("\t\t\treturn buf.toString();");
        insertRegion.add("\t\t}");
        insertRegion.add("");
        
        if (data.isSubClass()) {
            insertRegion.add("\t\t@Override");
        }
        insertRegion.add("\t\tprotected void toString(StringBuilder buf) {");
        if (data.isSubClass()) {
            insertRegion.add("\t\t\tsuper.toString(buf);");
        }
        for (int i = 0; i < nonDerived.size(); i++) {
            PropertyGen prop = nonDerived.get(i);
            String getter = nonDerived.get(i).generateBuilderFieldName();
            insertRegion.add("\t\t\tbuf.append(\"" + prop.getData().getPropertyName() +
                    "\").append('=').append(JodaBeanUtils.toString(" + getter + ")).append(',').append(' ');");
        }
        insertRegion.add("\t\t}");
        insertRegion.add("");
    }

    //-----------------------------------------------------------------------
    boolean isBean() {
        return data != null;
    }

    GeneratableBean getData() {
        return data;
    }

    BeanGenConfig getConfig() {
        return config;
    }

    String getFieldPrefix() {
        return config.getPrefix();
    }

    private List<PropertyGen> nonDerivedProperties() {
        List<PropertyGen> nonDerived = new ArrayList<PropertyGen>();
        for (PropertyGen prop : properties) {
            if (prop.getData().isDerived() == false) {
                nonDerived.add(prop);
            }
        }
        return nonDerived;
    }

}
