/*
 * Decompiled with CFR 0.152.
 */
package org.johnnei.javatorrent.internal.network.socket;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.sql.Date;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.johnnei.javatorrent.internal.utils.PrecisionTimer;
import org.johnnei.javatorrent.internal.utils.Sync;
import org.johnnei.javatorrent.internal.utp.UtpTimeout;
import org.johnnei.javatorrent.internal.utp.UtpWindow;
import org.johnnei.javatorrent.internal.utp.protocol.ConnectionState;
import org.johnnei.javatorrent.internal.utp.protocol.UtpAckHandler;
import org.johnnei.javatorrent.internal.utp.protocol.UtpInputStream;
import org.johnnei.javatorrent.internal.utp.protocol.UtpMultiplexer;
import org.johnnei.javatorrent.internal.utp.protocol.UtpOutputStream;
import org.johnnei.javatorrent.internal.utp.protocol.UtpPacket;
import org.johnnei.javatorrent.internal.utp.protocol.payload.DataPayload;
import org.johnnei.javatorrent.internal.utp.protocol.payload.FinPayload;
import org.johnnei.javatorrent.internal.utp.protocol.payload.IPayload;
import org.johnnei.javatorrent.internal.utp.protocol.payload.ResetPayload;
import org.johnnei.javatorrent.internal.utp.protocol.payload.StatePayload;
import org.johnnei.javatorrent.internal.utp.protocol.payload.SynPayload;
import org.johnnei.javatorrent.network.OutStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class UtpSocketImpl {
    private static final Logger LOGGER = LoggerFactory.getLogger(UtpSocketImpl.class);
    private static final int PACKET_OVERHEAD = 20;
    private static final int UNKNOWN_TIMESTAMP_DIFFERENCE = 0;
    private final Lock notifyLock = new ReentrantLock();
    private final Condition onPacketAcknowledged = this.notifyLock.newCondition();
    private final UtpMultiplexer utpMultiplexer;
    private Clock clock = Clock.systemDefaultZone();
    private PrecisionTimer timer = new PrecisionTimer();
    private SocketAddress socketAddress;
    private ConnectionState connectionState;
    private UtpAckHandler ackHandler;
    private Instant lastInteraction;
    private UtpTimeout timeout;
    private short endOfStreamSequenceNumber;
    private short sequenceNumberCounter;
    private short connectionIdReceive;
    private short connectionIdSend;
    private int packetSize;
    private int clientWindowSize;
    private UtpWindow window;
    private UtpInputStream inputStream;
    private UtpOutputStream outputStream;
    private short sequenceNumber;
    private int measuredDelay;
    private AtomicInteger statePackets = new AtomicInteger(0);
    private AtomicInteger dataPackets = new AtomicInteger(0);

    public UtpSocketImpl(UtpMultiplexer utpMultiplexer) {
        this.initDefaults();
        this.utpMultiplexer = utpMultiplexer;
        this.connectionIdReceive = (short)new Random().nextInt();
        this.connectionIdSend = (short)(this.connectionIdReceive + 1);
        this.sequenceNumberCounter = 1;
    }

    public UtpSocketImpl(UtpMultiplexer utpMultiplexer, SocketAddress socketAddress, short connectionId) {
        this.initDefaults();
        this.socketAddress = socketAddress;
        this.utpMultiplexer = utpMultiplexer;
        this.connectionIdReceive = (short)(connectionId + 1);
        this.connectionIdSend = connectionId;
        this.sequenceNumberCounter = (short)new Random().nextInt();
    }

    private void initDefaults() {
        this.connectionState = ConnectionState.CONNECTING;
        this.packetSize = 150;
        this.clientWindowSize = 150;
        this.timeout = new UtpTimeout();
        this.window = new UtpWindow(this);
        this.lastInteraction = this.clock.instant();
        this.ackHandler = new UtpAckHandler(this);
    }

    public void connect(InetSocketAddress endpoint) throws IOException {
        this.socketAddress = endpoint;
        this.connectionState = ConnectionState.CONNECTING;
        this.send(new SynPayload());
        Instant endTime = this.clock.instant().plusSeconds(10L);
        this.notifyLock.lock();
        try {
            while (this.connectionState != ConnectionState.CONNECTED && this.clock.instant().isBefore(endTime)) {
                this.onPacketAcknowledged.awaitUntil(Date.from(endTime));
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("Interruption while waiting for connection confirmation.", e);
        }
        finally {
            this.notifyLock.unlock();
        }
        if (this.connectionState != ConnectionState.CONNECTED) {
            throw new IOException("Endpoint did not respond to connection attempt");
        }
    }

    public void send(IPayload payload) throws IOException {
        this.send(new UtpPacket(this, payload));
    }

    private void send(UtpPacket packet) throws IOException {
        while (this.connectionState != ConnectionState.CLOSED && this.getAvailableWindowSize() < packet.getPacketSize()) {
            LOGGER.trace("Waiting to send packet (seq={}) of {} bytes. Window Status: {} / {} bytes", new Object[]{Short.toUnsignedInt(packet.getSequenceNumber()), packet.getPacketSize(), this.getBytesInFlight(), this.getSendWindowSize()});
            this.notifyLock.lock();
            try {
                this.onPacketAcknowledged.await(1L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new IOException("Interruption on writing packet", e);
            }
            finally {
                this.notifyLock.unlock();
            }
            if (packet.getPacketSize() <= Math.max(150, this.window.getSize())) continue;
            LOGGER.trace("Repacking {} it exceeds the maximum window size of {}", (Object)packet, (Object)this.window.getSize());
            byte[] unsentBytes = packet.repackage(this);
            this.send(packet);
            this.send(new DataPayload(unsentBytes));
            return;
        }
        this.sendUnbounded(packet);
    }

    public void sendUnbounded(UtpPacket packet) throws IOException {
        if (this.connectionState == ConnectionState.CLOSED) {
            throw new IOException("Socket is closed or reset during write.");
        }
        this.ackHandler.registerPacket(packet);
        this.doSend(packet);
    }

    public void onReceivedData() {
        if (this.connectionState == ConnectionState.CONNECTING && this.ackHandler.isInitialised()) {
            LOGGER.debug("Data packet has been received after SYN ack, connection is ok.");
            this.setConnectionState(ConnectionState.CONNECTED);
        }
    }

    private void doSend(UtpPacket packet) throws IOException {
        OutStream outStream = new OutStream();
        packet.write(this, outStream);
        packet.updateSentTime();
        byte[] buffer = outStream.toByteArray();
        this.lastInteraction = this.clock.instant();
        this.utpMultiplexer.send(new DatagramPacket(buffer, buffer.length, this.socketAddress));
        LOGGER.trace("Sent {} to {} ({} / {} bytes)", new Object[]{packet, this.socketAddress, this.ackHandler, this.getSendWindowSize()});
        if (packet.getType() == 2) {
            this.statePackets.incrementAndGet();
        } else if (packet.getType() == 0) {
            this.dataPackets.incrementAndGet();
        }
    }

    public UtpInputStream getInputStream() throws IOException {
        if (this.inputStream == null) {
            throw new IOException("Socket is not bound.");
        }
        return this.inputStream;
    }

    public UtpOutputStream getOutputStream() throws IOException {
        if (this.outputStream == null) {
            throw new IOException("Socket is not bound.");
        }
        return this.outputStream;
    }

    private int getAvailableWindowSize() {
        return this.getSendWindowSize() - this.getBytesInFlight();
    }

    public int getBytesInFlight() {
        return this.ackHandler.countBytesInFlight();
    }

    private int getSendWindowSize() {
        return Math.min(this.window.getSize(), this.clientWindowSize);
    }

    public void process(UtpPacket packet) throws IOException {
        int receiveTime = this.timer.getCurrentMicros();
        this.lastInteraction = this.clock.instant();
        LOGGER.trace("Received {} from {}", (Object)packet, (Object)this.socketAddress);
        this.clientWindowSize = packet.getWindowSize();
        Optional<UtpPacket> ackedPacket = this.ackHandler.onReceivedPacket(packet);
        ackedPacket.ifPresent(p -> {
            this.onPacketAcknowledged(receiveTime, packet, (UtpPacket)p);
            this.measuredDelay = Math.abs(receiveTime - packet.getTimestampMicroseconds());
        });
        packet.processPayload(this);
        Sync.signalAll(this.notifyLock, this.onPacketAcknowledged);
    }

    private void onPacketAcknowledged(int receiveTime, UtpPacket packet, UtpPacket ackedPacket) {
        LOGGER.trace("{} ACKed {} which was in flight. ({})", new Object[]{packet, ackedPacket, this.ackHandler});
        if (packet.getTimestampDifferenceMicroseconds() != 0) {
            this.window.update(packet);
            this.updatePacketSize();
        }
        if (ackedPacket.getType() == 4) {
            LOGGER.debug("Received ACK on SYN, connection is ok.");
            this.setConnectionState(ConnectionState.CONNECTED);
        }
        this.timeout.update(receiveTime, ackedPacket);
    }

    private void updatePacketSize() {
        this.packetSize = Math.max(150, this.window.getSize() / 10);
        LOGGER.trace("Packet Size scaled based on new window size: {} bytes", (Object)this.packetSize);
    }

    public void onReset() throws IOException {
        if (this.connectionState == ConnectionState.CLOSED) {
            return;
        }
        LOGGER.debug("Connection got reset.");
        UtpPacket resetPacket = new UtpPacket(this, new ResetPayload());
        this.setConnectionState(ConnectionState.CLOSED);
        this.doSend(resetPacket);
        this.notifyLock.lock();
        try {
            this.onPacketAcknowledged.signalAll();
        }
        finally {
            this.notifyLock.unlock();
        }
        this.ackHandler.onReset();
    }

    public void bindIoStreams(short sequenceNumber) {
        this.inputStream = new UtpInputStream(this, (short)(sequenceNumber + 1));
        this.outputStream = new UtpOutputStream(this);
    }

    public void handleTimeout() throws IOException {
        if (Duration.between(this.lastInteraction, this.clock.instant()).minus(this.timeout.getDuration()).isNegative()) {
            return;
        }
        int statePacketCount = this.statePackets.getAndSet(0);
        int dataPacketCount = this.dataPackets.getAndSet(0);
        LOGGER.debug("Socket has encountered a timeout after {}ms. Send Packets: {} Data, {} State.", new Object[]{this.timeout.getDuration().toMillis(), dataPacketCount, statePacketCount});
        this.packetSize = 150;
        this.window.onTimeout();
        this.sendUnbounded(new UtpPacket(this, new StatePayload()));
    }

    public void handleClose() {
        if (!this.connectionState.isClosedState()) {
            return;
        }
        if (this.getAcknowledgeNumber() == this.endOfStreamSequenceNumber && !this.ackHandler.hasPacketsInFlight()) {
            this.setConnectionState(ConnectionState.CLOSED);
            this.utpMultiplexer.cleanUpSocket(this);
        }
    }

    public void close() throws IOException {
        if (this.connectionState != ConnectionState.CONNECTED) {
            return;
        }
        this.setConnectionState(ConnectionState.DISCONNECTING);
        this.send(new FinPayload());
    }

    public boolean isInputShutdown() {
        return false;
    }

    public boolean isOutputShutdown() {
        return this.connectionState == ConnectionState.DISCONNECTING || this.connectionState == ConnectionState.CLOSED;
    }

    public short getAcknowledgeNumber() {
        return this.ackHandler.getAcknowledgeNumber();
    }

    public synchronized short nextSequenceNumber() {
        short s = this.sequenceNumberCounter;
        this.sequenceNumberCounter = (short)(s + 1);
        this.sequenceNumber = s;
        return this.sequenceNumber;
    }

    public short getReceivingConnectionId() {
        return this.connectionIdReceive;
    }

    public short getSendingConnectionId() {
        return this.connectionIdSend;
    }

    public int getMeasuredDelay() {
        return this.measuredDelay;
    }

    public int getWindowSize() {
        return this.utpMultiplexer.getReceiveBufferSize();
    }

    public int getPacketSize() {
        return this.packetSize - 20;
    }

    public ConnectionState getConnectionState() {
        return this.connectionState;
    }

    void setConnectionState(ConnectionState connectionState) {
        LOGGER.debug("Connection state transitioned from {} to {}", (Object)this.connectionState, (Object)connectionState);
        this.connectionState = connectionState;
    }

    public void setEndOfStreamSequenceNumber(short sequenceNumber) {
        this.endOfStreamSequenceNumber = sequenceNumber;
    }

    public short getEndOfStreamSequenceNumber() {
        return this.endOfStreamSequenceNumber;
    }

    public short getSequenceNumber() {
        return this.sequenceNumber;
    }

    public String toString() {
        return String.format("UtpSocketImpl[state=%s, window=%s, timeout=%sms, ackHandler=%s]", new Object[]{this.connectionState, this.window.getSize(), this.timeout.getDuration().toMillis(), this.ackHandler});
    }

    public static class Builder {
        private UtpMultiplexer utpMultiplexer;
        private SocketAddress socketAddress;

        public Builder setUtpMultiplexer(UtpMultiplexer utpMultiplexer) {
            this.utpMultiplexer = utpMultiplexer;
            return this;
        }

        public Builder setSocketAddress(SocketAddress socketAddress) {
            this.socketAddress = socketAddress;
            return this;
        }

        public UtpSocketImpl build() {
            return new UtpSocketImpl(this.utpMultiplexer);
        }

        public UtpSocketImpl build(short connectionId) {
            return new UtpSocketImpl(this.utpMultiplexer, this.socketAddress, connectionId);
        }
    }
}

