/*
 * Decompiled with CFR 0.152.
 */
package org.johnnei.javatorrent.internal.utp;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.DatagramChannel;
import java.time.Clock;
import java.time.Instant;
import java.util.Date;
import java.util.LinkedList;
import java.util.Objects;
import java.util.Queue;
import java.util.Random;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.johnnei.javatorrent.internal.utils.PrecisionTimer;
import org.johnnei.javatorrent.internal.utils.Sync;
import org.johnnei.javatorrent.internal.utp.Acknowledgement;
import org.johnnei.javatorrent.internal.utp.PacketAckHandler;
import org.johnnei.javatorrent.internal.utp.PacketLossHandler;
import org.johnnei.javatorrent.internal.utp.PacketSizeHandler;
import org.johnnei.javatorrent.internal.utp.SocketDelayHandler;
import org.johnnei.javatorrent.internal.utp.SocketTimeoutHandler;
import org.johnnei.javatorrent.internal.utp.SocketWindowHandler;
import org.johnnei.javatorrent.internal.utp.protocol.ConnectionState;
import org.johnnei.javatorrent.internal.utp.protocol.PacketType;
import org.johnnei.javatorrent.internal.utp.protocol.packet.DataPayload;
import org.johnnei.javatorrent.internal.utp.protocol.packet.FinPayload;
import org.johnnei.javatorrent.internal.utp.protocol.packet.Payload;
import org.johnnei.javatorrent.internal.utp.protocol.packet.StatePayload;
import org.johnnei.javatorrent.internal.utp.protocol.packet.SynPayload;
import org.johnnei.javatorrent.internal.utp.protocol.packet.UtpHeader;
import org.johnnei.javatorrent.internal.utp.protocol.packet.UtpPacket;
import org.johnnei.javatorrent.internal.utp.stream.PacketWriter;
import org.johnnei.javatorrent.internal.utp.stream.StreamState;
import org.johnnei.javatorrent.internal.utp.stream.UtpInputStream;
import org.johnnei.javatorrent.internal.utp.stream.UtpOutputStream;
import org.johnnei.javatorrent.network.socket.ISocket;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

public class UtpSocket
implements ISocket,
Closeable {
    private static final Logger LOGGER = LoggerFactory.getLogger(UtpSocket.class);
    private final Lock notifyLock = new ReentrantLock();
    private final Condition onStateChange = this.notifyLock.newCondition();
    private final Clock clock;
    private final PacketWriter packetWriter;
    private final PrecisionTimer precisionTimer;
    private final short sendConnectionId;
    private final DatagramChannel channel;
    private Instant nextWindowViolation;
    private short sequenceNumberCounter;
    private Short endOfStreamSequenceNumber;
    private ConnectionState connectionState;
    private final Queue<UtpPacket> resendQueue;
    private final Queue<Acknowledgement> acknowledgeQueue;
    private short lastSentAcknowledgeNumber;
    private final LinkedList<DataPayload> sendQueue;
    private PacketAckHandler packetAckHandler;
    private UtpOutputStream outputStream;
    private UtpInputStream inputStream;
    private SocketAddress remoteAddress;
    private SocketTimeoutHandler timeoutHandler;
    private PacketLossHandler packetLossHandler;
    private SocketWindowHandler windowHandler;
    private PacketSizeHandler packetSizeHandler;
    private final SocketDelayHandler delayHandler;
    private StreamState outputStreamState;
    private StreamState inputStreamState;

    public static UtpSocket createInitiatingSocket(DatagramChannel channel, short receiveConnectionId) {
        UtpSocket socket = new UtpSocket(channel, (short)(receiveConnectionId + 1));
        socket.sequenceNumberCounter = 0;
        socket.packetAckHandler = new PacketAckHandler(socket);
        return socket;
    }

    public static UtpSocket createRemoteConnecting(DatagramChannel channel, UtpPacket synPacket) {
        short sendConnectionId = synPacket.getHeader().getConnectionId();
        UtpSocket socket = new UtpSocket(channel, sendConnectionId);
        socket.sequenceNumberCounter = (short)new Random().nextInt();
        socket.packetAckHandler = new PacketAckHandler(socket, (short)(synPacket.getHeader().getSequenceNumber() - 1));
        return socket;
    }

    private UtpSocket(DatagramChannel channel, short sendConnectionId) {
        this.channel = channel;
        this.sendConnectionId = sendConnectionId;
        this.clock = Clock.systemDefaultZone();
        this.connectionState = ConnectionState.PENDING;
        this.acknowledgeQueue = new LinkedList<Acknowledgement>();
        this.sendQueue = new LinkedList();
        this.resendQueue = new LinkedList<UtpPacket>();
        this.packetWriter = new PacketWriter();
        this.precisionTimer = new PrecisionTimer();
        this.outputStream = new UtpOutputStream(this);
        this.timeoutHandler = new SocketTimeoutHandler(this.precisionTimer);
        this.packetLossHandler = new PacketLossHandler(this);
        this.windowHandler = new SocketWindowHandler();
        this.packetSizeHandler = new PacketSizeHandler(this.windowHandler);
        this.inputStreamState = StreamState.ACTIVE;
        this.outputStreamState = StreamState.ACTIVE;
        this.nextWindowViolation = this.clock.instant();
        this.delayHandler = new SocketDelayHandler(this.precisionTimer);
    }

    public void bind(SocketAddress remoteAddress) {
        this.remoteAddress = remoteAddress;
    }

    public void connect(InetSocketAddress endpoint) throws IOException {
        this.bind(endpoint);
        this.send(new SynPayload());
        this.connectionState = ConnectionState.SYN_SENT;
        Date timeout = Date.from(this.clock.instant().plusSeconds(10L));
        this.notifyLock.lock();
        try {
            while (this.connectionState != ConnectionState.CONNECTED) {
                if (this.onStateChange.awaitUntil(timeout)) continue;
                throw new IOException("Connection was not accepted within timeout.");
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("Connecting got interrupted.", e);
        }
        finally {
            this.notifyLock.unlock();
        }
    }

    public void onReceivedPacket(UtpPacket packet) {
        try (MDC.MDCCloseable ignored = MDC.putCloseable((String)"context", (String)Integer.toString(Short.toUnsignedInt(this.sendConnectionId)));){
            LOGGER.trace("Received [{}] packet [{}]", (Object)PacketType.getByType(packet.getHeader().getType()), (Object)Short.toUnsignedInt(packet.getHeader().getSequenceNumber()));
            this.packetAckHandler.onReceivedPacket(packet);
            this.packetLossHandler.onReceivedPacket(packet);
            this.timeoutHandler.onReceivedPacket();
            this.windowHandler.onReceivedPacket(packet).ifPresent(this.timeoutHandler::onAckedPacket);
            this.packetSizeHandler.onReceivedPacket(packet);
            this.delayHandler.onReceivedPacket(packet);
            packet.getPayload().onReceivedPayload(packet.getHeader(), this);
        }
    }

    public void send(ByteBuffer data) {
        this.sendQueue.add(new DataPayload(data));
    }

    public void resend(UtpPacket packet) {
        this.resendQueue.add(packet);
        this.windowHandler.onPacketLoss(packet);
        this.packetSizeHandler.onPacketLoss();
    }

    public void acknowledgePacket(Acknowledgement acknowledgement) {
        this.acknowledgeQueue.add(acknowledgement);
    }

    public void processSendQueue() throws IOException {
        try (MDC.MDCCloseable ignored = MDC.putCloseable((String)"context", (String)Integer.toString(Short.toUnsignedInt(this.sendConnectionId)));){
            boolean canSendMultiple;
            do {
                canSendMultiple = false;
                if (!this.resendQueue.isEmpty()) {
                    this.send(this.resendQueue.poll(), false);
                    canSendMultiple = true;
                    continue;
                }
                int maxPayloadSize = this.windowHandler.getMaxWindow() - this.windowHandler.getBytesInFlight() - 20;
                if (this.canSendNewPacket(maxPayloadSize)) {
                    this.send(this.sendQueue.poll());
                    canSendMultiple = true;
                    continue;
                }
                if (this.sendQueue.isEmpty() && this.outputStreamState == StreamState.SHUTDOWN_PENDING) {
                    this.send(new FinPayload());
                    this.outputStreamState = StreamState.SHUTDOWN;
                    continue;
                }
                if (this.acknowledgeQueue.isEmpty() || maxPayloadSize < 0) continue;
                this.sendStatePackets(maxPayloadSize);
            } while (canSendMultiple);
        }
    }

    private void sendStatePackets(int maxPayLoadSize) throws IOException {
        int remainingPackets = Math.min(this.acknowledgeQueue.size(), Math.max(1, maxPayLoadSize / 20));
        if (remainingPackets > 1) {
            LOGGER.trace("Sending out ST_STATE burst of [{}] packets.", (Object)remainingPackets);
        }
        while (remainingPackets > 0) {
            this.send(new StatePayload());
            --remainingPackets;
        }
    }

    private boolean canSendNewPacket(int maxPayloadSize) {
        if (this.sendQueue.isEmpty()) {
            return false;
        }
        int payloadSize = this.sendQueue.peek().getData().length;
        if (payloadSize <= maxPayloadSize) {
            this.mergePayloadsIfPossible(Math.min(this.packetSizeHandler.getPacketSize(), maxPayloadSize));
            return true;
        }
        if (!this.acknowledgeQueue.isEmpty()) {
            return false;
        }
        int maxWindow = this.windowHandler.getMaxWindow();
        if (payloadSize > maxWindow && maxWindow > 0 && this.clock.instant().isAfter(this.nextWindowViolation)) {
            int secondsToAdd = payloadSize / maxWindow;
            this.nextWindowViolation = this.clock.instant().plusSeconds(secondsToAdd);
            LOGGER.trace("Violating window of [{}] bytes by [{}] bytes (shortage: [{}]). Blocking exceeds for [{}] seconds", new Object[]{maxWindow, payloadSize, payloadSize - maxWindow, secondsToAdd});
            return true;
        }
        return false;
    }

    private void mergePayloadsIfPossible(int maxPayloadSize) {
        if (this.sendQueue.size() < 2) {
            return;
        }
        int combinedPayloads = 1;
        while (this.sendQueue.size() > 1) {
            DataPayload payloadOne = this.sendQueue.get(0);
            DataPayload payloadTwo = this.sendQueue.get(1);
            int payloadSum = payloadOne.getData().length + payloadTwo.getData().length;
            if (payloadSum > maxPayloadSize) break;
            this.sendQueue.poll();
            this.sendQueue.poll();
            ByteBuffer combinedPayload = ByteBuffer.allocate(payloadSum);
            combinedPayload.put(payloadOne.getData());
            combinedPayload.put(payloadTwo.getData());
            combinedPayload.flip();
            this.sendQueue.addFirst(new DataPayload(combinedPayload));
            ++combinedPayloads;
        }
        if (combinedPayloads > 1) {
            LOGGER.trace("Combined [{}] data payloads into 1.", (Object)combinedPayloads);
        }
    }

    public void processTimeout() {
        if (!this.timeoutHandler.isTimeoutExpired()) {
            return;
        }
        try (MDC.MDCCloseable ignored = MDC.putCloseable((String)"context", (String)Integer.toString(Short.toUnsignedInt(this.sendConnectionId)));){
            LOGGER.trace("Socket triggered timeout. Window: {} bytes. Bytes in flight: {}. Payload Size: {} bytes. Resend Queue: {} packets. Send Queue: {} packets (head: {}), Ack Queue: {} packets.", new Object[]{this.windowHandler.getMaxWindow(), this.windowHandler.getBytesInFlight(), this.getPacketPayloadSize(), this.resendQueue.size(), this.sendQueue.size(), this.sendQueue.peek(), this.acknowledgeQueue.size()});
            this.timeoutHandler.onTimeout();
            this.packetSizeHandler.onTimeout();
            this.windowHandler.onTimeout();
        }
    }

    private void send(Payload payload) throws IOException {
        UtpHeader header = new UtpHeader.Builder().setType(payload.getType().getTypeField()).setSequenceNumber(this.getPacketSequenceNumber(payload.getType())).setExtension((byte)0).setConnectionId(this.getSendConnectionId(payload.getType())).setWindowSize(this.windowHandler.getBytesInFlight()).build();
        UtpPacket packet = new UtpPacket(header, payload);
        this.send(packet, true);
    }

    private void send(UtpPacket packet, boolean renewAck) throws IOException {
        short ackNumber = this.lastSentAcknowledgeNumber;
        if (!renewAck) {
            ackNumber = packet.getHeader().getAcknowledgeNumber();
        } else if (!this.acknowledgeQueue.isEmpty()) {
            this.lastSentAcknowledgeNumber = ackNumber = this.acknowledgeQueue.poll().getSequenceNumber();
        }
        packet.getHeader().renew(ackNumber, this.precisionTimer.getCurrentMicros(), this.delayHandler.getMeasuredDelay());
        ByteBuffer buffer = this.packetWriter.write(packet);
        try (MDC.MDCCloseable ignored = MDC.putCloseable((String)"context", (String)Integer.toString(Short.toUnsignedInt(this.sendConnectionId)));){
            LOGGER.trace("Writing [{}] packet [{}] acking [{}] of [{}] bytes", new Object[]{PacketType.getByType(packet.getHeader().getType()), Short.toUnsignedInt(packet.getHeader().getSequenceNumber()), Short.toUnsignedInt(packet.getHeader().getAcknowledgeNumber()), buffer.limit()});
            this.channel.send(buffer, this.remoteAddress);
            this.packetLossHandler.onSentPacket(packet);
            this.timeoutHandler.onSentPacket();
            this.windowHandler.onSentPacket(packet);
            this.packetSizeHandler.onSentPacket(packet);
        }
        if (buffer.hasRemaining()) {
            throw new IOException("Write buffer utilization exceeded.");
        }
    }

    private short getPacketSequenceNumber(PacketType type) {
        if (type == PacketType.STATE && this.connectionState != ConnectionState.SYN_RECEIVED) {
            return this.sequenceNumberCounter;
        }
        this.sequenceNumberCounter = (short)(this.sequenceNumberCounter + 1);
        return this.sequenceNumberCounter;
    }

    private short getSendConnectionId(PacketType type) {
        if (type == PacketType.SYN) {
            return (short)(this.sendConnectionId - 1);
        }
        return this.sendConnectionId;
    }

    public int getPacketPayloadSize() {
        return this.packetSizeHandler.getPacketSize() - 20;
    }

    public InputStream getInputStream() throws IOException {
        return Objects.requireNonNull(this.inputStream, "Connection was not established yet");
    }

    public OutputStream getOutputStream() throws IOException {
        return this.outputStream;
    }

    @Override
    public void close() {
        this.flush();
        this.setConnectionState(ConnectionState.CLOSING);
        this.outputStreamState = StreamState.SHUTDOWN_PENDING;
    }

    public void submitData(short sequenceNumber, byte[] data) {
        this.inputStream.submitData(sequenceNumber, data);
    }

    public void shutdownInputStream(short sequenceNumber) {
        this.endOfStreamSequenceNumber = sequenceNumber;
        this.inputStreamState = StreamState.SHUTDOWN;
        this.setConnectionState(ConnectionState.CLOSING);
        if (this.outputStreamState == StreamState.ACTIVE) {
            this.close();
        }
    }

    public boolean isClosed() {
        return this.connectionState == ConnectionState.PENDING || this.connectionState == ConnectionState.CLOSED || this.connectionState == ConnectionState.RESET;
    }

    public boolean isInputShutdown() {
        return this.inputStreamState != StreamState.ACTIVE;
    }

    public boolean isOutputShutdown() {
        return this.outputStreamState != StreamState.ACTIVE;
    }

    public void flush() {
        this.outputStream.flush();
    }

    public boolean isShutdown() {
        return this.connectionState == ConnectionState.RESET || this.inputStreamState == StreamState.SHUTDOWN && this.inputStream.isCompleteUntil(this.endOfStreamSequenceNumber) && this.outputStreamState == StreamState.SHUTDOWN && this.windowHandler.getBytesInFlight() == 0;
    }

    public void setConnectionState(ConnectionState newState) {
        try (MDC.MDCCloseable ignored = MDC.putCloseable((String)"context", (String)Integer.toString(Short.toUnsignedInt(this.sendConnectionId)));){
            LOGGER.trace("Transitioning state from {} to {}", (Object)this.connectionState, (Object)newState);
        }
        this.connectionState = newState;
        if (this.connectionState == ConnectionState.CONNECTED) {
            this.inputStream = new UtpInputStream((short)(this.lastSentAcknowledgeNumber + 1));
        }
        Sync.signalAll((Lock)this.notifyLock, (Condition)this.onStateChange);
    }

    public ConnectionState getConnectionState() {
        return this.connectionState;
    }

    public void setAcknowledgeNumber(short acknowledgeNumber) {
        this.lastSentAcknowledgeNumber = acknowledgeNumber;
    }

    public String toString() {
        return String.format("UtpSocket[sendConnectionId=%s, remote=%s]", Short.toUnsignedInt(this.sendConnectionId), this.remoteAddress);
    }
}

