/*
 * Decompiled with CFR 0.152.
 */
package org.jooby.assets;

import com.google.common.base.CaseFormat;
import com.google.common.collect.ImmutableList;
import com.google.common.hash.Hashing;
import com.google.common.io.BaseEncoding;
import com.google.common.io.ByteStreams;
import com.google.common.io.Closeables;
import com.google.common.io.Files;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import com.typesafe.config.ConfigMergeable;
import com.typesafe.config.ConfigValue;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.jooby.Asset;
import org.jooby.MediaType;
import org.jooby.assets.AssetProcessor;
import org.jooby.assets.InMemoryAsset;
import org.jooby.internal.RoutePattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AssetCompiler {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final Map<String, List<AssetProcessor>> pipeline;
    private final Map<String, List<String>> fileset;
    private final Predicate<String> scripts;
    private final Predicate<String> styles;
    private final Config conf;
    private final Charset charset;
    private ClassLoader loader;

    public AssetCompiler(Config conf) throws Exception {
        this(conf.getClass().getClassLoader(), conf);
    }

    public AssetCompiler(ClassLoader loader, Config conf) throws Exception {
        this.loader = Objects.requireNonNull(loader, "Class loader is required.");
        this.conf = Objects.requireNonNull(conf, "Assets conf is required.");
        String basedir = conf.hasPath("assets.basedir") ? AssetCompiler.spath(conf.getString("assets.basedir")) : "";
        this.charset = Charset.forName(this.conf.getString("assets.charset"));
        this.fileset = this.conf.hasPath("assets.fileset") ? AssetCompiler.fileset(basedir, this.conf.getConfig("assets.fileset")) : Collections.emptyMap();
        this.scripts = AssetCompiler.predicate(this.conf, ".js", ".coffee", ".ts");
        this.styles = AssetCompiler.predicate(this.conf, ".css", ".scss", ".less");
        this.pipeline = this.fileset.size() > 0 ? AssetCompiler.pipeline(loader, this.conf.getConfig("assets")) : Collections.emptyMap();
    }

    public List<String> assets(String name) {
        return this.fileset.getOrDefault(name, Collections.emptyList());
    }

    public Set<String> keySet() {
        return this.fileset.keySet();
    }

    public Set<String> patterns() {
        return this.patterns(file -> true).map(v -> "/" + v + "/**").collect(Collectors.toCollection(LinkedHashSet::new));
    }

    public List<String> scripts(String name) {
        return this.assets(name).stream().filter(this.scripts).collect(Collectors.toList());
    }

    public List<String> styles(String name) {
        return this.assets(name).stream().filter(this.styles).collect(Collectors.toList());
    }

    public List<AssetProcessor> pipeline(String dist) {
        List<AssetProcessor> chain = this.pipeline.get(dist);
        if (chain == null) {
            throw new IllegalArgumentException("No pipeline for: " + dist);
        }
        return chain;
    }

    public Map<String, List<File>> build(String dist, File dir) throws Exception {
        LinkedHashMap<String, List<File>> output = new LinkedHashMap<String, List<File>>();
        for (String fset : this.keySet()) {
            List<String> files = this.assets(fset);
            this.log.info("compiling {}: ", (Object)fset);
            String css = this.compile(dist, files.stream().filter(this.styles).iterator(), MediaType.css, "");
            Path pcss = Paths.get(this.patterns(this.styles).findFirst().get(), fset + "." + this.sha1(css) + ".css");
            File fcss = dir.toPath().resolve(pcss).toFile();
            fcss.getParentFile().mkdirs();
            Files.write((CharSequence)css, (File)fcss, (Charset)this.charset);
            String js = this.compile(dist, files.stream().filter(this.scripts).iterator(), MediaType.js, ";");
            Path pjs = Paths.get(this.patterns(this.scripts).findFirst().get(), fset + "." + this.sha1(js) + ".js");
            File fjs = dir.toPath().resolve(pjs).toFile();
            fjs.getParentFile().mkdirs();
            Files.write((CharSequence)js, (File)fjs, (Charset)this.charset);
            this.log.info("{}", (Object)fcss);
            this.log.info("{}", (Object)fjs);
            output.put(fset, Arrays.asList(fcss, fjs));
        }
        return output;
    }

    public Asset build(Asset asset) throws Exception {
        MediaType type;
        if (this.pipeline.size() == 0) {
            return asset;
        }
        String filename = asset.path();
        if (this.scripts.test(filename)) {
            type = MediaType.js;
        } else if (this.styles.test(filename)) {
            type = MediaType.css;
        } else {
            return asset;
        }
        String output = this.compile("dev", filename, type, AssetCompiler.toString(asset.stream(), this.charset));
        return new InMemoryAsset(asset, output.getBytes(this.charset));
    }

    public String toString() {
        return this.fileset.toString();
    }

    private Stream<String> patterns(Predicate<String> filter) {
        return this.fileset.values().stream().flatMap(Collection::stream).filter(filter).map(path -> path.split("/")[1]);
    }

    private String compile(String env, Iterator<String> files, MediaType type, String sep) throws Exception {
        StringBuilder buff = new StringBuilder();
        while (files.hasNext()) {
            String file = files.next();
            this.log.info("  {}", (Object)file);
            buff.append(this.compile(env, file, type, AssetCompiler.readFile(this.loader, file, this.charset))).append(sep);
        }
        return buff.toString();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private String compile(String env, String filename, MediaType type, String input) throws Exception {
        Iterator<AssetProcessor> it = this.pipeline(env).iterator();
        String contents = input;
        while (it.hasNext()) {
            AssetProcessor processor = it.next();
            if (!processor.matches(type) || this.excludes(processor, filename)) continue;
            String pname = processor.name();
            long start = System.currentTimeMillis();
            try {
                this.log.debug("executing: {}", (Object)pname);
                contents = processor.process(filename, contents, this.conf);
            }
            finally {
                long end = System.currentTimeMillis();
                this.log.debug("{} took {}ms", (Object)pname, (Object)(end - start));
            }
        }
        return contents;
    }

    private String sha1(String source) {
        return BaseEncoding.base16().encode(Hashing.sha1().hashString((CharSequence)source, this.charset).asBytes()).substring(0, 8).toLowerCase();
    }

    private boolean excludes(AssetProcessor processor, String path) {
        Object value = processor.get("excludes");
        if (value == null) {
            return false;
        }
        List excludes = value instanceof List ? (List)value : ImmutableList.of((Object)value.toString());
        String spath = AssetCompiler.spath(path);
        return excludes.stream().map(it -> new RoutePattern("GET", it)).filter(pattern -> pattern.matcher("GET" + spath).matches()).findFirst().isPresent();
    }

    private static String readFile(ClassLoader loader, String path, Charset charset) throws IOException {
        String spath = path.startsWith("/") ? path.substring(1) : path;
        URL res = loader.getResource(spath);
        if (res == null) {
            throw new FileNotFoundException(spath);
        }
        return AssetCompiler.toString(res.openStream(), charset);
    }

    private static String toString(InputStream in, Charset charset) throws IOException {
        try {
            String string = new String(ByteStreams.toByteArray((InputStream)in), charset);
            return string;
        }
        finally {
            Closeables.closeQuietly((InputStream)in);
        }
    }

    private static Predicate<String> predicate(Config fileset, String ... extension) {
        String path = "assets" + extension[0];
        HashSet<String> extensions = new HashSet<String>();
        extensions.addAll(Arrays.asList(extension));
        if (fileset.hasPath(path)) {
            extensions.addAll(AssetCompiler.strlist(fileset.getAnyRef(path)));
        }
        return file -> {
            for (String ext : extensions) {
                if (!file.endsWith(ext)) continue;
                return true;
            }
            return false;
        };
    }

    private static Map<String, List<String>> fileset(String basedir, Config fileset) {
        HashMap<String, List<String>> result = new HashMap<String, List<String>>();
        fileset.entrySet().forEach(e -> {
            String[] key = AssetCompiler.unquote((String)e.getKey()).split("\\s*<\\s*");
            result.put(key[0], AssetCompiler.strlist(((ConfigValue)e.getValue()).unwrapped(), v -> basedir + AssetCompiler.spath(v)));
        });
        fileset.entrySet().forEach(e -> {
            String[] key = AssetCompiler.unquote((String)e.getKey()).split("\\s*<\\s*");
            if (key.length > 1) {
                ImmutableList.Builder resources = ImmutableList.builder();
                for (int i = key.length - 1; i >= 0; --i) {
                    resources.addAll((Iterable)result.get(key[i]));
                }
                result.put(key[0], (List<String>)resources.build());
            }
        });
        return result;
    }

    private static String unquote(String key) {
        return key.replace("\"", "");
    }

    private static Map<String, List<AssetProcessor>> pipeline(ClassLoader loader, Config conf) throws Exception {
        HashMap<String, List<AssetProcessor>> processors = new HashMap<String, List<AssetProcessor>>();
        processors.put("dev", Collections.emptyList());
        if (conf.hasPath("pipeline")) {
            Set<String> filter = conf.getConfig("pipeline").entrySet().stream().map(e -> (String)e.getKey()).collect(Collectors.toSet());
            filter.add("class");
            Set entrySet = conf.getConfig("pipeline").entrySet();
            for (Map.Entry entry : entrySet) {
                String env = AssetCompiler.unquote((String)entry.getKey());
                processors.put(env, AssetCompiler.processors(conf, loader, env, AssetCompiler.strlist(((ConfigValue)entry.getValue()).unwrapped()), filter));
            }
        }
        return processors;
    }

    private static List<AssetProcessor> processors(Config conf, ClassLoader loader, String env, List<String> names, Set<String> filter) throws Exception {
        LinkedHashMap<String, Class<AssetProcessor>> classes = new LinkedHashMap<String, Class<AssetProcessor>>();
        for (Map.Entry<String, String> entry : AssetCompiler.bind(conf, names).entrySet()) {
            classes.put(entry.getKey(), loader.loadClass(entry.getValue()));
        }
        return AssetCompiler.processors(conf, env, filter, classes);
    }

    private static List<AssetProcessor> processors(Config conf, String env, Set<String> filter, Map<String, Class<AssetProcessor>> classes) throws Exception {
        ArrayList<AssetProcessor> processors = new ArrayList<AssetProcessor>();
        Function<Config, Config> without = options -> {
            for (String path : filter) {
                options = options.withoutPath(path);
            }
            return options;
        };
        for (Map.Entry<String, Class<AssetProcessor>> entry : classes.entrySet()) {
            String name = entry.getKey();
            Class<AssetProcessor> clazz = entry.getValue();
            Config options2 = ConfigFactory.empty();
            if (conf.hasPath(name) && (options2 = conf.getConfig(name)).hasPath(env)) {
                options2 = options2.getConfig(env).withFallback((ConfigMergeable)options2);
            }
            AssetProcessor processor = clazz.newInstance();
            processor.set(without.apply(options2));
            processors.add(processor);
        }
        return processors;
    }

    private static Map<String, String> bind(Config conf, List<String> names) {
        LinkedHashMap<String, String> map = new LinkedHashMap<String, String>();
        names.forEach(name -> {
            String clazz = AssetCompiler.class.getPackage().getName() + "." + CaseFormat.LOWER_HYPHEN.to(CaseFormat.UPPER_CAMEL, name);
            if (conf.hasPath(name + ".class")) {
                clazz = conf.getString(name + ".class");
            }
            map.put((String)name, clazz);
        });
        return map;
    }

    private static List<String> strlist(Object value) {
        return AssetCompiler.strlist(value, v -> v);
    }

    private static List<String> strlist(Object value, Function<String, String> mapper) {
        ImmutableList.Builder list = ImmutableList.builder();
        if (value instanceof Collection) {
            ((Collection)value).forEach(v -> list.add(mapper.apply((String)v)));
        } else {
            list.add((Object)mapper.apply(value.toString()));
        }
        return list.build();
    }

    private static String spath(String path) {
        return path.startsWith("/") ? path : "/" + path;
    }
}

