/*
 * Decompiled with CFR 0.152.
 */
package org.jooby.assets;

import com.google.common.base.CaseFormat;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.hash.Hashing;
import com.google.common.io.BaseEncoding;
import com.google.common.io.ByteStreams;
import com.google.common.io.Closeables;
import com.google.common.io.Files;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import com.typesafe.config.ConfigMergeable;
import com.typesafe.config.ConfigValue;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javaslang.Predicates;
import javaslang.control.Try;
import org.jooby.Asset;
import org.jooby.MediaType;
import org.jooby.assets.AssetAggregator;
import org.jooby.assets.AssetClassLoader;
import org.jooby.assets.AssetOptions;
import org.jooby.assets.AssetProcessor;
import org.jooby.assets.InMemoryAsset;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AssetCompiler {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final Map<String, List<AssetProcessor>> pipeline;
    private final List<AssetAggregator> aggregators = new ArrayList<AssetAggregator>();
    private final Map<String, List<String>> fileset;
    private final Predicate<String> scripts;
    private final Predicate<String> styles;
    private final Config conf;
    private final Charset charset;
    private ClassLoader loader;

    public AssetCompiler(Config conf) throws Exception {
        this(conf.getClass().getClassLoader(), conf);
    }

    public AssetCompiler(ClassLoader loader, Config conf) throws Exception {
        this.loader = AssetClassLoader.classLoader(loader);
        this.conf = Objects.requireNonNull(conf, "Assets conf is required.");
        String basedir = conf.hasPath("assets.basedir") ? AssetCompiler.spath(conf.getString("assets.basedir")) : "";
        this.charset = Charset.forName(this.conf.getString("assets.charset"));
        this.fileset = this.conf.hasPath("assets.fileset") ? AssetCompiler.fileset(loader, basedir, this.conf, this.aggregators::add) : new HashMap<String, List<String>>();
        this.scripts = AssetCompiler.predicate(this.conf, ".js", ".coffee", ".ts");
        this.styles = AssetCompiler.predicate(this.conf, ".css", ".scss", ".sass", ".less");
        this.pipeline = this.fileset.size() > 0 ? AssetCompiler.pipeline(loader, this.conf.getConfig("assets")) : Collections.emptyMap();
    }

    public List<String> assets(String name) {
        return this.fileset.getOrDefault(name, Collections.emptyList());
    }

    public Set<String> fileset() {
        return this.fileset.keySet();
    }

    public Set<String> patterns() {
        return this.patterns(file -> !this.aggregators.stream().filter(it -> it.fileset().contains(file)).findFirst().isPresent()).map(v -> "/" + v + "/**").collect(Collectors.toCollection(LinkedHashSet::new));
    }

    public boolean contains(String path) {
        Predicate<List> filter = fs -> fs.stream().filter(it -> path.endsWith((String)it)).findFirst().isPresent();
        boolean generated = this.aggregators.stream().map(AssetAggregator::fileset).filter(filter).findFirst().isPresent();
        if (!generated) {
            return this.fileset.values().stream().filter(filter).findFirst().isPresent();
        }
        return false;
    }

    public List<String> scripts(String fileset) {
        return this.assets(fileset).stream().filter(this.scripts).collect(Collectors.toList());
    }

    public List<String> styles(String fileset) {
        return this.assets(fileset).stream().filter(this.styles).collect(Collectors.toList());
    }

    public List<AssetProcessor> pipeline(String dist) {
        List<AssetProcessor> chain = this.pipeline.get(dist);
        if (chain == null) {
            this.log.debug("no pipeline for: {}", (Object)dist);
            return Collections.emptyList();
        }
        return Collections.unmodifiableList(chain);
    }

    public List<AssetAggregator> aggregators() {
        return Collections.unmodifiableList(this.aggregators);
    }

    public Map<String, List<File>> build(String dist, File dir) throws Exception {
        LinkedHashMap<String, List<File>> output = new LinkedHashMap<String, List<File>>();
        this.log.info("{} aggregators: {}", (Object)dist, this.aggregators);
        this.aggregators(this.aggregators, this.conf);
        List<AssetProcessor> pipeline = this.pipeline(dist);
        this.log.info("{} pipeline: {}", (Object)dist, pipeline);
        for (String fset : this.fileset()) {
            List<String> files = this.assets(fset);
            this.log.info("compiling {}:", (Object)fset);
            String css = this.compile(pipeline, files.stream().filter(this.styles).iterator(), MediaType.css, "");
            Path cssSha1 = Paths.get(fset + "." + this.sha1(css) + ".css", new String[0]);
            Path pcss = this.patterns(this.styles).findFirst().map(p -> Paths.get(p, new String[0]).resolve(cssSha1)).orElse(cssSha1);
            File fcss = dir.toPath().resolve(pcss).toFile();
            fcss.getParentFile().mkdirs();
            ImmutableList.Builder outputbuilder = ImmutableList.builder();
            if (css.length() > 0) {
                Files.write((CharSequence)css, (File)fcss, (Charset)this.charset);
                outputbuilder.add((Object)fcss);
            }
            String js = this.compile(pipeline, files.stream().filter(this.scripts).iterator(), MediaType.js, ";");
            Path jsSha1 = Paths.get(fset + "." + this.sha1(js) + ".js", new String[0]);
            Path pjs = this.patterns(this.scripts).findFirst().map(p -> Paths.get(p, new String[0]).resolve(jsSha1)).orElse(jsSha1);
            File fjs = dir.toPath().resolve(pjs).toFile();
            fjs.getParentFile().mkdirs();
            if (js.length() > 0) {
                Files.write((CharSequence)js, (File)fjs, (Charset)this.charset);
                outputbuilder.add((Object)fjs);
            }
            ImmutableList fsoutput = outputbuilder.build();
            fsoutput.forEach(it -> this.log.info("{} {} ({})", new Object[]{it.getName(), AssetCompiler.humanReadableByteCount(it.length()), it}));
            output.put(fset, (List<File>)fsoutput);
        }
        return output;
    }

    private void aggregators(List<AssetAggregator> aggregators, Config conf) throws Exception {
        for (AssetAggregator it : aggregators) {
            this.log.info("applying {}", (Object)it);
            it.run(conf);
        }
    }

    public Asset build(Asset asset) throws Exception {
        MediaType type;
        if (this.pipeline.size() == 0) {
            return asset;
        }
        String filename = asset.path();
        if (this.scripts.test(filename)) {
            type = MediaType.js;
        } else if (this.styles.test(filename)) {
            type = MediaType.css;
        } else {
            return asset;
        }
        List<AssetProcessor> pipeline = this.pipeline("dev");
        String output = this.compile(pipeline, filename, type, AssetCompiler.toString(asset.stream(), this.charset));
        return new InMemoryAsset(asset, output.getBytes(this.charset));
    }

    public String toString() {
        return this.fileset.toString();
    }

    private Stream<String> patterns(Predicate<String> filter) {
        return this.fileset.values().stream().flatMap(Collection::stream).filter(filter).map(path -> path.split("/")[1]);
    }

    private String compile(List<AssetProcessor> pipeline, Iterator<String> files, MediaType type, String sep) throws Exception {
        StringBuilder buff = new StringBuilder();
        while (files.hasNext()) {
            String file = files.next();
            buff.append(this.compile(pipeline, file, type, AssetCompiler.readFile(this.loader, file, this.charset))).append(sep);
        }
        return buff.toString();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private String compile(List<AssetProcessor> pipeline, String filename, MediaType type, String input) throws Exception {
        this.log.info("  {}", (Object)filename);
        Iterator<AssetProcessor> it = pipeline.iterator();
        String contents = input;
        while (it.hasNext()) {
            AssetProcessor processor = it.next();
            if (!processor.matches(type) || processor.excludes(filename)) continue;
            String pname = processor.name();
            long start = System.currentTimeMillis();
            try {
                this.log.debug("    executing: {}", (Object)pname);
                contents = processor.process(filename, contents, this.conf);
            }
            finally {
                long end = System.currentTimeMillis();
                this.log.debug("    {} took {}ms", (Object)pname, (Object)(end - start));
            }
        }
        return contents;
    }

    private String sha1(String source) {
        return BaseEncoding.base16().encode(Hashing.sha1().hashString((CharSequence)source, this.charset).asBytes()).substring(0, 8).toLowerCase();
    }

    private static String readFile(ClassLoader loader, String path, Charset charset) throws IOException {
        String spath = path.startsWith("/") ? path.substring(1) : path;
        InputStream resource = loader.getResourceAsStream(spath);
        if (resource == null) {
            throw new FileNotFoundException(path);
        }
        return AssetCompiler.toString(resource, charset);
    }

    private static String toString(InputStream in, Charset charset) throws IOException {
        try {
            String string = new String(ByteStreams.toByteArray((InputStream)in), charset);
            return string;
        }
        finally {
            Closeables.closeQuietly((InputStream)in);
        }
    }

    private static Predicate<String> predicate(Config fileset, String ... extension) {
        String path = "assets" + extension[0];
        HashSet<String> extensions = new HashSet<String>();
        extensions.addAll(Arrays.asList(extension));
        if (fileset.hasPath(path)) {
            extensions.addAll(AssetCompiler.strlist(fileset.getAnyRef(path)));
        }
        return file -> {
            for (String ext : extensions) {
                if (!file.endsWith(ext)) continue;
                return true;
            }
            return false;
        };
    }

    private static Map<String, List<String>> fileset(ClassLoader loader, String basedir, Config conf, Consumer<AssetAggregator> aggregators) {
        HashMap raw = new HashMap();
        HashMap<String, List> graph = new HashMap<String, List>();
        Config assetconf = conf.getConfig("assets");
        Config fileset = assetconf.getConfig("fileset");
        fileset.entrySet().forEach(e -> {
            List key = Splitter.on((char)'<').trimResults().omitEmptyStrings().splitToList((CharSequence)AssetCompiler.unquote((String)e.getKey()));
            List<String> candidates = AssetCompiler.strlist(((ConfigValue)e.getValue()).unwrapped(), v -> basedir + AssetCompiler.spath(v));
            ArrayList values = new ArrayList();
            candidates.forEach(it -> Try.run(() -> AssetCompiler.processors(assetconf, loader, null, (List<String>)ImmutableList.of((Object)it.substring(1)), (Set<String>)ImmutableSet.of()).stream().filter(Predicates.instanceOf(AssetAggregator.class)).forEach(p -> {
                AssetAggregator a = (AssetAggregator)p;
                aggregators.accept(a);
                a.fileset().forEach(f -> values.add(AssetCompiler.spath(f)));
            })).onFailure(x -> values.add(it)));
            raw.put(key.get(0), values);
            graph.put((String)key.get(0), key);
        });
        HashMap<String, List<String>> resolved = new HashMap<String, List<String>>();
        graph.forEach((fs, deps) -> AssetCompiler.resolve(fs, deps, raw, graph, resolved));
        return resolved;
    }

    private static List<String> resolve(String fs, List<String> deps, Map<String, List<String>> raw, Map<String, List<String>> graph, Map<String, List<String>> resolved) {
        List<String> result = resolved.get(fs);
        if (result == null) {
            result = new ArrayList<String>();
            resolved.put(fs, result);
            for (int i = deps.size() - 1; i > 0; --i) {
                result.addAll(AssetCompiler.resolve(deps.get(i), graph.get(deps.get(i)), raw, graph, resolved));
            }
            result.addAll((Collection<String>)raw.get(fs));
        }
        return result;
    }

    private static String unquote(String key) {
        return key.replace("\"", "");
    }

    private static Map<String, List<AssetProcessor>> pipeline(ClassLoader loader, Config conf) throws Exception {
        HashMap<String, List<AssetProcessor>> processors = new HashMap<String, List<AssetProcessor>>();
        processors.put("dev", Collections.emptyList());
        if (conf.hasPath("pipeline")) {
            Set<String> filter = conf.getConfig("pipeline").entrySet().stream().map(e -> (String)e.getKey()).collect(Collectors.toSet());
            filter.add("class");
            Set entrySet = conf.getConfig("pipeline").entrySet();
            for (Map.Entry entry : entrySet) {
                String env = AssetCompiler.unquote((String)entry.getKey());
                processors.put(env, AssetCompiler.processors(conf, loader, env, AssetCompiler.strlist(((ConfigValue)entry.getValue()).unwrapped()), filter));
            }
        }
        return processors;
    }

    private static <T extends AssetOptions> List<T> processors(Config conf, ClassLoader loader, String env, List<String> names, Set<String> filter) throws Exception {
        LinkedHashMap<String, Class<T>> classes = new LinkedHashMap<String, Class<T>>();
        for (Map.Entry<String, String> entry : AssetCompiler.bind(conf, names).entrySet()) {
            classes.put(entry.getKey(), loader.loadClass(entry.getValue()));
        }
        return AssetCompiler.processors(conf, env, filter, classes);
    }

    private static <T extends AssetOptions> List<T> processors(Config conf, String env, Set<String> filter, Map<String, Class<T>> classes) throws Exception {
        ArrayList<AssetOptions> processors = new ArrayList<AssetOptions>();
        Function<Config, Config> without = options -> {
            for (String path : filter) {
                options = options.withoutPath(path);
            }
            return options;
        };
        for (Map.Entry<String, Class<T>> entry : classes.entrySet()) {
            String name = entry.getKey();
            Class<T> clazz = entry.getValue();
            Config options2 = ConfigFactory.empty();
            if (conf.hasPath(name)) {
                options2 = conf.getConfig(name);
                if (env != null && options2.hasPath(env)) {
                    options2 = options2.getConfig(env).withFallback((ConfigMergeable)options2);
                }
            }
            AssetOptions processor = (AssetOptions)clazz.newInstance();
            processor.set(without.apply(options2));
            processors.add(processor);
        }
        return processors;
    }

    private static Map<String, String> bind(Config conf, List<String> names) {
        LinkedHashMap<String, String> map = new LinkedHashMap<String, String>();
        names.forEach(name -> {
            String clazz = AssetCompiler.class.getPackage().getName() + "." + CaseFormat.LOWER_HYPHEN.to(CaseFormat.UPPER_CAMEL, name);
            if (conf.hasPath(name + ".class")) {
                clazz = conf.getString(name + ".class");
            }
            map.put((String)name, clazz);
        });
        return map;
    }

    private static List<String> strlist(Object value) {
        return AssetCompiler.strlist(value, v -> v);
    }

    private static List<String> strlist(Object value, Function<String, String> mapper) {
        ImmutableList.Builder list = ImmutableList.builder();
        if (value instanceof Collection) {
            ((Collection)value).forEach(v -> list.add(mapper.apply((String)v)));
        } else {
            list.add((Object)mapper.apply(value.toString()));
        }
        return list.build();
    }

    private static String spath(String path) {
        return path.charAt(0) == '/' ? path : "/" + path;
    }

    static String humanReadableByteCount(long bytes) {
        int unit = 1024;
        if (bytes < (long)unit) {
            return bytes + "b";
        }
        int exp = (int)(Math.log(bytes) / Math.log(unit));
        char pre = "kmgtpe".charAt(exp - 1);
        return String.format("%.1f%sb", (double)bytes / Math.pow(unit, exp), Character.valueOf(pre));
    }
}

