/**
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "{}"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright 2014 Edgar Espina
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package org.jooby.caffeine;

import static java.util.Objects.requireNonNull;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.stream.Collectors;

import org.jooby.Env;
import org.jooby.Jooby;
import org.jooby.Session;

import com.github.benmanes.caffeine.cache.AsyncLoadingCache;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.CaffeineSpec;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.google.inject.Binder;
import com.google.inject.Key;
import com.google.inject.TypeLiteral;
import com.google.inject.name.Names;
import com.google.inject.util.Types;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import com.typesafe.config.ConfigValueFactory;

/**
 * <h1>caffeine</h1>
 * <p>
 * {@link Cache} services from <a href="https://github.com/ben-manes/caffeine">Caffeine</a>
 * </p>
 *
 * <h2>usage</h2>
 *
 * <p>
 * App.java:
 * </p>
 * <pre>
 * {
 *   use(CaffeineCache.newCache());
 *
 *   get("/", req {@literal ->} {
 *     Cache cache = req.require(Cache.class);
 *     // do with cache...
 *   });
 * }
 * </pre>
 *
 * <h2>options</h2>
 *
 * <h3>cache configuration</h3>
 * <p>
 * A default cache will be created, if you need to control the number of entries, expire policy,
 * etc... set the <code>caffeine.cache</code> property in <code>application.conf</code>, like:
 * </p>
 *
 * <pre>
 * caffeine.cache {
 *   maximumSize = 10
 * }
 * </pre>
 *
 * or via {@link CaffeineSpec} syntax:
 *
 * <pre>
 * caffeine.cache = "maximumSize=10"
 * </pre>
 *
 * <h3>multiple caches</h3>
 * <p>
 * Just add entries to: <code>caffeine.</code>, like:
 * </p>
 *
 * <pre>
 * # default cache (don't need a name on require calls)
 * caffeine.cache = "maximumSize=10"
 *
 * caffeine.cache1 = "maximumSize=1"
 *
 * caffeine.cacheX = "maximumSize=100"
 * </pre>
 *
 * <pre>
 * {
 *   get("/", req {@literal ->} {
 *     Cache defcache = req.require(Cache.class);
 *
 *     Cache cache1 = req.require("cache1", Cache.class);
 *
 *     Cache cacheX = req.require("cacheX", Cache.class);
 *   });
 * }
 * </pre>
 *
 * <h3>type-safe caches</h3>
 * <p>
 * Type safe caches are provided by calling and creating a new {@link CaffeineCache} subclass:
 * </p>
 *
 * <pre>
 * {
 *   // please notes the {} at the line of the next line
 *   use(new CaffeineCache&lt;Integer, String&gt;() {});
 * }
 * </pre>
 *
 * <p>
 * Later, you can inject this cache in a type-safe manner:
 * </p>
 *
 * <pre>
 * &#64;Inject
 * public MyService(Cache&lt;Integer, String&gt; cache) {
 *  ...
 * }
 * </pre>
 *
 * <h2>session store</h2>
 * <p>
 * This module comes with a {@link Session.Store} implementation. In order to use it you need to
 * define a cache named <code>session</code> in your <code>application.conf</code> file:
 * </p>
 *
 * <pre>
 * caffeine.session = "maximumSize=10"
 * </pre>
 *
 * And set the {@link CaffeineSessionStore}:
 *
 * <pre>
 * {
 *   session(CaffeineSessionStore.class);
 * }
 * </pre>
 *
 * You can access to the ```session``` via name:
 *
 * <pre>
 * {
 *   get("/", req {@literal} {
 *     Cache cache = req.require("session", Cache.class);
 *   });
 * }
 * </pre>
 *
 * @author edgar
 * @author ben manes
 * @since 1.0.0.Final
 * @param <K> Cache key.
 * @param <V> Cache value.
 * @see CaffeineSessionStore
 */
public abstract class CaffeineCache<K, V> implements Jooby.Module {

  public interface Callback<K, V, C extends Cache<K, V>> {
    C apply(String name, Caffeine<K, V> builder);
  }

  public interface AsyncCallback<K, V> {
    AsyncLoadingCache<K, V> apply(String name, Caffeine<K, V> builder);
  }

  private static final String DEF = "caffeine";

  @SuppressWarnings("rawtypes")
  private BiFunction<String, Caffeine, Object> callback = (n, b) -> b.build();

  private final String name;

  /**
   * Creates a new {@link CaffeineCache} using the provided namespace.
   *
   * @param name Cache namespace.
   */
  public CaffeineCache(final String name) {
    this.name = name;
  }

  /**
   * Creates a new {@link CaffeineCache} using <code>caffeine</code> as cache namespace.
   */
  public CaffeineCache() {
    this(DEF);
  }

  /**
   * Creates a new {@link CaffeineCache} with {@link String} and {@link Object} types as key/value.
   *
   * @return A new {@link CaffeineCache}.
   */
  public static CaffeineCache<String, Object> newCache() {
    return new CaffeineCache<String, Object>() {
    };
  }

  /**
   * Configure a cache builder and creates a new {@link Cache}.
   *
   * @param configurer Configurer callback.
   * @return This instance.
   */
  public CaffeineCache<K, V> doWith(final Callback<K, V, Cache<K, V>> configurer) {
    requireNonNull(configurer, "Configurer callback is required.");
    this.callback = configurer::apply;
    return this;
  }

  /**
   * Configure a cache builder and creates a new {@link AsyncLoadingCache}.
   *
   * @param configurer Configurer callback.
   * @return This instance.
   */
  public CaffeineCache<K, V> doWithAsync(final AsyncCallback<K, V> configurer) {
    requireNonNull(configurer, "Configurer callback is required.");
    this.callback = configurer::apply;
    return this;
  }

  @SuppressWarnings({"unchecked", "rawtypes" })
  @Override
  public void configure(final Env env, final Config conf, final Binder binder) {
    Config gconf = conf.hasPath(name) ? conf : ConfigFactory.empty();

    gconf = gconf.withFallback(
        ConfigFactory.empty().withValue("caffeine.cache", ConfigValueFactory.fromAnyRef("")));

    gconf.getObject(name).unwrapped().forEach((name, spec) -> {
      Caffeine<K, V> cb = (Caffeine<K, V>) Caffeine.from(toSpec(spec));

      Object cache = callback.apply(name, cb);

      List<TypeLiteral> types = cacheType(name, cache, getClass().getGenericSuperclass());

      types.forEach(type -> {
        binder.bind(Key.get(type, Names.named(name)))
            .toInstance(cache);
        if (name.equals("cache")) {
          // unnamed cache for default cache
          binder.bind(type).toInstance(cache);
          // raw type for default cache
          binder.bind(type.getRawType()).toInstance(cache);
        }
        if (name.equals("session")) {
          binder.bind(Key.get(type, Names.named(name))).toInstance(cache);
          binder.bind(Key.get(type.getRawType(), Names.named(name))).toInstance(cache);
        }
      });
    });
  }

  @SuppressWarnings("rawtypes")
  private static List<TypeLiteral> cacheType(final String name, final Object cache,
      final Type superclass) {
    Class[] ctypes;
    if (cache instanceof AsyncLoadingCache) {
      ctypes = new Class[]{AsyncLoadingCache.class };
    } else if (cache instanceof LoadingCache) {
      ctypes = new Class[]{LoadingCache.class, Cache.class };
    } else {
      ctypes = new Class[]{Cache.class };
    }
    List<TypeLiteral> result = new ArrayList<>(ctypes.length);
    for (Class ctype : ctypes) {
      if (name.equals("session")) {
        result.add(TypeLiteral.get(Types.newParameterizedType(ctype, String.class, Session.class)));
      } else {
        result.add(TypeLiteral.get(Types.newParameterizedType(ctype, types(superclass))));
      }
    }
    return result;
  }

  @SuppressWarnings("unchecked")
  private String toSpec(final Object spec) {
    if (spec instanceof Map) {
      Map<String, Object> m = (Map<String, Object>) spec;
      return m.entrySet().stream()
          .map(e -> e.getKey() + "=" + e.getValue())
          .collect(Collectors.joining(","));
    }
    return spec.toString();
  }

  private static Type[] types(final Type superclass) {
    Type key = String.class;
    Type value = Object.class;
    if (superclass instanceof ParameterizedType) {
      ParameterizedType parameterized = (ParameterizedType) superclass;
      Type[] args = parameterized.getActualTypeArguments();
      key = args[0];
      value = args[1];
    }
    return new Type[]{key, value };
  }

}
