/**
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "{}"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright 2014 Edgar Espina
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package org.jooby.internal.jetty;

import com.google.common.primitives.Primitives;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigException;
import org.eclipse.jetty.alpn.server.ALPNServerConnectionFactory;
import org.eclipse.jetty.http2.server.HTTP2CServerConnectionFactory;
import org.eclipse.jetty.http2.server.HTTP2ServerConnectionFactory;
import org.eclipse.jetty.server.HttpConfiguration;
import org.eclipse.jetty.server.HttpConnectionFactory;
import org.eclipse.jetty.server.SecureRequestCustomizer;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.ServerConnector;
import org.eclipse.jetty.server.SslConnectionFactory;
import org.eclipse.jetty.server.handler.ContextHandler;
import org.eclipse.jetty.util.DecoratedObjectFactory;
import org.eclipse.jetty.util.ssl.SslContextFactory;
import org.eclipse.jetty.util.thread.QueuedThreadPool;
import org.eclipse.jetty.websocket.api.WebSocketBehavior;
import org.eclipse.jetty.websocket.api.WebSocketPolicy;
import org.eclipse.jetty.websocket.server.WebSocketServerFactory;
import org.jooby.funzy.Try;
import org.jooby.spi.HttpHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.inject.Inject;
import javax.inject.Provider;
import javax.net.ssl.SSLContext;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

public class JettyServer implements org.jooby.spi.Server {

  private static final String H2 = "h2";
  private static final String H2_17 = "h2-17";
  private static final String HTTP_1_1 = "http/1.1";

  private static final String JETTY_HTTP = "jetty.http";
  private static final String CONNECTOR = "connector";

  /** The logging system. */
  private final Logger log = LoggerFactory.getLogger(org.jooby.spi.Server.class);

  private Server server;

  @Inject
  public JettyServer(final HttpHandler handler, final Config conf,
      final Provider<SSLContext> sslCtx) {
    this.server = server(handler, conf, sslCtx);
  }

  private Server server(final HttpHandler handler, final Config conf,
      final Provider<SSLContext> sslCtx) {
    System.setProperty("org.eclipse.jetty.util.UrlEncoded.charset",
        conf.getString("jetty.url.charset"));

    System.setProperty("org.eclipse.jetty.server.Request.maxFormContentSize",
        conf.getBytes("server.http.MaxRequestSize").toString());

    QueuedThreadPool pool = conf(new QueuedThreadPool(), conf.getConfig("jetty.threads"),
        "jetty.threads");

    Server server = new Server(pool);
    server.setStopAtShutdown(false);

    // HTTP connector
    boolean http2 = conf.getBoolean("server.http2.enabled");

    ServerConnector http = http(server, conf.getConfig(JETTY_HTTP), JETTY_HTTP, http2);
    http.setPort(conf.getInt("application.port"));
    http.setHost(conf.getString("application.host"));

    if (conf.hasPath("application.securePort")) {

      ServerConnector https = https(server, conf.getConfig(JETTY_HTTP), JETTY_HTTP,
          sslCtx.get(), http2);
      https.setPort(conf.getInt("application.securePort"));

      server.addConnector(https);
    }

    server.addConnector(http);

    ContextHandler sch = new ContextHandler();
    sch.setAttribute(DecoratedObjectFactory.ATTR, new DecoratedObjectFactory());

    WebSocketPolicy wsConfig = conf(new WebSocketPolicy(WebSocketBehavior.SERVER),
        conf.getConfig("jetty.ws"), "jetty.ws");
    WebSocketServerFactory webSocketServerFactory = new WebSocketServerFactory(
        sch.getServletContext(), wsConfig);
    webSocketServerFactory.setCreator((req, rsp) -> {
      JettyWebSocket ws = new JettyWebSocket();
      req.getHttpServletRequest().setAttribute(JettyWebSocket.class.getName(), ws);
      return ws;
    });

    // always '/' context path is internally handle by jooby
    sch.setContextPath("/");
    sch.setHandler(new JettyHandler(handler, webSocketServerFactory, conf
        .getString("application.tmpdir"), conf.getBytes("jetty.FileSizeThreshold").intValue()));

    server.setHandler(sch);

    return server;
  }

  private ServerConnector http(final Server server, final Config conf, final String path,
      final boolean http2) {
    HttpConfiguration httpConfig = conf(new HttpConfiguration(), conf.withoutPath(CONNECTOR),
        path);

    ServerConnector connector;
    if (http2) {
      connector = new ServerConnector(server, new HttpConnectionFactory(httpConfig),
          new HTTP2CServerConnectionFactory(httpConfig));
    } else {
      connector = new ServerConnector(server, new HttpConnectionFactory(httpConfig));
    }

    return conf(connector, conf.getConfig(CONNECTOR), path + "." + CONNECTOR);
  }

  private ServerConnector https(final Server server, final Config conf, final String path,
      final SSLContext sslContext, final boolean http2) {

    HttpConfiguration httpConf = conf(new HttpConfiguration(), conf.withoutPath(CONNECTOR),
        path);

    SslContextFactory sslContextFactory = new SslContextFactory();
    sslContextFactory.setSslContext(sslContext);
    sslContextFactory.setIncludeProtocols("TLSv1.2");
    sslContextFactory.setIncludeCipherSuites("TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256");

    HttpConfiguration httpsConf = new HttpConfiguration(httpConf);
    httpsConf.addCustomizer(new SecureRequestCustomizer());

    HttpConnectionFactory https11 = new HttpConnectionFactory(httpsConf);

    if (http2) {
      ALPNServerConnectionFactory alpn = new ALPNServerConnectionFactory(H2, H2_17, HTTP_1_1);
      alpn.setDefaultProtocol(HTTP_1_1);

      HTTP2ServerConnectionFactory https2 = new HTTP2ServerConnectionFactory(httpsConf);

      ServerConnector connector = new ServerConnector(server,
          new SslConnectionFactory(sslContextFactory, "alpn"), alpn, https2, https11);

      return conf(connector, conf.getConfig(CONNECTOR), path + ".connector");
    } else {

      ServerConnector connector = new ServerConnector(server,
          new SslConnectionFactory(sslContextFactory, HTTP_1_1), https11);

      return conf(connector, conf.getConfig(CONNECTOR), path + ".connector");
    }
  }

  @Override
  public void start() throws Exception {
    server.start();
  }

  @Override
  public void join() throws InterruptedException {
    server.join();
  }

  @Override
  public void stop() throws Exception {
    server.stop();
  }

  @Override
  public Optional<Executor> executor() {
    return Optional.ofNullable(server.getThreadPool());
  }

  private void tryOption(final Object source, final Config config, final Method option) {
    Try.run(() -> {
      String optionName = option.getName().replace("set", "");
      Object optionValue = config.getAnyRef(optionName);
      Class<?> optionType = Primitives.wrap(option.getParameterTypes()[0]);
      if (Number.class.isAssignableFrom(optionType) && optionValue instanceof String) {
        // either a byte or time unit
        try {
          optionValue = config.getBytes(optionName);
        } catch (ConfigException.BadValue ex) {
          optionValue = config.getDuration(optionName, TimeUnit.MILLISECONDS);
        }
        if (optionType == Integer.class) {
          // to int
          optionValue = ((Number) optionValue).intValue();
        }
      }
      log.debug("{}.{}({})", source.getClass().getSimpleName(), option.getName(), optionValue);
      option.invoke(source, optionValue);
    }).unwrap(InvocationTargetException.class)
        .throwException();
  }

  private <T> T conf(final T source, final Config config, final String path) {
    Map<String, Method> methods = Arrays.stream(source.getClass().getMethods())
        .filter(m -> m.getName().startsWith("set") && m.getParameterCount() == 1)
        .collect(Collectors.toMap(Method::getName, Function.<Method>identity()));

    config.entrySet().forEach(entry -> {
      String key = "set" + entry.getKey();
      Method method = methods.get(key);
      if (method != null) {
        tryOption(source, config, method);
      } else {
        log.error("Unknown option: {}.{} for: {}", path, key, source.getClass().getName());
      }
    });

    return source;
  }

}
