/**
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "{}"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright 2014 Edgar Espina
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package org.jooby.micrometer;

import com.google.inject.Binder;
import com.netflix.spectator.atlas.AtlasConfig;
import com.typesafe.config.Config;
import io.micrometer.atlas.AtlasMeterRegistry;
import io.micrometer.core.instrument.Clock;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.composite.CompositeMeterRegistry;
import io.micrometer.core.instrument.dropwizard.DropwizardConfig;
import io.micrometer.core.instrument.dropwizard.DropwizardMeterRegistry;
import io.micrometer.core.instrument.simple.SimpleConfig;
import io.micrometer.core.instrument.simple.SimpleMeterRegistry;
import io.micrometer.datadog.DatadogConfig;
import io.micrometer.datadog.DatadogMeterRegistry;
import io.micrometer.ganglia.GangliaConfig;
import io.micrometer.ganglia.GangliaMeterRegistry;
import io.micrometer.graphite.GraphiteConfig;
import io.micrometer.graphite.GraphiteMeterRegistry;
import io.micrometer.influx.InfluxConfig;
import io.micrometer.influx.InfluxMeterRegistry;
import io.micrometer.jmx.JmxConfig;
import io.micrometer.jmx.JmxMeterRegistry;
import io.micrometer.newrelic.NewRelicConfig;
import io.micrometer.newrelic.NewRelicMeterRegistry;
import io.micrometer.prometheus.PrometheusConfig;
import io.micrometer.prometheus.PrometheusMeterRegistry;
import io.micrometer.signalfx.SignalFxConfig;
import io.micrometer.signalfx.SignalFxMeterRegistry;
import io.micrometer.statsd.StatsdConfig;
import io.micrometer.statsd.StatsdMeterRegistry;
import io.micrometer.wavefront.WavefrontConfig;
import io.micrometer.wavefront.WavefrontMeterRegistry;
import org.jooby.Env;
import org.jooby.Jooby;

import javax.annotation.Nonnull;
import java.util.ArrayList;
import java.util.List;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;

/**
 * <h1>micrometer</h1>
 * <p>
 *   <a href="https://micrometer.io/">Micrometer</a> provides a simple facade over the
 *   instrumentation clients for the most popular monitoring systems, allowing you to instrument
 *   your JVM-based application code without vendor lock-in. Think SLF4J, but for metrics.
 * </p>
 *
 * <h2>exposes</h2>
 * <ul>
 *   <li>One or more <code>MeterRegistry</code></li>
 * </ul>
 * <h2>usage</h2>
 *
 * <pre>
 * {@code
 * {
 *   use(new Micrometer());
 *
 *   // Timer example:
 *   use("*", (req, rsp, chain) -> {
 *     MeterRegistry registry = require(MeterRegistry.class);
 *     Timer timer = registry.timer("http.server.requests");
 *     timer.record(() -> chain.next(req, rsp));
 *   });
 * }
 * }
 * </pre>
 *
 * <h2>monitoring systems</h2>
 * <p>
 *   It is possible to attach one or more monitoring system. Here are some examples:
 * </p>
 *
 * <pre>
 * {@code
 * {
 *   use(new Micrometer()
 *     .atlas(conf -> {
 *       return new AtlasMeterRegistry(conf);
 *     })
 *     .prometheus(PrometheusMeterRegistry::new)
 *     // etc...
 *   );
 * }
 * }
 * </pre>
 *
 * <p>
 *   Jooby creates a <code>MeterRegistryConfig</code> for every available monitoring system. You
 *   control configuration properties via <code>.conf</code> file:
 * </p>
 *
 * application.conf
 * <pre>
 * {@code
 *   micrometer {
 *     atlas {
 *       uri: "http://localhost:7101/api/v1/publish"
 *     }
 *     prometheus {
 *       step: "PT1M"
 *     }
 *   }
 * }
 * </pre>
 *
 * <p>
 *   <a href="https://prometheus.io/">Prometheus</a> expects to scrape or poll individual app
 *   instances for metrics. Jooby provides a ready to use prometheus handler:
 * </p>
 *
 * <pre>
 * {@code
 *
 * import org.jooby.micrometer.PrometheusHandler;
 *
 * {
 *   use(new Micrometer()
 *      .prometheus(PrometheusMeterRegistry::new)
 *   );
 *
 *   get("/metrics", new PrometheusHandler());
 * }
 * }
 * </pre>
 *
 * <h2>timed annotation</h2>
 * <p>
 *   Jooby supports the <code>io.micrometer.core.annotation.Timed</code> annotation for MVC routes:
 * </p>
 *
 * <pre>
 * &#064;Path("/controller")
 * class Controller {
 *
 *   &#064;Timed("people.all")
 *   public People list() {
 *     ...
 *   }
 * }
 * </pre>
 *
 * App:
 *
 * <pre>
 * {@code
 *
 * import org.jooby.micrometer.TimedHandler;
 *
 * {
 *   use(new Micrometer());
 *
 *   use("*", new TimedHandler());
 * }
 * }
 * </pre>
 *
 * <p>
 *   The <code>TimedHandler</code> record all the controller methods with
 *   <code>io.micrometer.core.annotation.Timed</code> annotation.
 * </p>
 *
 * <h2>advanced options</h2>
 * <p>
 *   Advanced options are available via <code>doWith</code> method:
 * </p>
 *
 * <pre>
 * {@code
 * {
 *   use(new Micrometer()
 *     .doWith(registry -> {
 *        //work with registry
 *     })
 *   );
 * }
 * }
 * </pre>
 *
 * <p>
 *   That's all! Happy coding!!
 * </p>
 *
 * @author edgar
 * @since 1.5.0
 */
public class Micrometer implements Jooby.Module {

  private BiConsumer<CompositeMeterRegistry, Config> configurer;
  private Function simple;
  private Function atlas;
  private Function datadog;
  private Function ganglia;
  private Function graphite;
  private Function influx;
  private Function jmx;
  private Function newrelic;
  private Function prometheus;
  private Function signalfx;
  private Function statsd;
  private Function wavefront;
  private Function dropwizard;

  /**
   * Advanced configuration of main registry.
   *
   * @param configurer Configurer callback.
   * @return This module.
   */
  public Micrometer doWith(@Nonnull final BiConsumer<CompositeMeterRegistry, Config> configurer) {
    this.configurer = configurer;
    return this;
  }

  /**
   * Advanced configuration of main registry.
   *
   * @param configurer Configurer callback.
   * @return This module.
   */
  public Micrometer doWith(@Nonnull final Consumer<CompositeMeterRegistry> configurer) {
    return doWith((registry, conf) -> configurer.accept(registry));
  }

  /**
   * Add a simple registry.
   *
   * @param provider Registry provider.
   * @return This module.
   */
  public Micrometer simple(@Nonnull Function<SimpleConfig, SimpleMeterRegistry> provider) {
    this.simple = provider;
    return this;
  }

  /**
   * Add an atlas registry.
   *
   * @param provider Registry provider.
   * @return This module.
   */
  public Micrometer atlas(@Nonnull Function<AtlasConfig, AtlasMeterRegistry> provider) {
    this.atlas = provider;
    return this;
  }

  /**
   * Add a datadog registry.
   *
   * @param provider Registry provider.
   * @return This module.
   */
  public Micrometer datadog(@Nonnull Function<DatadogConfig, DatadogMeterRegistry> provider) {
    this.datadog = provider;
    return this;
  }

  /**
   * Add a dropwizard registry.
   *
   * @param provider Registry provider.
   * @return This module.
   */
  public Micrometer dropwizard(
      @Nonnull Function<DropwizardConfig, DropwizardMeterRegistry> provider) {
    this.dropwizard = provider;
    return this;
  }

  /**
   * Add a ganglia registry.
   *
   * @param provider Registry provider.
   * @return This module.
   */
  public Micrometer ganglia(@Nonnull Function<GangliaConfig, GangliaMeterRegistry> provider) {
    this.ganglia = provider;
    return this;
  }

  /**
   * Add a graphite registry.
   *
   * @param provider Registry provider.
   * @return This module.
   */
  public Micrometer graphite(@Nonnull Function<GraphiteConfig, GraphiteMeterRegistry> provider) {
    this.graphite = provider;
    return this;
  }

  /**
   * Add an influx registry.
   *
   * @param provider Registry provider.
   * @return This module.
   */
  public Micrometer influx(@Nonnull Function<InfluxConfig, InfluxMeterRegistry> provider) {
    this.influx = provider;
    return this;
  }

  /**
   * Add a jmx registry.
   *
   * @param provider Registry provider.
   * @return This module.
   */
  public Micrometer jmx(@Nonnull Function<JmxConfig, JmxMeterRegistry> provider) {
    this.jmx = provider;
    return this;
  }

  /**
   * Add a new-relic registry.
   *
   * @param provider Registry provider.
   * @return This module.
   */
  public Micrometer newrelic(@Nonnull Function<NewRelicConfig, NewRelicMeterRegistry> provider) {
    this.newrelic = provider;
    return this;
  }

  /**
   * Add a prometheus registry.
   *
   * @param provider Registry provider.
   * @return This module.
   */
  public Micrometer prometheus(
      @Nonnull Function<PrometheusConfig, PrometheusMeterRegistry> provider) {
    this.prometheus = provider;
    return this;
  }

  /**
   * Add a statsD registry.
   *
   * @param provider Registry provider.
   * @return This module.
   */
  public Micrometer statsd(@Nonnull Function<StatsdConfig, StatsdMeterRegistry> provider) {
    this.statsd = provider;
    return this;
  }

  /**
   * Add a signalfx registry.
   *
   * @param provider Registry provider.
   * @return This module.
   */
  public Micrometer signalfx(@Nonnull Function<SignalFxConfig, SignalFxMeterRegistry> provider) {
    this.signalfx = provider;
    return this;
  }

  /**
   * Add a wavefront registry.
   *
   * @param provider Registry provider.
   * @return This module.
   */
  public Micrometer wavefront(@Nonnull Function<WavefrontConfig, WavefrontMeterRegistry> provider) {
    this.wavefront = provider;
    return this;
  }

  @Override public void configure(Env env, Config conf, Binder binder) {

    CompositeMeterRegistry registry = new CompositeMeterRegistry();
    if (configurer != null) {
      configurer.accept(registry, conf);
    }

    List<MeterRegistry> registries = new ArrayList<>();

    if (simple != null) {
      registries.add((MeterRegistry) simple.apply(simple(conf)));
    }

    if (atlas != null) {
      registries.add((MeterRegistry) atlas.apply(atlas(conf)));
    }

    if (datadog != null) {
      registries.add((MeterRegistry) datadog.apply(datadog(conf)));
    }

    if (dropwizard != null) {
      registries.add((MeterRegistry) dropwizard.apply(dropwizard(conf)));
    }

    if (ganglia != null) {
      registries.add((MeterRegistry) ganglia.apply(ganglia(conf)));
    }

    if (graphite != null) {
      registries.add((MeterRegistry) graphite.apply(graphite(conf)));
    }

    if (influx != null) {
      registries.add((MeterRegistry) influx.apply(influx(conf)));
    }

    if (jmx != null) {
      registries.add((MeterRegistry) jmx.apply(jmx(conf)));
    }

    if (newrelic != null) {
      registries.add((MeterRegistry) newrelic.apply(newrelic(conf)));
    }

    if (prometheus != null) {
      registries.add((MeterRegistry) this.prometheus.apply(prometheus(conf)));
    }

    if (signalfx != null) {
      registries.add((MeterRegistry) signalfx.apply(signalfx(conf)));
    }

    if (statsd != null) {
      registries.add((MeterRegistry) statsd.apply(statsd(conf)));
    }

    if (wavefront != null) {
      registries.add((MeterRegistry) wavefront.apply(wavefront(conf)));
    }

    if (registries.size() + registry.getRegistries().size() == 0) {
      SimpleMeterRegistry r = new SimpleMeterRegistry(simple(conf), Clock.SYSTEM);
      registries.add(r);
    }

    registries.forEach(it -> {
      registry.add(it);

      Class type = it.getClass();
      binder.bind(type).toInstance(it);
    });

    binder.bind(MeterRegistry.class).toInstance(registry);
    binder.bind(CompositeMeterRegistry.class).toInstance(registry);

    env.onStop(registry::close);
  }

  static SimpleConfig simple(Config conf) {
    return key -> value(conf, "simple.", key);
  }

  static DropwizardConfig dropwizard(Config conf) {
    return new DropwizardConfig() {
      @Override public String prefix() {
        return "dropwizard";
      }

      @Override public String get(String key) {
        return value(conf, "dropwizard.", key);
      }
    };
  }

  static AtlasConfig atlas(Config conf) {
    return key -> value(conf, "atlas.", key);
  }

  static DatadogConfig datadog(Config conf) {
    return key -> value(conf, "datadog.", key);
  }

  static GangliaConfig ganglia(Config conf) {
    return key -> value(conf, "ganglia.", key);
  }

  static GraphiteConfig graphite(Config conf) {
    return key -> value(conf, "graphite.", key);
  }

  static InfluxConfig influx(Config conf) {
    return key -> value(conf, "influx.", key);
  }

  static JmxConfig jmx(Config conf) {
    return key -> value(conf, "jmx.", key);
  }

  static NewRelicConfig newrelic(Config conf) {
    return key -> value(conf, "newrelic.", key);
  }

  static PrometheusConfig prometheus(Config conf) {
    return key -> value(conf, "prometheus.", key);
  }

  static SignalFxConfig signalfx(Config conf) {
    return key -> value(conf, "signalfx.", key);
  }

  static StatsdConfig statsd(Config conf) {
    return key -> value(conf, "statsd.", key);
  }

  static WavefrontConfig wavefront(Config conf) {
    return key -> value(conf, "wavefront.", key);
  }

  static String value(Config conf, String prefix, String key) {
    String path = key;
    if (!path.startsWith(prefix)) {
      path = prefix + path;
    }
    path = "micrometer." + path;
    return conf.hasPath(path) ? conf.getString(path) : null;
  }

}
