/**
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "{}"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright 2014 Edgar Espina
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package org.jooby.pac4j;

import com.google.common.base.Strings;
import com.google.inject.Binder;
import com.google.inject.TypeLiteral;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import static java.util.Objects.requireNonNull;
import org.jooby.Env;
import org.jooby.Err;
import org.jooby.Jooby;
import org.jooby.Registry;
import org.jooby.Request;
import org.jooby.Route;
import org.jooby.Router;
import org.jooby.Status;
import org.jooby.funzy.Throwing;
import org.jooby.internal.pac4j2.Pac4jActionAdapter;
import org.jooby.internal.pac4j2.Pac4jAuthorizer;
import org.jooby.internal.pac4j2.Pac4jCallback;
import org.jooby.internal.pac4j2.Pac4jClientType;
import org.jooby.internal.pac4j2.Pac4jContext;
import org.jooby.internal.pac4j2.Pac4jLoginForm;
import org.jooby.internal.pac4j2.Pac4jLogout;
import org.jooby.internal.pac4j2.Pac4jProfileManager;
import org.jooby.internal.pac4j2.Pac4jSecurityFilter;
import org.jooby.internal.pac4j2.Pac4jSessionStore;
import org.pac4j.core.authorization.authorizer.Authorizer;
import org.pac4j.core.client.Client;
import org.pac4j.core.client.Clients;
import org.pac4j.core.context.WebContext;
import org.pac4j.core.context.session.SessionStore;
import org.pac4j.core.credentials.Credentials;
import org.pac4j.core.engine.CallbackLogic;
import org.pac4j.core.engine.DefaultCallbackLogic;
import org.pac4j.core.engine.DefaultLogoutLogic;
import org.pac4j.core.engine.DefaultSecurityLogic;
import org.pac4j.core.engine.LogoutLogic;
import org.pac4j.core.engine.SecurityLogic;
import org.pac4j.core.profile.CommonProfile;
import org.pac4j.core.profile.ProfileManager;
import org.pac4j.core.profile.UserProfile;
import org.pac4j.http.client.indirect.FormClient;
import org.pac4j.http.credentials.authenticator.test.SimpleTestUsernamePasswordAuthenticator;

import javax.inject.Provider;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * <h1>pac4j module</h1>
 * <p>
 * Authentication module via: <a href="https://github.com/pac4j/pac4j">pac4j</a>.
 * </p>
 *
 * <h2>usage</h2>
 *
 * <p>
 *   Display a basic login-form and restrict access to all the routes defined after the
 *   {@link Pac4j} module:
 * </p>
 *
 * <pre>{@code
 * {
 *   get("/public", () {@literal ->} {
 *     ...
 *   });
 *
 *   use(new Pac4j());
 *
 *   get("/private", () {@literal ->} {
 *     ...
 *   });
 *
 * }
 * }</pre>
 *
 * <h2>clients</h2>
 * <p>
 *   A <a href="http://www.pac4j.org/docs/clients.html">Client</a> represents an authentication
 *   mechanism. It performs the login process and returns (if successful) a user profile
 * </p>
 *
 * <p>
 *   Clients are configured at bootstrap time using the {@link Pac4j} DSL:
 * </p>
 *
 * <pre>{@code
 * {
 *   use(new Pac4j()
 *     .client(conf -> {
 *       return new FacebookClient(conf.getString("fb.key"), conf.getString("fb.secret"));
 *     })
 *   );
 * }
 * }</pre>
 *
 * <p>
 *   You can chain calls to add multiple clients:
 * </p>
 *
 * <pre>{@code
 * {
 *   use(new Pac4j()
 *     .client(conf -> {
 *       return new FormClient("/login", new SimpleTestSimpleTestUsernamePasswordAuthenticator());
 *     })
 *     .client(conf -> {
 *       return new FacebookClient(conf.getString("fb.key"), conf.getString("fb.secret"));
 *     })
 *     .client(conf -> {
 *       return new TwitterClient(conf.getString("twitter.key"), conf.getString("twitter.secret"));
 *     })
 *   );
 * }
 * }</pre>
 *
 * <h2>protecting urls</h2>
 * <p>
 *   By default Pac4j restrict access to all the routes defined after the Pac4j module. You can
 *   specify what url must be protected using a path pattern:
 * </p>
 *
 * <pre>{@code
 * {
 *   use(new Pac4j()
 *     .client("/admin/**", conf -> {
 *       return new FormClient("/login", new SimpleTestSimpleTestUsernamePasswordAuthenticator());
 *    }));
 * }
 * }</pre>
 *
 * <p>Now all the routes under <code>/admin</code> are protected by Pac4j.</p>
 *
 * <h2>user profile</h2>
 * <p>
 *   After login the user profile (current logged user) is accessible via <code>require</code> calls:
 * </p>
 *
 * <pre>{@code
 * {
 *   use(new Pac4j().form());
 *
 *   get("/profile", () -> {
 *     CommonProfile profile = require(CommonProfile.class);
 *     ...
 *   });
 * }
 * }</pre>
 *
 * <p>Access to specific profile type depends on the authentication client:</p>
 *
 * <pre>{@code
 * {
 *   use(new Pac4j()
 *     .client(conf -> {
 *       return new FacebookClient(conf.getString("fb.key"), conf.getString("fb.secret"));
 *     })
 *   );
 *
 *   get("/profile", () -> {
 *     FacebookProfile profile = require(FacebookProfile.class);
 *     ...
 *   });
 * }
 * }</pre>
 *
 * <p>Or if you prefer the pac4j API:</p>
 *
 * <pre>{@code
 * {
 *   use(new Pac4j()
 *     .client(conf -> {
 *       return new FacebookClient(conf.getString("fb.key"), conf.getString("fb.secret"));
 *     })
 *   );
 *
 *   get("/profile", req -> {
 *     ProfileManager pm = require(ProfileManager.class);
 *     List<CommonProfile> profiles = pm.getAll(req.ifSession().isPresent());
 *     ...
 *   });
 * }
 * }</pre>
 *
 * <h2>authorizer</h2>
 * <p>
 *   Authorizers are provided via <code>client</code> DSL. You can provider an instance of an
 *   auhtorizer or class reference to an authorizer.
 * </p>
 *
 * <pre>{@code
 * {
 *   use(new Pac4j()
 *     .client("*", MyAuthorizer.class, conf -> {
 *       return new FacebookClient(conf.getString("fb.key"), conf.getString("fb.secret"));
 *     })
 *   );
 * }
 * }</pre>
 *
 * <p>Here <code>MyAuthorizer</code> will be provisioned by Guice.</p>
 *
 * <h2>advanced usage</h2>
 * <p>
 *   For advanced usage is available via {@link Pac4j#doWith(Consumer)} method:
 * </p>
 *
 * <pre>{@code
 * {
 *   use(new Pac4j()
 *     .doWith(pac4j -> {
 *       pac4j.setSecurityLogic(...);
 *       pac4j.setHttpActionAdapter(...);
 *     })
 *   );
 * }
 * }</pre>
 *
 * @author edgar
 * @since 1.3.0
 */
public class Pac4j implements Jooby.Module {

  private static class ClientConfig {
    private String pattern;

    private String authorizer;

    private Client client;

    public ClientConfig(String pattern, String authorizer, Client client) {
      this.pattern = pattern;
      this.authorizer = authorizer;
      this.client = client;
    }
  }

  static final TypeLiteral<SessionStore<WebContext>> SSTORE = new TypeLiteral<SessionStore<WebContext>>() {
  };

  private BiConsumer<org.pac4j.core.config.Config, Config> configurer;

  private final org.pac4j.core.config.Config pac4j = new org.pac4j.core.config.Config();

  private List<Throwing.Function3<Config, Binder, AtomicReference<Registry>, ClientConfig>> clients = new ArrayList<>();

  private Function<Request, UserProfile> unauthenticated;

  private Set<Object> profileTypes = new HashSet<>();

  private boolean showDevLogin;

  private Boolean multiProfile;

  /**
   * Creates a new Pac4j module.
   */
  public Pac4j() {
  }

  /**
   * Configurer pa4j options, only necessary it you want to provide your own pac4j components.
   *
   * <pre>{@code
   * {
   *   use(new Pac4j()
   *     .doWith(pac4j -> {
   *       pac4j.setSecurityLogic(...);
   *       pac4j.setHttpActionAdapter(...);
   *     })
   *   );
   * }
   * }</pre>
   *
   * @param configurer Configurer callback.
   * @return This module.
   */
  public Pac4j doWith(Consumer<org.pac4j.core.config.Config> configurer) {
    return doWith((pac4j, conf) -> configurer.accept(pac4j));
  }

  /**
   * Configurer pa4j options, only necessary it you want to provide your own pac4j components.
   *
   * <pre>{@code
   * {
   *   use(new Pac4j()
   *     .doWith((pac4j, conf) -> {
   *       pac4j.setSecurityLogic(...);
   *       pac4j.setHttpActionAdapter(...);
   *     })
   *   );
   * }
   * }</pre>
   *
   * @param configurer Configurer callback.
   * @return This module.
   */
  public Pac4j doWith(BiConsumer<org.pac4j.core.config.Config, Config> configurer) {
    this.configurer = configurer;
    return this;
  }

  /**
   * Add a pac4j client and protected all the routes defined after the module:
   *
   * <pre>{@code
   * {
   *   use(new Pac4j()
   *     .client(conf -> {
   *       return new FacebookClient(conf.getString("fb.key"), conf.getString("fb.secret"));
   *     })
   *   );
   *
   *   // protected routes
   * }
   * }</pre>
   *
   * @param client Client provider.
   * @param <C> Credential type.
   * @param <U> User profile type.
   * @return This module.
   */
  public <C extends Credentials, U extends CommonProfile> Pac4j client(
      Function<Config, Client<C, U>> client) {
    return clientInternal("*", client, null);
  }

  /**
   * Add a pac4j client and protected all the routes defined after that matches the pattern:
   *
   * <pre>{@code
   * {
   *   use(new Pac4j()
   *     .client("/admin/**", conf -> {
   *       return new FacebookClient(conf.getString("fb.key"), conf.getString("fb.secret"));
   *     })
   *   );
   *
   *   // all routes at /admin are now protected.
   * }
   * }</pre>
   *
   * @param pattern Pattern to protect.
   * @param client Client provider.
   * @param <C> Credential type.
   * @param <U> User profile type.
   * @return This module.
   */
  public <C extends Credentials, U extends CommonProfile> Pac4j client(String pattern,
      Function<Config, Client<C, U>> client) {
    return clientInternal(pattern, client, null);
  }

  /**
   * Add a pac4j client, protected all the routes defined after that matches the pattern and attach
   * an authorizer:
   *
   * <pre>{@code
   * {
   *   use(new Pac4j()
   *     .client("*", MyAuthorizer.class, conf -> {
   *       return new FacebookClient(conf.getString("fb.key"), conf.getString("fb.secret"));
   *     })
   *   );
   * }
   * }</pre>
   *
   * <p>The authorizer will be provisioned by Guice.</p>
   *
   * @param pattern Pattern to protect.
   * @param authorizer Authorizer.
   * @param client Client provider.
   * @param <C> Credential type.
   * @param <U> User profile type.
   * @return This module.
   */
  public <C extends Credentials, U extends CommonProfile> Pac4j client(String pattern,
      Class<? extends Authorizer> authorizer, Function<Config, Client<C, U>> client) {
    return client(pattern, new Pac4jAuthorizer(authorizer), client);
  }

  /**
   * Add a pac4j client, protected all the routes defined after that matches the pattern and attach
   * an authorizer:
   *
   * <pre>{@code
   * {
   *   use(new Pac4j()
   *     .client("*", new MyAuthorizer(), conf -> {
   *       return new FacebookClient(conf.getString("fb.key"), conf.getString("fb.secret"));
   *     })
   *   );
   * }
   * }</pre>
   *
   * @param pattern Pattern to protect.
   * @param authorizer Authorizer.
   * @param client Client provider.
   * @param <C> Credential type.
   * @param <U> User profile type.
   * @return This module.
   */
  public <C extends Credentials, U extends CommonProfile> Pac4j client(String pattern,
      Authorizer<U> authorizer, Function<Config, Client<C, U>> client) {
    return clientInternal(pattern, client, authorizer);
  }

  /**
   * Set a default action which is execute when no user is logged in.
   *
   * <pre>{@code
   * {
   *   use(new Pac4j()
   *     .unauthenticated(req -> {
   *       UserProfile anonymous = ...
   *       return anonymous;
   *     })
   *   );
   *
   *   get("/", () -> {
   *     // might or might not be anonymous
   *     UserProfile profile = require(UserProfile.class);
   *     return ...;
   *   }
   * }
   * }</pre>
   *
   * The default action throws a <code>403</code> error.
   *
   *
   * @param provider Unauthenticated user provider.
   * @return This module.
   */
  public Pac4j unauthenticated(Function<Request, UserProfile> provider) {
    this.unauthenticated = requireNonNull(provider, "Unauthenticated provider required.");
    return this;
  }

  /**
   * Set a default action which is execute when no user is logged in.
   *
   * <pre>{@code
   * {
   *   use(new Pac4j()
   *     .unauthenticated(() -> {
   *       UserProfile anonymous = ...
   *       return anonymous;
   *     })
   *   );
   *
   *   get("/", () -> {
   *     // might or might not be anonymous
   *     UserProfile profile = require(UserProfile.class);
   *     return ...;
   *   }
   * }
   * }</pre>
   *
   * The default action throws a <code>403</code> error.
   *
   *
   * @param provider Unauthenticated user provider.
   * @return This module.
   */
  public Pac4j unauthenticated(Supplier<UserProfile> provider) {
    requireNonNull(provider, "Unauthenticated provider required.");
    return unauthenticated(req -> provider.get());
  }

  private <C extends Credentials, U extends CommonProfile> Pac4j clientInternal(String pattern,
      Function<Config, Client<C, U>> provider, Authorizer<U> authorizer) {
    clients.add((conf, binder, registry) -> {
      Client<C, U> client = provider.apply(conf);
      String authorizerName = null;
      if (authorizer != null) {
        authorizerName = authorizerName(authorizer);
        pac4j.addAuthorizer(authorizerName, authorizer);
      }
      Pac4jClientType.profileTypes(Pac4jClientType.clientType(client.getClass()), profile -> {
        if (profileTypes.add(profile)) {
          binder
              .bind(profile)
              .toProvider(profileProvider(registry, profile, unauthenticated));
        }
      });
      return new ClientConfig(pattern, authorizerName, client);
    });
    return this;
  }

  /**
   * Set pac4j option multiProfile.
   *
   * @param multiProfile True for multiprofile.
   * @return This module.
   */
  public Pac4j multiProfile(boolean multiProfile) {
    this.multiProfile = multiProfile;
    return this;
  }

  /**
   * Add a simple login form. Useful for development and quick startup.
   *
   * @return This module.
   */
  public Pac4j form() {
    return form("*");
  }

  /**
   * Add a simple login form. Useful for development and quick startup.
   *
   * @param pattern Pattern to protect.ver
   * @return This module.
   */
  public Pac4j form(String pattern) {
    return clientInternal(pattern, conf -> {
      showDevLogin = true;
      return new FormClient("/login", new SimpleTestUsernamePasswordAuthenticator());
    }, null);
  }

  @Override public void configure(Env env, Config conf, Binder binder) throws Throwable {
    String contextPath = conf.getString("application.path");
    String callbackPath = conf.getString("pac4j.callback.path");

    pac4j.setClients(
        new Clients(URI.create(conf.getString("pac4j.callback.url")).normalize().toString(),
            new ArrayList<>()));
    pac4j.setHttpActionAdapter(new Pac4jActionAdapter());

    if (configurer != null) {
      configurer.accept(pac4j, conf);
    }

    /** Pac4j Clients: (external clients via doWith + dsl clients) */
    Clients clients = pac4j.getClients();
    List<Client> clientList = new ArrayList<>(clients.getClients());
    if (clientList.size() == 0 && this.clients.size() == 0) {
      // default form
      form();
    }
    AtomicReference<Registry> registryRef = new AtomicReference<>();
    List<ClientConfig> securityRoutes = this.clients.stream()
        .map(fn -> fn.apply(conf, binder, registryRef))
        .collect(Collectors.toList());
    securityRoutes.forEach(it -> clientList.add(it.client));
    clients.setClients(clientList);

    boolean multiProfile = Optional.ofNullable(this.multiProfile).orElse(clientList.size() > 1);
    Router router = env.router();

    /** Config: */
    binder.bind(org.pac4j.core.config.Config.class).toInstance(pac4j);

    /** WebContext: */
    binder.bind(WebContext.class).to(Pac4jContext.class);

    /** Profile manager: */
    Function<WebContext, ProfileManager> pmf = pac4j.getProfileManagerFactory();
    if (pmf == null) {
      pmf = ProfileManager::new;
      pac4j.setProfileManagerFactory(pmf);
    }
    binder.bind(ProfileManager.class).toProvider(Pac4jProfileManager.class);

    /** Session store: */
    SessionStore<WebContext> sessionStore = pac4j.getSessionStore();
    if (sessionStore == null) {
      binder.bind(SSTORE).to(Pac4jSessionStore.class);
      binder.bind(SessionStore.class).to(Pac4jSessionStore.class);
    } else {
      binder.bind(SSTORE).toInstance(sessionStore);
      binder.bind(SessionStore.class).toInstance(sessionStore);
    }

    /** DEV Login form: */
    if (showDevLogin) {
      router.get("/login",
          new Pac4jLoginForm(Route.normalize(contextPath + callbackPath)))
          .name("pac4j(LoginForm)");
    }

    /** Callback: */
    CallbackLogic callbackLogic = pac4j.getCallbackLogic();
    if (callbackLogic == null) {
      callbackLogic = new DefaultCallbackLogic();
      pac4j.setCallbackLogic(callbackLogic);
    }
    String callbackRedirectTo = Optional
        .ofNullable(Strings.emptyToNull(conf.getString("pac4j.callback.redirectTo")))
        .orElse(contextPath);
    boolean renewSession = conf.getBoolean("pac4j.callback.renewSession");
    List<String> excludePaths = conf.getStringList("pac4j.excludePaths");
    router.use(conf.getString("pac4j.callback.method"), callbackPath,
        new Pac4jCallback(pac4j, callbackRedirectTo, multiProfile, renewSession))
        .excludes(excludePaths)
        .name("pac4j(Callback)");

    /** Security Filters: */
    Set<String> patterns = new LinkedHashSet<>();
    SecurityLogic securityLogic = pac4j.getSecurityLogic();
    if (securityLogic == null) {
      securityLogic = new DefaultSecurityLogic();
      pac4j.setSecurityLogic(securityLogic);
    }
    Map<String, Pac4jSecurityFilter> filters = new LinkedHashMap<>();
    patterns.add(callbackPath);
    securityRoutes.forEach(it -> {
      String pattern = Route.normalize(it.pattern);
      patterns.add(pattern);
      Pac4jSecurityFilter filter = filters.get(pattern);
      String clientName = it.client.getName();
      if (filter == null) {
        filter = new Pac4jSecurityFilter(pac4j, clientName, it.authorizer, null, multiProfile,
            patterns);
        filters.put(pattern, filter);
      } else {
        // multiple clients per pattern
        filter.addClient(clientName);
      }
    });

    filters.forEach((pattern, filter) -> {
      List<String> excludes = new ArrayList<>();
      excludes.addAll(excludePaths);
      excludes.addAll(patterns);
      excludes.remove(pattern);
      excludes.remove("/**");
      router.use(conf.getString("pac4j.securityFilter.method"), pattern, filter)
          .excludes(excludes)
          .name("pac4j(" + filter + ")");
    });

    /** Logout: */
    LogoutLogic logoutLogic = pac4j.getLogoutLogic();
    if (logoutLogic == null) {
      logoutLogic = new DefaultLogoutLogic<>();
      pac4j.setLogoutLogic(logoutLogic);
    }
    router.use(conf.getString("pac4j.logout.method"), conf.getString("pac4j.logout.path"),
        new Pac4jLogout(pac4j,
            conf.getString("pac4j.logout.redirectTo"),
            conf.getString("pac4j.logout.pattern"),
            conf.getBoolean("pac4j.logout.destroySession"),
            conf.getBoolean("pac4j.logout.local"),
            conf.getBoolean("pac4j.logout.central")))
        .name("pac4j(Logout)");

    /** Initialize Guice authorizers: */
    env.onStart(registry -> {
      registryRef.set(registry);
      pac4j.getAuthorizers().values().stream()
          .filter(Pac4jAuthorizer.class::isInstance)
          .map(Pac4jAuthorizer.class::cast)
          .forEach(fwd -> fwd.setRegistry(registry));
    });

    this.profileTypes.clear();
    this.profileTypes = null;

    this.clients.clear();
    this.clients = null;
  }

  @Override public Config config() {
    return ConfigFactory.parseResources(getClass(), "pac4j.conf");
  }

  static String authorizerName(Object authorizer) {
    if (authorizer instanceof Pac4jAuthorizer) {
      return ((Pac4jAuthorizer) authorizer).authorizer.getSimpleName();
    }
    return authorizer.getClass().getSimpleName();
  }

  static Provider profileProvider(AtomicReference<Registry> registry, Class profile,
      Function<Request, UserProfile> unauthenticated) {
    return () -> {
      Request req = registry.get().require(Request.class);
      ProfileManager pm = req.require(ProfileManager.class);
      Object result = pm.getAll(req.ifSession().isPresent()).stream()
          .filter(profile::isInstance)
          .findFirst()
          .orElse(null);
      if (result == null) {
        if (unauthenticated == null) {
          throw new Err(Status.FORBIDDEN, "Not found: " + profile.getSimpleName());
        }
        result = unauthenticated.apply(req);
      }
      return result;
    };
  }
}
