/**
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "{}"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright 2014 Edgar Espina
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package org.jooby.whoops;

import com.google.common.base.Throwables;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.ImmutableMap;
import com.google.common.io.ByteStreams;
import com.google.inject.Binder;
import com.mitchellbosecke.pebble.PebbleEngine;
import com.mitchellbosecke.pebble.loader.ClasspathLoader;
import com.mitchellbosecke.pebble.template.PebbleTemplate;
import com.typesafe.config.Config;
import org.jooby.Env;
import org.jooby.Err;
import org.jooby.Err.Handler;
import org.jooby.Jooby;
import org.jooby.MediaType;
import org.jooby.Route;
import org.jooby.Router;
import org.jooby.funzy.Throwing;
import org.jooby.funzy.Try;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

/**
 * <h1>whoops</h1>
 * <p>
 * Pretty error page that helps you debug your web application.
 * </p>
 * <p>
 * <strong>NOTE</strong>: This module is base on <a href="https://github.com/filp/whoops">whoops</a>
 * and uses the same front end resources.
 * </p>
 *
 * <h2>exports</h2>
 * <ul>
 * <li>A {@link Err.Handler pretty error page}</li>
 * </ul>
 *
 * <h2>usage</h2>
 *
 * <pre>{@code
 * {
 *   use(new Whoops());
 *
 *   get("/", req -> {
 *     throw new IllegalStateException("Something broken!");
 *   });
 * }
 * }</pre>
 *
 * <p>
 * The pretty error page handler is available in development mode:
 * <code>application.env = dev</code>
 * </p>
 *
 * <h2>custom err pages</h2>
 * <p>
 * The pretty error page is implemented via {@link Router#err(Handler)}. You might run into troubles
 * if your application require custom error pages. On those cases you probably won't use this module
 * or if apply one of the following options:
 * </p>
 *
 * <h3>whoops as last err handler</h3>
 * <p>
 * This option is useful if you have custom error pages on some specific exceptions:
 * </p>
 * <pre>{@code
 * {
 *   err(NotFoundException.class, (req, rsp, err) -> {
 *    // custom not found
 *   });
 *
 *   err(AccessDeniedException.class, (req, rsp, err) -> {
 *    // custom access denied
 *   });
 *
 *   // not handled it use whoops
 *   use(new Whoops());
 * }
 * }</pre>
 * <p>
 * Here the custom error page for <code>NotFoundException</code> or
 * <code>AccessDeniedException</code> will be render before the <code>Whoops</code> error handler.
 * </p>
 *
 * <h3>whoops on dev</h3>
 * <p>
 * This options will active <code>Whoops</code> in <code>dev</code> and the custom err pages in
 * <code>prod-like</code> environments:
 * </p>
 *
 * <pre>{@code
 * {
 *   on("dev", () -> {
 *     use(new Whoops());
 *   })
 *   .orElse(() -> {
 *     err((req, rsp, err) -> {
 *       // custom not found
 *     });
 *   });
 * }
 * }</pre>
 *
 * @author edgar
 * @since 1.0.0.CR4
 */
public class Whoops implements Jooby.Module {

  private static final String HANDLER = "org.jooby.internal.HttpHandlerImpl";

  private static final int SAMPLE_SIZE = 10;

  private static final BiFunction<Path, Integer, String> openWith = (p, l) -> "";

  /** The logging system. */
  private final Logger log = LoggerFactory.getLogger(getClass());

  private final int maxFrameSize;

  /**
   * Creates a new {@link Whoops} module.
   *
   * @param maxFrameSize Max number of frame to show in the pretty error page.
   */
  public Whoops(final int maxFrameSize) {
    this.maxFrameSize = maxFrameSize;
  }

  /**
   * Creates a new {@link Whoops} module with max frame size of 8.
   */
  public Whoops() {
    this(8);
  }

  @Override
  public void configure(final Env env, final Config conf, final Binder binder) {
    boolean whoops = conf.hasPath("whoops.enabled")
        ? conf.getBoolean("whoops.enabled")
        : "dev".equals(env.name());
    if (whoops) {
      ClassLoader loader = env.router().getClass().getClassLoader();
      Handler handler = prettyPage(loader, SourceLocator.local(), maxFrameSize, log);
      env.router().err(tryPage(handler, log));
    }
  }

  static Handler tryPage(final Handler handler, final Logger log) {
    return (req, rsp, ex) -> Try.run(() -> handler.handle(req, rsp, ex)).onFailure(cause -> log.debug("execution of pretty err page resulted in exception", cause));
  }

  private static Handler prettyPage(final ClassLoader loader, final SourceLocator locator,
      final int maxStackSize, final Logger log) {
    String css = readString(loader, "css/whoops.base.css");
    String clipboard = readString(loader, "js/clipboard.min.js");
    String js = readString(loader, "js/whoops.base.js");
    String zepto = readString(loader, "js/zepto.min.js");

    ClasspathLoader cpathloader = new ClasspathLoader();
    cpathloader.setPrefix("whoops");
    cpathloader.setSuffix(".html");
    PebbleEngine engine = new PebbleEngine.Builder()
        .loader(cpathloader)
        .templateCache(CacheBuilder.newBuilder().maximumSize(10).build())
        .build();

    /** Lazy compile template and keep it */
    Throwing.Function<String, PebbleTemplate> template = Throwing.
      throwingFunction(engine::getTemplate)
        .memoized();

    return (req, rsp, err) -> {
      // only html, ignore any other request and fallback to default handler
      if (req.accepts(MediaType.html).isPresent()) {
        // is this a real Err? or we just wrap it?
        Throwable cause = Optional.ofNullable(err.getCause()).orElse(err);
        // dump causes as a list
        List<Throwable> causal = Throwables.getCausalChain(cause);

        // get the cause (initial exception)
        Throwable head = causal.get(causal.size() - 1);
        String message = message(head);

        Map<String, Object> envdata = new LinkedHashMap<>();

        envdata.put("response", dump(() -> ImmutableMap.of("status", rsp.status().get())));

        /** route */
        envdata.put("route", dump(() -> {
          Route route = req.route();
          ImmutableMap.Builder<String, Object> map = ImmutableMap.builder();
          return map
              .put("method", route.method())
              .put("path", route.path())
              .put("path vars", route.vars())
              .put("pattern", route.pattern())
              .put("name", route.name())
              .put("attributes", route.attributes()).build();
        }));
        /** request params */
        envdata.put("request params", dump(() -> req.params().toMap()));
        /** request locals */
        envdata.put("request locals", dump(req::attributes));
        /** http headers */
        envdata.put("request headers", dump(req::headers));
        /** session */
        req.ifSession().ifPresent(s -> envdata.put("session", dump(s::attributes)));

        List<Map<String, Object>> frames = causal.stream().filter(it -> it != head)
            .map(it -> frame(loader, locator, it, it.getStackTrace()[0]))
            .collect(Collectors.toList());

        frames.addAll(frames(loader, locator, head));

        // truncate frames
        frames = frames.subList(0, Math.min(maxStackSize, frames.size()));

        Map<String, Object> context = ImmutableMap.<String, Object>builder()
            .put("stylesheet", css)
            .put("zepto", zepto)
            .put("clipboard", clipboard)
            .put("javascript", js)
            .put("chain", causal)
            .put("q", head.getClass().getName() + ": " + message)
            .put("message", message)
            .put("stacktrace", Throwables.getStackTraceAsString(cause))
            .put("frames", frames)
            .put("env", envdata)
            .build();
        Writer writer = new StringWriter();

        template.apply("layout").evaluate(writer, context);

        log.error("execution of: {}{} resulted in exception\nRoute:\n{}\n\nStacktrace:",
            req.method(), req.path(), req.route().print(6), err);
        rsp.type(MediaType.html).send(writer.toString());
      }
    };
  }

  private static String message(final Throwable cause) {
    if (cause instanceof Supplier) {
      return ((Supplier) cause).get().toString();
    }
    return Optional.ofNullable(cause.getMessage()).orElse("");
  }

  private static <T> Map<String, String> dump(final Supplier<Map<String, T>> hash) {
    return dump(hash, Object::toString);
  }

  private static <T> Map<String, String> dump(final Supplier<Map<String, T>> hash,
      final Function<T, String> map) {
    Map<String, String> data = new LinkedHashMap<>();
    hash.get().forEach((n, v) -> data.put(n, map.apply(v)));
    return data;
  }

  static String readString(final ClassLoader loader, final String path) {
    return Try.with(() -> loader.getResourceAsStream("whoops/" + path))
        .apply(stream -> new String(ByteStreams.toByteArray(stream), StandardCharsets.UTF_8))
        .get();
  }

  static List<Map<String, Object>> frames(final ClassLoader loader, final SourceLocator locator,
      final Throwable cause) {
    List<StackTraceElement> stacktrace = Arrays.asList(cause.getStackTrace());
    int limit = IntStream.range(0, stacktrace.size())
        .filter(i -> stacktrace.get(i).getClassName().equals(HANDLER)).findFirst()
        .orElse(stacktrace.size());
    return stacktrace.stream()
        // trunk stack at HttpHandlerImpl (skip servers stack)
        .limit(limit)
        .map(e -> frame(loader, locator, cause, e))
        .collect(Collectors.toList());
  }

  @SuppressWarnings("rawtypes")
  static Map<String, Object> frame(final ClassLoader loader, final SourceLocator locator,
      final Throwable cause, final StackTraceElement e) {
    int line = Math.max(e.getLineNumber(), 1);
    String className = e.getClassName();
    SourceLocator.Source source = locator.source(className);
    int[] range = source.range(line, SAMPLE_SIZE);
    int lineStart = range[0];
    int lineNth = line - lineStart;
    Path filePath = source.getPath();
    Optional<Class> clazz = findClass(loader, className);
    String filename = Optional.ofNullable(e.getFileName()).orElse("~unknown");
    return ImmutableMap.<String, Object>builder()
        .put("fileName", new File(filename).getName())
        .put("methodName", Optional.ofNullable(e.getMethodName()).orElse("~unknown"))
        .put("lineNumber", line)
        .put("lineStart", lineStart + 1)
        .put("lineNth", lineNth)
        .put("location", Optional.ofNullable(clazz.map(Whoops::locationOf)
            .orElse(new File(filename).getParent())).orElse(filename))
        .put("source", source.source(range[0], range[1]))
        .put("open", openWith.apply(filePath, line))
        .put("type", clazz.map(Class::getSimpleName).orElse(new File(filename).getName()))
        .put("comments", Arrays.asList(
            ImmutableMap.of(
                "context", cause.getClass().getName(),
                "text", message(cause))))
        .build();
  }

  @SuppressWarnings("rawtypes")
  static String locationOf(final Class clazz) {
    return Optional.ofNullable(clazz.getResource(clazz.getSimpleName() + ".class"))
        .map(url -> Try.apply(() -> {
          String path = url.getPath();
          int i = path.indexOf("!");
          if (i > 0) {
            // jar url
            String jar = path.substring(0, i);
            return jar.substring(Math.max(jar.lastIndexOf('/'), -1) + 1);
          }
          String cfile = clazz.getName().replace(".", "/") + ".class";
          String relativePath = path.replace(cfile, "");
          return new File(System.getProperty("user.dir"))
              .toPath()
              .relativize(Paths.get(relativePath).toFile().getCanonicalFile().toPath())
              .toString();
        }).orElse("~unknown"))
        .orElse("~unknown");
  }

  @SuppressWarnings("rawtypes")
  static Optional<Class> findClass(final ClassLoader loader, final String name) {
    return Stream.of(loader, Thread.currentThread().getContextClassLoader())
        // we don't care about exception
        .map(Throwing.<ClassLoader, Class>throwingFunction(cl -> cl.loadClass(name))
            .orElse((Class) null))
        .filter(Objects::nonNull)
        .findFirst();
  }

}
