/*
 * Copyright (c) 2017 Villu Ruusmann
 *
 * This file is part of JPMML-SkLearn
 *
 * JPMML-SkLearn is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JPMML-SkLearn is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with JPMML-SkLearn.  If not, see <http://www.gnu.org/licenses/>.
 */
package sklearn2pmml.preprocessing;

import java.util.Collections;
import java.util.List;

import org.dmg.pmml.Apply;
import org.dmg.pmml.DataType;
import org.dmg.pmml.DerivedField;
import org.dmg.pmml.Expression;
import org.dmg.pmml.Field;
import org.dmg.pmml.FieldRef;
import org.dmg.pmml.HasDefaultValue;
import org.dmg.pmml.HasMapMissingTo;
import org.dmg.pmml.InvalidValueTreatmentMethod;
import org.dmg.pmml.OpType;
import org.jpmml.converter.Feature;
import org.jpmml.converter.TypeUtil;
import org.jpmml.converter.ValueUtil;
import org.jpmml.python.DataFrameScope;
import org.jpmml.python.ExpressionTranslator;
import org.jpmml.python.Scope;
import org.jpmml.python.TypeInfo;
import org.jpmml.sklearn.SkLearnEncoder;
import sklearn.Transformer;
import sklearn.TransformerUtil;

public class ExpressionTransformer extends Transformer {

	public ExpressionTransformer(){
		this("sklearn2pmml.preprocessing", "ExpressionTransformer");
	}

	public ExpressionTransformer(String module, String name){
		super(module, name);
	}

	@Override
	public List<Feature> encodeFeatures(List<Feature> features, SkLearnEncoder encoder){
		String expr = getExpr();
		Object mapMissingTo = getMapMissingTo();
		Object defaultValue = getDefaultValue();
		InvalidValueTreatmentMethod invalidValueTreatment = parseInvalidValueTreatment(getInvalidValueTreatment());
		TypeInfo dtype = getDType();

		if(ValueUtil.isNaN(defaultValue)){
			defaultValue = null;
		} // End if

		if(ValueUtil.isNaN(mapMissingTo)){
			mapMissingTo = null;
		}

		Scope scope = new DataFrameScope("X", features);

		Expression expression = ExpressionTranslator.translate(expr, scope);

		if(mapMissingTo != null){
			HasMapMissingTo<?, Object> hasMapMissingTp = (HasMapMissingTo<?, Object>)expression;

			hasMapMissingTp.setMapMissingTo(ValueUtil.asString(mapMissingTo));
		} // End if

		if(defaultValue != null){
			HasDefaultValue<?, Object> hasDefaultValue = (HasDefaultValue<?, Object>)expression;

			hasDefaultValue.setDefaultValue(ValueUtil.asString(defaultValue));
		} // End if

		if(invalidValueTreatment != null){
			Apply apply = (Apply)expression;

			apply.setInvalidValueTreatment(invalidValueTreatment);
		}

		DataType dataType;

		if(dtype != null){
			dataType = dtype.getDataType();
		} else

		{
			dataType = TypeUtil.getDataType(expression, scope);

			if(dataType == null){
				dataType = DataType.DOUBLE;
			}
		}

		OpType opType = TransformerUtil.getOpType(dataType);

		// Detect identity field reference
		if((expression instanceof FieldRef) && (mapMissingTo == null)){
			FieldRef fieldRef = (FieldRef)expression;

			Feature feature = scope.resolveFeature(fieldRef.getField());
			if(feature != null){
				Field<?> field = feature.getField();

				if((field.getOpType() == opType) && (field.getDataType() == dataType)){
					return Collections.singletonList(feature);
				}
			}
		}

		DerivedField derivedField = encoder.createDerivedField(createFieldName("eval", expr), opType, dataType, expression);

		return Collections.singletonList(TransformerUtil.createFeature(derivedField, encoder));
	}

	public Object getDefaultValue(){
		return getOptionalScalar("default_value");
	}

	public ExpressionTransformer setDefaultValue(Object defaultValue){
		put("default_value", defaultValue);

		return this;
	}

	public TypeInfo getDType(){
		return super.getOptionalDType("dtype", true);
	}

	public ExpressionTransformer setDType(Object dtype){
		put("dtype", dtype);

		return this;
	}

	public String getExpr(){

		// SkLearn2PMML 0.31.0
		if(containsKey("expr_")){
			return getString("expr_");
		} else

		// SkLearn2PMML 0.31.1+
		{
			return getString("expr");
		}
	}

	public ExpressionTransformer setExpr(String expr){
		put("expr", expr);

		return this;
	}

	public String getInvalidValueTreatment(){
		return getOptionalString("invalid_value_treatment");
	}

	public ExpressionTransformer setInvalidValueTreatment(String invalidValueTreatment){
		put("invalid_value_treatment", invalidValueTreatment);

		return this;
	}

	public Object getMapMissingTo(){
		return getOptionalScalar("map_missing_to");
	}

	public ExpressionTransformer setMapMissingTo(Object mapMissingTo){
		put("map_missing_to", mapMissingTo);

		return this;
	}

	static
	private InvalidValueTreatmentMethod parseInvalidValueTreatment(String invalidValueTreatment){

		if(invalidValueTreatment == null){
			return null;
		}

		switch(invalidValueTreatment){
			case "return_invalid":
				return InvalidValueTreatmentMethod.RETURN_INVALID;
			case "as_missing":
				return InvalidValueTreatmentMethod.AS_MISSING;
			default:
				throw new IllegalArgumentException(invalidValueTreatment);
		}
	}
}