package org.jresearch.commons.gwt.app.client.mvc;

import java.util.Optional;
import java.util.Set;

import javax.annotation.Nonnull;

import org.fusesource.restygwt.client.Method;
import org.fusesource.restygwt.client.REST;
import org.jresearch.commons.gwt.app.client.mvc.event.AfterLogoutEvent;
import org.jresearch.commons.gwt.app.client.mvc.event.LogoutEvent;
import org.jresearch.commons.gwt.app.client.mvc.event.LogoutHandler;
import org.jresearch.commons.gwt.app.client.mvc.event.ProfileShowEvent;
import org.jresearch.commons.gwt.app.client.mvc.event.ProfileShowHandler;
import org.jresearch.commons.gwt.app.client.mvc.event.ProfileUpdateEvent;
import org.jresearch.commons.gwt.app.client.mvc.event.ProfileUpdateHandler;
import org.jresearch.commons.gwt.app.client.mvc.event.ResetPasswordEvent;
import org.jresearch.commons.gwt.app.client.mvc.event.ResetPasswordHandler;
import org.jresearch.commons.gwt.app.client.mvc.event.SignUpEvent;
import org.jresearch.commons.gwt.app.client.mvc.event.SignUpHandler;
import org.jresearch.commons.gwt.app.client.resource.AppRs;
import org.jresearch.commons.gwt.app.shared.model.user.UserProfileModel;
import org.jresearch.commons.gwt.client.app.AbstractAppController;
import org.jresearch.commons.gwt.client.app.IAppModule;
import org.jresearch.commons.gwt.client.mvc.AbstractMethodCallback;
import org.jresearch.commons.gwt.client.mvc.INotificator;
import org.jresearch.commons.gwt.client.mvc.ViewCommand;
import org.jresearch.commons.gwt.client.mvc.event.Bus;
import org.jresearch.commons.gwt.client.mvc.event.ErrorEvent;
import org.jresearch.commons.gwt.client.mvc.event.LoginEvent;
import org.jresearch.commons.gwt.client.mvc.event.LoginHandler;
import org.jresearch.commons.gwt.client.service.AppRestService;
import org.jresearch.commons.gwt.client.service.LocalizationRestService;
import org.jresearch.commons.gwt.client.widget.AuthTokenManager;
import org.jresearch.commons.gwt.flexess.client.service.FlexessRestService;
import org.jresearch.commons.gwt.flexess.shared.model.AuthData;
import org.jresearch.commons.gwt.shared.service.AccessDenidedException;
import org.jresearch.commons.gwt.shared.service.ExpiredAuthTokenRestException;
import org.jresearch.commons.gwt.shared.service.NoUserException;
import org.jresearch.commons.gwt.shared.service.WrongAuthTokenRestException;

import com.google.common.base.Joiner;
import com.google.gwt.inject.client.AsyncProvider;
import com.google.gwt.storage.client.Storage;
import com.google.inject.Inject;

public abstract class AbstractUserAppController<V extends AbstractUserAppView<? extends AbstractUserAppController<V>>> extends AbstractAppController<V> implements LoginHandler, ResetPasswordHandler, SignUpHandler, LogoutHandler, ProfileShowHandler, ProfileUpdateHandler, AuthTokenManager {

	public static final Storage STORAGE = Storage.getLocalStorageIfSupported();
	@Nonnull
	public static final String STORAGE_KEY_AUTH_PREF = "STORAGE_KEY_AUTH"; //$NON-NLS-1$
	@Nonnull
	public static final String STORAGE_KEY_REFRESH_PREF = "STORAGE_KEY_REFRESH"; //$NON-NLS-1$
	@Nonnull
	private static final Joiner KEY_JOINER = Joiner.on('_').skipNulls();

	@Nonnull
	private final String storageKeyAuth;
	@Nonnull
	private final String storageKeyRefresh;
	@Nonnull
	private final FlexessRestService service;

	private UserProfileModel profile;

	@Inject
	@Nonnull
	private INotificator notificator;

	public AbstractUserAppController(@Nonnull final String id, @Nonnull final AppRestService appService, @Nonnull final FlexessRestService service, @Nonnull final LocalizationRestService localizationService, @Nonnull final Set<? extends IAppModule> appModules, @Nonnull final AsyncProvider<V> view, @Nonnull final Bus bus, final boolean checkOffline) {
		super(id, appService, localizationService, appModules, view, bus, checkOffline);
		this.service = service;
		storageKeyAuth = KEY_JOINER.join(STORAGE_KEY_AUTH_PREF, id);
		storageKeyRefresh = KEY_JOINER.join(STORAGE_KEY_REFRESH_PREF, id);
		bus.addHandler(ProfileUpdateEvent.TYPE, this);
		bus.addHandler(ProfileShowEvent.TYPE, this);
		bus.addHandler(LogoutEvent.TYPE, this);
		bus.addHandler(LoginEvent.TYPE, this);
		bus.addHandler(ResetPasswordEvent.TYPE, this);
		bus.addHandler(SignUpEvent.TYPE, this);
	}

	@Override
	protected void initView() {
		super.initView();
		executeCommand(new ViewCommand<V>() {
			@Override
			public void execute(final V view) {
				view.disableUserControls();
			}
		});
	}

	@Override
	public void onError(final ErrorEvent event) {
		final Throwable ex = event.getException();
		if (ex instanceof NoUserException || ex instanceof AccessDenidedException) {
			notificator.showError(AppRs.MSG.shouldLogin());
			bus.fire(new LoginEvent(event.getCommand()));
		} else if (ex instanceof WrongAuthTokenRestException) {
			clearTokens();
		} else if (ex instanceof ExpiredAuthTokenRestException) {
			clearTokens();
			bus.fire(new LogoutEvent());
		} else {
			super.onError(event);
		}
	}

	@SuppressWarnings("all")
	@Override
	public void onLogout(final LogoutEvent event) {
		if (isOfflineMode()) {
			notificator.showNotification(AppRs.TXT.cantLogoutInOfflineMode());
		} else {
			REST.withCallback(new AbstractMethodCallback<Void>(bus) {
				@Override
				public void onSuccess(final Method method, final Void response) {
					clearTokens();
					bus.fire(new AfterLogoutEvent());
				}
			}).call(service).logOut();
		}
	}

	@Override
	public void onLogin(final LoginEvent event) {
		if (isOfflineMode()) {
			notificator.showNotification(AppRs.TXT.cantLoginInOfflineMode());
		} else {
			executeCommand(new ViewCommand<V>() {
				@Override
				public void execute(final V view) {
					if (profile == null) {
						view.showLoginDialog(event.getCommand());
					}
				}
			});
		}
	}

	@Override
	public void onSignUp(final SignUpEvent event) {
		final V v = getView();
		if (v != null) {
			v.showSignUpDialod(event.getCommand());
		}
	}

	@Override
	public void onReset(final ResetPasswordEvent event) {
		if (isOfflineMode()) {
			notificator.showNotification(AppRs.TXT.cantLogoutInOfflineMode());
		} else {
			final V v = getView();
			if (v != null) {
				v.showResetPasswordDialog(event.getCommand());
			}
		}
	}

	@Override
	public void onProfileUpdate(final ProfileUpdateEvent profileUpdateEvent) {
		profile = profileUpdateEvent.getUserProfileModel();
		executeCommand(new ViewCommand<V>() {
			@Override
			public void execute(final V view) {
				if (profileUpdateEvent.getUserProfileModel() != null) {
					view.enableUserControls();
				} else {
					view.disableUserControls();
				}
			}
		});
	}

	@Override
	public void onProfileShow(final ProfileShowEvent event) {
		final V v = getView();
		if (v != null) {
			v.showProfieDialog(event.getUserProfileModel());
		}
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.jresearch.commons.gwt.client.app.AbstractAppController#setOfflineMode
	 * (org.jresearch.commons.gwt.client.model.OfflineMode)
	 */
	@Override
	protected void onChangeOfflineMode() {
		super.onChangeOfflineMode();
		executeCommand(new ViewCommand<V>() {
			@Override
			public void execute(final V view) {
				if (isOfflineMode()) {
					view.disableUserControlsAsIs();
				} else {
					view.enableUserControlsAsIs();
				}
			}
		});
	}

	public void updateTokens(@Nonnull final AuthData authData) {
		if (STORAGE != null) {
			if (authData.getAuthTocken() != null) {
				STORAGE.setItem(storageKeyAuth, authData.getAuthTocken());
			} else {
				STORAGE.removeItem(storageKeyAuth);
			}
			if (authData.getRefreshTocken() != null) {
				STORAGE.setItem(storageKeyRefresh, authData.getRefreshTocken());
			} else {
				STORAGE.removeItem(storageKeyRefresh);
			}
		}
	}

	@Override
	public void clearTokens() {
		if (STORAGE != null) {
			STORAGE.removeItem(storageKeyAuth);
			STORAGE.removeItem(storageKeyRefresh);
		}
	}

	@Override
	public Optional<String> getAuthToken() {
		return getToken(storageKeyAuth);
	}

	@Override
	public Optional<String> getRefreshToken() {
		return getToken(storageKeyRefresh);
	}

	private static Optional<String> getToken(@Nonnull final String key) {
		final String token = STORAGE != null ? STORAGE.getItem(key) : null;
		return Optional.ofNullable(token);
	}

}
