package org.jresearch.commons.gwt.app.client;

import javax.annotation.Nonnull;

import org.jresearch.commons.gwt.app.client.mvc.event.AfterLoginEvent;
import org.jresearch.commons.gwt.app.client.mvc.event.AfterLoginHandler;
import org.jresearch.commons.gwt.app.client.mvc.event.AfterLogoutEvent;
import org.jresearch.commons.gwt.app.client.mvc.event.AfterLogoutHandler;
import org.jresearch.commons.gwt.app.client.mvc.event.ProfileUpdateEvent;
import org.jresearch.commons.gwt.app.shared.model.user.UserModel;
import org.jresearch.commons.gwt.app.shared.service.user.UserServiceAsync;
import org.jresearch.commons.gwt.client.mvc.AbstractCallback;
import org.jresearch.commons.gwt.client.mvc.event.Bus;
import org.jresearch.commons.gwt.shared.service.AnonymousUserException;

import com.google.inject.Inject;
import com.google.inject.Singleton;

/**
 * Holds the current user information.
 */
@Singleton
public class UserHolder implements AfterLogoutHandler, AfterLoginHandler {

	@Nonnull
	private final Bus bus;
	@Nonnull
	private final UserServiceAsync userService;

	private String userId;
	private UserModel user;

	@Inject
	public UserHolder(@Nonnull final UserServiceAsync userService, @Nonnull final Bus bus) {
		this.userService = userService;
		this.bus = bus;

		bus.addHandler(AfterLoginEvent.TYPE, this);
		bus.addHandler(AfterLogoutEvent.TYPE, this);

		updateUser();
	}

	private void updateUser() {
		userService.getCurrentUser(new AbstractCallback<UserModel>(bus) {
			@Override
			public void onSuccess(final UserModel result) {
				setUser(result);
			}

			@Override
			public void onFailure(final Throwable caught) {
				if (caught instanceof AnonymousUserException) {
					unsetUser();
				} else {
					super.onFailure(caught);
				}
			}
		});
	}

	public String getCurrentUserId() {
		return userId;
	}

	public UserModel getCurrentUser() {
		return user;
	}

	public void setUserId(final String userId) {
		this.userId = userId;
	}

	public void setUser(final UserModel user) {
		setUserId(user.getId().toString());
		this.user = user;
		bus.fire(new ProfileUpdateEvent(user.getUserProfile()));
	}

	public void unsetUserId() {
		userId = null;
	}

	public void unsetUser() {
		unsetUserId();
		user = null;
		bus.fire(new ProfileUpdateEvent(null));
	}

	@Override
	public void onAfterLogin(final AfterLoginEvent event) {
		final String newUserId = event.getUserId();
		if (!newUserId.equals(userId)) {
			unsetUser();
			setUserId(newUserId);
		}
		if (getCurrentUser() == null) {
			updateUser();
		}
	}

	@Override
	public void onAfterLogout(final AfterLogoutEvent afterLogoutEvent) {
		unsetUser();
	}

}
