package org.jresearch.commons.gwt.app.client.mvc;

import java.util.Set;

import javax.annotation.Nonnull;

import org.jresearch.commons.gwt.app.client.mvc.event.AfterLogoutEvent;
import org.jresearch.commons.gwt.app.client.mvc.event.LogoutEvent;
import org.jresearch.commons.gwt.app.client.mvc.event.LogoutHandler;
import org.jresearch.commons.gwt.app.client.mvc.event.ProfileShowEvent;
import org.jresearch.commons.gwt.app.client.mvc.event.ProfileShowHandler;
import org.jresearch.commons.gwt.app.client.mvc.event.ProfileUpdateEvent;
import org.jresearch.commons.gwt.app.client.mvc.event.ProfileUpdateHandler;
import org.jresearch.commons.gwt.app.client.mvc.event.ResetPasswordEvent;
import org.jresearch.commons.gwt.app.client.mvc.event.ResetPasswordHandler;
import org.jresearch.commons.gwt.app.client.mvc.event.SignUpEvent;
import org.jresearch.commons.gwt.app.client.mvc.event.SignUpHandler;
import org.jresearch.commons.gwt.app.client.resource.AppRs;
import org.jresearch.commons.gwt.app.shared.model.user.UserProfileModel;
import org.jresearch.commons.gwt.client.app.AbstractAppController;
import org.jresearch.commons.gwt.client.app.IAppModule;
import org.jresearch.commons.gwt.client.mvc.AbstractCallback;
import org.jresearch.commons.gwt.client.mvc.INotificator;
import org.jresearch.commons.gwt.client.mvc.ViewCommand;
import org.jresearch.commons.gwt.client.mvc.event.Bus;
import org.jresearch.commons.gwt.client.mvc.event.ErrorEvent;
import org.jresearch.commons.gwt.client.mvc.event.LoginEvent;
import org.jresearch.commons.gwt.client.mvc.event.LoginHandler;
import org.jresearch.commons.gwt.client.service.AppRestService;
import org.jresearch.commons.gwt.client.service.LocalizationRestService;
import org.jresearch.commons.gwt.flexess.shared.service.FlexessServiceAsync;
import org.jresearch.commons.gwt.shared.service.AccessDenidedException;
import org.jresearch.commons.gwt.shared.service.NoUserException;

import com.google.gwt.inject.client.AsyncProvider;
import com.google.inject.Inject;

public abstract class AbstractUserAppController<V extends AbstractUserAppView<? extends AbstractUserAppController<V>>> extends AbstractAppController<V> implements LoginHandler, ResetPasswordHandler, SignUpHandler, LogoutHandler, ProfileShowHandler, ProfileUpdateHandler {

	@Nonnull
	private final FlexessServiceAsync service;

	private UserProfileModel profile;

	@Inject
	@Nonnull
	private INotificator notificator;

	public AbstractUserAppController(@Nonnull final String id, @Nonnull final AppRestService appService, @Nonnull final FlexessServiceAsync service, @Nonnull final LocalizationRestService localizationService, @Nonnull final Set<IAppModule> appModules, @Nonnull final AsyncProvider<V> view, @Nonnull final Bus bus, final boolean checkOffline) {
		super(id, appService, localizationService, appModules, view, bus, checkOffline);
		this.service = service;
		bus.addHandler(ProfileUpdateEvent.TYPE, this);
		bus.addHandler(ProfileShowEvent.TYPE, this);
		bus.addHandler(LogoutEvent.TYPE, this);
		bus.addHandler(LoginEvent.TYPE, this);
		bus.addHandler(ResetPasswordEvent.TYPE, this);
		bus.addHandler(SignUpEvent.TYPE, this);
	}

	@Override
	protected void initView() {
		super.initView();
		executeCommand(new ViewCommand<V>() {
			@Override
			public void execute(final V view) {
				view.disableUserControls();
			}
		});
	}

	@Override
	public void onError(final ErrorEvent event) {
		final Throwable ex = event.getException();
		if (ex instanceof NoUserException || ex instanceof AccessDenidedException) {
			notificator.showError(AppRs.MSG.shouldLogin());
			bus.fire(new LoginEvent(event.getCommand()));
		} else {
			super.onError(event);
		}
	}

	@SuppressWarnings("all")
	@Override
	public void onLogout(final LogoutEvent event) {
		if (isOfflineMode()) {
			notificator.showNotification(AppRs.TXT.cantLogoutInOfflineMode());
		} else {
			service.logOut(new AbstractCallback<Void>(bus) {
				@Override
				public void onSuccess(final Void result) {
					bus.fire(new AfterLogoutEvent());
				}
			});
		}
	}

	@Override
	public void onLogin(final LoginEvent event) {
		if (isOfflineMode()) {
			notificator.showNotification(AppRs.TXT.cantLoginInOfflineMode());
		} else {
			executeCommand(new ViewCommand<V>() {
				@Override
				public void execute(final V view) {
					if (profile == null) {
						view.showLoginDialog(event.getCommand());
					}
				}
			});
		}
	}

	@Override
	public void onSignUp(final SignUpEvent event) {
		final V v = getView();
		if (v != null) {
			v.showSignUpDialod(event.getCommand());
		}
	}

	@Override
	public void onReset(final ResetPasswordEvent event) {
		if (isOfflineMode()) {
			notificator.showNotification(AppRs.TXT.cantLogoutInOfflineMode());
		} else {
			final V v = getView();
			if (v != null) {
				v.showResetPasswordDialog(event.getCommand());
			}
		}
	}

	@Override
	public void onProfileUpdate(final ProfileUpdateEvent profileUpdateEvent) {
		profile = profileUpdateEvent.getUserProfileModel();
		executeCommand(new ViewCommand<V>() {
			@Override
			public void execute(final V view) {
				if (profileUpdateEvent.getUserProfileModel() != null) {
					view.enableUserControls();
				} else {
					view.disableUserControls();
				}
			}
		});
	}

	@Override
	public void onProfileShow(final ProfileShowEvent event) {
		final V v = getView();
		if (v != null) {
			v.showProfieDialog(event.getUserProfileModel());
		}
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.jresearch.commons.gwt.client.app.AbstractAppController#setOfflineMode
	 * (org.jresearch.commons.gwt.client.model.OfflineMode)
	 */
	@Override
	protected void onChangeOfflineMode() {
		super.onChangeOfflineMode();
		executeCommand(new ViewCommand<V>() {
			@Override
			public void execute(final V view) {
				if (isOfflineMode()) {
					view.disableUserControlsAsIs();
				} else {
					view.enableUserControlsAsIs();
				}
			}
		});
	}

}
