package org.jresearch.commons.gwt.app.client.widget;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;

import org.fusesource.restygwt.client.Method;
import org.fusesource.restygwt.client.REST;
import org.jresearch.commons.gwt.app.client.mvc.event.AfterLoginEvent;
import org.jresearch.commons.gwt.app.client.mvc.event.AfterLoginHandler;
import org.jresearch.commons.gwt.app.client.mvc.event.ResetPasswordEvent;
import org.jresearch.commons.gwt.app.client.mvc.event.SignUpEvent;
import org.jresearch.commons.gwt.app.client.resource.AppRs;
import org.jresearch.commons.gwt.client.mvc.AbstractMethodCallback;
import org.jresearch.commons.gwt.client.widget.FieldDescriptor;
import org.jresearch.commons.gwt.flexess.shared.model.AuthData;
import org.jresearch.commons.gwt.flexess.shared.service.flexess.GwtAuthenticationData;
import org.jresearch.commons.gwt.flexess.shared.service.flexess.PasswordAuthenticationData;
import org.jresearch.commons.gwt.flexess.shared.service.flexess.SocialAuthenticationData;
import org.jresearch.commons.gwt.oauth2.client.mvc.event.SocialAuthenticationEvent;
import org.jresearch.commons.gwt.oauth2.client.mvc.event.SocialAuthenticationHandler;
import org.jresearch.commons.gwt.oauth2.client.widget.EnableAction;
import org.jresearch.commons.gwt.oauth2.client.widget.SocialAuthenticator;
import org.jresearch.commons.gwt.oauth2.shared.model.OauthConfiguration;
import org.jresearch.commons.gwt.oauth2.shared.model.SocialNetwork;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.PasswordTextBox;
import com.google.gwt.user.client.ui.TextBox;
import com.google.inject.Inject;

public class LoginDialog extends AbstractAuthDialog implements SocialAuthenticationHandler, AfterLoginHandler {

	@Inject
	protected SocialAuthenticator socialAuthenticator;

	protected TextBox userName;
	protected PasswordTextBox password;

	private Map<SocialNetwork, Button> socialNetworkControls;
	private SocialNetwork network;
	private String token;

	private ArrayList<FieldDescriptor<?, ?>> fields;

	@Inject
	protected LoginDialog() {
		super(AppRs.MSG.loginDialog());
	}

	@Override
	protected void createLinks(final ButtonPanel linkBar) {
		linkBar.add(reset = createEventLink(AppRs.MSG.resetPassword(), new ResetPasswordEvent(), false));
		linkBar.add(signUp = createEventLink(AppRs.MSG.register(), new SignUpEvent(), false));
	}

	@Override
	protected void createSocialLoginButtons(final ButtonPanel socialBar) {
		socialNetworkControls = Maps.newHashMap();
		for (final SocialNetwork net : EnumSet.allOf(SocialNetwork.class)) {
			final Button button = new Button(net.name());
			socialBar.add(button);
			button.setVisible(false);
			socialNetworkControls.put(net, button);
		}
	}

	private void enableSocialNetworkControl(final SocialNetwork net) {
		final Button button = socialNetworkControls.get(net);
		if (button != null) {
			configureSocialLoginButton(net, button);
		}
	}

	protected void configureSocialLoginButton(final SocialNetwork net, final Button button) {
		socialAuthenticator.enableNetwork(net, new EnableAction() {
			@Override
			public void configureControl(final OauthConfiguration configuration) {
				button.addClickHandler(new ClickHandler() {
					@Override
					public void onClick(final ClickEvent event) {
						socialAuthenticator.authenticate(configuration);
					}
				});
				button.setVisible(true);
			}
		});
	}

	@Override
	protected void init() {
		enableSocialNetworkAuthentication();
		super.init();
	}

	private void enableSocialNetworkAuthentication() {
		for (final SocialNetwork net : EnumSet.allOf(SocialNetwork.class)) {
			enableSocialNetworkControl(net);
		}
		bus.addHandler(SocialAuthenticationEvent.TYPE, this);
		bus.addHandler(AfterLoginEvent.TYPE, this);
	}

	@Override
	public void onAuthentication(final SocialAuthenticationEvent event) {
		// Handle only is active
		if (isVisible()) {
			network = event.getNetwork();
			token = event.getToken();
			onSubmit();
		}
	}

	@Override
	protected List<FieldDescriptor<?, ?>> getFields() {
		if (fields == null) {
			fields = Lists.newArrayList();

			userName = new TextBox();
			fields.add(new FieldDescriptor<>(AppRs.TXT.username(), userName));

			password = new PasswordTextBox();
			fields.add(new FieldDescriptor<>(AppRs.TXT.password(), password));
		}
		return fields;
	}

	@Override
	protected void reset() {
		network = null;
		token = null;
		super.reset();
	}

	@Override
	protected String getActionName() {
		return AppRs.MSG.loginAction();
	}

	@Override
	protected void doAction(final Label status) {
		final GwtAuthenticationData data = getAuthenticationData();
		REST.withCallback(new AbstractMethodCallback<AuthData>(bus) {
			@SuppressWarnings("incomplete-switch")
			@Override
			public void onSuccess(final Method method, final AuthData result) {
				switch (result.getAuthStatus()) {
				case Authenticated:
					LoginDialog.this.hide();
					final String userId = result.getUserId();
					if (userId == null) {
						throw new IllegalStateException("Null user id after authentication"); //$NON-NLS-1$
					}
					bus.fire(new AfterLoginEvent(userId));
					if (command != null) {
						Scheduler.get().scheduleDeferred(command);
					}
					break;
				case NotEnroled:
					status.setText(AppRs.MSG.notEnrolled());
					status.setVisible(true);
					getButtonBar().enable();
					getSocialBar().enable();
					break;
				case BadCredentials:
					status.setText(AppRs.MSG.badCredentials());
					status.setVisible(true);
					getButtonBar().enable();
					getSocialBar().enable();
					break;
				case NotAuthorized:
					status.setText(AppRs.MSG.notAuthorized());
					status.setVisible(true);
					getButtonBar().enable();
					getSocialBar().enable();
					break;
				case Reset:
					status.setText(AppRs.MSG.reset());
					status.setVisible(true);
					getButtonBar().enable();
					getSocialBar().enable();
					break;
				}
			}
		}).call(service).authenticate(data);

	}

	@Nonnull
	private GwtAuthenticationData getAuthenticationData() {
		GwtAuthenticationData result = new PasswordAuthenticationData(userName.getValue(), password.getValue());
		if (token != null && network != null) {
			result = new SocialAuthenticationData(network, token);
			token = null;
		}
		return result;
	}

	@Override
	public void onAfterLogin(final AfterLoginEvent event) {
		hide();
	}

	@Override
	protected void resetFields() {
		userName.setValue(null);
		password.setValue(null);
	}

}
