package org.jresearch.commons.gwt.client.mvc.event;

import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.jresearch.commons.gwt.client.base.resource.BaseRs;
import org.jresearch.commons.gwt.client.ga.GaMethods;
import org.jresearch.commons.gwt.shared.service.NoUserException;

import com.google.common.collect.Maps;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.History;
import com.google.inject.Inject;
import com.google.web.bindery.event.shared.Event;
import com.google.web.bindery.event.shared.Event.Type;
import com.google.web.bindery.event.shared.EventBus;
import com.google.web.bindery.event.shared.HandlerRegistration;

public class Bus {

	private static Logger logger = Logger.getLogger("org.jresearch.commons.gwt.client.mvc.event.Bus"); //$NON-NLS-1$

	private final Map<String, Event<?>> history = Maps.newHashMap();

	public Bus(final EventBus eventBus) {
		this();
		this.eventBus = eventBus;
	}

	public Bus() {
		History.addValueChangeHandler(new ValueChangeHandler<String>() {
			@SuppressWarnings("synthetic-access")
			@Override
			public void onValueChange(final ValueChangeEvent<String> event) {
				final String historyToken = event.getValue();
				if (history.containsKey(historyToken)) {
					fire(history.get(historyToken), true);
				}
			}
		});
	}

	@Inject
	private EventBus eventBus;

	public void fire(final Event<?> event) {
		fire(event, false);
	}

	public void fire(final Event<?> event, final Object source) {
		fire(event, source, false);
	}

	public void fire(final Event<?> event, final boolean disableHistory) {
		fire(event, null, disableHistory);
	}

	public void fire(final Event<?> event, final Object source, final boolean disableHistory) {
		logger.log(Level.FINER, BaseRs.FMT.logFire(event.toDebugString()), new Throwable());
		Scheduler.get().scheduleDeferred(new Command() {
			@Override
			public void execute() {
				fireImmediately(event, source, disableHistory);
			}
		});
	}

	public void fireImmediately(final Event<?> event) {
		fireImmediately(event, false);
	}

	public void fireImmediately(final Event<?> event, final Object source) {
		fireImmediately(event, source, false);
	}

	public void fireImmediately(final Event<?> event, final boolean disableHistory) {
		fireImmediately(event, null, disableHistory);
	}

	public void fireImmediately(final Event<?> event, final Object source, final boolean disableHistory) {
		logger.log(Level.FINE, BaseRs.FMT.logFireIm(event.toDebugString()));
		if (source == null) {
			eventBus.fireEvent(event);
		} else {
			eventBus.fireEventFromSource(event, source);
		}
		if (!disableHistory && event instanceof IHistoryEvent) {
			final IHistoryEvent hEvent = (IHistoryEvent) event;
			final String token = hEvent.getToken();
			GaMethods.trackPageview(hEvent.getPageName());
			history.put(token, event);
			History.newItem(token, false);
		}
	}

	/**
	 * @param type
	 * @param handler
	 * @return
	 * @see com.google.web.bindery.event.shared.EventBus#addHandler(com.google.web.bindery.event.shared.Event.Type,
	 *      java.lang.Object)
	 */
	public <H> HandlerRegistration addHandler(final Type<H> type, final H handler) {
		return eventBus.addHandler(type, handler);
	}

	/**
	 * @param type
	 * @param source
	 * @param handler
	 * @return
	 * @see com.google.web.bindery.event.shared.EventBus#addHandlerToSource(com.google.web.bindery.event.shared.Event.Type,
	 *      java.lang.Object, java.lang.Object)
	 */
	public <H> HandlerRegistration addHandlerToSource(final Type<H> type, final Object source, final H handler) {
		return eventBus.addHandlerToSource(type, source, handler);
	}

	/**
	 * Fire {@link ErrorEvent} with given template and parameters
	 *
	 * @param errorMessage
	 *            - error message to show
	 */
	public void fireError(final String errorMessage) {
		fire(new ErrorEvent(errorMessage));
	}

	/**
	 * Fire {@link ErrorEvent} with {@link NoUserException}
	 */
	public void fireNoUserError() {
		fire(new ErrorEvent(new NoUserException()));
	}

}
