/*
 * Copyright 2007 Future Earth, info@future-earth.eu
 * Copyright 2014 Stanislav Spiridonov, stas@jresearch.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package org.jresearch.commons.gwt.client.tool;

import static org.jresearch.commons.gwt.client.base.resource.BaseRs.*;

//NEED
import java.util.Date;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.jresearch.commons.gwt.client.base.resource.BaseRs;
import org.jresearch.commons.gwt.shared.model.time.GwtDateTimeModel;
import org.jresearch.commons.gwt.shared.model.time.GwtLocalDateModel;
import org.jresearch.commons.gwt.shared.model.time.GwtLocalDateTimeModel;
import org.jresearch.commons.gwt.shared.model.time.GwtLocalTimeModel;
import org.jresearch.commons.gwt.shared.tools.LocalTimeRange;
import org.jresearch.commons.gwt.shared.tools.MomentKey;
import org.jresearch.commons.gwt.shared.tools.SharedDates;
import org.jresearch.gwt.momentjs.lib.client.Duration;
import org.jresearch.gwt.momentjs.lib.client.Moment;

import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.primitives.Doubles;
import com.google.common.primitives.Ints;
import com.google.gwt.i18n.client.LocaleInfo;
import com.google.gwt.i18n.shared.DateTimeFormat;
import com.google.gwt.i18n.shared.DateTimeFormatInfo;
import com.google.gwt.i18n.shared.TimeZone;

public final class Dates {

	private static final String REMOVE_YEAR_PATTERN = "[^a-zA-Z]*y+[^a-zA-Z]*"; //$NON-NLS-1$

	@Nonnull
	private static final GwtLocalDateModel TIME_DAY = new GwtLocalDateModel(1, 1, 1900);
	@Nonnull
	private static final GwtLocalTimeModel MIDNIGHT = new GwtLocalTimeModel();

	@SuppressWarnings("nls")
	public enum MomentPrecision {
		YEAR("year"),
		MONTH("month"),
		WEEK("week"),
		DAY("day"),
		HOUR("hour"),
		MINUTE("minute"),
		SECOND("second"),
		MILLISECOND("");

		private final String shorthand;

		private MomentPrecision(final String shorthand) {
			this.shorthand = shorthand;
		}

		/**
		 * @return the shorthand
		 */
		public String getShorthand() {
			return shorthand;
		}
	}

	@SuppressWarnings("nls")
	public enum MomentInterval {
		CLOSE("()"),
		OPEN("[]"),
		OPEN_CLOSE("[)"),
		CLOSE_OPEN("(]");

		private final String shorthand;

		private MomentInterval(final String shorthand) {
			this.shorthand = shorthand;
		}

		/**
		 * @return the shorthand
		 */
		public String getShorthand() {
			return shorthand;
		}
	}

	static {
		Moment.init();
		Moment.locale(LocaleInfo.getCurrentLocale().getLocaleName());
	}

	public static long toMiliseconds(@Nonnull final GwtLocalDateTimeModel dayTime) {
		return toMiliseconds(dayTime.getDate()) + toMiliseconds(dayTime.getTime());
	}

	public static long toMiliseconds(@Nonnull final GwtLocalDateModel day) {
		return toMiliseconds(toMomentUtc(day));
	}

	public static int toMiliseconds(@Nonnull final GwtLocalTimeModel time) {
		return ((time.getHour() * 60 + time.getMinute()) * 60 + time.getSecond()) * 1000 + time.getMillisecond();
	}

	public static long toMiliseconds(@Nonnull final Moment moment) {
		return Double.valueOf(moment.valueOf()).longValue();
	}

	@Nonnull
	public static Date toDate(@Nonnull final GwtLocalDateModel day) {
		return toDate(day, MIDNIGHT);
	}

	@Nonnull
	public static Date toDate(@Nonnull final GwtLocalTimeModel time) {
		return toDate(TIME_DAY, time);
	}

	@Nonnull
	public static Date toDate(@Nonnull final GwtLocalDateTimeModel dateTime) {
		return toDate(dateTime.getDate(), dateTime.getTime());
	}

	@SuppressWarnings("deprecation")
	@Nonnull
	private static Date toDate(@Nonnull final GwtLocalDateModel day, @Nonnull final GwtLocalTimeModel time) {
		return new Date(day.getYear() - 1900, day.getMonth() - 1, day.getDay(), time.getHour(), time.getMinute(), time.getSecond());
	}

	@Nonnull
	public static GwtLocalDateModel toLocalDate(final long milliseconds) {
		return toLocalDate(Moment.utc(Long.valueOf(milliseconds).doubleValue()));
	}

	@SuppressWarnings("deprecation")
	@Nonnull
	public static GwtLocalDateModel toLocalDate(@Nonnull final Date date) {
		return new GwtLocalDateModel(date.getDate(), date.getMonth() + 1, date.getYear() + 1900);
	}

	@Nonnull
	public static GwtDateTimeModel toDateTime(@Nonnull final GwtLocalDateModel value) {
		return toDateTime(toMomentTz(value));
	}

	@Nonnull
	public static Moment toMomentUtc(@Nonnull final GwtLocalDateModel day) {
		return Moment.utc(new int[] { day.getYear(), day.getMonth() - 1, day.getDay() });
	}

	@Nonnull
	public static Moment toMomentTz(@Nonnull final GwtLocalDateModel day) {
		return Moment.moment(new int[] { day.getYear(), day.getMonth() - 1, day.getDay() });
	}

	@Nonnull
	public static Moment toMomentUtc(@Nonnull final GwtLocalDateTimeModel dayTime) {
		return toMomentUtc(dayTime.getDate(), dayTime.getTime());
	}

	@Nonnull
	public static Moment toMomentUtc(@Nonnull final GwtLocalTimeModel time) {
		return toMomentUtc(TIME_DAY, time);
	}

	@Nonnull
	public static Moment toMomentUtc(@Nonnull final GwtLocalDateModel day, @Nonnull final GwtLocalTimeModel time) {
		return Moment.utc(new int[] { day.getYear(), day.getMonth() - 1, day.getDay(), time.getHour(), time.getMinute(), time.getSecond(), time.getMillisecond() });
	}

	@Nonnull
	public static Moment toMomentUtc(@Nonnull final GwtLocalDateModel day, @Nonnull final GwtLocalTimeModel time, final MomentPrecision precision) {
		switch (precision) {
		case SECOND:
			return Moment.utc(new int[] { day.getYear(), day.getMonth() - 1, day.getDay(), time.getHour(), time.getMinute(), time.getSecond() });
		case MINUTE:
			return Moment.utc(new int[] { day.getYear(), day.getMonth() - 1, day.getDay(), time.getHour(), time.getMinute() });
		case HOUR:
			return Moment.utc(new int[] { day.getYear(), day.getMonth() - 1, day.getDay(), time.getHour() });
		case DAY:
			return Moment.utc(new int[] { day.getYear(), day.getMonth() - 1, day.getDay() });
		case MONTH:
			return Moment.utc(new int[] { day.getYear() });
		case YEAR:
			return Moment.utc(new int[] { day.getYear() });
		default:
			throw new IllegalArgumentException("Unsupported precision: " + precision); //$NON-NLS-1$
		}
	}

	@Nonnull
	public static GwtLocalDateModel toLocalDate(@Nonnull final Moment moment) {
		return new GwtLocalDateModel(moment.date(), moment.month() + 1, moment.year());
	}

	@Nonnull
	public static GwtLocalTimeModel toLocalTime(@Nonnull final Moment moment) {
		return new GwtLocalTimeModel(moment.hour(), moment.minute(), moment.second(), moment.millisecond());
	}

	@Nonnull
	public static GwtLocalDateTimeModel toLocalDateTime(@Nonnull final Moment moment) {
		return new GwtLocalDateTimeModel(toLocalDate(moment), toLocalTime(moment));
	}

	@Nonnull
	public static GwtDateTimeModel toDateTime(@Nonnull final Moment moment) {
		return new GwtDateTimeModel(toLocalDate(moment), toLocalTime(moment), moment.utcOffset() * 60);
	}

	@Nonnull
	private static final LoadingCache<GwtLocalDateModel, Integer> weelCountCache = CacheBuilder.newBuilder().expireAfterWrite(30, TimeUnit.SECONDS).build(new CacheLoader<GwtLocalDateModel, Integer>() {
		@SuppressWarnings("null")
		@Override
		public Integer load(final GwtLocalDateModel key) throws Exception {
			return Integer.valueOf(countWeeks(Moment.utc(), toMomentUtc(key)));
		}
	});

	private Dates() {
		super();
	}

	/**
	 * @return first day of year relative to the current with given offset
	 */
	@Nonnull
	public static GwtLocalDateModel getCurrentYearStartDay(final int yearOffset) {
		final GwtLocalDateModel result = startOfYear(today());
		return yearOffset == 0 ? result : change(result, yearOffset, MomentKey.YEARS);
	}

	@Nonnull
	public static GwtLocalDateModel startOfYear(@Nonnull final GwtLocalDateModel anyYearDay) {
		return toLocalDate(toMomentUtc(anyYearDay).startOf(MomentPrecision.YEAR.shorthand));
	}

	public static int getWeek(@Nonnull final GwtLocalDateModel date) {
		return toMomentUtc(date).week();
	}

	@Nonnull
	public static GwtLocalDateModel getCurrentWeekStartDay(final int weekOffset) {
		final GwtLocalDateModel result = firstDayOfWeek(today());
		return weekOffset == 0 ? result : change(result, weekOffset, MomentKey.WEEKS);
	}

	public static boolean isSameDay(final GwtLocalDateModel o, final GwtLocalDateModel t) {
		return o != null && t != null && o.getYear() == t.getYear() && o.getMonth() == t.getMonth() && o.getDay() == t.getDay();
	}

	public static boolean betweenTime(@Nonnull final GwtLocalTimeModel time, @Nonnull final GwtLocalTimeModel startTime, @Nonnull final GwtLocalTimeModel endTime) {
		return betweenTime(time, startTime, toMiliseconds(endTime));
	}

	/**
	 * @param duration
	 *            in milliseconds
	 */
	public static boolean betweenTime(@Nonnull final GwtLocalTimeModel time, @Nonnull final GwtLocalTimeModel startTime, final long duration) {
		final int tMs = toMiliseconds(time);
		final int miliseconds = toMiliseconds(startTime);
		return tMs >= miliseconds && tMs < miliseconds + duration;
	}

	public static boolean between(@Nonnull final GwtLocalDateTimeModel testTime, @Nonnull final GwtLocalDateTimeModel start, final GwtLocalDateTimeModel end) {
		return toMomentUtc(testTime).isBetween(toMomentUtc(start), toMomentUtc(end), MomentPrecision.SECOND.shorthand, MomentInterval.OPEN_CLOSE.shorthand);
	}

	/** start:start+duration inside of rangeStart:rangeEnd */
	public static boolean between(final long start, final long end, final long rangeStart, final long rangeEnd) {
		return start >= rangeStart && end <= rangeEnd;
	}

	/** start:start+duration inside of range */
	public static boolean between(@Nonnull final GwtLocalDateTimeModel testStart, final int testDuration, @Nonnull final GwtLocalDateModel rangeDate, @Nonnull final LocalTimeRange range) {
		if (isSameDay(testStart.getDate(), rangeDate)) {
			final Moment startMoment = toMomentUtc(testStart);
			final Moment endMoment = startMoment.copy().add(testDuration, MomentKey.MILLISECONDS.getShorthand());
			final Moment startRange = toMomentUtc(new GwtLocalDateTimeModel(rangeDate, range.getStart()));
			final Moment endRanget = startRange.copy().add(range.getDuration(), MomentKey.MILLISECONDS.getShorthand());

			return startMoment.isSameOrAfter(startRange) && endMoment.isSameOrBefore(endRanget);
		}
		return false;
	}

	/** if start-end overlap with day */
	public static boolean isOverlap(@Nonnull final GwtLocalDateModel start, @Nonnull final GwtLocalDateModel end, @Nonnull final GwtLocalDateModel day) {
		return isOverlap(start, end, day, SharedDates.DAY_LENGTH);
	}

	public static boolean isOverlap(@Nonnull final GwtLocalDateModel start1, final int days, @Nonnull final GwtLocalDateModel start2) {
		final Moment startDay = toMomentUtc(start1);
		final Moment endDay = startDay.copy().add(days, MomentKey.DAYS.getShorthand());
		final Moment checkDate = toMomentUtc(start2);
		return checkDate.isBetween(startDay, endDay, MomentPrecision.DAY.shorthand, MomentInterval.OPEN_CLOSE.shorthand);
	}

	public static boolean isOverlap(@Nonnull final GwtLocalDateModel start1, @Nonnull final GwtLocalDateModel end1, @Nonnull final GwtLocalDateModel start2, final int duration2) {
		return isOverlap(toMiliseconds(start1), diff(start1, end1), toMiliseconds(start2), duration2);
	}

	public static boolean isOverlap(@Nonnull final GwtLocalTimeModel start1, final int duration1, @Nonnull final GwtLocalTimeModel start2, final int duration2) {
		return isOverlap(toMiliseconds(start1), duration1, toMiliseconds(start2), duration2);
	}

	public static boolean isOverlap(@Nonnull final GwtLocalDateTimeModel start1, final int duration1, @Nonnull final GwtLocalDateTimeModel start2, final int duration2) {
		return isOverlap(toMiliseconds(start1), duration1, toMiliseconds(start2), duration2);
	}

	/**
	 * @param graceTime
	 *            allow to overlap time
	 */
	public static boolean isOverlap(final long s1, final int duration1, final long s2, final int duration2, final int graceTime) {
		final long f1 = s1 + duration1;
		final long f2 = s2 + duration2;
		return f2 - s1 > graceTime && f1 - s2 > graceTime && (s2 <= s1 && f2 > s1 || s2 > s1 && f2 <= f1 || s2 < f1 && f2 >= f1);
	}

	public static boolean isOverlap(final long start1, final int duration1, final long start2, final int duration2) {
		final long finish1 = start1 + duration1;
		final long finish2 = start2 + duration2;
		return start2 <= start1 && finish2 > start1 || start2 > start1 && finish2 <= finish1 && duration2 != 0 || start2 < finish1 && finish2 >= finish1 && duration1 != 0;
	}

	public static boolean isOverlap(@Nonnull final LocalTimeRange range1, @Nonnull final LocalTimeRange range2) {
		final long start1 = toMiliseconds(range1.getStart());
		final int duration1 = range1.getDuration();
		final long start2 = toMiliseconds(range2.getStart());
		final int duration2 = range2.getDuration();
		return isOverlap(start1, duration1, start2, duration2);
	}

	public static String getDatePatternWithoutYear() {
		final String pattern = LocaleInfo.getCurrentLocale()
				.getDateTimeFormatInfo()
				.dateFormat();
		return pattern.replaceAll(REMOVE_YEAR_PATTERN, ""); //$NON-NLS-1$
	}

	public static String getDateLongPatternWithoutYear() {
		final String pattern = LocaleInfo.getCurrentLocale()
				.getDateTimeFormatInfo()
				.dateFormatLong();
		return pattern.replaceAll(REMOVE_YEAR_PATTERN, ""); //$NON-NLS-1$
	}

	/**
	 * Find out if hours is overlap in single day
	 *
	 * @param range
	 *            - take into account only time part
	 * @param startTime
	 *            - take into account only time part
	 */
	public static boolean isHoursOverlap(final LocalTimeRange range, @Nonnull final GwtLocalTimeModel startTime, final int duration) {
		return isOverlap(range.getStart(), range.getDuration(), startTime, duration);
	}

	public static int compareDateTime(@Nonnull final GwtLocalDateTimeModel date1, @Nonnull final GwtLocalDateTimeModel date2, final MomentPrecision precision) {
		final Moment moment1 = toMomentUtc(date1.getDate(), date1.getTime(), precision);
		final Moment moment2 = toMomentUtc(date2.getDate(), date2.getTime(), precision);
		return Doubles.compare(moment1.valueOf(), moment2.valueOf());
	}

	public static long compareDate(@Nonnull final GwtLocalDateModel date1, @Nonnull final GwtLocalDateModel date2) {
		final int yearDiff = date1.getYear() - date2.getYear();
		if (yearDiff != 0) {
			return yearDiff;
		}
		final int monthDiff = date1.getMonth() - date2.getMonth();
		if (monthDiff != 0) {
			return monthDiff;
		}
		return date1.getDay() - date2.getDay();
	}

	public static long compareDate(@Nonnull final GwtLocalDateModel date1, @Nonnull final GwtLocalDateModel date2, final MomentPrecision precision) {
		long result = 0;
		switch (precision) {
		case MILLISECOND:
		case SECOND:
		case MINUTE:
		case HOUR:
		case DAY:
			result += date1.getDay() - date2.getDay();
			//$FALL-THROUGH$
		case MONTH:
			result += (date1.getMonth() - date2.getMonth()) * 31;
			//$FALL-THROUGH$
		case YEAR:
			result += (date1.getYear() - date2.getYear()) * 365;
			break;
		case WEEK:
			result = getWeek(date1) - getWeek(date2);
			result += (date1.getYear() - date2.getYear()) * 100;
			break;
		default:
		}
		return result;
	}

	public static int compareTime(@Nonnull final GwtLocalTimeModel date1, @Nonnull final GwtLocalTimeModel date2, final MomentPrecision precision) {
		int x = 1;
		switch (precision) {
		case HOUR:
			x *= 60;
			//$FALL-THROUGH$
		case MINUTE:
			x *= 60;
			//$FALL-THROUGH$
		case SECOND:
			x *= 1000;
			//$FALL-THROUGH$
		case MILLISECOND:
			break;
		case DAY:
			//$FALL-THROUGH$
		case MONTH:
			//$FALL-THROUGH$
		case YEAR:
			//$FALL-THROUGH$
		default:
			return 0;
		}
		return (toMiliseconds(date1) - toMiliseconds(date2)) / x;
	}

	public static int compareTime(@Nonnull final GwtLocalTimeModel date1, @Nonnull final GwtLocalTimeModel date2) {
		return compareTime(date1, date2, MomentPrecision.MILLISECOND);
	}

	/** Print ONLY date range ignore time */
	@Nonnull
	public static String printDateRange(@Nonnull final GwtLocalDateModel startTime, @Nonnull final GwtLocalDateModel endTime, final boolean printYear) {
		final DateTimeFormat fy = DateTimeFormat.getFormat(LocaleInfo.getCurrentLocale().getDateTimeFormatInfo().dateFormat());
		final DateTimeFormat f = DateTimeFormat.getFormat(getDatePatternWithoutYear());
		if (isSameDay(startTime, endTime)) {
			return printYear ? fy.format(toDate(startTime)) : f.format(toDate(startTime));
		} else if (isSameMonth(startTime, endTime)) {
			if (printYear) {
				return BaseRs.FMT.rangeSameMonthWithYear(startTime.getDay(), endTime.getDay(), printMonth(startTime), endTime.getYear());
			}
			return BaseRs.FMT.rangeSameMonth(startTime.getDay(), endTime.getDay(), printMonth(startTime));
		} else if (isSameYear(startTime, endTime)) {
			if (printYear) {
				return BaseRs.FMT.rangeWithYear(f.format(toDate(startTime)), f.format(toDate(endTime)), endTime.getYear());
			}
			return BaseRs.FMT.range(f.format(toDate(startTime)), f.format(toDate(endTime)));
		} else {
			if (printYear) {
				return BaseRs.FMT.range(fy.format(toDate(startTime)), fy.format(toDate(endTime)));
			}
			return BaseRs.FMT.range(f.format(toDate(startTime)), f.format(toDate(endTime)));
		}
	}

	/**
	 * Print date range from date plus days.
	 */
	public static String printDateRange(@Nonnull final GwtLocalDateModel startTime, final int days, final boolean printYear) {
		return printDateRange(startTime, changeDay(startTime, days - 1), printYear);
	}

	/**
	 * Returns the date time range, even if dates different
	 *
	 * @param duration
	 *            - in MS
	 */
	public static String printDateTimeRange(@Nonnull final GwtLocalDateTimeModel startTime, final int duration) {
		final String t = LocaleInfo.getCurrentLocale()
				.getDateTimeFormatInfo()
				.timeFormatShort();
		final Date startDate = toDate(startTime);
		if (duration == 0) {
			final DateTimeFormat ft = DateTimeFormat.getFormat(t);
			return ft.format(startDate);
		}
		final GwtLocalDateTimeModel endTime = change(startTime, duration);
		final Date endDate = toDate(endTime);
		if (isSameDay(startTime.getDate(), endTime.getDate())) {
			final DateTimeFormat ft = DateTimeFormat.getFormat(t);
			return BaseRs.FMT.rangeSameDay(ft.format(startDate), ft.format(endDate), printDayOfMonth(startTime.getDate()));
		}
		final String p = LocaleInfo.getCurrentLocale()
				.getDateTimeFormatInfo()
				.dateTimeShort(getDatePatternWithoutYear(), t);
		final DateTimeFormat fp = DateTimeFormat.getFormat(p);
		return BaseRs.FMT.range(fp.format(startDate), fp.format(endDate));
	}

	/**
	 * Prints full week day name by day index [0..6]
	 *
	 * @param localDay
	 *            - index of day from 0 to 6, 0 is fist day of week @return
	 *            localized weekday name
	 */
	@SuppressWarnings("null")
	@Nonnull
	public static String printWeekday(final int localDay) {
		final DateTimeFormatInfo info = LocaleInfo.getCurrentLocale().getDateTimeFormatInfo();
		final int requestedDay = localDay < 0 ? 0 : localDay > 6 ? 6 : localDay + info.firstDayOfTheWeek();
		return info.weekdaysFullStandalone()[requestedDay > 6 ? requestedDay - 7 : requestedDay];
	}

	public static String printWeek(@Nonnull final GwtLocalDateModel date) {
		return BaseRs.FMT.printWeek(Integer.valueOf(getWeek(date)), printYear(date));
	}

	/** 1-12 */
	public static String printMonth(final int month) {
		return LocaleInfo.getCurrentLocale().getDateTimeFormatInfo().monthsFullStandalone()[month - 1];
	}

	public static String printMonth(@Nonnull final GwtLocalDateModel date) {
		return printMonth(date.getMonth());
	}

	public static String printYear(@Nonnull final GwtLocalDateModel date) {
		return DateTimeFormat.getFormat(LocaleInfo.getCurrentLocale()
				.getDateTimeFormatInfo()
				.formatYear())
				.format(toDate(date));
	}

	@Nonnull
	public static String printTimeRange(@Nonnull final GwtLocalTimeModel startTime, @Nonnull final GwtLocalTimeModel endTime) {
		return isSameTime(startTime, endTime) ? printTime(startTime) : BaseRs.FMT.range(printTime(startTime), printTime(endTime));
	}

	public static String printTimeRange(@Nonnull final GwtLocalTimeModel startTime, final int duration) {
		return printTimeRange(startTime, new GwtLocalTimeModel(toMiliseconds(startTime) + duration));
	}

	/**
	 * Returns the ONLY the date
	 *
	 * @param date
	 *            to print @return only date string
	 */
	@SuppressWarnings("null")
	@Nonnull
	public static String printDateWoYear(@Nonnull final GwtLocalDateModel date) {
		final DateTimeFormat f = DateTimeFormat.getFormat(getDatePatternWithoutYear());
		return f.format(toDate(date));
	}

	/**
	 * Returns the ONLY the date
	 *
	 * @param dateTime
	 *            date to print
	 * @return only date string
	 */
	@Nonnull
	public static String printDate(@Nonnull final GwtLocalDateTimeModel dateTime) {
		return printDate(dateTime.getDate());
	}

	/**
	 * Returns the ONLY the date
	 *
	 * @param date
	 *            to print @return only date string
	 */
	@SuppressWarnings("null")
	@Nonnull
	public static String printDate(@Nonnull final GwtLocalDateModel date) {
		final String p = getFormatInfo().dateFormatShort();
		final DateTimeFormat f = DateTimeFormat.getFormat(p);
		return f.format(toDate(date));
	}

	/**
	 * Returns the ONLY the time
	 *
	 * @return only time string
	 */
	public static String printTime(final int hours, final int minutes) {
		return printTime(new GwtLocalTimeModel(hours, minutes));
	}

	/**
	 * Returns the ONLY the time
	 *
	 * @return only time string
	 */
	@SuppressWarnings("null")
	@Nonnull
	public static String printTime(@Nonnull final GwtLocalTimeModel time) {
		final String p = getFormatInfo().timeFormatShort();
		final DateTimeFormat f = DateTimeFormat.getFormat(p);
		return f.format(toDate(time));
	}

	public static String printTime(@Nonnull final GwtLocalDateTimeModel time) {
		return printTime(time.getTime());
	}

	/**
	 * Returns the Date string
	 *
	 * @param date
	 *            to print
	 *
	 * @return date string
	 */
	@SuppressWarnings("null")
	@Nonnull
	public static String print(@Nonnull final GwtLocalDateModel date, @Nonnull final String format) {
		return DateTimeFormat.getFormat(format).format(toDate(date));
	}

	@Nonnull
	public static String printDateTime(@Nonnull final GwtLocalDateTimeModel dateTime) {
		return printDateTime(dateTime, null);
	}

	@Nonnull
	public static String printDateTime(@Nonnull final GwtLocalDateModel day, @Nonnull final GwtLocalTimeModel time) {
		return printDateTime(new GwtLocalDateTimeModel(day, time), null);
	}

	@SuppressWarnings("null")
	@Nonnull
	private static String printDateTimeFs(@Nonnull final GwtLocalDateTimeModel date, @Nullable final TimeZone timeZone) {
		final String d = "yMMdd"; //$NON-NLS-1$
		final String t = "HHmmss"; //$NON-NLS-1$
		return DateTimeFormat.getFormat(getFormatInfo().dateTimeShort(t, d))
				.format(toDate(date), timeZone);
	}

	@SuppressWarnings("null")
	@Nonnull
	private static String printDateTime(@Nonnull final GwtLocalDateTimeModel date, @Nullable final TimeZone timeZone) {
		final String t = getFormatInfo().timeFormatShort();
		final String d = getFormatInfo().dateFormatShort();
		return DateTimeFormat.getFormat(getFormatInfo().dateTimeShort(t, d))
				.format(toDate(date), timeZone);
	}

	@Nonnull
	public static String printDateTime(@Nonnull final GwtDateTimeModel date) {
		return printDateTime(date, getTimeZone(date));
	}

	@Nonnull
	public static String printDateTimeFs(@Nonnull final GwtDateTimeModel date) {
		return printDateTimeFs(date, getTimeZone(date));
	}

	@SuppressWarnings("null")
	@Nonnull
	public static TimeZone getTimeZone(@Nonnull final GwtDateTimeModel date) {
		return com.google.gwt.i18n.client.TimeZone.createTimeZone(Ints.saturatedCast(date.getOffset() / (60 * 1000)));
	}

	public static String printDayOfMonth(@Nonnull final GwtLocalDateModel date) {
		return DateTimeFormat.getFormat(LocaleInfo.getCurrentLocale()
				.getDateTimeFormatInfo()
				.formatMonthAbbrevDay())
				.format(toDate(date));
	}

	/** Print duration with specified precision */
	public static String printDuration(final long duration, final MomentPrecision precision) {
		return printDuration(duration, precision, TXT.year(), TXT.years(), TXT.month(), TXT.months(), TXT.day(), TXT.days(), TXT.hour(), TXT.hours(), TXT.minute(), TXT.minutes(), TXT.second(), TXT.seconds(), TXT.millisecond(), TXT.milliseconds());
	}

	/** Print duration with specified precision */
	public static String printDuration(final long duration, final MomentPrecision precision, final String y, final String ys, final String m, final String ms, final String d, final String ds, final String h, final String hs, final String mi, final String mis, final String s, final String ss, final String ml, final String mls) {
		long du = duration;
		final long years = du / precision(MomentPrecision.YEAR);
		final long months = isEnought(MomentPrecision.MONTH, precision) ? 0 : (du -= precision(MomentPrecision.YEAR) * years) / precision(MomentPrecision.MONTH);
		final long days = isEnought(MomentPrecision.DAY, precision) ? 0 : (du -= precision(MomentPrecision.MONTH) * months) / precision(MomentPrecision.DAY);
		final long hours = isEnought(MomentPrecision.HOUR, precision) ? 0 : (du -= precision(MomentPrecision.DAY) * days) / precision(MomentPrecision.HOUR);
		final long minutes = isEnought(MomentPrecision.MINUTE, precision) ? 0 : (du -= precision(MomentPrecision.HOUR) * hours) / precision(MomentPrecision.MINUTE);
		final long seconds = isEnought(MomentPrecision.SECOND, precision) ? 0 : (du -= precision(MomentPrecision.MINUTE) * minutes) / precision(MomentPrecision.SECOND);
		final long milliseconds = isEnought(MomentPrecision.MILLISECOND, precision) ? 0 : (du -= precision(MomentPrecision.SECOND) * seconds) / precision(MomentPrecision.MILLISECOND);
		final String join = Joiner.on(", ") //$NON-NLS-1$
				.skipNulls()
				.join(str(years, y, ys), str(months, m, ms), str(days, d, ds), str(hours, h, hs), str(minutes, mi, mis), str(seconds, s, ss), str(milliseconds, ml, mls));
		return join.isEmpty() ? defaultDuration(precision, ys, ms, ds, hs, mis, ss, mls) : join;
	}

	/**
	 * Return string with default duration for specified precision
	 *
	 * @param precision
	 *            - target precision
	 * @return the default (usually zero) string
	 */
	private static String defaultDuration(final MomentPrecision precision, final String years, final String months, final String days, final String hours, final String minutes, final String seconds, final String milliseconds) {
		switch (precision) {
		case YEAR:
			return "0 " + years; //$NON-NLS-1$
		case MONTH:
			return "0 " + months; //$NON-NLS-1$
		case DAY:
			return "0 " + days; //$NON-NLS-1$
		case HOUR:
			return "0 " + hours; //$NON-NLS-1$
		case MINUTE:
			return "0 " + minutes; //$NON-NLS-1$
		case SECOND:
			return "0 " + seconds; //$NON-NLS-1$
		default:
			return "0 " + milliseconds; //$NON-NLS-1$
		}
	}

	/**
	 * Calculate if specified precision is reached
	 *
	 * @param current
	 *            - current precision
	 * @param target
	 *            - target precision
	 */
	private static boolean isEnought(final MomentPrecision current, final MomentPrecision target) {
		return target.compareTo(current) > 0;
	}

	private static String str(final long unit, final String single, final String multiple) {
		return Strings.emptyToNull(BaseRs.FMT.str(Ints.saturatedCast(unit), single, multiple));
	}

	public static int precision(final MomentPrecision unit) {
		switch (unit) {
		case YEAR:
			return 365 * 24 * 60 * 60 * 1000;
		case MONTH:
			return 30 * 24 * 60 * 60 * 1000;
		case WEEK:
			return 7 * 24 * 60 * 60 * 1000;
		case DAY:
			return 24 * 60 * 60 * 1000;
		case HOUR:
			return 60 * 60 * 1000;
		case MINUTE:
			return 60 * 1000;
		case SECOND:
			return 60 * 1000;
		default:
			return 1000;
		}
	}

	public static boolean isSameTime(@Nonnull final GwtLocalTimeModel oneTime, @Nonnull final GwtLocalTimeModel secondTime) {
		return compareTime(oneTime, secondTime, MomentPrecision.MINUTE) == 0;
	}

	public static boolean isSameWeek(@Nonnull final GwtLocalDateModel oneDate, @Nonnull final GwtLocalDateModel anotherDate) {
		return getWeek(oneDate) == getWeek(anotherDate) && isSameYear(oneDate, anotherDate);
	}

	public static boolean isSameMonth(@Nonnull final GwtLocalDateModel oneDate, @Nonnull final GwtLocalDateModel anotherDate) {
		return oneDate.getMonth() == anotherDate.getMonth() && isSameYear(oneDate, anotherDate);
	}

	public static boolean isSameYear(@Nonnull final GwtLocalDateModel oneDate, @Nonnull final GwtLocalDateModel anotherDate) {
		return oneDate.getYear() == anotherDate.getYear();
	}

	public static DateTimeFormatInfo getFormatInfo() {
		return LocaleInfo.getCurrentLocale()
				.getDateTimeFormatInfo();
	}

	@Nonnull
	public static GwtLocalDateModel getSurroundStart(@Nonnull final GwtLocalDateModel date, final MomentPrecision precision) {
		int day = date.getDay();
		int month = date.getMonth();
		switch (precision) {
		case YEAR:
			month = 1;
			//$FALL-THROUGH$
		case MONTH:
			day = 1;
			return new GwtLocalDateModel(day, month, date.getYear());
		case WEEK:
			return firstDayOfWeek(date);
		case DAY:
		case HOUR:
		case MINUTE:
		case SECOND:
		case MILLISECOND:
		default:
			return date;
		}
	}

	@Nonnull
	public static GwtLocalTimeModel change(@Nonnull final GwtLocalTimeModel time, final int amount) {
		int value = toMiliseconds(time) + amount;
		if (value >= SharedDates.DAY_LENGTH) {
			value -= SharedDates.DAY_LENGTH;
		} else if (value < 0) {
			value += SharedDates.DAY_LENGTH;
		}
		return new GwtLocalTimeModel(value);
	}

	@Nonnull
	public static GwtLocalDateTimeModel change(@Nonnull final GwtLocalDateTimeModel dateTime, final int amount) {
		return change(dateTime, amount, MomentKey.MILLISECONDS);
	}

	@Nonnull
	public static GwtLocalDateModel changeDay(@Nonnull final GwtLocalDateModel date, final int amount) {
		return amount == 0 ? date : change(date, amount, MomentKey.DAYS);
	}

	@Nonnull
	public static GwtLocalDateModel change(@Nonnull final GwtLocalDateModel date, final int amount, @Nonnull final MomentKey momentKey) {
		return amount == 0 ? date : toLocalDate(toMomentUtc(date).add(amount, momentKey.getShorthand()));
	}

	@Nonnull
	public static GwtLocalDateTimeModel change(@Nonnull final GwtLocalDateTimeModel date, final int amount, @Nonnull final MomentKey momentKey) {
		return amount == 0 ? date : toLocalDateTime(toMomentUtc(date).add(amount, momentKey.getShorthand()));
	}

	@Nonnull
	public static GwtLocalTimeModel change(@Nonnull final GwtLocalTimeModel time, final int amount, final MomentKey momentKey) {
		return amount == 0 ? time : toLocalTime(toMomentUtc(time).add(amount, momentKey.getShorthand()));
	}

	public static int toMiliseconds(final int amount, final MomentKey field) {
		switch (field) {
		case DAYS:
			return amount * SharedDates.DAY_LENGTH;
		case HOURS:
			return amount * 60 * 60 * 1000;
		case MINUTES:
			return amount * 60 * 1000;
		case SECONDS:
			return amount * 1000;
		case MILLISECONDS:
			return amount;
		case YEARS:
		case MONTHS:
		default:
			throw new IllegalArgumentException(BaseRs.FMT.wrongDurationKey(field));
		}
	}

	/**
	 * Calculates if the date is in the past
	 *
	 * @return <code>true</code> if past
	 */
	public static boolean isPast(@Nonnull final GwtLocalDateModel date) {
		return compareDate(date, today()) < 0;
	}

	/**
	 * Merge to overlap ranges
	 *
	 * @return single range with early start and later end
	 */
	@Nonnull
	public static LocalTimeRange merge(@Nonnull final LocalTimeRange range1, @Nonnull final LocalTimeRange range2) {
		return new LocalTimeRange() {
			@Override
			public GwtLocalTimeModel getStart() {
				return early(range1.getStart(), range2.getStart());
			}

			@Override
			public int getDuration() {
				final GwtLocalTimeModel end1 = Dates.change(range1.getStart(), range1.getDuration());
				final GwtLocalTimeModel end2 = Dates.change(range2.getStart(), range2.getDuration());
				return toMiliseconds(later(end1, end2)) - toMiliseconds(getStart());
			}
		};
	}

	/**
	 * @return the early date from two given
	 */
	@Nonnull
	public static GwtLocalTimeModel early(@Nonnull final GwtLocalTimeModel start, @Nonnull final GwtLocalTimeModel start2) {
		return earlyTime(start, start2) ? start : start2;
	}

	public static boolean earlyTime(@Nonnull final GwtLocalTimeModel oneTime, @Nonnull final GwtLocalTimeModel anotherTime) {
		return toMiliseconds(oneTime) < toMiliseconds(anotherTime);
	}

	/**
	 * @return the early date from two given
	 */
	@Nonnull
	public static GwtLocalTimeModel later(@Nonnull final GwtLocalTimeModel end, @Nonnull final GwtLocalTimeModel end2) {
		return !earlyTime(end, end2) ? end : end2;
	}

	@Nonnull
	public static GwtLocalDateModel today() {
		return toLocalDate(getLocalMomentNow());
	}

	@Nonnull
	public static GwtLocalTimeModel now() {
		return toLocalTime(getLocalMomentNow());
	}

	@Nonnull
	public static GwtDateTimeModel todayNow() {
		return toDateTime(Moment.moment());
	}

	@Nonnull
	public static GwtLocalDateTimeModel todayNowLocal() {
		return toLocalDateTime(getLocalMomentNow());
	}

	@Nonnull
	private static Moment getLocalMomentNow() {
		return Moment.utc().add(Moment.moment().utcOffset(), MomentKey.MINUTES.getShorthand());
	}

	@Nonnull
	public static GwtLocalDateModel clone(@Nonnull final GwtLocalDateModel date) {
		return new GwtLocalDateModel(date);
	}

	public static String format(@Nonnull final GwtLocalDateModel date, @Nonnull final DateTimeFormat format) {
		return format.format(toDate(date));
	}

	/**
	 * Counts weeks between today and given date if date is in current week
	 * return 0.
	 */
	public static int countWeeks(@Nonnull final GwtLocalDateModel date) {
		return weelCountCache.getUnchecked(date).intValue();
	}

	public static int countWeeks(@Nonnull final GwtLocalDateModel date1, @Nonnull final GwtLocalDateModel date2) {
		return countWeeks(toMomentUtc(date1), toMomentUtc(date2));
	}

	/**
	 * Counts weeks between today and given date if date is in current week
	 * return 0.
	 */
	private static int countWeeks(@Nonnull final Moment date1, @Nonnull final Moment date2) {
		final Moment mon1 = date1.weekday(0).startOf(MomentPrecision.DAY.shorthand);
		final Moment mon2 = date2.weekday(0).startOf(MomentPrecision.DAY.shorthand);
		return mon2.diff(mon1, MomentKey.WEEKS.getShorthand());
	}

	public static boolean isToday(@Nonnull final GwtLocalDateModel day) {
		return compareDate(today(), day) == 0;
	}

	/**
	 * Calculate first day of the first week in the given month
	 *
	 * @param anyMonthDay
	 *            - day to select month
	 * @return first day of the first week. The day may belongs to the previous
	 *         month (in 90%)
	 */
	@Nonnull
	public static GwtLocalDateModel firstDayOfTheFirstMonthWeek(@Nonnull final GwtLocalDateModel anyMonthDay) {
		final Moment moment = toMomentUtc(anyMonthDay);
		// Set to the first month day, than local week day (0 is a first day)
		return toLocalDate(moment.date(1).weekday(0));
	}

	/**
	 * Calculate first day of a week
	 *
	 * @param anyWeekDay
	 *            - day to select week
	 * @return first day of the week.
	 */
	@Nonnull
	public static GwtLocalDateModel firstDayOfWeek(@Nonnull final GwtLocalDateModel anyWeekDay) {
		final Moment moment = toMomentUtc(anyWeekDay);
		// Set local week day (0 is a first day)
		return toLocalDate(moment.weekday(0));
	}

	public static int weekday(@Nonnull final GwtLocalDateModel date) {
		return toMomentUtc(date).weekday();
	}

	public static int isoWeekday(@Nonnull final GwtLocalDateModel date) {
		return toMomentUtc(date).isoWeekday();
	}

	public static int dayOfYear(@Nonnull final GwtLocalDateModel date) {
		return toMomentUtc(date).dayOfYear();
	}

	public static int dayOfMonth(@Nonnull final GwtLocalDateModel date) {
		return toMomentUtc(date).date();
	}

	/** Milliseconds from start to finish */
	public static int diff(@Nonnull final GwtLocalDateModel start, @Nonnull final GwtLocalDateModel end) {
		return -toMomentUtc(start).diff(toMomentUtc(end), MomentKey.MILLISECONDS.getShorthand());
	}

	/** Milliseconds from start to finish */
	public static int diff(@Nonnull final GwtLocalTimeModel start, @Nonnull final GwtLocalTimeModel end) {
		return -toMomentUtc(start).diff(toMomentUtc(end), MomentKey.MILLISECONDS.getShorthand());
	}

	/** Milliseconds from start to finish */
	public static int diff(@Nonnull final GwtLocalDateTimeModel start, @Nonnull final GwtLocalDateTimeModel end) {
		return -toMomentUtc(start).diff(toMomentUtc(end), MomentKey.MILLISECONDS.getShorthand());
	}

	public static GwtLocalTimeModel endRangeTime(@Nonnull final LocalTimeRange range) {
		return change(range.getStart(), range.getDuration());
	}

	@Nonnull
	public static GwtLocalTimeModel with(@Nonnull final GwtLocalTimeModel time, final MomentKey field, final int value) {
		final Moment moment = toMomentUtc(time);
		return toLocalTime(moment.set(field.getShorthand(), value));
	}

	public static Duration duration(final int amount, final MomentKey key) {
		return Moment.duration(amount, key.getShorthand());
	}

}
