package org.jresearch.commons.gwt.client.app;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.annotation.Nonnull;

import org.jresearch.commons.gwt.client.model.localization.LocaleModel;
import org.jresearch.commons.gwt.client.mvc.AbstractController;
import org.jresearch.commons.gwt.client.mvc.ViewCommand;
import org.jresearch.commons.gwt.client.mvc.event.AboutEvent;
import org.jresearch.commons.gwt.client.mvc.event.AboutHandler;
import org.jresearch.commons.gwt.client.mvc.event.Bus;
import org.jresearch.commons.gwt.client.mvc.event.ErrorEvent;
import org.jresearch.commons.gwt.client.mvc.event.ErrorHandler;
import org.jresearch.commons.gwt.client.mvc.event.InitEvent;
import org.jresearch.commons.gwt.client.mvc.event.InitHandler;
import org.jresearch.commons.gwt.client.mvc.event.SetLanguageEvent;
import org.jresearch.commons.gwt.client.mvc.event.SetLanguageHandler;
import org.jresearch.commons.gwt.client.mvc.event.module.ModuleEvent;
import org.jresearch.commons.gwt.client.mvc.event.module.ModuleHandler;
import org.jresearch.commons.gwt.client.service.localization.ILocalizationServiceAsync;

import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.http.client.UrlBuilder;
import com.google.gwt.i18n.client.LocaleInfo;
import com.google.gwt.inject.client.AsyncProvider;
import com.google.gwt.user.client.Cookies;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.Window.Location;
import com.google.gwt.user.client.rpc.AsyncCallback;

public abstract class AbstractAppController<V extends AbstractAppView<? extends AbstractAppController<V>>> extends AbstractController<V>implements AboutHandler, SetLanguageHandler, InitHandler, ErrorHandler, ModuleHandler {

    private static Logger logger = Logger.getLogger("org.jresearch.commons.gwt.app.client.mvc.AbstractAppController"); //$NON-NLS-1$

    @Nonnull
    private final ILocalizationServiceAsync localizationService;

    final List<IAppModule> modules;

    private String activeModule;

    public AbstractAppController(@Nonnull final String id, @Nonnull final ILocalizationServiceAsync localizationService, @Nonnull final Set<IAppModule> appModules, @Nonnull final AsyncProvider<V> view, @Nonnull final Bus bus) {
        super(id, bus, view);
        this.localizationService = localizationService;

        modules = new ArrayList<>(appModules);

        if (!modules.isEmpty()) {
            activeModule = modules.get(0)
                    .getModuleId();
        }

        bus.addHandler(InitEvent.TYPE, this);
        bus.addHandler(SetLanguageEvent.TYPE, this);
        bus.addHandler(AboutEvent.TYPE, this);
        bus.addHandler(ErrorEvent.TYPE, this);
        bus.addHandler(ModuleEvent.TYPE, this);

        // Handle empty anchor
        History.addValueChangeHandler(new ValueChangeHandler<String>() {
            @SuppressWarnings("synthetic-access")
            @Override
            public void onValueChange(final ValueChangeEvent<String> event) {
                if (!modules.isEmpty()) {
                    String historyToken = event.getValue();
                    // if (historyToken == null || historyToken.isEmpty()) {
                    historyToken = modules.get(0)
                            .getModuleId();
                    // }
                    bus.fire(new ModuleEvent(historyToken));
                }
            }
        });

    }

    @Override
    public void onInit(final InitEvent initEvent) {
        initView();
        showView();
    }

    protected void initView() {
        executeCommandWithLoad(new ViewCommand<V>() {
            @Override
            public void execute(final V view) {
                view.initModules(modules);
            }
        });
    }

    @Override
    public void onError(final ErrorEvent event) {
        final Throwable ex = event.getException();
        logger.log(Level.SEVERE, ex.getLocalizedMessage(), ex);
        final V view = getView();
        if (view != null) {
            view.onError(event);
        }
    }

    @Override
    @SuppressWarnings({ "deprecation" })
    public void onSetLanguage(final SetLanguageEvent event) {
        final LocaleModel locale = event.getLocale();
        // report to the server
        localizationService.setLocale(locale, getLanguageChangedCallback());
        final String cookieName = LocaleInfo.getLocaleCookieName();
        final String queryParam = LocaleInfo.getLocaleQueryParam();
        if (cookieName == null && queryParam == null) {
            // if there is no way for us to affect the locale, don nothing
            return;
        }

        if (cookieName != null) {
            // expire in one year
            final Date expires = new Date();
            expires.setYear(expires.getYear() + 1);
            Cookies.setCookie(cookieName, locale.getLocaleName(), expires);
        }
        if (queryParam != null) {
            final UrlBuilder builder = Location.createUrlBuilder()
                    .setParameter(queryParam, locale.getLocaleName());
            Window.Location.replace(builder.buildString());
        } else {
            // If we are using only cookies, just reload
            Window.Location.reload();
        }
    }

    /**
     * Can be overwrote to return more specific server language change callback
     *
     * @return default callback
     */
    @SuppressWarnings("static-method")
    protected AsyncCallback<Void> getLanguageChangedCallback() {
        return new AsyncCallback<Void>() {
            @Override
            public void onSuccess(final Void result) {
                // do nothing
            }

            @Override
            public void onFailure(final Throwable caught) {
                // do nothing
            }
        };
    }

    @Override
    public void onAbout(final AboutEvent event) {
        executeCommand(new ViewCommand<V>() {
            @Override
            public void execute(final V view) {
                view.showAbout();
            }
        });
    }

    @Override
    public AbstractController<?> getParentController() {
        return null;
    }

    public void handleAboutClick() {
        bus.fire(new AboutEvent());
    }

    @Override
    public void onModule(final ModuleEvent event) {
        activeModule = event.getModuleId();
        executeCommand(new ViewCommand<V>() {
            @Override
            public void execute(final V view) {
                view.switchToModule(activeModule);
            }
        });
    }

    public String getActiveModuleId() {
        return activeModule;
    }

    /**
     * Receive all
     */
    @Override
    public String getModuleId() {
        return ModuleEvent.ANY_ID;
    }

}
