/*
 * Copyright 2007 Future Earth, info@future-earth.eu
 * Copyright 2014 Stanislav Spiridonov, stas@jresearch.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package org.jresearch.commons.gwt.shared.tools;

import static org.jresearch.commons.gwt.client.base.resource.BaseRs.*;

import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;

import javax.annotation.Nonnull;

import org.jresearch.commons.gwt.client.model.time.GwtLocalDateModel;
import org.jresearch.commons.gwt.client.model.time.GwtLocalDateTimeModel;
import org.jresearch.commons.gwt.client.model.time.GwtLocalTimeModel;
import org.jresearch.commons.gwt.client.model.time.LocalDateModel;
import org.jresearch.commons.gwt.client.model.time.LocalTimeModel;

import com.google.common.base.Joiner;
import com.google.gwt.core.shared.GWT;
import com.google.gwt.i18n.client.LocaleInfo;
import com.google.gwt.i18n.shared.DateTimeFormat;
import com.google.gwt.i18n.shared.DateTimeFormatInfo;

public final class Dates {

    public static final long DAY_LENGTH = 24l * 60 * 60 * 1000;
    private static final String REMOVE_YEAR_PATTERN = "[^\\p{Alpha}]*y+[^\\p{Alpha}]*"; //$NON-NLS-1$

    private Dates() {
        super();
    }

    public static GwtLocalDateTimeModel localDateTime(final Date dateTime) {
        return new GwtLocalDateTimeModel(dateTime);
    }

    public static GwtLocalDateTimeModel localDateTime(@Nonnull final Date date, @Nonnull final Date time) {
        return new GwtLocalDateTimeModel(date, time);
    }

    public static Calendar createCalendar(final Date date) {
        final Calendar result = createCalendar();
        result.setTime(date);
        return result;
    }

    public static Calendar createCalendar(final GwtLocalDateModel date) {
        final Calendar result = createCalendar();
        result.setTime(date.getDate());
        return result;
    }

    public static Calendar createCalendar(@Nonnull final Date date, final Date time) {
        final Calendar result = createCalendar(date, Calendar.DAY_OF_MONTH);
        if (time != null) {
            final Calendar timeCalendar = createTimeCalendar(time, Calendar.MILLISECOND);
            result.add(Calendar.HOUR_OF_DAY, timeCalendar.get(Calendar.HOUR_OF_DAY));
            result.add(Calendar.MINUTE, timeCalendar.get(Calendar.MINUTE));
            result.add(Calendar.SECOND, timeCalendar.get(Calendar.SECOND));
            result.add(Calendar.MILLISECOND, timeCalendar.get(Calendar.MILLISECOND));
        }
        return result;
    }

    /**
     * Create {@link Calendar} with specific precision
     *
     * @param date
     * @param precision
     *            - one of the following Calendar constant (
     *            {@link Calendar#YEAR}, {@link Calendar#MONTH},
     *            {@link Calendar#DAY_OF_MONTH}, {@link Calendar#HOUR_OF_DAY} or
     *            {@link Calendar#HOUR}, {@link Calendar#MINUTE},
     *            {@link Calendar#SECOND}, {@link Calendar#MILLISECOND})
     * @return
     */
    public static Calendar createCalendar(@Nonnull final Date date, final int precision) {
        final Calendar result = createCalendar(date);
        switch (precision) {
        case Calendar.YEAR:
            result.set(Calendar.MONTH, 0);
            //$FALL-THROUGH$
        case Calendar.MONTH:
            result.set(Calendar.DAY_OF_MONTH, 1);
            //$FALL-THROUGH$
        case Calendar.DAY_OF_MONTH:
            result.set(Calendar.HOUR_OF_DAY, 0);
            //$FALL-THROUGH$
        case Calendar.HOUR_OF_DAY:
        case Calendar.HOUR:
            result.set(Calendar.MINUTE, 0);
            //$FALL-THROUGH$
        case Calendar.MINUTE:
            result.set(Calendar.SECOND, 0);
            //$FALL-THROUGH$
        case Calendar.SECOND:
            result.set(Calendar.MILLISECOND, 0);
            //$FALL-THROUGH$
        default:
        }
        return result;
    }

    /**
     * Create dateless {@link Calendar} with specific precision. Dateless means
     * that year, month and day have zero values
     *
     * @param date
     * @param precision
     *            - one of the following Calendar constant (
     *            {@link Calendar#HOUR_OF_DAY} or {@link Calendar#HOUR},
     *            {@link Calendar#MINUTE}, {@link Calendar#SECOND},
     *            {@link Calendar#MILLISECOND})
     * @return
     */
    public static Calendar createTimeCalendar(@Nonnull final Date date, final int precision) {
        final Calendar result = createCalendar(date);
        result.set(Calendar.YEAR, 0);
        result.set(Calendar.MONTH, 0);
        result.set(Calendar.DAY_OF_MONTH, 1);
        switch (precision) {
        case Calendar.HOUR_OF_DAY:
        case Calendar.HOUR:
            result.set(Calendar.MINUTE, 0);
            //$FALL-THROUGH$
        case Calendar.MINUTE:
            result.set(Calendar.SECOND, 0);
            //$FALL-THROUGH$
        case Calendar.SECOND:
            result.set(Calendar.MILLISECOND, 0);
            //$FALL-THROUGH$
        default:
        }
        return result;
    }

    public static Calendar createCalendar() {
        final Calendar result = new GregorianCalendar();
        result.setFirstDayOfWeek(findFirstDayOfWeek());
        result.setMinimalDaysInFirstWeek(4);
        return result;
    }

    private static int findFirstDayOfWeek() {
        // In GWT week days starts form 0
        return GWT.isClient() ? LocaleInfo.getCurrentLocale()
                .getDateTimeFormatInfo()
                .firstDayOfTheWeek() + 1 : Calendar.MONDAY;
    }

    @Nonnull
    public static Calendar getWeekStartDay(@Nonnull final Date someDayInWeek) {
        final Calendar helper = createCalendar(someDayInWeek);
        helper.set(Calendar.DAY_OF_WEEK, helper.getFirstDayOfWeek());
        return helper;
    }

    @Nonnull
    public static Calendar getWeekEndDay(@Nonnull final Date someDayInWeek) {
        final Calendar helper = getWeekStartDay(someDayInWeek);
        helper.add(Calendar.WEEK_OF_YEAR, 1);
        helper.add(Calendar.DAY_OF_YEAR, -1);
        return helper;
    }

    public static void setTime(final Calendar calendar, final int hours, final int minutes) {
        calendar.set(Calendar.HOUR_OF_DAY, hours);
        calendar.set(Calendar.MINUTE, minutes);
        calendar.set(Calendar.SECOND, minutes == 59 ? 59 : 0);
        calendar.set(Calendar.MILLISECOND, minutes == 59 ? 999 : 0);
    }

    @SuppressWarnings("deprecation")
    public static boolean isSameDay(final Date one, final Date two) {
        if (one == null || two == null) {
            return false;
        }

        if (one.getYear() != two.getYear()) {
            return false;
        }
        if (one.getMonth() != two.getMonth()) {
            return false;
        }
        if (one.getDate() != two.getDate()) {
            return false;
        }
        return true;
    }

    public static boolean isSameDay(final GwtLocalDateModel o, final GwtLocalDateModel t) {
        return o != null && t != null && o.getYear() == t.getYear() && o.getMonth() == t.getMonth() && o.getDate() == t.getDate();
    }

    public static boolean isSameDay(final Calendar one, final Calendar two) {
        if (one != null && two != null) {
            if (one.get(Calendar.YEAR) == two.get(Calendar.YEAR) && one.get(Calendar.DAY_OF_YEAR) == two.get(Calendar.DAY_OF_YEAR)) {
                return true;
            }
        }
        return false;
    }

    /**
     * This methis checks wheter the Date are the Same or the second is the 00
     * minute on the next day.
     *
     * @param one
     *            - The First Date
     * @param two
     *            - The Second date
     * @return
     */
    @SuppressWarnings("deprecation")
    public static boolean isSameDayOrNextDayZeroMinutes(final Date one, final Date two) {
        if (one == null || two == null) {
            return false;
        }

        if (one.getYear() != two.getYear()) {
            return false;
        }
        if (one.getMonth() != two.getMonth()) {
            return false;
        }
        if (one.getDate() != two.getDate()) {
            return !nextDayZeroMinutes(one, two);
        }
        return true;
    }

    /**
     * Check if the second date is the first minute of the next day.
     *
     * @param one
     *            The first date.
     * @param two
     *            The second date.
     * @return true of false
     */
    @SuppressWarnings("deprecation")
    public static boolean nextDayZeroMinutes(final Date one, final Date two) {
        if (one == null || two == null) {
            return false;
        }

        if (one.getYear() != two.getYear()) {
            return false;
        }
        if (one.getMonth() != two.getMonth()) {
            return false;
        }
        if (one.getDate() + 1 != two.getDate()) {
            return false;
        }
        if (0 != two.getHours() || 0 != two.getMinutes()) {
            return false;
        }
        return true;
    }

    @SuppressWarnings("deprecation")
    public static boolean zeroMinutes(final Date two) {
        if (two == null) {
            return false;
        }
        if (0 == two.getHours() && 0 == two.getMinutes()) {
            return true;
        }
        return false;
    }

    /**
     * Check only the time part, date part ignore (testTime:testTime+duration)
     * between [startDate:endDate]
     */
    public static boolean betweenTime(final Date testTime, final long duration, final Date startTime, final Date endTime) {
        final long dayTime = getDayTime(testTime);
        return between(dayTime, dayTime + duration, getDayTime(startTime), getDayTime(endTime));
    }

    /** testDate between startDate:endDate+duration */
    public static boolean between(final Date testDate, final Date startDate, final Date endDate, final long duration) {
        return between(testDate.getTime(), 0, startDate.getTime(), endDate.getTime() + duration);
    }

    /** start:end inside of range */
    public static boolean between(final Date start, final Date end, final ITimeRange range) {
        return between(start, end.getTime() - start.getTime(), range);
    }

    /** start:start+duration inside of range */
    public static boolean between(final Date start, final long duration, final ITimeRange range) {
        return between(start.getTime(), duration, range);
    }

    /** start:start+duration inside of range */
    public static boolean between(final long start, final long duration, final ITimeRange range) {
        return between(start, start + duration, range.getStart()
                .getTime(), range.getEnd()
                        .getTime());
    }

    /** start:start+duration inside of rangeStart:rangeEnd */
    public static boolean between(final long start, final long end, final long rangeStart, final long rangeEnd) {
        return start >= rangeStart && end <= rangeEnd;
    }

    /** start:start+duration inside of rangeStart:rangeEnd */
    public static boolean between(final Date start, final long duration, final Date rangeStrart, final Date rangeEnd) {
        return between(start.getTime(), start.getTime() + duration, rangeStrart.getTime(), rangeEnd.getTime());
    }

    /** if start-end overlap with day */
    public static boolean isOverlap(final Date start, final Date end, final Date day) {
        final Calendar helper = createCalendar(day);
        setTime(helper, 0, 0);
        return isOverlap(start, end, helper.getTime(), Dates.DAY_LENGTH);
    }

    public static boolean isOverlap(final GwtLocalDateModel start1, final GwtLocalDateModel end1, final GwtLocalDateModel start2, final long duration2) {
        return isOverlap(start1.getDate(), end1.getDate(), start2.getDate(), duration2);
    }

    public static boolean isOverlap(final Date start1, final Date end1, final Date start2, final long duration2) {
        return isOverlap(start1, end1.getTime() - start1.getTime(), start2, duration2);
    }

    /***
     *
     * @param start1
     *            start of the 1st interval
     * @param duration1
     *            duration of the 1st interval
     * @param start2
     *            start of the 2nd interval
     * @param duration2
     *            duration of the 2nd interval
     * @return
     */
    public static boolean isOverlap(final Date start1, final long duration1, final Date start2, final long duration2) {
        return isOverlap(start1.getTime(), duration1, start2.getTime(), duration2);
    }

    /**
     *
     * @param s1
     * @param duration1
     * @param s2
     * @param duration2
     * @param graceTime
     *            allow to overlap time
     * @return
     */
    public static boolean isOverlap(final long s1, final long duration1, final long s2, final long duration2, final long graceTime) {
        final long f1 = s1 + duration1;
        final long f2 = s2 + duration2;
        return f2 - s1 > graceTime && f1 - s2 > graceTime && (s2 <= s1 && f2 > s1 || s2 > s1 && f2 <= f1 || s2 < f1 && f2 >= f1);
    }

    public static boolean isOverlap(final long start1, final long duration1, final long start2, final long duration2) {
        final long finish1 = start1 + duration1;
        final long finish2 = start2 + duration2;
        return start2 <= start1 && finish2 > start1 || start2 > start1 && finish2 <= finish1 || start2 < finish1 && finish2 >= finish1;
    }

    public static boolean isOverlap(final ITimeRange range1, final ITimeRange range2) {
        final long start1 = range1.getStart()
                .getTime();
        final long duration1 = range1.getEnd()
                .getTime() - start1;
        final long start2 = range2.getStart()
                .getTime();
        final long duration2 = range2.getEnd()
                .getTime() - start2;
        return isOverlap(start1, duration1, start2, duration2);
    }

    public static boolean isOverlap(final Date start1, final long duration1, final ITimeRange range) {
        final long start2 = range.getStart()
                .getTime();
        final long duration2 = range.getEnd()
                .getTime() - start2;
        return isOverlap(start1.getTime(), duration1, start2, duration2);
    }

    /**
     *
     * @param start1
     * @param duration1
     * @param range
     * @param graceTime
     *            - allow to overlap time
     * @return
     */
    public static boolean isOverlap(final Date start1, final long duration1, final ITimeRange range, final long graceTime) {
        final long start2 = range.getStart()
                .getTime();
        final long duration2 = range.getEnd()
                .getTime() - start2;
        return isOverlap(start1.getTime(), duration1, start2, duration2, graceTime);
    }

    public static boolean isOverlap(final Date start, final long duration, final List<? extends ITimeRange> occupiedTime) {
        for (final ITimeRange range : occupiedTime) {
            if (isOverlap(start, duration, range)) {
                return true;
            }
        }
        return false;
    }

    public static boolean isOverlap(final Date start, final long duration, final List<? extends ITimeRange> occupiedTime, final long graceTime) {
        for (final ITimeRange range : occupiedTime) {
            if (isOverlap(start, duration, range, graceTime)) {
                return true;
            }
        }
        return false;
    }

    public static Date getDate(final Date start, final long duration) {
        return new Date(start.getTime() + duration);
    }

    public static String getDatePatternWithoutYear() {
        final String pattern = LocaleInfo.getCurrentLocale()
                .getDateTimeFormatInfo()
                .dateFormat();
        return pattern.replaceAll(REMOVE_YEAR_PATTERN, ""); //$NON-NLS-1$
    }

    public static String getDateLongPatternWithoutYear() {
        final String pattern = LocaleInfo.getCurrentLocale()
                .getDateTimeFormatInfo()
                .dateFormatLong();
        return pattern.replaceAll(REMOVE_YEAR_PATTERN, ""); //$NON-NLS-1$
    }

    /**
     * Find out if hours is overlap in single day
     *
     * @param day
     *            - day to check
     * @param range
     *            - take into account only time part
     * @param startTime
     *            - take into account only time part
     * @param duration
     * @return
     */
    public static boolean isHoursOverlap(final Date day, final ITimeRange range, final Date startTime, final long duration) {
        return isOverlap(setTime(day, range.getStart()), setTime(day, range.getEnd()), setTime(day, startTime), duration);
    }

    /**
     * Sets the time of day from the time parameter
     *
     * @param day
     *            - to set the time
     * @param time
     *            - time to set
     * @return new {@link Date} with date from day and time from time
     */
    public static Date setTime(final Date day, final Date time) {
        final Calendar dayHelper = createCalendar(day);
        final Calendar timeHelper = createCalendar(time);
        setTime(dayHelper, timeHelper.get(Calendar.HOUR_OF_DAY), timeHelper.get(Calendar.MINUTE));
        return dayHelper.getTime();
    }

    /**
     * Calculate time from midnight
     *
     * @param time
     *            any date
     * @return amount of MILLISECONDs from midnight
     */
    public static long getDayTime(final Date time) {
        return getDayTime(createCalendar(time));
    }

    /**
     * Calculate time from midnight
     *
     * @param time
     *            any date
     * @return amount of MILLISECONDs from midnight
     */
    public static long getDayTime(final Calendar time) {
        return ((time.get(Calendar.HOUR_OF_DAY) * 60 + time.get(Calendar.MINUTE)) * 60 + time.get(Calendar.SECOND)) * 1000 + time.get(Calendar.MILLISECOND);
    }

    /**
     * Set and return date with time from midnight
     *
     * @param dayTyme
     *            amount of MILLISECONDs from midnight
     * @return date with specific time (the date is undefined, depends from
     *         implementation)
     */
    public static Date setDayTime(final long dayTyme) {
        final Calendar helper = createCalendar();
        final long fullSeconds = dayTyme / 1000;
        final long fullMinutes = fullSeconds / 60;
        helper.set(Calendar.HOUR_OF_DAY, (int) ((fullMinutes / 60) % 24));
        helper.set(Calendar.MINUTE, (int) (fullMinutes % 60));
        helper.set(Calendar.SECOND, (int) (fullSeconds % 60));
        helper.set(Calendar.MILLISECOND, (int) (dayTyme % 1000));
        return helper.getTime();
    }

    /**
     * Compare two date with specified precision
     *
     * @param date1
     * @param date2
     * @param precision
     *            - one of the following Calendar constant (
     *            {@link Calendar#YEAR}, {@link Calendar#MONTH},
     *            {@link Calendar#DAY_OF_MONTH}, {@link Calendar#HOUR_OF_DAY} or
     *            {@link Calendar#HOUR}, {@link Calendar#MINUTE},
     *            {@link Calendar#SECOND}, {@link Calendar#MILLISECOND})
     * @return the value 0 if the time represented by the date2 is equal to the
     *         time represented by the date1; a value less than 0 if the time of
     *         the date1 is before the time represented by the date2; and a
     *         value greater than 0 if the time of the date1 is after the time
     *         represented by the date2.
     */
    public static long compareDateTime(@Nonnull final Date date1, @Nonnull final Date date2, final int precision) {
        final Calendar dayHelper1 = createCalendar(date1, precision);
        final Calendar dayHelper2 = createCalendar(date2, precision);
        return dayHelper1.getTimeInMillis() - dayHelper2.getTimeInMillis();
    }

    public static long compareDate(@Nonnull final GwtLocalDateModel date1, @Nonnull final GwtLocalDateModel date2) {
        return (date1.getYear() - date2.getYear()) + (date1.getMonth() - date2.getMonth()) + (date1.getDay() - date2.getDay());
    }

    /**
     * Compare the time part from given dates with specified precision
     *
     * @param date1
     * @param date2
     * @param precision
     *            - one of the following Calendar constant (
     *            {@link Calendar#HOUR_OF_DAY} or {@link Calendar#HOUR},
     *            {@link Calendar#MINUTE}, {@link Calendar#SECOND},
     *            {@link Calendar#MILLISECOND})
     * @return the value 0 if the time represented by the date2 is equal to the
     *         time represented by the date1; a value less than 0 if the time of
     *         the date1 is before the time represented by the date2; and a
     *         value greater than 0 if the time of the date1 is after the time
     *         represented by the date2.
     */
    public static long compareTime(@Nonnull final Date date1, @Nonnull final Date date2, final int precision) {
        final Calendar dayHelper1 = createTimeCalendar(date1, precision);
        final Calendar dayHelper2 = createTimeCalendar(date2, precision);
        return dayHelper1.getTimeInMillis() - dayHelper2.getTimeInMillis();
    }

    /**
     * Print ONLY date range ignore time
     *
     * @param startTime
     * @param endTime
     * @return
     */
    @SuppressWarnings("boxing")
    public static String printDateRange(@Nonnull final Date startTime, @Nonnull final Date endTime, final boolean printYear) {
        final DateTimeFormat fy = DateTimeFormat.getFormat(LocaleInfo.getCurrentLocale()
                .getDateTimeFormatInfo()
                .dateFormat());
        final DateTimeFormat f = DateTimeFormat.getFormat(getDatePatternWithoutYear());
        if (isSameDay(startTime, endTime)) {
            return printYear ? fy.format(startTime) : f.format(startTime);
        } else if (isSameMonth(startTime, endTime)) {
            if (printYear) {
                return Strings.format("{0}-{1} {2} {3}", getField(startTime, Calendar.DAY_OF_MONTH), getField(endTime, Calendar.DAY_OF_MONTH), printMonth(startTime), getField(endTime, Calendar.YEAR)); //$NON-NLS-1$
            }
            return Strings.format("{0}-{1} {2}", getField(startTime, Calendar.DAY_OF_MONTH), getField(endTime, Calendar.DAY_OF_MONTH), printMonth(startTime)); //$NON-NLS-1$
        } else if (isSameYear(startTime, endTime)) {
            if (printYear) {
                return Strings.format("{0} - {1} {2}", f.format(startTime), f.format(endTime), getField(endTime, Calendar.YEAR)); //$NON-NLS-1$
            }
            return Strings.format("{0} - {1}", f.format(startTime), f.format(endTime)); //$NON-NLS-1$
        } else {
            if (printYear) {
                return Strings.format("{0} - {1}", fy.format(startTime), fy.format(endTime)); //$NON-NLS-1$
            }
            return Strings.format("{0} - {1}", f.format(startTime), f.format(endTime)); //$NON-NLS-1$
        }
    }

    /**
     * Print ONLY date range ignore time. The year does not print.
     *
     * {@link Dates#printDateRange(Date, Date, boolean)} to print with year
     *
     * @param startTime
     * @param endTime
     * @return
     */
    public static String printDateRange(@Nonnull final Date startTime, @Nonnull final Date endTime) {
        return printDateRange(startTime, endTime, false);
    }

    public static String printDateTimeRange(final Date start, final long duration) {
        return printDateTimeRange(start, new Date(start.getTime() + duration));
    }

    public static String printDateTimeRange(final GwtLocalDateModel start, final long duration) {
        return printDateTimeRange(start.getDate(), new Date(start.getDate().getTime() + duration));
    }

    public static String printDateTimeRange(@Nonnull final GwtLocalDateModel start, @Nonnull final GwtLocalDateModel end) {
        return printDateTimeRange(start.getDate(), end.getDate());
    }

    /**
     * Returns the date time range, even if dates different
     *
     * @param startTime
     * @param endTime
     * @return
     */
    public static String printDateTimeRange(@Nonnull final Date startTime, @Nonnull final Date endTime) {
        final String d = getDatePatternWithoutYear();
        final String t = LocaleInfo.getCurrentLocale()
                .getDateTimeFormatInfo()
                .timeFormatShort();
        if (isSameTime(startTime, endTime)) {
            final DateTimeFormat ft = DateTimeFormat.getFormat(t);
            return ft.format(startTime);
        } else if (isSameDay(startTime, endTime)) {
            final DateTimeFormat ft = DateTimeFormat.getFormat(t);
            return Strings.format("{0}-{1} {2}", ft.format(startTime), ft.format(endTime), printDayOfMonth(startTime)); //$NON-NLS-1$
        } else {
            final String p = LocaleInfo.getCurrentLocale()
                    .getDateTimeFormatInfo()
                    .dateTimeShort(d, t);
            final DateTimeFormat fp = DateTimeFormat.getFormat(p);
            return Strings.format("{0} - {1}", fp.format(startTime), fp.format(endTime)); //$NON-NLS-1$ ;
        }
    }

    /**
     * Prints full week day name by day index [0..6]
     *
     * @param dayIndex
     *            - index of day from 0 to 6, 0 is fist day of week
     * @return localized weekday name
     */
    @SuppressWarnings("null")
    @Nonnull
    public static String printWeekday(final int dayIndex) {
        final DateTimeFormatInfo info = LocaleInfo.getCurrentLocale().getDateTimeFormatInfo();
        final int requestedDay = dayIndex < 0 ? 0 : dayIndex > 6 ? 6 : dayIndex + info.firstDayOfTheWeek();
        return info.weekdaysFullStandalone()[requestedDay > 6 ? requestedDay - 7 : requestedDay];
    }

    public static String printMonth(@Nonnull final Date date) {
        return DateTimeFormat.getFormat(LocaleInfo.getCurrentLocale()
                .getDateTimeFormatInfo()
                .formatMonthFull())
                .format(date);
    }

    public static String printYear(final Date date) {
        return DateTimeFormat.getFormat(LocaleInfo.getCurrentLocale()
                .getDateTimeFormatInfo()
                .formatYear())
                .format(date);
    }

    public static int getField(@Nonnull final Date date, final int field) {
        return createCalendar(date).get(field);
    }

    /**
     * Returns the ONLY time range, even if dates different
     *
     * @param startTime
     * @param endTime
     * @return
     */
    public static String printTimeRange(@Nonnull final Date startTime, @Nonnull final Date endTime) {
        return isSameTime(startTime, endTime) ? printTime(startTime) : Strings.format("{0} - {1}", printTime(startTime), printTime(endTime)); //$NON-NLS-1$ ;
    }

    /**
     * Returns the ONLY the date
     *
     * @param date
     *            to print
     * @return only date string
     */
    @SuppressWarnings("null")
    @Nonnull
    public static String printDate(@Nonnull final Date date) {
        final String p = getFormatInfo().dateFormatShort();
        final DateTimeFormat f = DateTimeFormat.getFormat(p);
        return f.format(date);
    }

    /**
     * Returns the ONLY the time
     *
     * @param time
     * @return only time string
     */
    public static String printTime(@Nonnull final Date time) {
        final String p = getFormatInfo().timeFormatShort();
        final DateTimeFormat f = DateTimeFormat.getFormat(p);
        return f.format(time);
    }

    /**
     * Returns the Date/Time
     *
     * @param date
     *            to print
     * @return date/time string
     */
    public static String printDateTime(@Nonnull final Date date) {
        final String t = getFormatInfo().timeFormatShort();
        final String d = getFormatInfo().dateFormatShort();
        return DateTimeFormat.getFormat(getFormatInfo().dateTimeShort(t, d))
                .format(date);
    }

    public static String printDayOfMonth(@Nonnull final Date date) {
        return DateTimeFormat.getFormat(LocaleInfo.getCurrentLocale()
                .getDateTimeFormatInfo()
                .formatMonthAbbrevDay())
                .format(date);
    }

    /** Print duration with specified precision */
    public static String printDuration(final int duration, final int unit) {
        int d = duration;
        final int years = d / precision(Calendar.YEAR);
        final int months = (d -= precision(Calendar.YEAR) * years) / precision(Calendar.MONTH);
        final int days = (d -= precision(Calendar.MONTH) * months) / precision(Calendar.DAY_OF_MONTH);
        final int hours = (d -= precision(Calendar.DAY_OF_MONTH) * days) / precision(Calendar.HOUR);
        final int minutes = (d -= precision(Calendar.HOUR) * hours) / precision(Calendar.MINUTE);
        final int seconds = (d -= precision(Calendar.MINUTE) * minutes) / precision(Calendar.SECOND);
        final int milliseconds = (d -= precision(Calendar.SECOND) * seconds) / precision(Calendar.MILLISECOND);
        final String join = Joiner.on(", ") //$NON-NLS-1$
                .skipNulls()
                .join(str(years, TXT.year(), TXT.years()), str(months, TXT.month(), TXT.months()), str(days, TXT.day(), TXT.days()), str(hours, TXT.hour(), TXT.hours()), str(minutes, TXT.minute(), TXT.minutes()), str(seconds, TXT.second(), TXT.seconds()), str(milliseconds, TXT.millisecond(), TXT.milliseconds()));
        return join.isEmpty() ? TXT.undefined() : join;
    }

    private static String str(final int unit, final String single, final String multiple) {
        switch (unit) {
        case 0:
            return null;
        case 1:
            return String.valueOf(unit) + " " + single; //$NON-NLS-1$
        default:
            return String.valueOf(unit) + " " + multiple; //$NON-NLS-1$
        }
    }

    public static int precision(final int unit) {
        switch (unit) {
        case Calendar.YEAR:
            return 365 * 24 * 60 * 60 * 1000;
        case Calendar.MONTH:
            return 30 * 24 * 60 * 60 * 1000;
        case Calendar.DAY_OF_MONTH:
            return 24 * 60 * 60 * 1000;
        case Calendar.HOUR_OF_DAY:
        case Calendar.HOUR:
            return 60 * 60 * 1000;
        case Calendar.MINUTE:
            return 60 * 1000;
        case Calendar.SECOND:
            return 60 * 1000;
        default:
            return 1000;
        }
    }

    public static boolean isSameTime(@Nonnull final Date oneTime, @Nonnull final Date secondTime) {
        return compareTime(oneTime, secondTime, Calendar.MINUTE) == 0;
    }

    /** Take into account only time part */
    public static boolean isSameTime(@Nonnull final Date startTime, @Nonnull final Date endTime, final int precision) {
        return compareTime(startTime, endTime, precision) == 0;
    }

    public static boolean isSameDateTime(@Nonnull final Date startTime, @Nonnull final Date endTime, final int precision) {
        return compareDateTime(startTime, endTime, precision) == 0;
    }

    public static boolean isSameMonth(@Nonnull final Date startTime, @Nonnull final Date endTime) {
        return compareDateTime(startTime, endTime, Calendar.MONTH) == 0;
    }

    public static boolean isSameYear(@Nonnull final Date startTime, @Nonnull final Date endTime) {
        return compareDateTime(startTime, endTime, Calendar.YEAR) == 0;
    }

    public static DateTimeFormatInfo getFormatInfo() {
        return LocaleInfo.getCurrentLocale()
                .getDateTimeFormatInfo();
    }

    /**
     * Return the end date of surround the current time period
     *
     * @param period
     *            - one of the following Calendar constant (
     *            {@link Calendar#YEAR}, {@link Calendar#MONTH},
     *            {@link Calendar#DAY_OF_MONTH}, {@link Calendar#HOUR_OF_DAY} or
     *            {@link Calendar#HOUR}, {@link Calendar#MINUTE},
     *            {@link Calendar#SECOND}, {@link Calendar#MILLISECOND})
     * @return
     */
    public static Date getSurroundEnd(final int period) {
        return getSurroundEnd(new Date(), period);
    }

    /**
     * Return the start date of surround the current time period
     *
     * @param period
     *            - one of the following Calendar constant (
     *            {@link Calendar#YEAR}, {@link Calendar#MONTH},
     *            {@link Calendar#DAY_OF_MONTH}, {@link Calendar#HOUR_OF_DAY} or
     *            {@link Calendar#HOUR}, {@link Calendar#MINUTE},
     *            {@link Calendar#SECOND}, {@link Calendar#MILLISECOND})
     * @return
     */
    public static Date getSurroundStart(final int period) {
        return getSurroundStart(new Date(), period);
    }

    /**
     * Return the end date of surround the given time period
     *
     * @param date
     *            date to surround
     * @param period
     *            - one of the following Calendar constant (
     *            {@link Calendar#YEAR}, {@link Calendar#MONTH},
     *            {@link Calendar#DAY_OF_MONTH}, {@link Calendar#HOUR_OF_DAY} or
     *            {@link Calendar#HOUR}, {@link Calendar#MINUTE},
     *            {@link Calendar#SECOND})
     * @return
     */
    public static Date getSurroundEnd(@Nonnull final Date date, final int period) {
        final Calendar calendar = createCalendar(date, period);
        calendar.add(period, 1);
        calendar.add(Calendar.MILLISECOND, -1);
        return calendar.getTime();
    }

    public static int getPrev(final int period) {
        switch (period) {
        case Calendar.YEAR:
            return Calendar.MONTH;
        case Calendar.MONTH:
            return Calendar.DAY_OF_MONTH;
        case Calendar.DAY_OF_MONTH:
            return Calendar.HOUR_OF_DAY;
        case Calendar.HOUR_OF_DAY:
        case Calendar.HOUR:
            return Calendar.MINUTE;
        case Calendar.MINUTE:
            return Calendar.SECOND;
        case Calendar.SECOND:
            return Calendar.MILLISECOND;
        default:
            throw new IllegalStateException("No pevious for miliseconds"); //$NON-NLS-1$
        }
    }

    public static int getNext(final int period) {
        switch (period) {
        case Calendar.MONTH:
            return Calendar.YEAR;
        case Calendar.DAY_OF_MONTH:
            return Calendar.MONTH;
        case Calendar.HOUR_OF_DAY:
        case Calendar.HOUR:
            return Calendar.DAY_OF_MONTH;
        case Calendar.MINUTE:
            return Calendar.HOUR_OF_DAY;
        case Calendar.SECOND:
            return Calendar.MINUTE;
        case Calendar.MILLISECOND:
            return Calendar.SECOND;
        default:
            throw new IllegalStateException("No next for year"); //$NON-NLS-1$
        }
    }

    /**
     * Return the start date of surround the given time period
     *
     * @param date
     *            date to surround
     * @param period
     *            - one of the following Calendar constant (
     *            {@link Calendar#YEAR}, {@link Calendar#MONTH},
     *            {@link Calendar#DAY_OF_MONTH}, {@link Calendar#HOUR_OF_DAY} or
     *            {@link Calendar#HOUR}, {@link Calendar#MINUTE},
     *            {@link Calendar#SECOND}, {@link Calendar#MILLISECOND})
     * @return
     */
    public static Date getSurroundStart(@Nonnull final Date date, final int period) {
        final Calendar calendar = createCalendar(date, period);
        return calendar.getTime();
    }

    /**
     * See {@link Calendar#add(int, int)}
     *
     * @param date
     *            - date to change
     * @param amount
     *            - amount of the change
     * @param field
     *            - field of date to apply change
     * @return new {@link Date}
     */
    @SuppressWarnings("null")
    @Nonnull
    public static Date change(final Date date, final int amount, final int field) {
        final Calendar calendar = createCalendar(date);
        calendar.add(field, amount);
        return calendar.getTime();
    }

    /**
     * Add/remove from given date the specified amount im ms
     *
     * @param date
     *            - date to change
     * @param amount
     *            - amount of the change
     * @return new {@link Date}
     */
    public static Date change(final Date date, final long amount) {
        return new Date(date.getTime() + amount);
    }

    @Nonnull
    public static GwtLocalTimeModel change(final GwtLocalTimeModel time, final long amount) {
        long value = time.getMiliseconds() + amount;
        if (value >= DAY_LENGTH) {
            value -= DAY_LENGTH;
        } else if (value < 0) {
            value += DAY_LENGTH;
        }
        return new GwtLocalTimeModel(value);
    }

    public static GwtLocalDateModel changeDay(final GwtLocalDateModel date, final int amount) {
        final Calendar helper = createCalendar(date);
        helper.add(Calendar.DAY_OF_MONTH, amount);
        return new GwtLocalDateModel(helper);
    }

    public static GwtLocalDateModel change(final GwtLocalDateModel date, final int amount, final int field) {
        final Calendar helper = createCalendar(date);
        helper.add(field, amount);
        return new GwtLocalDateModel(helper);
    }

    /**
     * Constructs from {@link LocalDateModel} and {@link LocalTimeModel} single
     * Date
     *
     * @param date
     *            - date ONLY to use
     * @param time
     *            - time ONLY to use
     * @return date combined from given date and time
     */
    public static Date getDate(final LocalDateModel date, final LocalTimeModel time) {
        return createCalendar(date.getDate(), time.getDate()).getTime();
    }

    /**
     * Calculates is date in the past with specific precision
     *
     * @param date
     * @param precision
     *            - one of the following Calendar constant (
     *            {@link Calendar#YEAR}, {@link Calendar#MONTH},
     *            {@link Calendar#DAY_OF_MONTH}, {@link Calendar#HOUR_OF_DAY} or
     *            {@link Calendar#HOUR}, {@link Calendar#MINUTE},
     *            {@link Calendar#SECOND}, {@link Calendar#MILLISECOND})
     * @return
     */
    public static boolean isPast(@Nonnull final Date date, final int precision) {
        return precision == Calendar.MILLISECOND ? System.currentTimeMillis() >= date.getTime() : compareDateTime(date, new Date(), precision) < 0;
    }

    /**
     * Calculates if the date is in the past
     *
     * @param date
     * @return <code>true</code> if past
     */
    public static boolean isPast(@Nonnull final GwtLocalDateModel date) {
        return compareDate(date, today()) < 0;
    }

    /**
     * Merge to overlap ranges
     *
     * @param range1
     * @param range2
     * @return single range with early start and later end
     */
    public static ITimeRange merge(final ITimeRange range1, final ITimeRange range2) {
        return new ITimeRange() {
            @Override
            public Date getStart() {
                return early(range1.getStart(), range2.getStart());
            }

            @Override
            public Date getEnd() {
                return later(range1.getEnd(), range2.getEnd());
            }
        };
    }

    /**
     * @return the early date from two given
     */
    @Nonnull
    public static Date early(@Nonnull final Date start, @Nonnull final Date start2) {
        return start.getTime() < start2.getTime() ? start : start2;
    }

    /**
     * @return the early date from two given
     */
    @Nonnull
    public static Date later(@Nonnull final Date end, @Nonnull final Date end2) {
        return end.getTime() >= end2.getTime() ? end : end2;
    }

    public static long getDuration(final ITimeRange range) {
        return range.getEnd()
                .getTime() - range.getStart()
                        .getTime();
    }

    /**
     * Checks if date is end of the day (23.59.59.999)
     *
     * @param date
     * @return
     */
    public static boolean isDayEnd(@Nonnull final Date date) {
        final Calendar helper = createCalendar(date);
        final int h = helper.get(Calendar.HOUR_OF_DAY);
        final int m = helper.get(Calendar.MINUTE);
        final int s = helper.get(Calendar.SECOND);
        return h == 23 && m == 59 && s == 59;
    }

    /**
     * Checks if date is start of the day (0.0.0.0)
     *
     * @param date
     * @return
     */
    public static boolean isDayStart(final Date date) {
        final Calendar helper = createCalendar(date);
        final int h = helper.get(Calendar.HOUR_OF_DAY);
        final int m = helper.get(Calendar.MINUTE);
        final int s = helper.get(Calendar.SECOND);
        return h == 0 && m == 0 && s == 0;
    }

    @Nonnull
    public static GwtLocalDateModel today() {
        return new GwtLocalDateModel(createCalendar());
    }

    @Nonnull
    public static GwtLocalDateModel clone(final GwtLocalDateModel date) {
        return new GwtLocalDateModel(date);
    }

    public static String format(final GwtLocalDateModel date, final DateTimeFormat format) {
        return format.format(date.getDate());
    }

    /**
     * Counts weeks between today and given date if date is in current week
     * return 0.
     */
    public static int countWeeks(final GwtLocalDateModel date) {
        return countWeeks(today().getDate(), date.getDate());
    }

    /**
     * Counts weeks between today and given date if date is in current week
     * return 0.
     */
    public static int countWeeks(@Nonnull final Date date1, @Nonnull final Date date2) {
        if (date2.before(date1)) {
            return -countWeeks(date2, date1);
        }
        final Calendar cal1 = getWeekStartDay(date1);
        setTime(cal1, 0, 0);
        final Calendar cal2 = getWeekStartDay(date2);
        setTime(cal2, 0, 0);
        int weeks = 0;
        final Date b = cal2.getTime();
        while (cal1.getTime().before(b)) {
            // add another week
            cal1.add(Calendar.WEEK_OF_YEAR, 1);
            weeks++;
        }
        return weeks;
    }

    public static int getCurrentWeek() {
        return createCalendar().get(Calendar.WEEK_OF_YEAR);
    }

}
