package org.jresearch.commons.gwt.client.mvc;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.jresearch.commons.gwt.client.mvc.event.Bus;

import com.google.gwt.inject.client.AsyncProvider;

@SuppressWarnings("rawtypes")
public abstract class AbstractController<V extends AbstractView> extends AbstractViewlessController {

	@Nonnull
	private final AsyncProvider<V> viewProvider;
	private V view;
	private final List<ViewCommand<V>> delayedCommands = new ArrayList<>();
	private boolean loading = false;

	public AbstractController(@Nonnull final String id, @Nonnull final Bus bus, @Nonnull final AsyncProvider<V> view) {
		super(id, bus);
		this.viewProvider = view;
	}

	protected void showView() {
		if (view == null) {
			executeCommandWithLoad(new ViewCommand<V>() {
				@Override
				public void execute(final V v) {
					v.showContent();
				}
			});
		}
	}

	protected void executeCommandWithLoad(@Nonnull final ViewCommand<V> command) {
		if (view == null) {
			delayedCommands.add(command);
			if (!loading) {
				loading = true;
				viewProvider.get(new AbstractCallback<V>(bus) {
					@Override
					public void onSuccess(final V result) {
						view = result;
						loading = false;
						onViewLoad();
					}

					@Override
					public void onFailure(final Throwable caught) {
						loading = false;
						delayedCommands.clear();
						super.onFailure(caught);
					}
				});
			}
		} else {
			command.execute(view);
		}
	}

	public void executeCommand(@Nonnull final ViewCommand<V> command) {
		if (view == null) {
			delayedCommands.add(command);
		} else {
			command.execute(view);
		}
	}

	@Nullable
	public V getView() {
		return view;
	}

	protected void onViewLoad() {
		for (final ViewCommand<V> command : delayedCommands) {
			command.execute(view);
		}
	}

	/**
	 * @return the parentController
	 */
	abstract public AbstractController<?> getParentController();

	public void onViewCreate() {
		// do notfing
	}

}
