package org.jresearch.commons.gwt.client.app;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.annotation.Nonnull;

import org.jresearch.commons.gwt.client.model.localization.LocaleModel;
import org.jresearch.commons.gwt.client.mvc.AbstractController;
import org.jresearch.commons.gwt.client.mvc.ViewCommand;
import org.jresearch.commons.gwt.client.mvc.event.AboutEvent;
import org.jresearch.commons.gwt.client.mvc.event.AboutHandler;
import org.jresearch.commons.gwt.client.mvc.event.Bus;
import org.jresearch.commons.gwt.client.mvc.event.ErrorEvent;
import org.jresearch.commons.gwt.client.mvc.event.ErrorHandler;
import org.jresearch.commons.gwt.client.mvc.event.InitEvent;
import org.jresearch.commons.gwt.client.mvc.event.InitHandler;
import org.jresearch.commons.gwt.client.mvc.event.SetLanguageEvent;
import org.jresearch.commons.gwt.client.mvc.event.SetLanguageHandler;
import org.jresearch.commons.gwt.client.mvc.event.module.ModuleEvent;
import org.jresearch.commons.gwt.client.mvc.event.module.ModuleHandler;
import org.jresearch.commons.gwt.client.service.AppServiceAsync;
import org.jresearch.commons.gwt.client.service.localization.ILocalizationServiceAsync;
import org.jresearch.commons.gwt.client.tool.DeferredTask;

import com.google.gwt.http.client.UrlBuilder;
import com.google.gwt.i18n.client.LocaleInfo;
import com.google.gwt.inject.client.AsyncProvider;
import com.google.gwt.user.client.Cookies;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.Window.Location;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.rpc.StatusCodeException;

public abstract class AbstractAppController<V extends AbstractAppView<? extends AbstractAppController<V>>> extends AbstractController<V> implements AboutHandler, SetLanguageHandler, InitHandler, ErrorHandler, ModuleHandler {

	private static Logger logger = Logger.getLogger("org.jresearch.commons.gwt.app.client.mvc.AbstractAppController"); //$NON-NLS-1$

	// 15 seconds
	public static final int ONLINE_REFRESH_INTERVEALL = 15 * 1000;
	// 5 seconds
	public static final int OFFLINE_REFRESH_INTERVEALL = 5 * 1000;

	@Nonnull
	private final ILocalizationServiceAsync localizationService;
	final List<IAppModule> modules;
	private String activeModule;
	/** current offline/online mode */
	private boolean offlineMode = false;
	/** is check in progress (offline/online mode) */
	private boolean checking = false;
	/** is possible offline (offline/online mode) */
	private boolean semiOffline = false;
	/** is possible online (offline/online mode) */
	private boolean semiOnline = false;
	@Nonnull
	private final AppServiceAsync appService;
	private final DeferredTask refreshTask = new DeferredTask() {
		@Override
		public void run() {
			updateOfflineMode();
		}
	};

	public AbstractAppController(@Nonnull final String id, @Nonnull final AppServiceAsync appService, @Nonnull final ILocalizationServiceAsync localizationService, @Nonnull final Set<IAppModule> appModules, @Nonnull final AsyncProvider<V> view, @Nonnull final Bus bus) {
		super(id, bus, view);
		this.appService = appService;
		this.localizationService = localizationService;

		modules = new ArrayList<>(appModules);

		if (!modules.isEmpty()) {
			activeModule = modules.get(0)
					.getModuleId();
		}

		bus.addHandler(InitEvent.TYPE, this);
		bus.addHandler(SetLanguageEvent.TYPE, this);
		bus.addHandler(AboutEvent.TYPE, this);
		bus.addHandler(ErrorEvent.TYPE, this);
		bus.addHandler(ModuleEvent.TYPE, this);

//		// Handle empty anchor
//		History.addValueChangeHandler(new ValueChangeHandler<String>() {
//			@SuppressWarnings("synthetic-access")
//			@Override
//			public void onValueChange(final ValueChangeEvent<String> event) {
//				if (!modules.isEmpty()) {
//					String historyToken = event.getValue();
//					if (Strings.isEmpty(historyToken) && !modules.isEmpty()) {
//						historyToken = modules.get(0).getModuleId();
//					}
//					bus.fire(new ModuleEvent(historyToken));
//				}
//			}
//		});

		refreshTask.defer(ONLINE_REFRESH_INTERVEALL);
	}

	@Override
	public void onInit(final InitEvent initEvent) {
		initView();
		showView();
	}

	protected void initView() {
		executeCommandWithLoad(new ViewCommand<V>() {
			@Override
			public void execute(final V view) {
				view.initModules(modules);
			}
		});
	}

	@Override
	public void onError(final ErrorEvent event) {
		final Throwable ex = event.getException();
		if (ex instanceof StatusCodeException) {
			final StatusCodeException exception = (StatusCodeException) ex;
			if (exception.getStatusCode() == 0) {
				if (!isOfflineMode()) {
					updateOfflineMode();
				}
				return;
			}
		}
		logger.log(Level.SEVERE, ex.getLocalizedMessage(), ex);
		final V view = getView();
		if (view != null) {
			view.onError(event);
		}
	}

	// Check twice if the mode really changed
	private void updateOfflineMode() {
		if (!checking) {
			checking = true;
			appService.checkOffline(new AsyncCallback<Void>() {
				@Override
				public void onSuccess(final Void result) {
					int refreshInterval = ONLINE_REFRESH_INTERVEALL;
					if (isOfflineMode() && !semiOnline) {
						refreshInterval = OFFLINE_REFRESH_INTERVEALL;
						semiOnline = true;
					} else {
						setOfflineMode(false);
						semiOnline = false;
					}
					semiOffline = false;
					refreshTask.defer(refreshInterval);
					checking = false;
				}

				@Override
				public void onFailure(final Throwable caught) {
					if (!isOfflineMode() && !semiOffline) {
						semiOffline = true;
					} else {
						setOfflineMode(true);
						semiOffline = false;
					}
					semiOnline = false;
					refreshTask.defer(OFFLINE_REFRESH_INTERVEALL);
					checking = false;
				}
			});
		}

	}

	protected void setOfflineMode(final boolean offline) {
		if (offlineMode != offline) {
			offlineMode = offline;
			onChangeOfflineMode();
		}
	}

	/**
	 * Call on set/unset the offline mode for application
	 *
	 * Application controllers that override this method have to call supper
	 *
	 */
	protected void onChangeOfflineMode() {
		final V view = getView();
		if (view != null) {
			view.onChangeOfflineMode(isOfflineMode());
			if (isOfflineMode()) {
				// switch to home if current module does not support offline
				if (!modules.isEmpty() && activeModule != null && !supportsOffline(activeModule)) {
					bus.fire(new ModuleEvent(modules.get(0).getModuleId()), true);
				}
				// hide all unsupported modules
				for (final IAppModule module : modules) {
					if (!module.supportsOffline()) {
						view.hideModule(module.getModuleId());
					}
				}
			} else {
				for (final IAppModule module : modules) {
					view.showModule(module.getModuleId());
				}
			}
		}
	}

	@Override
	@SuppressWarnings({ "deprecation" })
	public void onSetLanguage(final SetLanguageEvent event) {
		final LocaleModel locale = event.getLocale();
		// report to the server
		localizationService.setLocale(locale, getLanguageChangedCallback());
		final String cookieName = LocaleInfo.getLocaleCookieName();
		final String queryParam = LocaleInfo.getLocaleQueryParam();
		if (cookieName == null && queryParam == null) {
			// if there is no way for us to affect the locale, don nothing
			return;
		}

		if (cookieName != null) {
			// expire in one year
			final Date expires = new Date();
			expires.setYear(expires.getYear() + 1);
			Cookies.setCookie(cookieName, locale.getLocaleName(), expires);
		}
		if (queryParam != null) {
			final UrlBuilder builder = Location.createUrlBuilder()
					.setParameter(queryParam, locale.getLocaleName());
			Window.Location.replace(builder.buildString());
		} else {
			// If we are using only cookies, just reload
			Window.Location.reload();
		}
	}

	/**
	 * Can be overwrote to return more specific server language change callback
	 *
	 * @return default callback
	 */
	@SuppressWarnings("static-method")
	protected AsyncCallback<Void> getLanguageChangedCallback() {
		return new AsyncCallback<Void>() {
			@Override
			public void onSuccess(final Void result) {
				// do nothing
			}

			@Override
			public void onFailure(final Throwable caught) {
				// do nothing
			}
		};
	}

	@Override
	public void onAbout(final AboutEvent event) {
		executeCommand(new ViewCommand<V>() {
			@Override
			public void execute(final V view) {
				view.showAbout();
			}
		});
	}

	@Override
	public AbstractController<?> getParentController() {
		return null;
	}

	public void handleAboutClick() {
		bus.fire(new AboutEvent());
	}

	@Override
	public void onModule(final ModuleEvent event) {
		if (!isOfflineMode() || supportsOffline(event.getModuleId())) {
			executeCommand(new ViewCommand<V>() {
				@Override
				public void execute(final V view) {
					if (view.switchToModule(event.getModuleId())) {
						activeModule = event.getModuleId();
					}
				}
			});
		}
	}

	/**
	 * @param moduleId
	 * @return
	 */
	private boolean supportsOffline(@Nonnull final String moduleId) {
		for (final IAppModule module : modules) {
			if (moduleId.equals(module.getModuleId())) {
				return module.supportsOffline();
			}
		}
		return false;
	}

	public String getActiveModuleId() {
		return activeModule;
	}

	/**
	 * Receive all events for any module id
	 */
	@Override
	public String getModuleId() {
		return ModuleEvent.ANY_ID;
	}

	/**
	 * @return the offlineMode
	 */
	public boolean isOfflineMode() {
		return offlineMode;
	}

}
