package org.jresearch.commons.gwt.shared.tools;

import static java.lang.Math.*;

public class Colors {

    private Colors() {
        super();
    }

    public static Rgb html2rgb(final String color) {
        return new Rgb(Integer.parseInt(color.substring(1, 3), 16), Integer.parseInt(color.substring(3, 5), 16), Integer.parseInt(color.substring(5, 7), 16));
    }

    public static String rgb2html(final Rgb color) {
        return new StringBuilder("#").append(toHexString(color.getR())).append(toHexString(color.getG())).append(toHexString(color.getB())).toString(); //$NON-NLS-1$
    }

    public static String hsl2html(final Hsl hsl) {
        return rgb2html(hsl2rgb(hsl));
    }

    private static String toHexString(final int v) {
        final StringBuilder sb = new StringBuilder(Integer.toHexString(v).toUpperCase());
        if (sb.length() < 2) {
            sb.insert(0, '0'); // pad with leading zero if needed
        }
        return sb.toString();
    }

    public static Lab rgb2Lab(final Rgb color) {
        final double rf = f1(color.getR());
        final double gf = f1(color.getG());
        final double bf = f1(color.getB());

        final double X = (0.4124f * rf + 0.3576f * gf + 0.1805f * bf) / 95.0429;
        final double Y = (0.2126f * rf + 0.7152f * gf + 0.0722f * bf) / 100.0;
        final double Z = (0.0193f * rf + 0.1192f * gf + 0.9505f * bf) / 108.89;

        final double fX = f2(X);
        final double fY = f2(Y);
        final double fZ = f2(Z);

        return new Lab((116f * fY) - 16f, 500f * (fX - fY), 200f * (fY - fZ));
    }

    private static double f1(final double v) {
        return ((v > 0.04045) ? (Math.exp(Math.log((v + 0.055) / 1.055) * 2.4)) : v / 12.92) * 100;
    }

    private static double f2(final double v) {
        return v > 0.008856f ? Math.exp(Math.log(v) / 3) : 7.787 * v + 16 / 116;
    }

    public static double distance(final Rgb color1, final Rgb color2) {
        return distance(rgb2Lab(color1), rgb2Lab(color2));
    }

    public static double distance(final Lab lab1, final Lab lab2) {
        return sqrt(pow(lab1.getL() - lab2.getL(), 2) + pow(lab1.getA() - lab2.getA(), 2) + pow(lab1.getB() - lab2.getB(), 2));
    }

    public static Rgb hsl2rgb(final Hsl hsl) {
        final double hue = hsl.getH() / 360.0;
        final double saturation = hsl.getS() / 100.0;
        final double lightness = hsl.getL() / 100.0;
        double r;
        double g;
        double b;
        double q;
        double p;

        if (saturation == 0) {
            r = g = b = lightness * 255;
        } else {
            q = lightness < 0.5 ? lightness * (1 + saturation) : lightness + saturation - saturation * lightness;
            p = 2 * lightness - q;

            r = 255 * hue2rgb(p, q, hue + 1.0 / 3.0);
            g = 255 * hue2rgb(p, q, hue);
            b = 255 * hue2rgb(p, q, hue - 1.0 / 3.0);
        }

        return new Rgb((int) Math.round(r), (int) Math.round(g), (int) Math.round(b));
    }

    private static double hue2rgb(final double p, final double q, final double t) {
        final double fixT = t < 0 ? t + 1 : t > 1 ? t - 1 : t;
        if (6 * fixT < 1) {
            return p + (q - p) * 6 * fixT;
        }
        if (2 * fixT < 1) {
            return q;
        }
        if (3 * fixT < 2) {
            return p + (q - p) * (2.0 / 3.0 - fixT) * 6;
        }
        return p;
    }

    static public Hsl rgb2hsl(final Rgb rgb) {
        final double red = rgb.getR() / 255.0;
        final double green = rgb.getG() / 255.0;
        final double blue = rgb.getB() / 255.0;

        final double min = Math.min(red, Math.min(green, blue));
        final double max = Math.max(red, Math.max(green, blue));
        final double delta = max - min;

        double h = 0;
        double s;
        final double l = (max + min) / 2f;

        if (delta == 0) {
            s = 0;
        } else {
            s = l < 0.5 ? delta / (max + min) : delta / (2 - max - min);

            final double delta_r = (((max - red) / 6.0) + delta / 2.0) / delta;
            final double delta_g = (((max - green) / 6.0) + delta / 2.0) / delta;
            final double delta_b = (((max - blue) / 6.0) + delta / 2.0) / delta;

            h = red == max ? delta_b - delta_g : green == max ? 1 / 3.0 + delta_r - delta_b : 2 / 3.0 + delta_g - delta_r;
            h = h < 0 ? h + 1 : h > 1 ? h - 1 : h;
        }
        return new Hsl((int) Math.round(360 * h), (int) Math.round(s * 100), (int) Math.round(l * 100));
    }

}
