package org.jresearch.commons.gwt.client.model.time;

import java.util.Calendar;
import java.util.Date;

import javax.annotation.Nonnull;

import org.jresearch.commons.gwt.shared.tools.Dates;

import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.collect.Range;
import com.google.common.primitives.Ints;
import com.google.gwt.i18n.shared.TimeZone;

public class GwtDateTimeModel extends GwtLocalDateTimeModel {

    private static final long serialVersionUID = 5840767318129944964L;

    /** 23:59:59.999 */
    public static final long BOUND = 999 + (59 + (59 + 23 * 60) * 60) * 1000;
    @SuppressWarnings("null")
    public static final Range<Long> RANGE = Range.open(Long.valueOf(-BOUND), Long.valueOf(BOUND));

    /**
     * the offset in millis from UTC, from -23:59:59.999 to +23:59:59.999
     */
    private long offset;

    public GwtDateTimeModel() {
        this(Dates.createCalendar());
    }

    @SuppressWarnings("deprecation")
    public GwtDateTimeModel(final Calendar date) {
        this(new GwtLocalDateModel(date), new GwtLocalTimeModel(date), date.getTime().getTimezoneOffset() * 60 * 1000);
    }

    public GwtDateTimeModel(final Date date) {
        this(date, date);
    }

    @SuppressWarnings("deprecation")
    public GwtDateTimeModel(final Date date, final Date time) {
        this(new GwtLocalDateModel(date), new GwtLocalTimeModel(time), date.getTimezoneOffset() * 60 * 1000);
    }

    /**
     * @param millisOffset the offset in millis from UTC, from -23:59:59.999 to
     * +23:59:59.999
     */
    public GwtDateTimeModel(@Nonnull final GwtLocalDateModel date, @Nonnull final GwtLocalTimeModel time, final long millisOffset) {
        super(date, time);
        setOffset(millisOffset);
    }

    @Override
    public Date toDate() {
        return new Date(getDate().toDate().getTime() + getTime().getMiliseconds());
    }

    /**
     * @return the offset
     */
    public long getOffset() {
        return offset;
    }

    /**
     * @param offset the offset to set
     */
    @SuppressWarnings("null")
    public void setOffset(final long offset) {
        Preconditions.checkArgument(RANGE.contains(Long.valueOf(offset)));
        this.offset = offset;
    }

    public TimeZone getTimeZone() {
        return com.google.gwt.i18n.client.TimeZone.createTimeZone(Ints.saturatedCast(offset / (60 * 1000)));
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("super", super.toString()) //$NON-NLS-1$
                .add("offset", offset) //$NON-NLS-1$
                .toString();
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(Integer.valueOf(super.hashCode()), Long.valueOf(getOffset()));
    }

    @Override
    public boolean equals(final Object object) {
        if (object instanceof GwtDateTimeModel) {
            if (!super.equals(object))
                return false;
            final GwtDateTimeModel that = (GwtDateTimeModel) object;
            return Objects.equal(Long.valueOf(this.getOffset()), Long.valueOf(that.getOffset()));
        }
        return false;
    }

}
