/*
 * Copyright 2007 Future Earth, info@future-earth.eu
 * Copyright 2014 Stanislav Spiridonov, stas@jresearch.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package org.jresearch.commons.gwt.client.tool;

import static org.jresearch.commons.gwt.client.base.resource.BaseRs.*;

import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.jresearch.commons.gwt.client.base.resource.BaseRs;
import org.jresearch.commons.gwt.shared.model.time.GwtDateTimeModel;
import org.jresearch.commons.gwt.shared.model.time.GwtLocalDateModel;
import org.jresearch.commons.gwt.shared.model.time.GwtLocalDateTimeModel;
import org.jresearch.commons.gwt.shared.model.time.GwtLocalTimeModel;
import org.jresearch.commons.gwt.shared.tools.ITimeRange;
import org.jresearch.commons.gwt.shared.tools.LocalTimeRange;
import org.jresearch.commons.gwt.shared.tools.SharedDates;

import com.google.common.base.Joiner;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.primitives.Ints;
import com.google.gwt.core.shared.GWT;
import com.google.gwt.i18n.client.LocaleInfo;
import com.google.gwt.i18n.shared.DateTimeFormat;
import com.google.gwt.i18n.shared.DateTimeFormatInfo;
import com.google.gwt.i18n.shared.TimeZone;

public final class Dates {

    private static final String REMOVE_YEAR_PATTERN = "[^a-zA-Z]*y+[^a-zA-Z]*"; //$NON-NLS-1$

    public static final int MINUTE = Calendar.MINUTE;
    public static final int MONTH = Calendar.MONTH;
    public static final int DAY_OF_MONTH = Calendar.DAY_OF_MONTH;
    public static final int DAY_OF_WEEK = Calendar.DAY_OF_WEEK;
    public static final int YEAR = Calendar.YEAR;
    public static final int HOUR_OF_DAY = Calendar.HOUR_OF_DAY;
    public static final int HOUR = Calendar.HOUR;
    public static final int SECOND = Calendar.SECOND;
    public static final int MILLISECOND = Calendar.MILLISECOND;

    @Nonnull
    private static final LoadingCache<GwtLocalDateModel, Integer> weelCountCache = CacheBuilder.newBuilder().expireAfterWrite(30, TimeUnit.SECONDS).build(new CacheLoader<GwtLocalDateModel, Integer>() {
        @Override
        public Integer load(final GwtLocalDateModel key) throws Exception {
            return Integer.valueOf(countWeeks(today().toDate(), key.toDate()));
        }
    });

    private Dates() {
        super();
    }

    @Nonnull
    public static GwtLocalDateTimeModel createLocalDateTime(@Nonnull final Date dateTime) {
        return new GwtLocalDateTimeModel(createLocalDate(dateTime), createLocalTime(dateTime));
    }

    @Nonnull
    public static GwtLocalDateTimeModel createLocalDateTime(@Nonnull final Date date, @Nonnull final Date time) {
        return new GwtLocalDateTimeModel(createLocalDate(date), createLocalTime(time));
    }

    @Nonnull
    public static Calendar createCalendar(@Nonnull final Date date) {
        final Calendar result = createCalendar();
        result.setTime(date);
        return result;
    }

    @Nonnull
    public static Calendar createCalendar(@Nonnull final GwtLocalDateModel date) {
        final Calendar result = createCalendar();
        result.clear();
        result.set(YEAR, date.getYear());
        result.set(MONTH, date.getMonth() - 1);
        result.set(Calendar.DATE, date.getDay());
        return result;
    }

    @Nonnull
    public static Calendar createCalendar(@Nonnull final Date date, @Nullable final Date time) {
        final Calendar result = createCalendar(date, Calendar.DAY_OF_MONTH);
        if (time != null) {
            final Calendar timeCalendar = createTimeCalendar(time, Calendar.MILLISECOND);
            result.add(Calendar.HOUR_OF_DAY, timeCalendar.get(Calendar.HOUR_OF_DAY));
            result.add(Calendar.MINUTE, timeCalendar.get(Calendar.MINUTE));
            result.add(Calendar.SECOND, timeCalendar.get(Calendar.SECOND));
            result.add(Calendar.MILLISECOND, timeCalendar.get(Calendar.MILLISECOND));
        }
        return result;
    }

    /**
     * Create {@link Calendar} with specific precision
     *
     * @param date
     * @param precision
     *            - one of the following Calendar constant (
     *            {@link Calendar#YEAR}, {@link Calendar#MONTH},
     *            {@link Calendar#DAY_OF_MONTH}, {@link Calendar#HOUR_OF_DAY} or
     *            {@link Calendar#HOUR}, {@link Calendar#MINUTE},
     *            {@link Calendar#SECOND}, {@link Calendar#MILLISECOND}) @return
     */
    @Nonnull
    public static Calendar createCalendar(@Nonnull final Date date, final int precision) {
        final Calendar result = createCalendar(date);
        switch (precision) {
        case Calendar.YEAR:
            result.set(Calendar.MONTH, 0);
            //$FALL-THROUGH$
        case Calendar.MONTH:
            result.set(Calendar.DAY_OF_MONTH, 1);
            //$FALL-THROUGH$
        case Calendar.DAY_OF_MONTH:
            result.set(Calendar.HOUR_OF_DAY, 0);
            //$FALL-THROUGH$
        case Calendar.HOUR_OF_DAY:
        case Calendar.HOUR:
            result.set(Calendar.MINUTE, 0);
            //$FALL-THROUGH$
        case Calendar.MINUTE:
            result.set(Calendar.SECOND, 0);
            //$FALL-THROUGH$
        case Calendar.SECOND:
            result.set(Calendar.MILLISECOND, 0);
            //$FALL-THROUGH$
        default:
        }
        return result;
    }

    public static Calendar createCalendar(@Nonnull final GwtLocalDateModel date, final int precision) {
        final Calendar result = createCalendar();
        result.clear();
        switch (precision) {
        case Calendar.SECOND:
        case Calendar.MINUTE:
        case Calendar.HOUR_OF_DAY:
        case Calendar.HOUR:
        case Calendar.DAY_OF_MONTH:
            result.set(Calendar.DAY_OF_MONTH, date.getDay());
            //$FALL-THROUGH$
        case Calendar.MONTH:
            result.set(Calendar.MONTH, date.getMonth() - 1);
            //$FALL-THROUGH$
        case Calendar.YEAR:
            result.set(Calendar.YEAR, date.getYear());
            //$FALL-THROUGH$
        default:
        }
        return result;
    }

    /**
     * Create dateless {@link Calendar} with specific precision. Dateless means
     * that year, month and day have zero values
     *
     * @param date
     * @param precision
     *            - one of the following Calendar constant (
     *            {@link Calendar#HOUR_OF_DAY} or {@link Calendar#HOUR},
     *            {@link Calendar#MINUTE}, {@link Calendar#SECOND},
     *            {@link Calendar#MILLISECOND}) @return
     */
    public static Calendar createTimeCalendar(@Nonnull final Date date, final int precision) {
        final Calendar result = createCalendar(date);
        result.set(Calendar.YEAR, 0);
        result.set(Calendar.MONTH, 0);
        result.set(Calendar.DAY_OF_MONTH, 1);
        switch (precision) {
        case Calendar.HOUR_OF_DAY:
        case Calendar.HOUR:
            result.set(Calendar.MINUTE, 0);
            //$FALL-THROUGH$
        case Calendar.MINUTE:
            result.set(Calendar.SECOND, 0);
            //$FALL-THROUGH$
        case Calendar.SECOND:
            result.set(Calendar.MILLISECOND, 0);
            //$FALL-THROUGH$
        default:
        }
        return result;
    }

    @Nonnull
    public static Calendar createCalendar() {
        final Calendar result = new GregorianCalendar();
        result.setFirstDayOfWeek(findFirstDayOfWeek());
        result.setMinimalDaysInFirstWeek(4);
        return result;
    }

    @Nonnull
    public static GwtLocalDateModel createLocalDate(@Nonnull final Date date) {
        return new GwtLocalDateModel(date.getDate(), date.getMonth() + 1, date.getYear() + 1900);
    }

    @Nonnull
    public static GwtLocalTimeModel createLocalTime(@Nonnull final Date date) {
        return new GwtLocalTimeModel(date.getHours(), date.getMinutes(), date.getSeconds());
    }

    private static int findFirstDayOfWeek() {
        // In GWT week days starts form 0
        return GWT.isClient() ? LocaleInfo.getCurrentLocale()
                .getDateTimeFormatInfo()
                .firstDayOfTheWeek() + 1 : Calendar.MONDAY;
    }

    public static int getCurrentWeek() {
        return getWeek(createCalendar());
    }

    public static int getWeek(final GwtLocalDateModel date) {
        return getWeek(createCalendar(date.toDate()));
    }

    public static int getWeek(final Calendar date) {
        return date.get(Calendar.WEEK_OF_YEAR);
    }

    @Nonnull
    public static GwtLocalDateModel getCurrentWeekStartDay() {
        return getCurrentWeekStartDay(0);
    }

    @Nonnull
    public static GwtLocalDateModel getCurrentWeekStartDay(final int weekOffset) {
        final Calendar helper = createCalendar();
        helper.set(Calendar.DAY_OF_WEEK, helper.getFirstDayOfWeek());
        if (weekOffset != 0) {
            helper.add(Calendar.WEEK_OF_YEAR, weekOffset);
        }
        return new GwtLocalDateModel(helper);
    }

    @Nonnull
    public static Calendar getWeekStartDay(@Nonnull final Date someDayInWeek) {
        final Calendar helper = createCalendar(someDayInWeek);
        helper.set(Calendar.DAY_OF_WEEK, helper.getFirstDayOfWeek());
        return helper;
    }

    @Nonnull
    public static Calendar getWeekEndDay(@Nonnull final Date someDayInWeek) {
        final Calendar helper = getWeekStartDay(someDayInWeek);
        helper.add(Calendar.WEEK_OF_YEAR, 1);
        helper.add(Calendar.DAY_OF_YEAR, -1);
        return helper;
    }

    public static void setTime(final Calendar calendar, final int hours, final int minutes) {
        calendar.set(Calendar.HOUR_OF_DAY, hours);
        calendar.set(Calendar.MINUTE, minutes);
        calendar.set(Calendar.SECOND, minutes == 59 ? 59 : 0);
        calendar.set(Calendar.MILLISECOND, minutes == 59 ? 999 : 0);
    }

    @SuppressWarnings("deprecation")
    public static boolean isSameDay(final Date one, final Date two) {
        if (one == null || two == null) {
            return false;
        }

        if (one.getYear() != two.getYear()) {
            return false;
        }
        if (one.getMonth() != two.getMonth()) {
            return false;
        }
        if (one.getDate() != two.getDate()) {
            return false;
        }
        return true;
    }

    public static boolean isSameDay(final GwtLocalDateModel o, final GwtLocalDateModel t) {
        return o != null && t != null && o.getYear() == t.getYear() && o.getMonth() == t.getMonth() && o.getDay() == t.getDay();
    }

    public static boolean isSameDay(final Calendar one, final Calendar two) {
        if (one != null && two != null) {
            if (one.get(Calendar.YEAR) == two.get(Calendar.YEAR) && one.get(Calendar.DAY_OF_YEAR) == two.get(Calendar.DAY_OF_YEAR)) {
                return true;
            }
        }
        return false;
    }

    /**
     * This method checks wheter the Date are the Same or the second is the 00
     * minute on the next day.
     *
     * @param one
     *            - The First Date @param two - The Second date @return
     */
    @SuppressWarnings("deprecation")
    public static boolean isSameDayOrNextDayZeroMinutes(final Date one, final Date two) {
        if (one == null || two == null) {
            return false;
        }

        if (one.getYear() != two.getYear()) {
            return false;
        }
        if (one.getMonth() != two.getMonth()) {
            return false;
        }
        if (one.getDate() != two.getDate()) {
            return !nextDayZeroMinutes(one, two);
        }
        return true;
    }

    /**
     * Check if the second date is the first minute of the next day.
     *
     * @param one
     *            The first date. @param two The second date. @return true of
     *            false
     */
    @SuppressWarnings("deprecation")
    public static boolean nextDayZeroMinutes(final Date one, final Date two) {
        if (one == null || two == null) {
            return false;
        }

        if (one.getYear() != two.getYear()) {
            return false;
        }
        if (one.getMonth() != two.getMonth()) {
            return false;
        }
        if (one.getDate() + 1 != two.getDate()) {
            return false;
        }
        if (0 != two.getHours() || 0 != two.getMinutes()) {
            return false;
        }
        return true;
    }

    @SuppressWarnings("deprecation")
    public static boolean zeroMinutes(final Date two) {
        if (two == null) {
            return false;
        }
        if (0 == two.getHours() && 0 == two.getMinutes()) {
            return true;
        }
        return false;
    }

    /**
     * Check only the time part, date part ignore (testTime:testTime+duration)
     * between [startDate:endDate]
     */
    public static boolean betweenTime(@Nonnull final Date testTime, final long duration, @Nonnull final Date startTime, @Nonnull final Date endTime) {
        final long dayTime = getDayTime(testTime);
        return between(dayTime, dayTime + duration, getDayTime(startTime), getDayTime(endTime));
    }

    public static boolean betweenTime(final GwtLocalTimeModel time, final GwtLocalTimeModel startTime, final GwtLocalTimeModel endTime) {
        final int tMs = time.getMiliseconds();
        return tMs >= startTime.getMiliseconds() && tMs < endTime.getMiliseconds();
    }

    /** testDate between startDate:endDate+duration */
    public static boolean between(final Date testDate, final Date startDate, final Date endDate, final long duration) {
        return between(testDate.getTime(), 0, startDate.getTime(), endDate.getTime() + duration);
    }

    /** start:end inside of range */
    public static boolean between(final Date start, final Date end, final ITimeRange range) {
        return between(start, end.getTime() - start.getTime(), range);
    }

    /** start:start+duration inside of range */
    public static boolean between(final Date start, final long duration, final ITimeRange range) {
        return between(start.getTime(), duration, range);
    }

    /** start:start+duration inside of range */
    public static boolean between(final long start, final long duration, final ITimeRange range) {
        return between(start, start + duration, range.getStart().getTime(), range.getEnd().getTime());
    }

    /** start:start+duration inside of rangeStart:rangeEnd */
    public static boolean between(final long start, final long end, final long rangeStart, final long rangeEnd) {
        return start >= rangeStart && end <= rangeEnd;
    }

    /** start:start+duration inside of rangeStart:rangeEnd */
    public static boolean between(final Date start, final long duration, final Date rangeStrart, final Date rangeEnd) {
        return between(start.getTime(), start.getTime() + duration, rangeStrart.getTime(), rangeEnd.getTime());
    }

    /** if start-end overlap with day */
    public static boolean isOverlap(final Date start, final Date end, @Nonnull final Date day) {
        final Calendar helper = createCalendar(day);
        setTime(helper, 0, 0);
        return isOverlap(start, end, helper.getTime(), SharedDates.DAY_LENGTH);
    }

    /** if start-end overlap with day */
    public static boolean isOverlap(final GwtLocalDateModel start, final GwtLocalDateModel end, @Nonnull final GwtLocalDateModel day) {
        return isOverlap(start, end, day, SharedDates.DAY_LENGTH);
    }

    public static boolean isOverlap(final GwtLocalDateModel start1, final GwtLocalDateModel end1, final GwtLocalDateModel start2, final long duration2) {
        return isOverlap(start1.toDate(), end1.toDate(), start2.toDate(), duration2);
    }

    public static boolean isOverlap(final Date start1, final Date end1, final Date start2, final long duration2) {
        return isOverlap(start1, end1.getTime() - start1.getTime(), start2, duration2);
    }

    /**
     *
     * @param start1
     *            start of the 1st interval @param duration1 duration of the 1st
     *            interval @param start2 start of the 2nd interval @param
     *            duration2 duration of the 2nd interval @return
     */
    public static boolean isOverlap(final Date start1, final long duration1, final Date start2, final long duration2) {
        return isOverlap(start1.getTime(), duration1, start2.getTime(), duration2);
    }

    public static boolean isOverlap(@Nonnull final GwtLocalTimeModel start1, final long duration1, @Nonnull final GwtLocalTimeModel start2, final long duration2) {
        return isOverlap(start1.getMiliseconds(), duration1, start2.getMiliseconds(), duration2);
    }

    public static boolean isOverlap(@Nonnull final GwtLocalDateTimeModel start1, final long duration1, @Nonnull final GwtLocalDateTimeModel start2, final long duration2) {
        return isOverlap(start1.getMiliseconds(), duration1, start2.getMiliseconds(), duration2);
    }

    /**
     *
     * @param s1
     * @param duration1
     * @param s2
     * @param duration2
     * @param graceTime
     *            allow to overlap time @return
     */
    public static boolean isOverlap(final long s1, final long duration1, final long s2, final long duration2, final long graceTime) {
        final long f1 = s1 + duration1;
        final long f2 = s2 + duration2;
        return f2 - s1 > graceTime && f1 - s2 > graceTime && (s2 <= s1 && f2 > s1 || s2 > s1 && f2 <= f1 || s2 < f1 && f2 >= f1);
    }

    public static boolean isOverlap(final long start1, final long duration1, final long start2, final long duration2) {
        final long finish1 = start1 + duration1;
        final long finish2 = start2 + duration2;
        return start2 <= start1 && finish2 > start1 || start2 > start1 && finish2 <= finish1 && duration2 != 0 || start2 < finish1 && finish2 >= finish1 && duration1 != 0;
    }

    public static boolean isOverlap(final ITimeRange range1, final ITimeRange range2) {
        final long start1 = range1.getStart().getTime();
        final long duration1 = range1.getEnd().getTime() - start1;
        final long start2 = range2.getStart().getTime();
        final long duration2 = range2.getEnd().getTime() - start2;
        return isOverlap(start1, duration1, start2, duration2);
    }

    public static boolean isOverlap(@Nonnull final LocalTimeRange range1, @Nonnull final LocalTimeRange range2) {
        final long start1 = range1.getStart().getMiliseconds();
        final long duration1 = range1.getDuration();
        final long start2 = range2.getStart().getMiliseconds();
        final long duration2 = range2.getDuration();
        return isOverlap(start1, duration1, start2, duration2);
    }

    public static boolean isOverlap(final Date start1, final long duration1, final ITimeRange range) {
        final long start2 = range.getStart()
                .getTime();
        final long duration2 = range.getEnd()
                .getTime() - start2;
        return isOverlap(start1.getTime(), duration1, start2, duration2);
    }

    public static boolean isOverlap(final GwtLocalDateTimeModel start1, final long duration1, final ITimeRange range) {
        final long start2 = range.getStart()
                .getTime();
        final long duration2 = range.getEnd()
                .getTime() - start2;
        return isOverlap(start1.getMiliseconds(), duration1, start2, duration2);
    }

    /**
     * @param start1
     * @param duration1
     * @param range
     * @param graceTime
     *            - allow to overlap time @return
     */
    public static boolean isOverlap(final Date start1, final long duration1, final ITimeRange range, final long graceTime) {
        final long start2 = range.getStart()
                .getTime();
        final long duration2 = range.getEnd()
                .getTime() - start2;
        return isOverlap(start1.getTime(), duration1, start2, duration2, graceTime);
    }

    /**
     * @param start1
     * @param duration1
     * @param range
     * @param graceTime
     *            - allow to overlap time @return
     */
    public static boolean isOverlap(final GwtLocalDateTimeModel start1, final long duration1, final ITimeRange range, final long graceTime) {
        final long start2 = range.getStart()
                .getTime();
        final long duration2 = range.getEnd()
                .getTime() - start2;
        return isOverlap(start1.getMiliseconds(), duration1, start2, duration2, graceTime);
    }

    public static boolean isOverlap(final Date start, final long duration, final List<? extends ITimeRange> ranges) {
        for (final ITimeRange range : ranges) {
            if (isOverlap(start, duration, range)) {
                return true;
            }
        }
        return false;
    }

    public static boolean isOverlap(@Nonnull final GwtLocalDateTimeModel start, final long duration, @Nonnull final List<? extends ITimeRange> ranges) {
        for (final ITimeRange range : ranges) {
            if (isOverlap(start, duration, range)) {
                return true;
            }
        }
        return false;
    }

    public static boolean isOverlap(final Date start, final long duration, final List<? extends ITimeRange> occupiedTime, final long graceTime) {
        for (final ITimeRange range : occupiedTime) {
            if (isOverlap(start, duration, range, graceTime)) {
                return true;
            }
        }
        return false;
    }

    public static boolean isOverlap(final GwtLocalDateTimeModel start, final long duration, final List<? extends ITimeRange> occupiedTime, final long graceTime) {
        for (final ITimeRange range : occupiedTime) {
            if (isOverlap(start, duration, range, graceTime)) {
                return true;
            }
        }
        return false;
    }

    public static Date getDate(final Date start, final long duration) {
        return new Date(start.getTime() + duration);
    }

    public static String getDatePatternWithoutYear() {
        final String pattern = LocaleInfo.getCurrentLocale()
                .getDateTimeFormatInfo()
                .dateFormat();
        return pattern.replaceAll(REMOVE_YEAR_PATTERN, ""); //$NON-NLS-1$
    }

    public static String getDateLongPatternWithoutYear() {
        final String pattern = LocaleInfo.getCurrentLocale()
                .getDateTimeFormatInfo()
                .dateFormatLong();
        return pattern.replaceAll(REMOVE_YEAR_PATTERN, ""); //$NON-NLS-1$
    }

    /**
     * Find out if hours is overlap in single day
     *
     * @param day
     *            - day to check @param range - take into account only time
     *            part @param startTime - take into account only time
     *            part @param duration @return
     */
    public static boolean isHoursOverlap(@Nonnull final Date day, final ITimeRange range, @Nonnull final Date startTime, final long duration) {
        return isOverlap(setTime(day, range.getStart()), setTime(day, range.getEnd()), setTime(day, startTime), duration);
    }

    /**
     * Sets the time of day from the time parameter
     *
     * @param day
     *            - to set the time @param time - time to set @return new
     *            {@link Date} with date from day and time from time
     */
    public static Date setTime(@Nonnull final Date day, @Nonnull final Date time) {
        final Calendar dayHelper = createCalendar(day);
        final Calendar timeHelper = createCalendar(time);
        setTime(dayHelper, timeHelper.get(Calendar.HOUR_OF_DAY), timeHelper.get(Calendar.MINUTE));
        return dayHelper.getTime();
    }

    /**
     * Calculate time from midnight
     *
     * @param time
     *            any date @return amount of MILLISECONDs from midnight
     */
    public static long getDayTime(@Nonnull final Date time) {
        return SharedDates.getDayTime(createCalendar(time));
    }

    // /**
    // * Set and return date with time from midnight
    // *
    // * @param dayTyme
    // * amount of MILLISECONDs from midnight @return date with
    // * specific time (the date is undefined, depends from
    // * implementation)
    // */
    // public static Date setDayTime(final long dayTyme) {
    // final Calendar helper = createCalendar();
    // final long fullSeconds = dayTyme / 1000;
    // final long fullMinutes = fullSeconds / 60;
    // helper.set(Calendar.HOUR_OF_DAY, (int) ((fullMinutes / 60) % 24));
    // helper.set(Calendar.MINUTE, (int) (fullMinutes % 60));
    // helper.set(Calendar.SECOND, (int) (fullSeconds % 60));
    // helper.set(Calendar.MILLISECOND, (int) (dayTyme % 1000));
    // return helper.getTime();
    // }

    /**
     * Compare two date with specified precision
     *
     * @param date1
     * @param date2
     * @param precision
     *            - one of the following Calendar constant (
     *            {@link Calendar#YEAR}, {@link Calendar#MONTH},
     *            {@link Calendar#DAY_OF_MONTH}, {@link Calendar#HOUR_OF_DAY} or
     *            {@link Calendar#HOUR}, {@link Calendar#MINUTE},
     *            {@link Calendar#SECOND}, {@link Calendar#MILLISECOND}) @return
     *            the value 0 if the time represented by the date2 is equal to
     *            the time represented by the date1; a value less than 0 if the
     *            time of the date1 is before the time represented by the date2;
     *            and a value greater than 0 if the time of the date1 is after
     *            the time represented by the date2.
     */
    public static long compareDateTime(@Nonnull final Date date1, @Nonnull final Date date2, final int precision) {
        final Calendar dayHelper1 = createCalendar(date1, precision);
        final Calendar dayHelper2 = createCalendar(date2, precision);
        return dayHelper1.getTimeInMillis() - dayHelper2.getTimeInMillis();
    }

    public static long compareDateTime(@Nonnull final GwtLocalDateTimeModel date1, @Nonnull final GwtLocalDateTimeModel date2, final int precision) {
        final Calendar dayHelper1 = createCalendar(date1.toDate(), precision);
        final Calendar dayHelper2 = createCalendar(date2.toDate(), precision);
        return dayHelper1.getTimeInMillis() - dayHelper2.getTimeInMillis();
    }

    public static long compareDate(@Nonnull final GwtLocalDateModel date1, @Nonnull final GwtLocalDateModel date2) {
        return (date1.getYear() - date2.getYear()) + (date1.getMonth() - date2.getMonth()) + (date1.getDay() - date2.getDay());
    }

    public static long compareDateTime(@Nonnull final GwtLocalDateModel date1, @Nonnull final GwtLocalDateModel date2, final int precision) {
        return compareDate(date1, date2, precision);
    }

    public static long compareDate(@Nonnull final GwtLocalDateModel date1, @Nonnull final GwtLocalDateModel date2, final int precision) {
        long result = 0;
        switch (precision) {
        case Calendar.MILLISECOND:
        case Calendar.SECOND:
        case Calendar.MINUTE:
        case Calendar.HOUR_OF_DAY:
        case Calendar.HOUR:
        case Calendar.DAY_OF_MONTH:
            result += date1.getDay() - date2.getDay();
            //$FALL-THROUGH$
        case Calendar.MONTH:
            result += date1.getMonth() - date2.getMonth();
            //$FALL-THROUGH$
        case Calendar.YEAR:
            result += date1.getYear() - date2.getYear();
            //$FALL-THROUGH$
        default:
        }
        return result;
    }

    /**
     * Compare the time part from given dates with specified precision
     *
     * @param date1
     * @param date2
     * @param precision
     *            - one of the following Calendar constant (
     *            {@link Calendar#HOUR_OF_DAY} or {@link Calendar#HOUR},
     *            {@link Calendar#MINUTE}, {@link Calendar#SECOND},
     *            {@link Calendar#MILLISECOND}) @return the value 0 if the time
     *            represented by the date2 is equal to the time represented by
     *            the date1; a value less than 0 if the time of the date1 is
     *            before the time represented by the date2; and a value greater
     *            than 0 if the time of the date1 is after the time represented
     *            by the date2.
     */
    public static long compareTime(@Nonnull final Date date1, @Nonnull final Date date2, final int precision) {
        final Calendar dayHelper1 = createTimeCalendar(date1, precision);
        final Calendar dayHelper2 = createTimeCalendar(date2, precision);
        return dayHelper1.getTimeInMillis() - dayHelper2.getTimeInMillis();
    }

    public static long compareTime(@Nonnull final GwtLocalTimeModel date1, @Nonnull final GwtLocalTimeModel date2, final int precision) {
        long x = 1;
        switch (precision) {
        case Calendar.HOUR_OF_DAY:
        case Calendar.HOUR:
            x *= 60;
            //$FALL-THROUGH$
        case Calendar.MINUTE:
            x *= 60;
            //$FALL-THROUGH$
        case Calendar.SECOND:
            x *= 1000;
            //$FALL-THROUGH$
        case Calendar.MILLISECOND:
            break;
        case Calendar.DAY_OF_MONTH:
            //$FALL-THROUGH$
        case Calendar.MONTH:
            //$FALL-THROUGH$
        case Calendar.YEAR:
            //$FALL-THROUGH$
        default:
            return 0;
        }
        return (date1.getMiliseconds() - date2.getMiliseconds()) / x;
    }

    /**
     * Print ONLY date range ignore time
     *
     * @param startTime
     * @param endTime
     * @return
     */
    @SuppressWarnings("null")
    @Nonnull
    public static String printDateRange(@Nonnull final Date startTime, @Nonnull final Date endTime, final boolean printYear) {
        final DateTimeFormat fy = DateTimeFormat.getFormat(LocaleInfo.getCurrentLocale()
                .getDateTimeFormatInfo()
                .dateFormat());
        final DateTimeFormat f = DateTimeFormat.getFormat(getDatePatternWithoutYear());
        if (isSameDay(startTime, endTime)) {
            return printYear ? fy.format(startTime) : f.format(startTime);
        } else if (isSameMonth(startTime, endTime)) {
            if (printYear) {
                return BaseRs.FMT.rangeSameMonthWithYear(getField(startTime, Calendar.DAY_OF_MONTH), getField(endTime, Calendar.DAY_OF_MONTH), printMonth(startTime), getField(endTime, Calendar.YEAR));
            }
            return BaseRs.FMT.rangeSameMonth(getField(startTime, Calendar.DAY_OF_MONTH), getField(endTime, Calendar.DAY_OF_MONTH), printMonth(startTime));
        } else if (isSameYear(startTime, endTime)) {
            if (printYear) {
                return BaseRs.FMT.rangeWithYear(f.format(startTime), f.format(endTime), getField(endTime, Calendar.YEAR));
            }
            return BaseRs.FMT.range(f.format(startTime), f.format(endTime));
        } else {
            if (printYear) {
                return BaseRs.FMT.range(fy.format(startTime), fy.format(endTime));
            }
            return BaseRs.FMT.range(f.format(startTime), f.format(endTime));
        }
    }

    /**
     * Print ONLY date range ignore time
     *
     * @param startTime
     * @param endTime
     * @return
     */
    @Nonnull
    public static String printDateRange(@Nonnull final GwtLocalDateModel startTime, @Nonnull final GwtLocalDateModel endTime, final boolean printYear) {
        final DateTimeFormat fy = DateTimeFormat.getFormat(LocaleInfo.getCurrentLocale().getDateTimeFormatInfo().dateFormat());
        final DateTimeFormat f = DateTimeFormat.getFormat(getDatePatternWithoutYear());
        if (isSameDay(startTime, endTime)) {
            return printYear ? fy.format(startTime.toDate()) : f.format(startTime.toDate());
        } else if (isSameMonth(startTime, endTime)) {
            if (printYear) {
                return BaseRs.FMT.rangeSameMonthWithYear(startTime.getDay(), endTime.getDay(), printMonth(startTime), endTime.getYear());
            }
            return BaseRs.FMT.rangeSameMonth(getField(startTime, Calendar.DAY_OF_MONTH), getField(endTime, Calendar.DAY_OF_MONTH), printMonth(startTime));
        } else if (isSameYear(startTime, endTime)) {
            if (printYear) {
                return BaseRs.FMT.rangeWithYear(f.format(startTime.toDate()), f.format(endTime.toDate()), endTime.getYear());
            }
            return BaseRs.FMT.range(f.format(startTime.toDate()), f.format(endTime.toDate()));
        } else {
            if (printYear) {
                return BaseRs.FMT.range(fy.format(startTime.toDate()), fy.format(endTime.toDate()));
            }
            return BaseRs.FMT.range(f.format(startTime.toDate()), f.format(endTime.toDate()));
        }
    }

    /**
     * Print ONLY date range ignore time. The year does not print.
     *
     * {@link #printDateRange(Date, Date, boolean)} to print with year
     *
     * @param startTime
     * @param endTime
     * @return
     */
    public static String printDateRange(@Nonnull final Date startTime, @Nonnull final Date endTime) {
        return printDateRange(startTime, endTime, false);
    }

    /**
     * Print date range. The year does not print.
     *
     * {@link #printDateRange(Date, Date, boolean)} to print with year
     *
     * @param startTime
     * @param endTime
     * @return
     */
    public static String printDateRange(@Nonnull final GwtLocalDateModel startTime, final int days) {
        return printDateRange(startTime.toDate(), changeDay(startTime, days).toDate(), false);
    }

    /**
     * Print date range from date plus days.
     *
     * @param startTime
     * @param days
     * @param printYear
     * @return
     */
    public static String printDateRange(@Nonnull final GwtLocalDateModel startTime, final int days, final boolean printYear) {
        return printDateRange(startTime.toDate(), changeDay(startTime, days - 1).toDate(), printYear);
    }

    public static String printDateTimeRange(final Date start, final long duration) {
        return printDateTimeRange(start, new Date(start.getTime() + duration));
    }

    public static String printDateTimeRange(final GwtLocalDateTimeModel start, final long duration) {
        return printDateTimeRange(start.toDate(), new Date(start.getMiliseconds() + duration));
    }

    public static String printDateTimeRange(final GwtLocalDateModel start, final long duration) {
        return printDateTimeRange(start.toDate(), new Date(start.toDate().getTime() + duration));
    }

    public static String printDateTimeRange(@Nonnull final GwtLocalDateModel start, @Nonnull final GwtLocalDateModel end) {
        return printDateTimeRange(start.toDate(), end.toDate());
    }

    /**
     * Returns the date time range, even if dates different
     *
     * @param startTime
     * @param endTime
     * @return
     */
    public static String printDateTimeRange(@Nonnull final Date startTime, @Nonnull final Date endTime) {
        final String d = getDatePatternWithoutYear();
        final String t = LocaleInfo.getCurrentLocale()
                .getDateTimeFormatInfo()
                .timeFormatShort();
        if (isSameTime(startTime, endTime)) {
            final DateTimeFormat ft = DateTimeFormat.getFormat(t);
            return ft.format(startTime);
        } else if (isSameDay(startTime, endTime)) {
            final DateTimeFormat ft = DateTimeFormat.getFormat(t);
            return BaseRs.FMT.rangeSameDay(ft.format(startTime), ft.format(endTime), printDayOfMonth(startTime));
        } else {
            final String p = LocaleInfo.getCurrentLocale()
                    .getDateTimeFormatInfo()
                    .dateTimeShort(d, t);
            final DateTimeFormat fp = DateTimeFormat.getFormat(p);
            return BaseRs.FMT.range(fp.format(startTime), fp.format(endTime)); // ;
        }
    }

    /**
     * Prints full week day name by day index [0..6]
     *
     * @param dayIndex
     *            - index of day from 0 to 6, 0 is fist day of week @return
     *            localized weekday name
     */
    @SuppressWarnings("null")
    @Nonnull
    public static String printWeekday(final int dayIndex) {
        final DateTimeFormatInfo info = LocaleInfo.getCurrentLocale().getDateTimeFormatInfo();
        final int requestedDay = dayIndex < 0 ? 0 : dayIndex > 6 ? 6 : dayIndex + info.firstDayOfTheWeek();
        return info.weekdaysFullStandalone()[requestedDay > 6 ? requestedDay - 7 : requestedDay];
    }

    public static String printWeek(final GwtLocalDateModel date) {
        return BaseRs.FMT.printWeek(Integer.valueOf(getWeek(date)), printYear(date));
    }

    public static String printMonth(@Nonnull final Date date) {
        return DateTimeFormat.getFormat(LocaleInfo.getCurrentLocale()
                .getDateTimeFormatInfo()
                .formatMonthFull())
                .format(date);
    }

    public static String printMonth(@Nonnull final GwtLocalDateModel date) {
        return printMonth(date.toDate());
    }

    public static String printYear(final GwtLocalDateModel date) {
        return printYear(date.toDate());
    }

    public static String printYear(final Date date) {
        return DateTimeFormat.getFormat(LocaleInfo.getCurrentLocale()
                .getDateTimeFormatInfo()
                .formatYear())
                .format(date);
    }

    public static int getField(@Nonnull final Date date, final int field) {
        return createCalendar(date).get(field);
    }

    public static int getField(@Nonnull final GwtLocalDateModel date, final int field) {
        return createCalendar(date).get(field);
    }

    /**
     * Returns the ONLY time range, even if dates different
     *
     * @param startTime
     * @param endTime
     * @return
     */
    public static String printTimeRange(@Nonnull final Date startTime, @Nonnull final Date endTime) {
        return isSameTime(startTime, endTime) ? printTime(startTime) : BaseRs.FMT.range(printTime(startTime), printTime(endTime));
    }

    public static String printTimeRange(@Nonnull final GwtLocalTimeModel startTime, @Nonnull final GwtLocalTimeModel endTime) {
        return isSameTime(startTime, endTime) ? printTime(startTime) : BaseRs.FMT.range(printTime(startTime), printTime(endTime));
    }

    public static String printTimeRange(@Nonnull final GwtLocalTimeModel startTime, final int duration) {
        return printTimeRange(startTime, new GwtLocalTimeModel(startTime.getMiliseconds() + duration));
    }

    /**
     * Returns the ONLY the date
     *
     * @param date
     *            to print @return only date string
     */
    @Nonnull
    public static String printDate(@Nonnull final Date date) {
        final String p = getFormatInfo().dateFormatShort();
        final DateTimeFormat f = DateTimeFormat.getFormat(p);
        return f.format(date);
    }

    /**
     * Returns the ONLY the time
     *
     * @param time
     * @return only time string
     */
    public static String printTime(@Nonnull final Date time) {
        final String p = getFormatInfo().timeFormatShort();
        final DateTimeFormat f = DateTimeFormat.getFormat(p);
        return f.format(time);
    }

    public static String printTime(@Nonnull final GwtLocalTimeModel time) {
        return printTime(toDate(time));
    }

    /**
     * Returns the Date/Time
     *
     * @param date
     *            to print @return date/time string
     */
    @Nonnull
    public static String printDateTime(@Nonnull final Date date) {
        return printDateTime(date, null);
    }

    /**
     * Returns the Date/Time
     *
     * @param date
     *            to print @return date/time string
     */
    @SuppressWarnings("null")
    @Nonnull
    public static String printDateTime(@Nonnull final Date date, @Nullable final TimeZone timeZone) {
        final String t = getFormatInfo().timeFormatShort();
        final String d = getFormatInfo().dateFormatShort();
        return DateTimeFormat.getFormat(getFormatInfo().dateTimeShort(t, d))
                .format(date, timeZone);
    }

    @Nonnull
    public static String printDateTime(@Nonnull final GwtDateTimeModel date) {
        return printDateTime(date.toDate(), getTimeZone(date));
    }

    @SuppressWarnings("null")
    @Nonnull
    public static TimeZone getTimeZone(@Nonnull final GwtDateTimeModel date) {
        return com.google.gwt.i18n.client.TimeZone.createTimeZone(Ints.saturatedCast(date.getOffset() / (60 * 1000)));
    }

    /**
     * @param startDate
     * @param startTime
     * @return
     */
    @Nonnull
    public static String printDateTime(@Nonnull final GwtLocalDateModel startDate, @Nonnull final GwtLocalTimeModel startTime) {
        return printDateTime(new GwtLocalDateTimeModel(startDate, startTime));
    }

    @Nonnull
    public static String printDateTime(@Nonnull final GwtLocalDateTimeModel date) {
        return printDateTime(date.toDate());
    }

    public static String printDayOfMonth(@Nonnull final Date date) {
        return DateTimeFormat.getFormat(LocaleInfo.getCurrentLocale()
                .getDateTimeFormatInfo()
                .formatMonthAbbrevDay())
                .format(date);
    }

    public static String printDayOfMonth(@Nonnull final GwtLocalDateModel date) {
        return printDayOfMonth(date.toDate());
    }

    /** Print duration with specified precision */
    public static String printDuration(final int duration, final int precision) {
        return printDuration(duration, precision, TXT.year(), TXT.years(), TXT.month(), TXT.months(), TXT.day(), TXT.days(), TXT.hour(), TXT.hours(), TXT.minute(), TXT.minutes(), TXT.second(), TXT.seconds(), TXT.millisecond(), TXT.milliseconds());
    }

    /** Print duration with specified precision */
    public static String printDuration(final int duration, final int precision, final String y, final String ys, final String m, final String ms, final String d, final String ds, final String h, final String hs, final String mi, final String mis, final String s, final String ss, final String ml, final String mls) {
        int du = duration;
        final int years = du / precision(Calendar.YEAR);
        final int months = isEnought(Calendar.MONTH, precision) ? 0 : (du -= precision(Calendar.YEAR) * years) / precision(Calendar.MONTH);
        final int days = isEnought(Calendar.DAY_OF_MONTH, precision) ? 0 : (du -= precision(Calendar.MONTH) * months) / precision(Calendar.DAY_OF_MONTH);
        final int hours = isEnought(Calendar.HOUR, precision) ? 0 : (du -= precision(Calendar.DAY_OF_MONTH) * days) / precision(Calendar.HOUR);
        final int minutes = isEnought(Calendar.MINUTE, precision) ? 0 : (du -= precision(Calendar.HOUR) * hours) / precision(Calendar.MINUTE);
        final int seconds = isEnought(Calendar.SECOND, precision) ? 0 : (du -= precision(Calendar.MINUTE) * minutes) / precision(Calendar.SECOND);
        final int milliseconds = isEnought(Calendar.MILLISECOND, precision) ? 0 : (du -= precision(Calendar.SECOND) * seconds) / precision(Calendar.MILLISECOND);
        final String join = Joiner.on(", ") //$NON-NLS-1$
                .skipNulls()
                .join(str(years, y, ys), str(months, m, ms), str(days, d, ds), str(hours, h, hs), str(minutes, mi, mis), str(seconds, s, ss), str(milliseconds, ml, mls));
        return join.isEmpty() ? defaultDuration(precision, ys, ms, ds, hs, mis, ss, mls) : join;
    }

    /**
     * Return string with default duration for specified precision
     *
     * @param precision
     *            - target precision
     * @return the default (usually zero) string
     */
    private static String defaultDuration(final int precision, final String years, final String months, final String days, final String hours, final String minutes, final String seconds, final String milliseconds) {
        switch (precision) {
        case Calendar.YEAR:
            return "0 " + years; //$NON-NLS-1$
        case Calendar.MONTH:
            return "0 " + months; //$NON-NLS-1$
        case Calendar.DAY_OF_MONTH:
            return "0 " + days; //$NON-NLS-1$
        case Calendar.HOUR_OF_DAY:
        case Calendar.HOUR:
            return "0 " + hours; //$NON-NLS-1$
        case Calendar.MINUTE:
            return "0 " + minutes; //$NON-NLS-1$
        case Calendar.SECOND:
            return "0 " + seconds; //$NON-NLS-1$
        default:
            return "0 " + milliseconds; //$NON-NLS-1$
        }
    }

    /**
     * Calculate if specified precision is reached
     *
     * @param current
     *            - current precision
     * @param target
     *            - target precision
     * @return
     */
    private static boolean isEnought(final int current, final int target) {
        return target < current;
    }

    /** A valid partial-time as defined in [RFC 3339] */
    @SuppressWarnings("null")
    @Nonnull
    public static Date parseTimeToDate(@Nonnull final String value) {
        final String p = getFormatInfo().timeFormatShort();
        final DateTimeFormat f = DateTimeFormat.getFormat(p);
        return f.parse(value);
    }

    /** A valid partial-time as defined in [RFC 3339] */
    @Nonnull
    public static GwtLocalTimeModel parseTime(@Nonnull final String value) {
        return new GwtLocalTimeModel(createCalendar(parseTimeToDate(value)));
    }

    private static String str(final int unit, final String single, final String multiple) {
        switch (unit) {
        case 0:
            return null;
        case 1:
            return String.valueOf(unit) + " " + single; //$NON-NLS-1$
        default:
            return String.valueOf(unit) + " " + multiple; //$NON-NLS-1$
        }
    }

    public static int precision(final int unit) {
        switch (unit) {
        case Calendar.YEAR:
            return 365 * 24 * 60 * 60 * 1000;
        case Calendar.MONTH:
            return 30 * 24 * 60 * 60 * 1000;
        case Calendar.DAY_OF_YEAR:
        case Calendar.DAY_OF_MONTH:
        case Calendar.DAY_OF_WEEK:
        case Calendar.DAY_OF_WEEK_IN_MONTH:
            return 24 * 60 * 60 * 1000;
        case Calendar.HOUR_OF_DAY:
        case Calendar.HOUR:
            return 60 * 60 * 1000;
        case Calendar.MINUTE:
            return 60 * 1000;
        case Calendar.SECOND:
            return 60 * 1000;
        default:
            return 1000;
        }
    }

    public static boolean isSameTime(@Nonnull final GwtLocalTimeModel oneTime, @Nonnull final GwtLocalTimeModel secondTime) {
        return compareTime(oneTime, secondTime, Calendar.MINUTE) == 0;
    }

    public static boolean isSameTime(@Nonnull final Date oneTime, @Nonnull final Date secondTime) {
        return compareTime(oneTime, secondTime, Calendar.MINUTE) == 0;
    }

    /** Take into account only time part */
    public static boolean isSameTime(@Nonnull final Date startTime, @Nonnull final Date endTime, final int precision) {
        return compareTime(startTime, endTime, precision) == 0;
    }

    public static boolean isSameDateTime(@Nonnull final Date startTime, @Nonnull final Date endTime, final int precision) {
        return compareDateTime(startTime, endTime, precision) == 0;
    }

    public static boolean isSameWeek(@Nonnull final GwtLocalDateModel oneDate, @Nonnull final GwtLocalDateModel anotherDate) {
        return compareDateTime(oneDate.toDate(), anotherDate.toDate(), Calendar.WEEK_OF_YEAR) == 0;
    }

    public static boolean isSameMonth(@Nonnull final Date startTime, @Nonnull final Date endTime) {
        return compareDateTime(startTime, endTime, Calendar.MONTH) == 0;
    }

    public static boolean isSameMonth(@Nonnull final GwtLocalDateModel startTime, @Nonnull final GwtLocalDateModel endTime) {
        return compareDateTime(startTime, endTime, Calendar.MONTH) == 0;
    }

    public static boolean isSameYear(@Nonnull final Date startTime, @Nonnull final Date endTime) {
        return compareDateTime(startTime, endTime, Calendar.YEAR) == 0;
    }

    public static boolean isSameYear(@Nonnull final GwtLocalDateModel startTime, @Nonnull final GwtLocalDateModel endTime) {
        return compareDateTime(startTime, endTime, Calendar.YEAR) == 0;
    }

    public static DateTimeFormatInfo getFormatInfo() {
        return LocaleInfo.getCurrentLocale()
                .getDateTimeFormatInfo();
    }

    /**
     * Return the end date of surround the current time period
     *
     * @param period
     *            - one of the following Calendar constant (
     *            {@link Calendar#YEAR}, {@link Calendar#MONTH},
     *            {@link Calendar#DAY_OF_MONTH}, {@link Calendar#HOUR_OF_DAY} or
     *            {@link Calendar#HOUR}, {@link Calendar#MINUTE},
     *            {@link Calendar#SECOND}, {@link Calendar#MILLISECOND}) @return
     */
    public static Date getSurroundEnd(final int period) {
        return getSurroundEnd(new Date(), period);
    }

    /**
     * Return the start date of surround the current time period
     *
     * @param period
     *            - one of the following Calendar constant (
     *            {@link Calendar#YEAR}, {@link Calendar#MONTH},
     *            {@link Calendar#DAY_OF_MONTH}, {@link Calendar#HOUR_OF_DAY} or
     *            {@link Calendar#HOUR}, {@link Calendar#MINUTE},
     *            {@link Calendar#SECOND}, {@link Calendar#MILLISECOND}) @return
     */
    public static Date getSurroundStart(final int period) {
        return getSurroundStart(new Date(), period);
    }

    /**
     * Return the end date of surround the given time period
     *
     * @param date
     *            date to surround @param period - one of the following Calendar
     *            constant ( {@link Calendar#YEAR}, {@link Calendar#MONTH},
     *            {@link Calendar#DAY_OF_MONTH}, {@link Calendar#HOUR_OF_DAY} or
     *            {@link Calendar#HOUR}, {@link Calendar#MINUTE},
     *            {@link Calendar#SECOND}) @return
     */
    public static Date getSurroundEnd(@Nonnull final Date date, final int period) {
        final Calendar calendar = createCalendar(date, period);
        calendar.add(period, 1);
        calendar.add(Calendar.MILLISECOND, -1);
        return calendar.getTime();
    }

    @Nonnull
    public static GwtLocalDateModel getSurroundEnd(@Nonnull final GwtLocalDateModel date, final int period) {
        final Calendar calendar = createCalendar(date, period);
        calendar.add(period, 1);
        calendar.add(Calendar.MILLISECOND, -1);
        return new GwtLocalDateModel(calendar);
    }

    public static int getPrev(final int period) {
        switch (period) {
        case Calendar.YEAR:
            return Calendar.MONTH;
        case Calendar.MONTH:
            return Calendar.DAY_OF_MONTH;
        case Calendar.DAY_OF_MONTH:
            return Calendar.HOUR_OF_DAY;
        case Calendar.HOUR_OF_DAY:
        case Calendar.HOUR:
            return Calendar.MINUTE;
        case Calendar.MINUTE:
            return Calendar.SECOND;
        case Calendar.SECOND:
            return Calendar.MILLISECOND;
        default:
            throw new IllegalStateException("No pevious for miliseconds"); //$NON-NLS-1$
        }
    }

    public static int getNext(final int period) {
        switch (period) {
        case Calendar.MONTH:
            return Calendar.YEAR;
        case Calendar.DAY_OF_MONTH:
            return Calendar.MONTH;
        case Calendar.HOUR_OF_DAY:
        case Calendar.HOUR:
            return Calendar.DAY_OF_MONTH;
        case Calendar.MINUTE:
            return Calendar.HOUR_OF_DAY;
        case Calendar.SECOND:
            return Calendar.MINUTE;
        case Calendar.MILLISECOND:
            return Calendar.SECOND;
        default:
            throw new IllegalStateException("No next for year"); //$NON-NLS-1$
        }
    }

    /**
     * Return the start date of surround the given time period
     *
     * @param date
     *            date to surround @param period - one of the following Calendar
     *            constant ( {@link Calendar#YEAR}, {@link Calendar#MONTH},
     *            {@link Calendar#DAY_OF_MONTH}, {@link Calendar#HOUR_OF_DAY} or
     *            {@link Calendar#HOUR}, {@link Calendar#MINUTE},
     *            {@link Calendar#SECOND}, {@link Calendar#MILLISECOND}) @return
     */
    public static Date getSurroundStart(@Nonnull final Date date, final int period) {
        final Calendar calendar = createCalendar(date, period);
        return calendar.getTime();
    }

    @Nonnull
    public static GwtLocalDateModel getSurroundStart(@Nonnull final GwtLocalDateModel date, final int period) {
        int day = date.getDay();
        int month = date.getMonth();
        switch (period) {
        case Calendar.YEAR:
            month = 1;
            //$FALL-THROUGH$
        case Calendar.MONTH:
            day = 1;
            break;
        case Calendar.DAY_OF_MONTH:
        case Calendar.HOUR_OF_DAY:
        case Calendar.HOUR:
        case Calendar.MINUTE:
        case Calendar.SECOND:
        case Calendar.MILLISECOND:
        default:
        }
        return new GwtLocalDateModel(day, month, date.getYear());
    }

    /**
     * See {@link Calendar#add(int, int)}
     *
     * @param date
     *            - date to change @param amount - amount of the change @param
     *            field - field of date to apply change @return new {@link Date}
     */
    @SuppressWarnings("null")
    @Nonnull
    public static Date change(final Date date, final int amount, final int field) {
        final Calendar calendar = createCalendar(date);
        calendar.add(field, amount);
        return calendar.getTime();
    }

    /**
     * Add/remove from given date the specified amount im ms
     *
     * @param date
     *            - date to change @param amount - amount of the change @return
     *            new {@link Date}
     */
    @Nonnull
    public static Date change(@Nonnull final Date date, final long amount) {
        return new Date(date.getTime() + amount);
    }

    @Nonnull
    public static GwtLocalTimeModel change(final GwtLocalTimeModel time, final int amount) {
        int value = time.getMiliseconds() + amount;
        if (value >= SharedDates.DAY_LENGTH) {
            value -= SharedDates.DAY_LENGTH;
        } else if (value < 0) {
            value += SharedDates.DAY_LENGTH;
        }
        return new GwtLocalTimeModel(value);
    }

    @Nonnull
    public static GwtLocalDateTimeModel change(final GwtLocalDateTimeModel dateTime, final long amount) {
        return new GwtLocalDateTimeModel(createCalendar(new Date(dateTime.getMiliseconds() + amount)));
    }

    @Nonnull
    public static GwtLocalDateModel changeDay(@Nonnull final GwtLocalDateModel date, final int amount) {
        return amount == 0 ? date : change(date, amount, Calendar.DAY_OF_MONTH);
    }

    @Nonnull
    public static GwtLocalDateModel change(@Nonnull final GwtLocalDateModel date, final int amount, final int field) {
        final Calendar helper = createCalendar(date);
        helper.add(field, amount);
        return new GwtLocalDateModel(helper);
    }

    @Nonnull
    public static GwtLocalTimeModel change(@Nonnull final GwtLocalTimeModel time, final int amount, final int field) {
        return new GwtLocalTimeModel(time.getMiliseconds() + toMiliseconds(amount, field));
    }

    public static int toMiliseconds(final int amount, final int field) {
        switch (field) {
        case Calendar.YEAR:
        case Calendar.MONTH:
            return 0;
        case Calendar.DAY_OF_YEAR:
        case Calendar.DAY_OF_MONTH:
        case Calendar.DAY_OF_WEEK:
        case Calendar.DAY_OF_WEEK_IN_MONTH:
            return amount * SharedDates.DAY_LENGTH;
        case Calendar.HOUR_OF_DAY:
        case Calendar.HOUR:
            return amount * 60 * 60 * 1000;
        case Calendar.MINUTE:
            return amount * 60 * 1000;
        case Calendar.SECOND:
            return amount * 1000;
        case Calendar.MILLISECOND:
            return amount;
        default:
        }
        return 0;
    }

    /**
     * Calculates is date in the past with specific precision
     *
     * @param date
     * @param precision
     *            - one of the following Calendar constant (
     *            {@link Calendar#YEAR}, {@link Calendar#MONTH},
     *            {@link Calendar#DAY_OF_MONTH}, {@link Calendar#HOUR_OF_DAY} or
     *            {@link Calendar#HOUR}, {@link Calendar#MINUTE},
     *            {@link Calendar#SECOND}, {@link Calendar#MILLISECOND}) @return
     */
    public static boolean isPast(@Nonnull final Date date, final int precision) {
        return precision == Calendar.MILLISECOND ? System.currentTimeMillis() >= date.getTime() : compareDateTime(date, new Date(), precision) < 0;
    }

    public static boolean isTodayPast(@Nonnull final GwtLocalTimeModel date, final int precision) {
        return precision == Calendar.MILLISECOND ? System.currentTimeMillis() >= date.getMiliseconds() : compareTime(date, new GwtLocalTimeModel(), precision) < 0;
    }

    public static boolean isPast(@Nonnull final GwtLocalDateTimeModel date, final int precision) {
        return precision == Calendar.MILLISECOND ? System.currentTimeMillis() >= date.getMiliseconds() : compareDateTime(date, new GwtLocalDateTimeModel(createCalendar()), precision) < 0;
    }

    /**
     * Calculates is date in the past with specific precision
     *
     * @param date
     * @param precision
     *            - one of the following Calendar constant (
     *            {@link Calendar#YEAR}, {@link Calendar#MONTH},
     *            {@link Calendar#DAY_OF_MONTH}
     *
     * @return
     */
    public static boolean isPast(@Nonnull final GwtLocalDateModel date, final int precision) {
        return compareDate(date, new GwtLocalDateModel(createCalendar()), precision) < 0;
    }

    /**
     * Calculates if the date is in the past
     *
     * @param date
     * @return <code>true</code> if past
     */
    public static boolean isPast(@Nonnull final GwtLocalDateModel date) {
        return compareDate(date, today()) < 0;
    }

    /**
     * Merge to overlap ranges
     *
     * @param range1
     * @param range2
     * @return single range with early start and later end
     */
    public static ITimeRange merge(final ITimeRange range1, final ITimeRange range2) {
        return new ITimeRange() {
            @Override
            public Date getStart() {
                return early(range1.getStart(), range2.getStart());
            }

            @Override
            public Date getEnd() {
                return later(range1.getEnd(), range2.getEnd());
            }
        };
    }

    /**
     * Merge to overlap ranges
     *
     * @param range1
     * @param range2
     * @return single range with early start and later end
     */
    @Nonnull
    public static LocalTimeRange merge(@Nonnull final LocalTimeRange range1, @Nonnull final LocalTimeRange range2) {
        return new LocalTimeRange() {
            @Override
            public GwtLocalTimeModel getStart() {
                return early(range1.getStart(), range2.getStart());
            }

            @Override
            public int getDuration() {
                final GwtLocalTimeModel end1 = Dates.change(range1.getStart(), range1.getDuration());
                final GwtLocalTimeModel end2 = Dates.change(range2.getStart(), range2.getDuration());
                return later(end1, end2).getMiliseconds() - getStart().getMiliseconds();
            }
        };
    }

    /**
     * @return the early date from two given
     */
    @Nonnull
    public static Date early(@Nonnull final Date start, @Nonnull final Date start2) {
        return start.getTime() < start2.getTime() ? start : start2;
    }

    /**
     * @return the early date from two given
     */
    @Nonnull
    public static GwtLocalTimeModel early(@Nonnull final GwtLocalTimeModel start, @Nonnull final GwtLocalTimeModel start2) {
        return start.getMiliseconds() < start2.getMiliseconds() ? start : start2;
    }

    /**
     * @return the early date from two given
     */
    @Nonnull
    public static Date later(@Nonnull final Date end, @Nonnull final Date end2) {
        return end.getTime() >= end2.getTime() ? end : end2;
    }

    /**
     * @return the early date from two given
     */
    @Nonnull
    public static GwtLocalTimeModel later(@Nonnull final GwtLocalTimeModel end, @Nonnull final GwtLocalTimeModel end2) {
        return end.getMiliseconds() >= end2.getMiliseconds() ? end : end2;
    }

    public static long getDuration(final ITimeRange range) {
        return range.getEnd().getTime() - range.getStart().getTime();
    }

    /**
     * Checks if date is end of the day (23.59.59.999)
     *
     * @param date
     * @return
     */
    public static boolean isDayEnd(@Nonnull final Date date) {
        final Calendar helper = createCalendar(date);
        final int h = helper.get(Calendar.HOUR_OF_DAY);
        final int m = helper.get(Calendar.MINUTE);
        final int s = helper.get(Calendar.SECOND);
        return h == 23 && m == 59 && s == 59;
    }

    /**
     * Checks if date is start of the day (0.0.0.0)
     *
     * @param date
     * @return
     */
    public static boolean isDayStart(@Nonnull final Date date) {
        final Calendar helper = createCalendar(date);
        final int h = helper.get(Calendar.HOUR_OF_DAY);
        final int m = helper.get(Calendar.MINUTE);
        final int s = helper.get(Calendar.SECOND);
        return h == 0 && m == 0 && s == 0;
    }

    @Nonnull
    public static GwtLocalDateModel today() {
        return new GwtLocalDateModel(createCalendar());
    }

    @Nonnull
    public static GwtLocalTimeModel now() {
        return new GwtLocalTimeModel(createCalendar());
    }

    @Nonnull
    public static GwtLocalDateTimeModel todayNow() {
        return new GwtLocalDateTimeModel(createCalendar());
    }

    @Nonnull
    public static GwtLocalDateModel clone(@Nonnull final GwtLocalDateModel date) {
        return new GwtLocalDateModel(date);
    }

    public static String format(final GwtLocalDateModel date, final DateTimeFormat format) {
        return format.format(date.toDate());
    }

    /**
     * Counts weeks between today and given date if date is in current week
     * return 0.
     */
    public static int countWeeks(@Nonnull final GwtLocalDateModel date) {
        return weelCountCache.getUnchecked(date).intValue();
    }

    /**
     * Counts weeks between today and given date if date is in current week
     * return 0.
     */
    public static int countWeeks(@Nonnull final Date date) {
        return weelCountCache.getUnchecked(new GwtLocalDateModel(createCalendar(date))).intValue();
    }

    /**
     * Counts weeks between today and given date if date is in current week
     * return 0.
     */
    public static int countWeeks(@Nonnull final Date date1, @Nonnull final Date date2) {
        if (date2.before(date1)) {
            return -countWeeks(date2, date1);
        }
        final Calendar cal1 = getWeekStartDay(date1);
        setTime(cal1, 0, 0);
        final Calendar cal2 = getWeekStartDay(date2);
        setTime(cal2, 0, 0);
        int weeks = 0;
        final Date b = cal2.getTime();
        while (cal1.getTime().before(b)) {
            // add another week
            cal1.add(Calendar.WEEK_OF_YEAR, 1);
            weeks++;
        }
        return weeks;
    }

    /**
     * Print duration with specified precision
     *
     * @param duration
     *            - duration to print
     * @param precision
     *            - precision
     * @return Formated string with current locale
     */
    public static String printDuration(final long duration, final int precision) {
        return printDuration(Ints.saturatedCast(duration), precision);
    }

    @Nonnull
    public static Date toDate(@Nonnull final GwtLocalTimeModel time) {
        return setDayTime(time.getMiliseconds());
    }

    // @SuppressWarnings("null")
    // @Nonnull
    // public static Date toDate(@Nonnull final GwtLocalDateTimeModel dateTime)
    // {
    // dateTime.toDate();
    // final Calendar helper = createCalendar(getCurrentWeekStartDay());
    // setDayTime(helper, dateTime.getTime().getMiliseconds());
    // return helper.getTime();
    // }

    /**
     * Set and return date with time from midnight
     *
     * @param dayTyme
     *            amount of MILLISECONDs from midnight @return date with
     *            specific time (the date is undefined, depends from
     *            implementation)
     */
    @SuppressWarnings("null")
    @Nonnull
    public static Date setDayTime(final long dayTyme) {
        final Calendar helper = createCalendar();
        setDayTime(helper, dayTyme);
        return helper.getTime();
    }

    private static void setDayTime(@Nonnull final Calendar helper, final long dayTyme) {
        final long fullSeconds = dayTyme / 1000;
        final long fullMinutes = fullSeconds / 60;
        helper.set(Calendar.HOUR_OF_DAY, (int) ((fullMinutes / 60) % 24));
        helper.set(Calendar.MINUTE, (int) (fullMinutes % 60));
        helper.set(Calendar.SECOND, (int) (fullSeconds % 60));
        helper.set(Calendar.MILLISECOND, (int) (dayTyme % 1000));
    }

    public static int getWeekDay(final GwtLocalDateTimeModel date) {
        final Calendar calendar = createCalendar(date.getDate());
        final int dayOfWeek = calendar.get(DAY_OF_WEEK) - 1;
        return dayOfWeek == 0 ? 7 : dayOfWeek;
    }

}
