package org.jresearch.commons.gwt.client.mvc;

import java.util.Iterator;

import javax.annotation.Nonnull;

import org.jresearch.commons.gwt.client.base.resource.BaseRs;
import org.jresearch.commons.gwt.client.mvc.INotificator.NotificationType;
import org.jresearch.commons.gwt.client.mvc.event.ErrorEvent;

import com.google.gwt.core.client.Scheduler;
import com.google.gwt.safehtml.shared.SafeHtml;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.ui.HasWidgets;
import com.google.gwt.user.client.ui.HasWidgets.ForIsWidget;
import com.google.gwt.user.client.ui.RootLayoutPanel;
import com.google.gwt.user.client.ui.Widget;
import com.google.inject.Inject;

@SuppressWarnings("rawtypes")
public abstract class AbstractView<C extends AbstractController> {

    @Nonnull
    @Inject
    private INotificator notificator;

    @Nonnull
    private final C controller;

    public AbstractView(@Nonnull final C controller) {
        this.controller = controller;
        Scheduler.get().scheduleDeferred(new Command() {
            @Override
            public void execute() {
                controller.onViewCreate();
            }
        });
    }

    /**
     * @param viewId
     *            - id of a view which request container
     * @return {@link ForIsWidget} which should be used by the child view to
     *         content rendering. <code>null</code> if no child allowed
     */
    @SuppressWarnings({ "static-method" })
    public HasWidgets.ForIsWidget getChildContainer(final String viewId) {
        return null;
    }

    /**
     * @return the parentContainer
     */
    protected HasWidgets.ForIsWidget getParentContainer() {
        final AbstractController parentController = controller.getParentController();
        final AbstractView<?> parentView = parentController == null ? null : parentController.getView();
        return parentView == null ? null : parentView.getChildContainer(getId());
    }

    public void showContent() {
        showContent(getContent());
    }

    private void showContent(@Nonnull final Widget content) {
        final HasWidgets.ForIsWidget parentContainer = getParentContainer();
        if (parentContainer == null) {
            getRootContainer().add(content);
        } else {
            final Iterator<Widget> iterator = parentContainer.iterator();
            final Widget item = iterator.hasNext() ? iterator.next() : null;
            if (!content.equals(item)) {
                parentContainer.clear();
                parentContainer.add(content);
            }
        }
    }

    @Nonnull
    protected HasWidgets.ForIsWidget getRootContainer() {
        return RootLayoutPanel.get();
    }

    /**
     * @return the whole view content to add them to parent container
     */
    @Nonnull
    abstract public Widget getContent();

    /**
     * @return the id
     */
    public String getId() {
        return controller.getId();
    }

    public void onError(final ErrorEvent event) {
        final Throwable data = event.getException();
        final String message = BaseRs.TXT.error(data.getCause() != null ? data.getCause() : data);
        notificator.showNotification(NotificationType.error, message);
    }

    public void showInfo(final SafeHtml safeHtml) {
        notificator.showNotification(NotificationType.info, safeHtml.asString());
    }

}
