package org.jresearch.commons.gwt.oauth2.client.widget;

import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;

import org.fusesource.restygwt.client.Method;
import org.fusesource.restygwt.client.MethodCallback;
import org.fusesource.restygwt.client.REST;
import org.jresearch.commons.gwt.client.mvc.AbstractMethodCallback;
import org.jresearch.commons.gwt.client.mvc.event.Bus;
import org.jresearch.commons.gwt.client.mvc.event.ErrorEvent;
import org.jresearch.commons.gwt.client.tool.GwtDeferredTask;
import org.jresearch.commons.gwt.oauth2.client.mvc.event.SocialAuthenticationEvent;
import org.jresearch.commons.gwt.oauth2.client.service.Oauth2RestService;
import org.jresearch.commons.gwt.oauth2.shared.model.OauthConfiguration;
import org.jresearch.commons.gwt.oauth2.shared.model.SocialNetwork;

import com.google.api.gwt.oauth2.client.Auth;
import com.google.api.gwt.oauth2.client.AuthRequest;
import com.google.common.collect.Maps;
import com.google.gwt.core.client.Callback;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.ScheduledCommand;
import com.google.inject.Inject;
import com.google.inject.Singleton;

@Singleton
public class SocialAuthenticator {

	@Nonnull
	private final Oauth2RestService oauth2service;
	@Nonnull
	private final Bus bus;
	@Nonnull
	private final Map<SocialNetwork, OauthConfiguration> loaded = Maps.newHashMap();

	@Inject
	protected SocialAuthenticator(@Nonnull final Oauth2RestService oauth2service, @Nonnull final Bus bus) {
		this.oauth2service = oauth2service;
		this.bus = bus;
		new GwtDeferredTask(this::loadConfigurations).defer(1);
		// loadConfigurations();
	}

	private void loadConfigurations() {
		REST.withCallback(new AbstractMethodCallback<List<OauthConfiguration>>(bus) {
			@Override
			public void onFailure(final Method method, final Throwable exception) {
				// load something to break requests
				loaded.put(SocialNetwork.standalone, null);
				super.onFailure(method, exception);
			}

			@Override
			public void onSuccess(final Method method, final List<OauthConfiguration> result) {
				for (final OauthConfiguration conf : result) {
					loaded.put(conf.getNetwork(), conf);
				}
			}
		}).call(oauth2service).getServiceConfigurations();
	}

	public void authenticate(final OauthConfiguration configuration) {
		final AuthRequest req = new AuthRequest(configuration.getAuthUrl(), configuration.getClientId()).withScopes(configuration.getScopes());
		Auth.get().login(req, new Callback<String, Throwable>() {
			@Override
			public void onSuccess(final String token) {
				bus.fire(new SocialAuthenticationEvent(configuration.getNetwork(), token));
			}

			@Override
			public void onFailure(final Throwable caught) {
				bus.fire(new ErrorEvent(caught));
			}
		});
	}

	public void enableNetwork(final SocialNetwork network, final EnableAction action) {
		new EnableNetworkCommand(network, action).execute();
	}

	public void accociateWithCurrentUser(@Nonnull final SocialNetwork network, @Nonnull final String authenticationToken, final MethodCallback<Void> callback) {
		REST.withCallback(callback).call(oauth2service).associateWithCurrentUser(network, authenticationToken);
	}

	class EnableNetworkCommand implements ScheduledCommand {

		private final EnableAction action;
		private final SocialNetwork network;

		public EnableNetworkCommand(final SocialNetwork network, final EnableAction action) {
			this.network = network;
			this.action = action;
		}

		@Override
		public void execute() {
			if (loaded.isEmpty()) {
				Scheduler.get().scheduleDeferred(this);
			} else {
				final OauthConfiguration result = loaded.get(network);
				if (result != null) {
					action.configureControl(result);
				}
			}
		}

	}
}
