package org.jresearch.commons.gwt.oauth2.client.widget;

import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.inject.Singleton;

import org.jresearch.commons.gwt.client.mvc.AbstractCallback;
import org.jresearch.commons.gwt.client.mvc.event.Bus;
import org.jresearch.commons.gwt.client.mvc.event.ErrorEvent;
import org.jresearch.commons.gwt.oauth2.client.model.OauthConfiguration;
import org.jresearch.commons.gwt.oauth2.client.model.SocialNetwork;
import org.jresearch.commons.gwt.oauth2.client.mvc.event.SocialAuthenticationEvent;
import org.jresearch.commons.gwt.oauth2.client.service.oauth2.IOauth2ServiceAsync;

import com.google.api.gwt.oauth2.client.Auth;
import com.google.api.gwt.oauth2.client.AuthRequest;
import com.google.common.collect.Maps;
import com.google.gwt.core.client.Callback;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.ScheduledCommand;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.inject.Inject;

public class SocialAuthenticator {

	@Nonnull
	private final IOauth2ServiceAsync oauth2service;
	@Nonnull
	private final Bus bus;
	@Nonnull
	private final Map<SocialNetwork, OauthConfiguration> loaded = Maps.newHashMap();

	@Inject
	@Singleton
	protected SocialAuthenticator(@Nonnull final IOauth2ServiceAsync oauth2service, @Nonnull final Bus bus) {
		this.oauth2service = oauth2service;
		this.bus = bus;
		loadConfigurations();
	}

	private void loadConfigurations() {
		oauth2service.getServiceConfigurations(new AbstractCallback<List<OauthConfiguration>>(bus) {
			@Override
			public void onSuccess(final List<OauthConfiguration> result) {
				for (final OauthConfiguration conf : result) {
					loaded.put(conf.getNetwork(), conf);
				}
			}

			@Override
			public void onFailure(final Throwable caught) {
				// load something to break requests
				loaded.put(SocialNetwork.standalone, null);
				super.onFailure(caught);
			}
		});
	}

	public void authenticate(final OauthConfiguration configuration) {
		final AuthRequest req = new AuthRequest(configuration.getAuthUrl(), configuration.getClientId()).withScopes(configuration.getScopes());
		Auth.get().login(req, new Callback<String, Throwable>() {
			@Override
			public void onSuccess(final String token) {
				bus.fire(new SocialAuthenticationEvent(configuration.getNetwork(), token));
			}

			@Override
			public void onFailure(final Throwable caught) {
				bus.fire(new ErrorEvent(caught));
			}
		});
	}

	public void enableNetwork(final SocialNetwork network, final EnableAction action) {
		new EnableNetworkCommand(network, action).execute();
	}

	public void accociateWithCurrentUser(final SocialNetwork network, final String authenticationToken, final AsyncCallback<Void> callback) {
		oauth2service.associateWithCurrentUser(network, authenticationToken, callback);
	}

	class EnableNetworkCommand implements ScheduledCommand {

		private final EnableAction action;
		private final SocialNetwork network;

		public EnableNetworkCommand(final SocialNetwork network, final EnableAction action) {
			this.network = network;
			this.action = action;
		}

		@Override
		public void execute() {
			if (loaded.isEmpty()) {
				Scheduler.get().scheduleDeferred(this);
			} else {
				final OauthConfiguration result = loaded.get(network);
				if (result != null) {
					action.configureControl(result);
				}
			}
		}

	}
}
