package org.jresearch.commons.gwt.shared.model.localization;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Lists;
import com.google.gwt.core.shared.GWT;
import com.google.gwt.i18n.client.LocaleInfo;

public abstract class LocaleTool {

	// public static String getContainerName(final String propertyName) {
	// return Strings.format("({0})", propertyName); //$NON-NLS-1$
	// }

	@Nonnull
	public static LocaleModel getCurrentLocale(final boolean onlyLanguage) {
		final String localeName = GWT.isClient() ? LocaleInfo.getCurrentLocale().getLocaleName() : LocaleModel.DEFAULT_NAME;
		return LocaleTool.createLocaleModel(localeName, onlyLanguage);
	}

	public static String get(final Map<String, LangModel> langContainer, final boolean onlyLanguage) {
		return get(langContainer, getCurrentLocale(onlyLanguage));
	}

	public static String get(final Map<String, LangModel> langContainer, final LocaleModel locale) {
		if (locale == null) {
			return null;
		}
		final LocaleModel newLocale = checkForRoot(locale);
		final LangModel value = langContainer.get(newLocale.getLocaleName());
		return value != null ? value.getValue() : get(langContainer, newLocale.up());
	}

	public static String getRaw(final Map<String, LangModel> langContainer, final boolean onlyLanguage) {
		return getRaw(langContainer, getCurrentLocale(onlyLanguage));
	}

	public static String getRaw(final Map<String, LangModel> langContainer, final LocaleModel locale) {
		if (locale == null) {
			return null;
		}
		final LocaleModel newLocale = checkForRoot(locale);
		final LangModel value = langContainer.get(newLocale.getLocaleName());
		return value != null ? value.getValue() : null;
	}

	public static void setAll(@Nonnull final Map<String, LangModel> langContainer, final Collection<LangModel> values) {
		for (final LangModel value : values) {
			final LocaleModel locale = value.getLocale();
			if (locale != null) {
				set(langContainer, locale, value.getValue());
			}
		}
	}

	public static Collection<LangModel> getAll(final Map<String, LangModel> langContainer) {
		final List<LangModel> result = Lists.newArrayList();
		for (final LangModel obj : langContainer.values()) {
			result.add(obj);
		}
		return result;
	}

	public static LangModel set(@Nonnull final Map<String, LangModel> langContainer, final String value, final boolean onlyLanguage) {
		return set(langContainer, getCurrentLocale(onlyLanguage), value);
	}

	public static LangModel set(@Nonnull final Map<String, LangModel> langContainer, @Nonnull final LocaleModel locale, final String value) {
		final LocaleModel newLocale = checkForRoot(locale);
		return langContainer.put(newLocale.getLocaleName(), createLangModel(newLocale, value));
	}

	public static LangModel createLangModel(@Nonnull final LocaleModel locale, final String value) {
		if (value == null) {
			return null;
		}
		final LangModel langModel = new LangModel();
		langModel.setLocale(checkForRoot(locale));
		langModel.setValue(value);
		return langModel;
	}

	/**
	 * Check if given model should be replaced by root model.
	 *
	 * <i>It will be done in case of model contains only the language value and
	 * only if such value is DEFAULT or DEFAULT_LANG</i>
	 *
	 * @param model
	 *            - model to check
	 * @return - the original model or root
	 */
	@SuppressWarnings("null")
	@Nonnull
	public static LocaleModel checkForRoot(@Nonnull final LocaleModel model) {
		final String l = model.getLanguage();
		final boolean c = model.getCountry().length() != 0;
		final boolean v = model.getVariant().length() != 0;
		return !(c || v) && (LocaleModel.DEFAULT_NAME.equals(l) || LocaleModel.DEFAULT_LANG.equals(l)) ? LocaleModel.DEFAULT : model;
	}

	@Nonnull
	public static LocaleModel createLocaleModel(final String localeName, final boolean onlyLanguage) {
		final LocaleModel result = new LocaleModel();
		final String[] split = localeName.split(String.valueOf(LocaleModel.SEP));
		try {
			result.setLanguage(split[0]);
			if (!onlyLanguage) {
				result.setCountry(split[1]);
				result.setVariant(split[2]);
			}
		} catch (final ArrayIndexOutOfBoundsException e) {
			// ignore it
		}
		return result;
	}

}
