package org.jresearch.commons.gwt.client.widget;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.jresearch.commons.gwt.client.util.resource.UtilRs;

import com.google.gwt.event.dom.client.KeyCodes;
import com.google.gwt.safehtml.shared.SafeHtml;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.Event.NativePreviewEvent;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.DialogBox;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.HTML;

public class MessageBox extends DialogBox {

	@Nonnull
	public static final SafeHtml cancelButton = UtilRs.TXT.cancelButton();
	@Nonnull
	public static final SafeHtml okButton = UtilRs.TXT.okButton();
	@Nonnull
	public static final SafeHtml yesButton = UtilRs.TXT.yesButton();
	@Nonnull
	public static final SafeHtml noButton = UtilRs.TXT.noButton();

	@Nonnull
	private HTML content;
	private Button cancel;
	@Nonnull
	private ButtonPanel panel;
	@Nullable
	private SafeHtml cancelTxt;

	public MessageBox(@Nonnull final SafeHtml title, @Nonnull final SafeHtml text, @Nonnull final Action... actions) {
		this(actions);
		setHTML(title);
		content.setText(text.asString());
	}

	public MessageBox(@Nonnull final Action... actions) {
		final FlowPanel main = new FlowPanel();
		setModal(true);
		main.add(content = new HTML());
		main.add(panel = createActionButtons(actions));
		setWidget(main);
		setGlassEnabled(true);
	}

	/**
	 * Sets the text for cancel button. <code>null</code> for default. Default
	 * is {@link MessageBox#cancelButton} if several actions or
	 * {@link MessageBox#okButton} if no actions defined
	 *
	 * You can use predefined text {@link MessageBox#yesButton},
	 * {@link MessageBox#noButton}
	 *
	 * @param cancelTxt
	 */
	public void setCancelText(@Nullable final SafeHtml cancelTxt) {
		this.cancelTxt = cancelTxt;
		updateCancel(panel.getWidgetCount() == 1);
	}

	public void setActions(@Nonnull final Action... actions) {
		createButtons(panel, actions);
	}

	@Nonnull
	protected ButtonPanel createActionButtons(@Nonnull final Action... actions) {
		final ButtonPanel actionBar = new ButtonPanel();
		createButtons(actionBar, actions);
		return actionBar;
	}

	protected void createButtons(@Nonnull final ButtonPanel p, @Nonnull final Action... actions) {
		p.clear();
		for (final Action action : actions) {
			p.add(createButton(action));
		}
		updateCancel(actions.length != 0);
		p.add(cancel);
	}

	void updateCancel(final boolean actionsPresents) {
		final SafeHtml btnText = cancelTxt != null ? cancelTxt : actionsPresents ? cancelButton : okButton;
		if (cancel == null) {
			cancel = createButton(new Action(btnText, this::close));
		} else {
			cancel.setHTML(btnText);
		}
	}

	protected Button createButton(final Action action) {
		return new Button(action.getActionName(), e -> close(action.getHandler()));
	}

	private void close(final Command toDo) {
		hide(false);
		if (toDo != null) {
			toDo.execute();
		}
	}

	public void show(@Nonnull final Action... actions) {
		if (actions.length != 0) {
			createButtons(panel, actions);
		}
		center();
	}

	public void show(@Nonnull final SafeHtml title, @Nonnull final SafeHtml text, @Nonnull final Action... actions) {
		createButtons(panel, actions);
		show(title, text);
	}

	public void show(@Nonnull final SafeHtml title, @Nonnull final SafeHtml text) {
		setHTML(title);
		content.setHTML(text);
		center();
	}

	private void close() {
		hide(true);
	}

	@Override
	protected void onPreviewNativeEvent(final NativePreviewEvent event) {
		super.onPreviewNativeEvent(event);
		if (Event.ONKEYDOWN == event.getTypeInt() && event.getNativeEvent().getKeyCode() == KeyCodes.KEY_ESCAPE) {
			close();
		}
	}

}
