package org.jresearch.commons.gwt.client.widget.colorpicker;

import org.jresearch.commons.gwt.client.tool.DeferredTask;
import org.jresearch.commons.gwt.shared.tools.Colors;
import org.jresearch.commons.gwt.shared.tools.Hsl;

import com.google.gwt.canvas.client.Canvas;
import com.google.gwt.canvas.dom.client.Context2d;
import com.google.gwt.event.dom.client.MouseDownEvent;
import com.google.gwt.event.dom.client.MouseDownHandler;
import com.google.gwt.event.dom.client.MouseMoveEvent;
import com.google.gwt.event.dom.client.MouseMoveHandler;
import com.google.gwt.event.dom.client.MouseOutEvent;
import com.google.gwt.event.dom.client.MouseOutHandler;
import com.google.gwt.event.dom.client.MouseUpEvent;
import com.google.gwt.event.dom.client.MouseUpHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.ui.Composite;

/**
 * Original code was taken from
 * http://www.subshell.com/en/subshell/blog/Implementing-a-Color-Picker-Dialog-
 * With-Canvas-and-GWT100.html
 *
 * @author
 * http://www.subshell.com/en/subshell/blog/Implementing-a-Color-Picker-Dialog-
 * With-Canvas-and-GWT100.html
 *
 * @author kot
 */
public class HuePicker extends Composite {

    private final Canvas canvas;
    private int handleY = 90;
    private boolean mouseDown;

    private final DeferredTask updateDown = new DeferredTask() {
        @Override
        public void run() {
            mouseDown = false;
        }
    };

    public static HuePicker createIfSupported() {
        return Canvas.isSupported() ? new HuePicker() : null;
    }

    private HuePicker() {
        canvas = Canvas.createIfSupported();
        canvas.setPixelSize(26, 180);
        canvas.setCoordinateSpaceHeight(180);
        canvas.setCoordinateSpaceWidth(26);

        initWidget(canvas);

        canvas.addMouseDownHandler(new MouseDownHandler() {
            @Override
            public void onMouseDown(final MouseDownEvent event) {
                handleY = event.getRelativeY(canvas.getElement());
                drawGradient();
                fireHueChanged(getHue());

                mouseDown = true;
            }
        });
        canvas.addMouseMoveHandler(new MouseMoveHandler() {
            @Override
            public void onMouseMove(final MouseMoveEvent event) {
                if (mouseDown) {
                    updateDown.cancel();
                    handleY = event.getRelativeY(canvas.getElement());
                    drawGradient();
                    fireHueChanged(getHue());
                }
            }
        });
        canvas.addMouseUpHandler(new MouseUpHandler() {
            @Override
            public void onMouseUp(final MouseUpEvent event) {
                mouseDown = false;
            }
        });
        canvas.addMouseOutHandler(new MouseOutHandler() {
            @Override
            public void onMouseOut(final MouseOutEvent event) {
                updateDown.defer(1500);
            }
        });
    }

    @Override
    protected void onAttach() {
        super.onAttach();
        drawGradient();
    }

    private void drawGradient() {
        final Context2d ctx = canvas.getContext2d();

        // draw gradient
        ctx.setFillStyle("#ffffff"); //$NON-NLS-1$
        ctx.fillRect(0, 0, 26, 180);
        final Hsl hsl = new Hsl(0, 100, 50);
        for (int y = 0; y <= 179; y++) {
            hsl.setH(y * 2);
            ctx.setFillStyle(Colors.hsl2html(hsl));
            ctx.fillRect(3, y, 20, 1);
        }

        // draw handle
        if (handleY >= 0) {
            ctx.setFillStyle("#000000"); //$NON-NLS-1$

            ctx.beginPath();
            ctx.moveTo(3, handleY);
            ctx.lineTo(0, handleY - 3);
            ctx.lineTo(0, handleY + 3);
            ctx.closePath();
            ctx.fill();

            ctx.moveTo(23, handleY);
            ctx.lineTo(26, handleY - 3);
            ctx.lineTo(26, handleY + 3);
            ctx.closePath();
            ctx.fill();
        }
    }

    public HandlerRegistration addHueChangedHandler(final IHueChangedHandler handler) {
        return addHandler(handler, HueChangedEvent.getType());
    }

    private void fireHueChanged(final int hue) {
        fireEvent(new HueChangedEvent(hue));
    }

    public int getHue() {
        return handleY * 2;
    }

    public void setHue(final int hue) {
        handleY = (int) Math.min(Math.max(Math.round(hue / 2d), 0d), 179d);
        drawGradient();
        fireHueChanged(hue);
    }
}
