package org.jresearch.commons.gwt.client.widget.colorpicker;

import org.jresearch.commons.gwt.client.tool.DeferredTask;
import org.jresearch.commons.gwt.shared.tools.Colors;
import org.jresearch.commons.gwt.shared.tools.Hsl;
import org.jresearch.commons.gwt.shared.tools.Rgb;

import com.google.gwt.canvas.client.Canvas;
import com.google.gwt.canvas.dom.client.CanvasGradient;
import com.google.gwt.canvas.dom.client.CanvasPixelArray;
import com.google.gwt.canvas.dom.client.Context2d;
import com.google.gwt.canvas.dom.client.ImageData;
import com.google.gwt.event.dom.client.MouseDownEvent;
import com.google.gwt.event.dom.client.MouseDownHandler;
import com.google.gwt.event.dom.client.MouseMoveEvent;
import com.google.gwt.event.dom.client.MouseMoveHandler;
import com.google.gwt.event.dom.client.MouseOutEvent;
import com.google.gwt.event.dom.client.MouseOutHandler;
import com.google.gwt.event.dom.client.MouseUpEvent;
import com.google.gwt.event.dom.client.MouseUpHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.ui.Composite;

/**
 * Original code was taken from
 * http://www.subshell.com/en/subshell/blog/Implementing-a-Color-Picker-Dialog-
 * With-Canvas-and-GWT100.html
 *
 * @author
 * http://www.subshell.com/en/subshell/blog/Implementing-a-Color-Picker-Dialog-
 * With-Canvas-and-GWT100.html
 *
 * @author kot
 */
public class SaturationLightnessPicker extends Composite {
    private final Canvas canvas;
    private int hue = 180;
    private int handleX = 90;
    private int handleY = 90;
    private boolean mouseDown;

    private final DeferredTask updateDown = new DeferredTask() {
        @Override
        public void run() {
            mouseDown = false;
        }
    };

    public static SaturationLightnessPicker createIfSupported() {
        return Canvas.isSupported() ? new SaturationLightnessPicker() : null;
    }

    private SaturationLightnessPicker() {
        canvas = Canvas.createIfSupported();
        canvas.setCoordinateSpaceHeight(180);
        canvas.setCoordinateSpaceWidth(180);
        canvas.setPixelSize(180, 180);

        initWidget(canvas);

        canvas.addMouseDownHandler(new MouseDownHandler() {
            @Override
            public void onMouseDown(final MouseDownEvent event) {
                handleX = event.getRelativeX(canvas.getElement());
                handleY = event.getRelativeY(canvas.getElement());
                drawGradient(false);
                final String color = getColorAtPixel(handleX, handleY);
                drawGradient(true);
                fireColorChanged(color);

                mouseDown = true;
            }
        });
        canvas.addMouseMoveHandler(new MouseMoveHandler() {
            @Override
            public void onMouseMove(final MouseMoveEvent event) {
                if (mouseDown) {
                    updateDown.cancel();
                    handleX = event.getRelativeX(canvas.getElement());
                    handleY = event.getRelativeY(canvas.getElement());
                    drawGradient(false);
                    final String color = getColorAtPixel(handleX, handleY);
                    drawGradient(true);
                    fireColorChanged(color);
                }
            }
        });
        canvas.addMouseUpHandler(new MouseUpHandler() {
            @Override
            public void onMouseUp(final MouseUpEvent event) {
                mouseDown = false;
            }
        });
        canvas.addMouseOutHandler(new MouseOutHandler() {
            @Override
            public void onMouseOut(final MouseOutEvent event) {
                updateDown.defer(1500);
            }
        });
    }

    @Override
    protected void onAttach() {
        super.onAttach();
        drawGradient(true);
    }

    private void drawGradient(final boolean drawHandle) {
        final Context2d ctx = canvas.getContext2d();

        // draw gradient
        final Hsl hsl = new Hsl(hue, 0, 50);
        for (int x = 0; x <= 179; x++) {
            hsl.setS(Math.round(x * 100 / 179));
            final CanvasGradient grad = ctx.createLinearGradient(x, 0, x, 179);
            grad.addColorStop(0, "#000000"); //$NON-NLS-1$
            grad.addColorStop(0.5, Colors.hsl2html(hsl));
            grad.addColorStop(1, "#ffffff"); //$NON-NLS-1$
            ctx.setFillStyle(grad);
            ctx.fillRect(x, 0, 1, 180);
        }

        // draw handle
        if (drawHandle) {
            ctx.beginPath();
            ctx.arc(handleX, handleY, 4, 0, Math.PI * 2, false);
            ctx.closePath();
            ctx.setFillStyle("#ffffff"); //$NON-NLS-1$
            ctx.fill();

            ctx.beginPath();
            ctx.arc(handleX, handleY, 3, 0, Math.PI * 2, false);
            ctx.closePath();
            ctx.setFillStyle("#000000"); //$NON-NLS-1$
            ctx.fill();
        }
    }

    public HandlerRegistration addColorChangedHandler(final IColorChangedHandler handler) {
        return addHandler(handler, ColorChangedEvent.getType());
    }

    private void fireColorChanged(final String color) {
        fireEvent(new ColorChangedEvent(color));
    }

    private String getColorAtPixel(final int x, final int y) {
        final int fixedX = Math.max(Math.min(x, 179), 0);
        final int fixedy = Math.max(Math.min(y, 179), 0);
        final Context2d ctx = canvas.getContext2d();
        final ImageData imageData = ctx.getImageData(fixedX, fixedy, 1, 1);
        final CanvasPixelArray data = imageData.getData();
        return Colors.rgb2html(new Rgb(data.get(0), data.get(1), data.get(2)));
    }

    public void setHue(final int hue) {
        this.hue = hue;
        drawGradient(false);
        final String color = getColorAtPixel(handleX, handleY);
        drawGradient(true);
        fireColorChanged(color);
    }

    public String getColor() {
        drawGradient(false);
        final String color = getColorAtPixel(handleX, handleY);
        drawGradient(true);
        return color;
    }

    public void setColor(final String color) {
        final Rgb rgb = Colors.html2rgb(color);
        final Hsl hsl = Colors.rgb2hsl(rgb);
        hue = hsl.getH();
        handleX = (int) Math.min(Math.max(Math.round(hsl.getS() * 180d / 100d), 0), 179);
        handleY = (int) Math.min(Math.max(Math.round(hsl.getL() * 180d / 100d), 0), 179);
        drawGradient(true);
        fireColorChanged(color);
    }
}
