package org.jresearch.commons.gwt.client.widget.colorpicker;

import org.jresearch.commons.gwt.shared.tools.Colors;
import org.jresearch.commons.gwt.shared.tools.Hsl;
import org.jresearch.commons.gwt.shared.tools.Rgb;

import com.google.gwt.canvas.client.Canvas;
import com.google.gwt.canvas.dom.client.Context2d;
import com.google.gwt.editor.client.LeafValueEditor;
import com.google.gwt.uibinder.client.UiBinder;
import com.google.gwt.uibinder.client.UiField;
import com.google.gwt.uibinder.client.UiHandler;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.LazyPanel;
import com.google.gwt.user.client.ui.TextBox;
import com.google.gwt.user.client.ui.Widget;
import com.google.inject.Inject;

/**
 * Original code was taken from
 * http://www.subshell.com/en/subshell/blog/Implementing-a-Color-Picker-Dialog-
 * With-Canvas-and-GWT100.html
 *
 * @author http://www.subshell.com/en/subshell/blog/Implementing-a-Color-Picker-Dialog-
 *         With-Canvas-and-GWT100.html
 *
 * @author kot
 */
public class ColorPicker extends Composite implements LeafValueEditor<String> {

    //@formatter:off
    interface Binder extends UiBinder<Widget, ColorPicker> { /* nothing */ }
    //@formatter:on

    @UiField(provided = true)
    Canvas color;
    @UiField()
    TextBox textColor;
    @UiField
    SaturationLightnessPicker slPicker;
    @UiField
    HuePicker huePicker;

    @UiField
    LazyPanel modern;
    @UiField
    LazyPanel legacy;
    private final boolean canvasSupported;

    @Inject
    public ColorPicker(final Binder binder) {
        canvasSupported = Canvas.isSupported();
        color = Canvas.createIfSupported();
        initWidget(binder.createAndBindUi(this));
        modern.setVisible(canvasSupported);
        legacy.setVisible(!canvasSupported);
    }

    @UiHandler("huePicker")
    void onChangeHue(final HueChangedEvent event) {
        if (canvasSupported) {
            slPicker.setHue(event.getHue());
        }
    }

    @UiHandler("slPicker")
    void onChangeColor(final ColorChangedEvent event) {
        if (canvasSupported) {
            final Context2d ctx = color.getContext2d();
            ctx.setFillStyle(event.getColor());
            ctx.fillRect(0, 0, color.getCoordinateSpaceWidth(), color.getCoordinateSpaceHeight());
        } else {
            textColor.setValue(event.getColor());
        }
    }

    public void setColor(final String color) {
        if (canvasSupported) {
            final Context2d ctx = this.color.getContext2d();
            ctx.setFillStyle(color);
            ctx.fillRect(0, 0, this.color.getCoordinateSpaceWidth(), this.color.getCoordinateSpaceHeight());
            final Rgb rgb = Colors.html2rgb(color);
            final Hsl hsl = Colors.rgb2hsl(rgb);
            huePicker.setHue(hsl.getH());
            slPicker.setColor(color);
        } else {
            textColor.setValue(color);
        }
    }

    public String getColor() {
        return canvasSupported ? slPicker.getColor() : textColor.getValue();
    }

    @Override
    public void setValue(final String value) {
        setColor(value);
    }

    @Override
    public String getValue() {
        return getColor();
    }

}
