package org.jresearch.commons.gwt.app.client.widget;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;

import org.jresearch.commons.gwt.app.client.mvc.event.AfterLoginEvent;
import org.jresearch.commons.gwt.app.client.mvc.event.AfterLoginHandler;
import org.jresearch.commons.gwt.app.client.mvc.event.ResetPasswordEvent;
import org.jresearch.commons.gwt.app.client.mvc.event.SignUpEvent;
import org.jresearch.commons.gwt.app.client.resource.AppRs;
import org.jresearch.commons.gwt.client.mvc.AbstractCallback;
import org.jresearch.commons.gwt.flexess.client.service.flexess.IGwtAuthenticationData;
import org.jresearch.commons.gwt.flexess.client.service.flexess.PasswordAuthenticationData;
import org.jresearch.commons.gwt.flexess.client.service.flexess.SocialAuthenticationData;
import org.jresearch.commons.gwt.flexess.shared.model.AuthData;
import org.jresearch.commons.gwt.oauth2.client.model.OauthConfiguration;
import org.jresearch.commons.gwt.oauth2.client.mvc.event.SocialAuthenticationEvent;
import org.jresearch.commons.gwt.oauth2.client.mvc.event.SocialAuthenticationHandler;
import org.jresearch.commons.gwt.oauth2.client.widget.EnableAction;
import org.jresearch.commons.gwt.oauth2.client.widget.SocialAuthenticator;
import org.jresearch.commons.gwt.oauth2.shared.model.SocialNetwork;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gwt.core.client.Scheduler;
import com.google.inject.Inject;
import com.sencha.gxt.widget.core.client.Status;
import com.sencha.gxt.widget.core.client.button.ButtonBar;
import com.sencha.gxt.widget.core.client.button.TextButton;
import com.sencha.gxt.widget.core.client.event.SelectEvent;
import com.sencha.gxt.widget.core.client.event.SelectEvent.SelectHandler;
import com.sencha.gxt.widget.core.client.form.FieldLabel;
import com.sencha.gxt.widget.core.client.form.PasswordField;
import com.sencha.gxt.widget.core.client.form.TextField;

public class LoginDialog extends AbstractAuthDialog implements SocialAuthenticationHandler, AfterLoginHandler {

	@Inject protected SocialAuthenticator socialAuthenticator;

	protected TextField userName;
	protected PasswordField password;

	private Map<SocialNetwork, TextButton> socialNetworkControls;
	private SocialNetwork network;
	private String token;

	private ArrayList<FieldLabel> fields;

	private boolean resetEnable;
	private boolean registerEnable;

	@Inject
	protected LoginDialog() {
		super(AppRs.MSG.loginDialog());
	}

	@Override
	protected void createLinks(final ButtonBar linkBar) {
		reset = createEventLink(AppRs.MSG.resetPassword(), new ResetPasswordEvent());
		linkBar.add(reset);
		reset.disable();
		signUp = createEventLink(AppRs.MSG.register(), new SignUpEvent());
		linkBar.add(signUp);
		signUp.disable();
	}

	@Override
	protected void createSocialLoginButtons(final ButtonBar socialBar) {
		socialNetworkControls = Maps.newHashMap();
		for (final SocialNetwork net : EnumSet.allOf(SocialNetwork.class)) {
			final TextButton button = new TextButton(net.name());
			socialBar.add(button);
			button.hide();
			button.disable();
			socialNetworkControls.put(net, button);
		}
	}

	private void enableSocialNetworkControl(final SocialNetwork net) {
		final TextButton button = socialNetworkControls.get(net);
		if (button != null) {
			configureSocialLoginButton(net, button);
		}
	}

	protected void configureSocialLoginButton(final SocialNetwork net, final TextButton button) {
		socialAuthenticator.enableNetwork(net, new EnableAction() {
			@Override
			public void configureControl(final OauthConfiguration configuration) {
				button.addSelectHandler(new SelectHandler() {
					@Override
					public void onSelect(final SelectEvent event) {
						socialAuthenticator.authenticate(configuration);
					}
				});
				button.show();
				button.enable();
			}
		});
	}

	@Override
	protected void init() {
		enableSocialNetworkAuthentication();
		super.init();
	}

	private void enableSocialNetworkAuthentication() {
		for (final SocialNetwork net : EnumSet.allOf(SocialNetwork.class)) {
			enableSocialNetworkControl(net);
		}
		bus.addHandler(SocialAuthenticationEvent.TYPE, this);
		bus.addHandler(AfterLoginEvent.TYPE, this);
	}

	@Override
	public void onAuthentication(final SocialAuthenticationEvent event) {
		// Handle only is active
		if (isVisible()) {
			network = event.getNetwork();
			token = event.getToken();
			onSubmit();
		}
	}

	@Override
	protected List<FieldLabel> getFields() {
		if (fields == null) {
			fields = Lists.newArrayList();

			userName = new TextField();
			fields.add(new FieldLabel(userName, AppRs.MSG.username()));

			password = new PasswordField();
			// password.setMinLength(4);
			fields.add(new FieldLabel(password, AppRs.MSG.password()));
		}
		return fields;
	}

	@Override
	protected void reset() {
		network = null;
		token = null;
		super.reset();
	}

	@Override
	protected String getActionName() {
		return AppRs.MSG.loginAction();
	}

	@Override
	protected void doAction(final Status status) {
		final IGwtAuthenticationData<?> data = getAuthenticationData();
		service.authenticate(data, new AbstractCallback<AuthData>(bus) {
			@SuppressWarnings("incomplete-switch")
			@Override
			public void onSuccess(final AuthData result) {
				switch (result.getAuthStatus()) {
				case Authenticated:
					LoginDialog.this.hide();
					final String userId = result.getUserId();
					if (userId == null) {
						throw new IllegalStateException("Null user id after authentication"); //$NON-NLS-1$
					}
					bus.fire(new AfterLoginEvent(userId));
					if (command != null) {
						Scheduler.get().scheduleDeferred(command);
					}
					break;
				case NotEnroled:
					status.clearStatus(AppRs.MSG.notEnrolled());
					getButtonBar().enable();
					getSocialBar().enable();
					break;
				case BadCredentials:
					status.clearStatus(AppRs.MSG.badCredentials());
					getButtonBar().enable();
					getSocialBar().enable();
					break;
				case NotAuthorized:
					status.clearStatus(AppRs.MSG.notAuthorized());
					getButtonBar().enable();
					getSocialBar().enable();
					break;
				case Reset:
					status.clearStatus(AppRs.MSG.reset());
					getButtonBar().enable();
					getSocialBar().enable();
					break;
				}
			}
		});
	}

	private IGwtAuthenticationData<?> getAuthenticationData() {
		IGwtAuthenticationData<?> result = new PasswordAuthenticationData(userName.getValue(), password.getValue());
		if (token != null && network != null) {
			result = new SocialAuthenticationData(network, token);
			token = null;
		}
		return result;
	}

	@Override
	public void onAfterLogin(final AfterLoginEvent event) {
		hide();
	}

	@Override
	protected void resetFields() {
		userName.reset();
		password.reset();
	}

	/**
	 * @return the resetEnable
	 */
	public boolean isResetEnable() {
		return resetEnable;
	}

	/**
	 * @param resetEnable
	 *            the resetEnable to set
	 */
	public void setResetEnable(final boolean resetEnable) {
		this.resetEnable = resetEnable;
		reset.setEnabled(resetEnable);
		reset.setVisible(resetEnable);
	}

	/**
	 * @return the registerEnable
	 */
	public boolean isRegisterEnable() {
		return registerEnable;
	}

	/**
	 * @param registerEnable
	 *            the registerEnable to set
	 */
	public void setRegisterEnable(final boolean registerEnable) {
		this.registerEnable = registerEnable;
		signUp.setEnabled(resetEnable);
		signUp.setVisible(resetEnable);
	}

}
