/*
 * Decompiled with CFR 0.152.
 */
package org.jsqrl.server;

import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.HashSet;
import net.i2p.crypto.eddsa.EdDSAEngine;
import net.i2p.crypto.eddsa.EdDSAPublicKey;
import net.i2p.crypto.eddsa.spec.EdDSANamedCurveTable;
import net.i2p.crypto.eddsa.spec.EdDSAParameterSpec;
import net.i2p.crypto.eddsa.spec.EdDSAPublicKeySpec;
import org.jsqrl.config.SqrlConfig;
import org.jsqrl.error.SqrlException;
import org.jsqrl.model.SqrlAuthResponse;
import org.jsqrl.model.SqrlClientRequest;
import org.jsqrl.model.SqrlCommand;
import org.jsqrl.model.SqrlOptionFlag;
import org.jsqrl.model.TransactionInformationFlag;
import org.jsqrl.nut.SqrlNut;
import org.jsqrl.service.SqrlAuthenticationService;
import org.jsqrl.service.SqrlNutService;
import org.jsqrl.service.SqrlUserService;
import org.jsqrl.util.SqrlUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JSqrlServer {
    private static final Logger log = LoggerFactory.getLogger(JSqrlServer.class);
    private final SqrlUserService userService;
    private final SqrlAuthenticationService sqrlAuthenticationService;
    private final SqrlConfig config;
    private final SqrlNutService nutService;
    private final EdDSAParameterSpec edDsaSpec;

    public JSqrlServer(SqrlUserService userService, SqrlAuthenticationService sqrlAuthenticationService, SqrlConfig config, SqrlNutService nutService) {
        this.userService = userService;
        this.sqrlAuthenticationService = sqrlAuthenticationService;
        this.config = config;
        this.nutService = nutService;
        this.edDsaSpec = EdDSANamedCurveTable.getByName((String)"ed25519-sha-512");
    }

    public String createAuthenticationRequest(String ipAddress, Boolean qr) {
        SqrlNut nut = this.nutService.createNut(ipAddress, qr);
        String nutString = this.nutService.getNutString(nut);
        this.sqrlAuthenticationService.createAuthenticationRequest(nutString, ipAddress);
        log.debug("Creating nut {}", (Object)nutString);
        return nutString;
    }

    public Boolean checkAuthenticationStatus(String nut, String ipAddress) {
        return this.nutService.nutBelongsToIp(nut, ipAddress) != false && this.sqrlAuthenticationService.getAuthenticatedSqrlIdentityKey(nut, ipAddress) != null;
    }

    public SqrlAuthResponse handleClientRequest(SqrlClientRequest request, String nut, String ipAddress) {
        EdDSAEngine verifier;
        SqrlNut requestNut = this.nutService.createNutFromString(nut);
        log.debug("Handling client request for nut {}", (Object)nut);
        SqrlNut responseNut = this.nutService.createNut(ipAddress, requestNut.isQr());
        String responseNutString = this.nutService.getNutString(responseNut);
        String sukResponse = null;
        if (!request.getRequestVersion().equals(this.config.getSqrlVersion())) {
            return this.createResponse(responseNutString, null, TransactionInformationFlag.CLIENT_FAILURE);
        }
        try {
            verifier = new EdDSAEngine(MessageDigest.getInstance("SHA-512"));
            this.verifyIdSignature(request, (Signature)verifier);
            this.verifyPreviousIdSignature(request, (Signature)verifier);
        }
        catch (NoSuchAlgorithmException | SqrlException e) {
            log.debug("Unable to verify signature", (Throwable)e);
            return this.createResponse(responseNutString, null, TransactionInformationFlag.CLIENT_FAILURE);
        }
        String identityKey = request.getIdentityKey();
        String previousIdentityKey = request.getPreviousIdentityKey();
        HashSet<TransactionInformationFlag> tifs = new HashSet<TransactionInformationFlag>();
        Long nutAge = Duration.between(requestNut.getCreated(), LocalDateTime.now()).getSeconds();
        if (nutAge > this.config.getNutExpirationSeconds()) {
            tifs.add(TransactionInformationFlag.TRANSIENT_ERROR);
        } else {
            SqrlCommand command;
            this.sqrlAuthenticationService.linkNut(nut, responseNutString);
            if (requestNut.checkIpMatch(responseNut).booleanValue()) {
                tifs.add(TransactionInformationFlag.IP_MATCHED);
            }
            Object sqrlUser = this.userService.getUserBySqrlKey(identityKey);
            Boolean sqrlEnabled = true;
            if (sqrlUser != null) {
                tifs.add(TransactionInformationFlag.ID_MATCH);
            } else if (previousIdentityKey != null && (sqrlUser = this.userService.getUserBySqrlKey(previousIdentityKey)) != null) {
                this.userService.updateIdentityKey(previousIdentityKey, identityKey);
                tifs.add(TransactionInformationFlag.PREVIOUS_ID_MATCH);
            }
            if (sqrlUser != null && request.getOptionFlags().contains((Object)SqrlOptionFlag.SERVER_UNLOCK_KEY)) {
                sukResponse = sqrlUser.getServerUnlockKey();
            }
            if (sqrlUser != null && !sqrlUser.sqrlEnabled().booleanValue()) {
                sqrlEnabled = false;
                tifs.add(TransactionInformationFlag.SQRL_DISABLED);
            }
            if ((command = request.getCommand()) == null) {
                tifs.add(TransactionInformationFlag.FUNCTION_NOT_SUPPORTED);
            } else if (command != SqrlCommand.QUERY || !sqrlEnabled.booleanValue()) {
                if (command == SqrlCommand.IDENT && sqrlEnabled.booleanValue()) {
                    if (sqrlUser == null) {
                        this.userService.registerSqrlUser(identityKey, request.getServerUnlockKey(), request.getVerifyUnlockKey());
                    }
                    this.sqrlAuthenticationService.authenticateNut(responseNutString, identityKey);
                    tifs.add(TransactionInformationFlag.ID_MATCH);
                } else if (command == SqrlCommand.DISABLE && sqrlEnabled.booleanValue()) {
                    this.userService.disableSqrlUser(identityKey);
                } else if (command == SqrlCommand.REMOVE && sqrlEnabled.booleanValue()) {
                    if (sqrlUser != null) {
                        this.verifyUnlockRequestSignature(request, sqrlUser.getVerifyUnlockKey(), (Signature)verifier);
                        this.userService.removeSqrlUser(identityKey);
                    } else {
                        tifs.add(TransactionInformationFlag.CLIENT_FAILURE);
                    }
                } else if (command == SqrlCommand.ENABLE) {
                    if (sqrlUser != null) {
                        this.verifyUnlockRequestSignature(request, sqrlUser.getVerifyUnlockKey(), (Signature)verifier);
                        this.userService.enableSqrlUser(identityKey);
                    } else {
                        tifs.add(TransactionInformationFlag.CLIENT_FAILURE);
                    }
                }
            }
        }
        SqrlAuthResponse response = this.createResponse(responseNutString, sukResponse, tifs.toArray(new TransactionInformationFlag[tifs.size()]));
        log.debug("Response: {}", (Object)response);
        return response;
    }

    private SqrlAuthResponse createResponse(String nut, String suk, TransactionInformationFlag ... tifs) {
        return SqrlAuthResponse.builder().nut(nut).qry(this.config.getSqrlBaseUri() + "?nut=" + nut).addTifs(tifs).ver(this.config.getSqrlVersion()).suk(suk).build();
    }

    private void verifyIdSignature(SqrlClientRequest request, Signature verifier) {
        this.verifySqrlRequestSignature(request, verifier, SqrlUtil.base64UrlDecode(request.getIdentityKey()), request.getDecodedIdentitySignature(), "Unable to verify ID Signature");
    }

    private void verifyUnlockRequestSignature(SqrlClientRequest request, String verifyUnlockKey, Signature verifier) {
        this.verifySqrlRequestSignature(request, verifier, SqrlUtil.base64UrlDecode(verifyUnlockKey), request.getDecodedUnlockRequestSignature(), "Unable to verify Unlock Request Signature");
    }

    private void verifyPreviousIdSignature(SqrlClientRequest request, Signature verifier) {
        if (request.getPreviousIdentityKey() != null) {
            this.verifySqrlRequestSignature(request, verifier, SqrlUtil.base64UrlDecode(request.getPreviousIdentityKey()), request.getDecodedPreviousIdSignature(), "Unable to verify Previous ID Signature");
        }
    }

    private void verifySqrlRequestSignature(SqrlClientRequest request, Signature verifier, byte[] key, byte[] signature, String errorMessage) {
        byte[] requestMessage = (request.getClient() + request.getServer()).getBytes();
        try {
            if (!this.verifyEdDSASignature(verifier, key, requestMessage, signature).booleanValue()) {
                throw new SqrlException(errorMessage);
            }
        }
        catch (InvalidKeyException | SignatureException e) {
            throw new SqrlException("Unable to verify message signature", e);
        }
    }

    private Boolean verifyEdDSASignature(Signature verifier, byte[] key, byte[] message, byte[] signature) throws InvalidKeyException, SignatureException {
        verifier.initVerify((PublicKey)new EdDSAPublicKey(new EdDSAPublicKeySpec(key, this.edDsaSpec)));
        verifier.update(message);
        return verifier.verify(signature);
    }
}

