/**
 * The MIT License
 *
 * Copyright (c) 2007-2010, Sun Microsystems, Inc., Kohsuke Kawaguchi, Erik Ramfelt,
 *                          Henrik Lynggaard, Peter Liljenberg, Andrew Bayer, Vincent Latombe
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package hudson.plugins.clearcase.history;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;

/**
 * A filter that chains a collection of filters. It fill filter all elements that get filtered by any of underlying
 * filters
 * 
 * @author vlatombe
 */
public class FilterChain implements Filter {

    private final Collection<Filter> filters;

    public FilterChain(Collection<Filter> filters) {
        super();
        this.filters = (filters != null) ? filters : new ArrayList<Filter>();
    }

    @Override
    public boolean accept(HistoryEntry element) {
        for (Filter f : filters) {
            if (!f.accept(element)) {
                return false;
            }
        }
        return true;
    }

    @Override
    public boolean requiresMinorEvents() {
        for (Filter f : filters) {
            if (f.requiresMinorEvents()) {
                return true;
            }
        }
        return false;
    }

    public Collection<Filter> getFilters() {
        return Collections.unmodifiableCollection(filters);
    }

}
