package hudson.plugins.downstream_ext;

import hudson.model.AbstractBuild;
import hudson.model.AbstractProject;
import hudson.model.Action;
import hudson.model.Cause;
import hudson.model.Run;
import hudson.model.TaskListener;
import hudson.model.DependencyGraph.Dependency;
import hudson.util.LogTaskListener;

import java.io.PrintStream;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Defines a dependency introduced by the downstream-ext plugin.
 * 
 * @author kutzi
 */
public class DownstreamDependency extends Dependency {

	private static final Logger LOGGER = Logger.getLogger(DownstreamDependency.class.getName());
	
	private final DownstreamTrigger trigger;

	public DownstreamDependency(AbstractProject<?, ?> upstream, AbstractProject<?, ?> downstream,
			DownstreamTrigger trigger) {
		super(upstream, downstream);
		this.trigger = trigger;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	@SuppressWarnings("unchecked")
	public boolean shouldTriggerBuild(AbstractBuild build,
			TaskListener listener, List<Action> actions) {
		PrintStream logger = listener.getLogger();
		if(trigger.getStrategy().evaluate(trigger.getThreshold(), build.getResult())) {
            AbstractProject p = getDownstreamProject();
                
            if(trigger.isOnlyIfSCMChanges()) {
            	if (p.getScm().requiresWorkspaceForPolling()) {
            		// Downstream project locks workspace while building.
            		// If polled synchronously this could make the upstream build
            		// lock for a possibly long time.
            		// See HUDSON-5406
            		logger.println(Messages.DownstreamTrigger_StartedAsynchPoll(p.getName()));
            		Thread t = new Thread(new PollRunner(p, new Cause.UpstreamCause((Run<?,?>)build)));
            		t.start();
            		return false;
            	}
            	
            	if (p.pollSCMChanges(listener)) {
            		return true;
            	} else {
            		logger.println(Messages.DownstreamTrigger_NoSCMChanges(p.getName()));
            		return false;
            	}
            }
            return true;
		} else {
			logger.println(Messages.DownstreamTrigger_ConditionNotMet(trigger.getStrategy().getDisplayName(),
					trigger.getThreshold()));
			return false;
		}
	}

	// Technically it'd be safe to not override equals
	// since superclass implements it well.
	// But maybe that changes in the future.
	@Override
	public boolean equals(Object obj) {
		if (!(obj instanceof DownstreamDependency)) {
			return false;
		}
		
		// Currently, there can be only one downstream-ext dependency per project
		// If that'd change later we must check the trigger instance here, too.
		
		return super.equals(obj);
	}
	
	@SuppressWarnings("unchecked")
	private static class PollRunner implements Runnable {

		private final AbstractProject project;
		private final Cause cause;
		private final TaskListener taskListener;

		public PollRunner(AbstractProject p, Cause cause) {
			this.project = p;
			this.cause = cause;
			this.taskListener = new LogTaskListener(LOGGER, Level.INFO);
		}
		
		@Override
		public void run() {
			if(this.project.pollSCMChanges(this.taskListener)) {
				this.project.scheduleBuild(this.cause);
			}
		}
	}
}
