/*
 *	P4Java - java integration with Perforce SCM
 *	Copyright (C) 2007-,  Mike Wille, Tek42
 *
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Lesser General Public
 *	License as published by the Free Software Foundation; either
 *	version 2.1 of the License, or (at your option) any later version.
 *
 *	This library is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *	Lesser General Public License for more details.
 *
 *	You should have received a copy of the GNU Lesser General Public
 *	License along with this library; if not, write to the Free Software
 *	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *	You can contact the author at:
 *
 *	Web:	http://tek42.com
 *	Email:	mike@tek42.com
 *	Mail:	755 W Big Beaver Road
 *			Suite 1110
 *			Troy, MI 48084
 */

package com.tek42.perforce.parse;

import java.io.Writer;

import com.tek42.perforce.PerforceException;

/**
 * Interface for parsing perforce output into a concrete object and also for saving the object back to perforce. 
 * <p>
 * The pattern for using this template to build an object is:
 * <ol>
 * <li>Retrieve commands to execute via {@link #getBuildCmd(String)}</li>
 * <li>Execute those commands and capture the output.</li>
 * <li>Pass the output to {@link #build(StringBuilder)}</li>
 * </ol>
 * <p>And conversely for saving:
 * <ol>
 * <li>Retrieve commands to execute via {@link Builder#getSaveCmd(Object)}</li>
 * <li>Execute those commands and retrieve a Writer for I/O</li>
 * <li>Pass the writer and the object to save to {@link #save(Object, Writer)}</li>
 * </ol>
 * 
 * @author Mike Wille
 */
public interface Builder<T> {
	/**
	 * This should return the command line tokens to execute for retrieving an object from Perforce. For example:
	 * String[] { "p4", "user", "-o", "mwille" }
	 * 
	 * @param id
	 *            The ID or Name of the object we are working on. In the case of a changelist, the changelist number. In
	 *            the case of a user, the username.
	 * @return A 1D string array of tokens to execute.
	 */
	public String[] getBuildCmd(String p4exe, String id);

	/**
	 * This should assemble an object from a string value. Format of the string is dependent on the object we are
	 * building and is unspecified.
	 * 
	 * @param sb
	 *            The StringBuilder containing the string value.
	 * @return The resulting object
	 * @throws PerforceException
	 *             If the format is invalid or other errors occur.
	 */
	public T build(StringBuilder sb) throws PerforceException;

	/**
	 * The converse of {@link #getBuildCmd(String)} this should return the command line tokens to execute for saving an
	 * object to Perforce. For example: String[] { "p4", "user", "-i", "mwille" }
	 * <p>
	 * Note, although the object being saved is passed to this method, this method does not need to do anything with it.
	 * 
	 * @return A 1D string array of tokens to execute
	 * @param obj	The object that is being saved, useful if propert(ies) are needed for the save command to be generated.
	 */
	public String[] getSaveCmd(String p4exe, T obj);

	/**
	 * Tells the AbstractPerforceTemplate whether or not this builder will write data on Standard Input to the perforce
	 * command specified in getSaveCmd().  Currently, this only applies to saving as their is no writing required for
	 * building.
	 *
	 * @return	True if standard input should be opened and this builder's save() method called.  False otherwise.
	 */
	public boolean requiresStandardInput();
	
	/**
	 * The converse of {@link #build(StringBuilder)} this should take an object and disassemble it for writing to the
	 * Perforce server. The specification of what is written to the Writer is dependant on the object being saved.
	 * 
	 * @param obj
	 *            The object to be saved
	 * @param writer
	 *            The Writer to write the string representation to
	 * @throws PerforceException
	 *             If the object is invalid or there is an issue with writing
	 */
	public void save(T obj, Writer writer) throws PerforceException;

}
