package dk.hlyh.ciplugins.projecthealth;

import hudson.model.AbstractProject;
import hudson.model.Action;
import hudson.model.Hudson;
import hudson.model.Result;
import hudson.model.Run;
import hudson.tasks.junit.CaseResult;
import hudson.tasks.test.AbstractTestResultAction;
import hudson.util.RunList;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import org.kohsuke.stapler.Stapler;

/**
 * Shows statistics about project health and which test cases fail the most.
 * 
 * @author Henrik Lynggaard Hansen <henrik@hlyh.dk>
 */
public final class ProjectHealthProjectAction implements Action {

    private static final Logger LOG = Logger.getLogger(ProjectHealthProjectAction.class.getName());
    
    private final AbstractProject<?, ?> project;
    private Hudson hudson;
    private volatile List<TestFailure> failures;
    private volatile List<Run<?, ?>> builds;
    private volatile Run<?, ?> firstbuild;
    private volatile Run<?, ?> lastbuild;
    private volatile String currentSelection;

    public ProjectHealthProjectAction(AbstractProject<?, ?> project) {
        super();
        this.project = project;
        this.hudson = Hudson.getInstance();
    }

    public String getIconFileName() {
        return "graph.gif";
    }

    public String getDisplayName() {
        return "Project Health";
    }

    public String getUrlName() {
        return "project-health";
    }

    public final AbstractProject<?, ?> getProject() {
        return project;
    }
    
    
    /**
     * The a list of all known builds for building the dropdown lists.
     * Note: This method also does caching of the builds relevant for this report.
     * It is done here because it is the firt method called by the page. 
     * @return A list of all the builds
     */
    public RunList<?> getAllBuilds() {
        RunList<?> allBuilds = project.getBuilds();

        // note this method will also cache all relevant information
        
        String optionParam = Stapler.getCurrentRequest().getParameter("option");
        String numOfBuildsParam = Stapler.getCurrentRequest().getParameter("num");
        String firstBuildParam = Stapler.getCurrentRequest().getParameter("firstBuild");
        String oldestBuildParam = Stapler.getCurrentRequest().getParameter("oldestBuild");

        LOG.fine("Project health report called with: [option=" + optionParam + ", numOfBuilds=" + numOfBuildsParam+", firstBuild="+firstBuildParam + ", oldestBuikd="+ oldestBuildParam+"]");
        
        if (optionParam != null && optionParam.equalsIgnoreCase("recent")) {
            try {
                int limit = Integer.parseInt(numOfBuildsParam);
                builds = (List<Run<?, ?>>) allBuilds.subList(0, limit);
            } catch (NumberFormatException e) {
                builds = (List<Run<?, ?>>) allBuilds;

            }
        } else if (optionParam != null && optionParam.equalsIgnoreCase("period")) {
            try {
                // +/- one to make the search inclusive
                long first = Long.parseLong(firstBuildParam);
                long oldest = Long.parseLong(oldestBuildParam);
                // if the user has swaped the builds, swap them back
                if (oldest > first) {
                    long tmp = oldest;
                    oldest = first;
                    first = tmp;
                }
                //make search inclusive
                first +=1;
                oldest -= 1;
                builds = (List<Run<?, ?>>) allBuilds.byTimestamp(oldest, first);
            } catch (NumberFormatException e) {
                builds = (List<Run<?, ?>>) allBuilds;
            }
        } else {
            // last resort all builds
            builds = (List<Run<?, ?>>) allBuilds;
        }
        firstbuild = builds.get(0);
        lastbuild = builds.get(builds.size() - 1);

        return allBuilds;
    }

    /**
     * Get the number of builds included in the report
     * @return The number of builds
     */
    public int getNumberOfBuilds() {
        return builds.size();
    }

    /**
     * Get the number of failed builds in the report. A build is considered failed if
     * the result is worse than SUCCESS i.e unstable builds are considered failed.
     * @return The number of failed builds.
     */
    public int getNumberOfFailedBuilds() {
        Map<String, TestFailure> resultMap = new HashMap<String, TestFailure>();
        int numberOfFailed = 0;

        for (Run<?, ?> build : builds) {
            if (!build.isBuilding()) {
                if (build.getResult().isWorseThan(Result.SUCCESS)) {
                    numberOfFailed++;
                    getFailuresForRun(resultMap, build);
                }
            }
        }
        failures = new ArrayList<TestFailure>(resultMap.values());
        Collections.sort(failures);
        return numberOfFailed;
    }

    public int getNumberOfSuccessBuilds() {
        return builds.size() - getNumberOfFailedBuilds();
    }

    public boolean getHasFailures() {
        return getFailures().size() > 0;
    }

    public String getCurrentSelection() {
        return currentSelection;
    }

    public Run<?, ?> getFirstbuild() {
        return firstbuild;
    }

    public Run<?, ?> getLastbuild() {
        return lastbuild;
    }
    

    public String getOverviewPie() {
        int totalBuilds = getNumberOfSuccessBuilds() + getNumberOfFailedBuilds();
        int successPct = (getNumberOfSuccessBuilds() * 100) / totalBuilds;
        int failedPct = (getNumberOfFailedBuilds() * 100) / totalBuilds;

        String result = "https://chart.googleapis.com/chart?cht=p&chs=250x100";
        result += "&chd=t:" + successPct + "," + failedPct;
        result += "&chl=Success|Failure";
        result += "&chco=729FCF,EF2929";
        result += "&chtt=Project%20health";
        return result;
    }

    public String getFailurePie() {
        String result = "https://chart.googleapis.com/chart?cht=p&chs=850x300";
        String legends = null;
        String datas = null;
        int totalFailures = 0;
        for (TestFailure failure : getFailures()) {
            totalFailures += failure.getCount();
        }

        for (TestFailure failure : getFailures()) {
            int failurePct = (failure.getCount() * 100) / totalFailures;
            String shortName = failure.getTestcase().substring(failure.getTestcase().lastIndexOf(".") + 1);
            if (legends == null) {
                legends = shortName;
            } else {
                legends += "|" + shortName;
            }
            if (datas == null) {

                datas = "" + failurePct;
            } else {
                datas += "," + failurePct;
            }

        }
        result += "&chd=t:" + datas;
        result += "&chl=" + legends;
        result += "&chtt=Test+Failures";
        result += "&chco=EF2929";
        return result;
    }

    public List<TestFailure> getFailures() {
        return failures;
    }

    private void getFailuresForRun(Map<String, TestFailure> failures, Run<?, ?> build) {
        AbstractTestResultAction<?> tests = build.getAction(AbstractTestResultAction.class);
        if (tests == null) {
            return;
        }

        List<CaseResult> failedTests = tests.getFailedTests();
        for (CaseResult result : failedTests) {
            String name = result.getFullName();
            TestFailure failure = failures.get(name);
            if (failure == null) {
                failure = new TestFailure();
                failure.setTestcase(name);
                failure.setCount(0);
                failures.put(name, failure);
            }
            failure.setCount(failure.getCount() + 1);
        }
    }
}
