package dk.hlyh.ciplugins.projecthealth;

import hudson.model.AbstractProject;
import hudson.model.Action;
import hudson.model.Hudson;
import hudson.model.Result;
import hudson.model.Run;
import hudson.tasks.junit.CaseResult;
import hudson.tasks.test.AbstractTestResultAction;
import hudson.util.RunList;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import org.kohsuke.stapler.Stapler;

/**
 * Shows statistics about project health and which test cases fail the most.
 * 
 * @author Henrik Lynggaard Hansen <henrik@hlyh.dk>
 */
public final class ProjectHealthProjectAction implements Action {

    private static final Logger LOG = Logger.getLogger(ProjectHealthProjectAction.class.getName());
    // instance members
    private final AbstractProject<?, ?> project;
    private final Hudson hudson;
    // cached
    private transient volatile List<Run<?, ?>> builds;
    private transient volatile long successCount;
    private transient volatile long failureCount;
    private transient volatile long totalTestFailures;
    private transient volatile List<TestFailure> failures;
    private transient volatile Run<?, ?> firstbuild;
    private transient volatile Run<?, ?> lastbuild;

    public ProjectHealthProjectAction(AbstractProject<?, ?> project) {
        super();
        this.project = project;
        this.hudson = Hudson.getInstance();
    }

    public String getIconFileName() {
        return "graph.gif";
    }

    public String getDisplayName() {
        return "Project Health";
    }

    public String getUrlName() {
        return "project-health";
    }

    public final AbstractProject<?, ?> getProject() {
        return project;
    }

    /**
     * The a list of all known builds for building the dropdown lists.
     * Note: This method also does caching of the builds relevant for this report.
     * It is done here because it is the firt method called by the page. 
     * @return A list of all the builds
     */
    public RunList<?> getAllBuilds() {
        RunList<?> allBuilds = project.getBuilds();

        // init caches
        resetCaches();
        setBuildLimit(allBuilds);
        calculateTestFailures();

        return allBuilds;
    }

    /**
     * Get the number of builds included in the report
     * @return The number of builds
     */
    public int getNumberOfBuilds() {
        return builds.size();
    }

    /**
     * Get the number of failed builds in the report. A build is considered failed if
     * the result is worse than SUCCESS i.e unstable builds are considered failed.
     * @return The number of failed builds.
     */
    public long getNumberOfFailedBuilds() {
        return failureCount;
    }

    public long getNumberOfSuccessBuilds() {
        return successCount;
    }

    public boolean getHasFailures() {
        return totalTestFailures > 0;
    }

    public Run<?, ?> getFirstbuild() {
        return firstbuild;
    }

    public Run<?, ?> getLastbuild() {
        return lastbuild;
    }

    public String getOverviewPie() {
        long successPct = (getNumberOfSuccessBuilds() * 100) / getNumberOfBuilds();
        long failedPct = (getNumberOfFailedBuilds() * 100) / getNumberOfBuilds();

        String result = "https://chart.googleapis.com/chart?cht=p&chs=250x100";
        result += "&chd=t:" + successPct + "," + failedPct;
        result += "&chl=Success|Failure";
        result += "&chco=729FCF,EF2929";
        result += "&chtt=Project%20health";
        return result;
    }

    public String getFailurePie() {
        StringBuilder result = new StringBuilder(1024);
        StringBuilder legends = new StringBuilder(1024);
        StringBuilder datas = new StringBuilder(1024);
        

        boolean first = true;
        for (TestFailure failure : getFailures()) {
            long failurePct = (failure.getCount() * 100) / totalTestFailures;
            String shortName = failure.getTestcase().substring(failure.getTestcase().lastIndexOf(".") + 1);
            if (first) {
                legends.append(shortName);
                datas.append(failurePct);
            } else {
                legends.append("|").append(shortName);
                datas.append(",").append(failurePct);
            }
            first=false;
        }
        result.append("https://chart.googleapis.com/chart?cht=p&chs=850x300");
        result.append("&chd=t:").append(datas);
        result.append("&chl=").append(legends);
        result.append("&chtt=Test+Failures&chco=EF2929");
        return result.toString();
    }

    public List<TestFailure> getFailures() {
        return failures;
    }

    private void resetCaches() {
        builds = null;
        successCount = 0;
        failureCount = 0;
        failures = null;
        firstbuild = null;
        lastbuild = null;
    }

    private void setBuildLimit(RunList<?> allBuilds) {
        // note this method will also cache all relevant information

        String optionParam = Stapler.getCurrentRequest().getParameter("option");
        String numOfBuildsParam = Stapler.getCurrentRequest().getParameter("num");
        String firstBuildParam = Stapler.getCurrentRequest().getParameter("firstBuild");
        String oldestBuildParam = Stapler.getCurrentRequest().getParameter("oldestBuild");

        LOG.fine("Project health report called with: [option=" + optionParam + ", numOfBuilds=" + numOfBuildsParam + ", firstBuild=" + firstBuildParam + ", oldestBuikd=" + oldestBuildParam + "]");

        if (optionParam != null && optionParam.equalsIgnoreCase("recent")) {
            try {
                int limit = Integer.parseInt(numOfBuildsParam);
                builds = (List<Run<?, ?>>) allBuilds.subList(0, limit);
            } catch (NumberFormatException e) {
                builds = (List<Run<?, ?>>) allBuilds;

            }
        } else if (optionParam != null && optionParam.equalsIgnoreCase("period")) {
            try {
                // +/- one to make the search inclusive
                long first = Long.parseLong(firstBuildParam);
                long oldest = Long.parseLong(oldestBuildParam);
                // if the user has swaped the builds, swap them back
                if (oldest > first) {
                    long tmp = oldest;
                    oldest = first;
                    first = tmp;
                }
                //make search inclusive
                first += 1;
                oldest -= 1;
                builds = (List<Run<?, ?>>) allBuilds.byTimestamp(oldest, first);
            } catch (NumberFormatException e) {
                builds = (List<Run<?, ?>>) allBuilds;
            }
        } else {
            // last resort all builds
            builds = (List<Run<?, ?>>) allBuilds;
        }
        firstbuild = builds.get(0);
        lastbuild = builds.get(builds.size() - 1);
    }

    private void calculateTestFailures() {
        Map<String, TestFailure> resultMap = new HashMap<String, TestFailure>();

        for (Run<?, ?> build : builds) {
            if (!build.isBuilding()) {
                if (build.getResult().isWorseThan(Result.SUCCESS)) {
                    failureCount++;
                    getFailuresForRun(resultMap, build);
                } else {
                    successCount++;
                }
            }
        }
        failures = new ArrayList<TestFailure>(resultMap.values());
        Collections.sort(failures);
    }

    private void getFailuresForRun(Map<String, TestFailure> failures, Run<?, ?> build) {
        AbstractTestResultAction<?> tests = build.getAction(AbstractTestResultAction.class);
        if (tests == null) {
            return;
        }

        List<CaseResult> failedTests = tests.getFailedTests();
        for (CaseResult result : failedTests) {
            String name = result.getFullName();
            TestFailure failure = failures.get(name);
            if (failure == null) {
                failure = new TestFailure();
                failure.setTestcase(name);
                failure.setCount(0);
                failures.put(name, failure);
            }
            failure.setCount(failure.getCount() + 1);
            totalTestFailures += 1;
        }
    }
}
