/*
* Copyright 2008-2010 Nokia Siemens Networks Oyj
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/
package hudson.plugins.robot;

import hudson.FilePath;
import hudson.model.AbstractBuild;
import hudson.model.DirectoryBrowserSupport;
import hudson.plugins.robot.graph.RobotGraph;
import hudson.plugins.robot.helper.NumberHelper;
import hudson.plugins.robot.model.RobotResult;
import hudson.plugins.robot.model.RobotResultStatistics;
import hudson.util.ChartUtil;
import hudson.util.Graph;
import hudson.util.ChartUtil.NumberOnlyBuildLabel;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.TreeSet;

import javax.servlet.ServletException;

import org.apache.commons.lang.StringUtils;
import org.jfree.data.category.CategoryDataset;
import org.jfree.data.category.DefaultCategoryDataset;
import org.kohsuke.stapler.StaplerRequest;
import org.kohsuke.stapler.StaplerResponse;

public class RobotBuildAction extends AbstractRobotAction {

	private static final String GRAPH_YLABEL = "Number of testcases";
	private static final String GRAPH_XLABEL = "Build number";
	private AbstractBuild<?, ?> build;
	private RobotResult result;
	private String reportFileName;
	private String outputPath;


	public RobotBuildAction(AbstractBuild<?, ?> build, RobotResult result,
			String outputPath, String reportFileName) {
		this.build = build;
		this.result = result;
		this.outputPath = outputPath;
		this.reportFileName = reportFileName;
	}

	public AbstractBuild<?, ?> getBuild() {
		return build;
	}

	public double getPassPercentage(){
		double percentage = (double) result.getOverallStats().get(1).getPass() / result.getOverallStats().get(1).getTotal() * 100;
		return NumberHelper.roundToDecimals(percentage, 1);
	}
	
	public double getCriticalPassPercentage() {
		double percentage = (double) result.getOverallStats().get(0).getPass() / result.getOverallStats().get(0).getTotal() * 100;
		return NumberHelper.roundToDecimals(percentage, 1);
	}
	
	public long getAllPassed() {
		return result.getOverallStats().get(1).getPass();
	}

	public long getAllFailed() {
		return result.getOverallStats().get(1).getFail();
	}

	public long getAllTotal() {
		return getAllPassed() + getAllFailed();
	}

	public long getCriticalFailed() {
		return result.getOverallStats().get(0).getFail();
	}

	public long getCriticalPassed() {
		return result.getOverallStats().get(0).getPass();
	}

	public long getCriticalTotal() {
		return getCriticalPassed() + getCriticalFailed();
	}


	public List<RobotResultStatistics> getOverallStats() {
		return result.getOverallStats();
	}

	public RobotResult getResult() {
		return result;
	}

	public void setResult(RobotResult result) {
		this.result = result;
	}

	public void setBuild(AbstractBuild<?, ?> build) {
		this.build = build;
	}
	
	public String getReportFileName(){
		//Check for empty name for backwards compatibility
		return StringUtils.isBlank(reportFileName) ? RobotPublisher.DEFAULT_REPORT_FILE : reportFileName;
	}


	public void doGraph(StaplerRequest req, StaplerResponse rsp)
			throws IOException {
		if (ChartUtil.awtProblemCause != null) {
			rsp.sendRedirect2(req.getContextPath() + "/images/headless.png");
			return;
		}

		Calendar t = build.getTimestamp();

		if (req.checkIfModified(t, rsp))
			return;
		
		Graph g = new RobotGraph(getBuild(), getDataSet(), GRAPH_YLABEL,
				GRAPH_XLABEL, 500, 200);
		g.doPng(req, rsp);
	}

	public void doDynamic(StaplerRequest req, StaplerResponse rsp)
			throws IOException, ServletException, InterruptedException {
		String indexFile = getReportFileName();
		FilePath robotDir = getRobotDir();
		
		if(!new FilePath(robotDir, indexFile).exists()){
			rsp.sendRedirect2("notfound");
			return;
		}
		
		DirectoryBrowserSupport dbs = new DirectoryBrowserSupport(this,
				getRobotDir(), getDisplayName(),
				"folder.gif", false);
		dbs.setIndexFileName(indexFile);
		dbs.generateResponse(req, rsp, this);
	}

	private CategoryDataset getDataSet() {
		List<Number> values = new ArrayList<Number>();
		List<String> rows = new ArrayList<String>();
		List<NumberOnlyBuildLabel> columns = new ArrayList<NumberOnlyBuildLabel>();

		for (AbstractBuild<?, ?> build = this.build; build != null; build = build
				.getPreviousBuild()) {
			RobotBuildAction action = build.getAction(this.getClass());

			Number failed = 0, passed = 0;
			if (action != null) {
				failed = action.getAllFailed();
				passed = action.getAllPassed();
			}

			// default 'zero value' must be set over zero to circumvent
			// JFreeChart stacked area rendering problem with zero values
			if (failed.intValue() < 1)
				failed = 0.01f;
			if (passed.intValue() < 1)
				passed = 0.01f;

			ChartUtil.NumberOnlyBuildLabel label = new ChartUtil.NumberOnlyBuildLabel(
					build);

			values.add(passed);
			rows.add("Passed");
			columns.add(label);

			values.add(failed);
			rows.add("Failed");
			columns.add(label);
		}

		// Code from DataSetBuilder, reversed row order for passed tests to go
		// first into dataset for nicer order when rendered in chart
		DefaultCategoryDataset dataset = new DefaultCategoryDataset();

		TreeSet<String> rowSet = new TreeSet<String>(rows);
		TreeSet<ChartUtil.NumberOnlyBuildLabel> colSet = new TreeSet<ChartUtil.NumberOnlyBuildLabel>(
				columns);

		Comparable[] _rows = rowSet.toArray(new Comparable[rowSet.size()]);
		Comparable[] _cols = colSet.toArray(new Comparable[colSet.size()]);

		// insert rows and columns in the right order, reverse rows
		for (int i = _rows.length - 1; i >= 0; i--)
			dataset.setValue(null, _rows[i], _cols[0]);
		for (Comparable c : _cols)
			dataset.setValue(null, _rows[0], c);

		for (int i = 0; i < values.size(); i++)
			dataset.addValue(values.get(i), rows.get(i), columns.get(i));
		return dataset;
	}
	
	protected FilePath getRobotDir() {
		FilePath rootDir = new FilePath(build.getRootDir());
		if (StringUtils.isNotBlank(outputPath))
			return new FilePath(rootDir, outputPath);
        return rootDir;
	}
}