/*
* Copyright 2008-2010 Nokia Siemens Networks Oyj
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/
package hudson.plugins.robot;

import hudson.model.AbstractBuild;
import hudson.model.AbstractProject;
import hudson.plugins.robot.graph.RobotGraph;
import hudson.util.ChartUtil;
import hudson.util.Graph;
import hudson.util.ChartUtil.NumberOnlyBuildLabel;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.TreeSet;

import org.jfree.data.category.CategoryDataset;
import org.jfree.data.category.DefaultCategoryDataset;
import org.kohsuke.stapler.StaplerRequest;
import org.kohsuke.stapler.StaplerResponse;

public class RobotProjectAction  extends AbstractRobotAction {

	private AbstractProject<?, ?> project;

	public RobotProjectAction(AbstractProject<?, ?> project) {
		this.project = project;
	}
	
	public AbstractProject<?, ?> getProject() {
		return project;
	}

	public final boolean isDisplayGraph() {
		if (project.getBuilds().size() > 0) 
			return true;
		
		return false;
	}
	
	public RobotBuildAction getLastBuildAction(){
		return getLastFinishedBuild().getAction(RobotBuildAction.class);
	}

	public AbstractBuild<?, ?> getLastFinishedBuild() {
		AbstractBuild<?, ?> lastBuild = (AbstractBuild<?, ?>) project
				.getLastBuild();
		while (lastBuild != null
				&& (lastBuild.isBuilding() || lastBuild
						.getAction(RobotBuildAction.class) == null)) {
			lastBuild = lastBuild.getPreviousBuild();
		}
		return lastBuild;
	}

	public Integer getLastResultBuildNumber() {
		AbstractBuild<?, ?> lastBuild = getLastFinishedBuild();
		if (lastBuild != null)
			return lastBuild.getNumber();
		return null;
	}

	public void doGraph(StaplerRequest req, StaplerResponse rsp)
			throws IOException {
		if (ChartUtil.awtProblemCause != null) {
			rsp.sendRedirect2(req.getContextPath() + "/images/headless.png");
			return;
		}

		Calendar t = project.getLastCompletedBuild().getTimestamp();

		if (req.checkIfModified(t, rsp))
			return;
		
		Graph g = new RobotGraph(project.getLastBuild(), getDataSet(), "Number of testcases",
				"Build number", 500, 200);
		g.doPng(req, rsp);
	}

	public void doIndex(StaplerRequest req, StaplerResponse rsp)
			throws IOException {
		Integer buildNumber = getLastResultBuildNumber();
		if (buildNumber == null) {
			rsp.sendRedirect2("nodata");
		} else {
			rsp.sendRedirect2("../" + buildNumber + "/" + getUrlName());
		}
	}
	
	private CategoryDataset getDataSet() {
		List<Number> values = new ArrayList<Number>();
		List<String> rows = new ArrayList<String>();
		List<NumberOnlyBuildLabel> columns = new ArrayList<NumberOnlyBuildLabel>();

		for (AbstractBuild<?, ?> build = project.getLastBuild(); build != null; build = build
				.getPreviousBuild()) {
			RobotBuildAction action = build.getAction(RobotBuildAction.class);

			Number failed = 0, passed = 0;
			if (action != null) {
				failed = action.getAllFailed();
				passed = action.getAllPassed();
			}

			// default 'zero value' must be set over zero to circumvent
			// JFreeChart stacked area rendering problem with zero values
			if (failed.intValue() < 1)
				failed = 0.01f;
			if (passed.intValue() < 1)
				passed = 0.01f;

			ChartUtil.NumberOnlyBuildLabel label = new ChartUtil.NumberOnlyBuildLabel(
					build);

			values.add(passed);
			rows.add("Passed");
			columns.add(label);

			values.add(failed);
			rows.add("Failed");
			columns.add(label);
		}

		// Code from DataSetBuilder, reversed row order for passed tests to go
		// first into dataset for nicer order when rendered in chart
		DefaultCategoryDataset dataset = new DefaultCategoryDataset();

		TreeSet<String> rowSet = new TreeSet<String>(rows);
		TreeSet<ChartUtil.NumberOnlyBuildLabel> colSet = new TreeSet<ChartUtil.NumberOnlyBuildLabel>(
				columns);

		Comparable[] _rows = rowSet.toArray(new Comparable[rowSet.size()]);
		Comparable[] _cols = colSet.toArray(new Comparable[colSet.size()]);

		// insert rows and columns in the right order, reverse rows
		for (int i = _rows.length - 1; i >= 0; i--)
			dataset.setValue(null, _rows[i], _cols[0]);
		for (Comparable c : _cols)
			dataset.setValue(null, _rows[0], c);

		for (int i = 0; i < values.size(); i++)
			dataset.addValue(values.get(i), rows.get(i), columns.get(i));
		return dataset;
	}
}
